// Copyright (c) 2025 Valentin Lobstein (Chocapikk) <balgogan@protonmail.com>
//
// Permission is hereby granted, free of charge, to any person obtaining a copy of
// this software and associated documentation files (the "Software"), to deal in
// the Software without restriction, including without limitation the rights to
// use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
// the Software, and to permit persons to whom the Software is furnished to do so,
// subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS
// FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
// COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
// IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
// CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

package vulnerability

import (
	"encoding/json"
	"fmt"
	"os"
	"strings"
	"sync"

	"github.com/Chocapikk/wpprobe/internal/file"
	"github.com/Chocapikk/wpprobe/internal/logger"
	versionpkg "github.com/Chocapikk/wpprobe/internal/version"
)

// Global cache for vulnerability data - loaded once and shared across all scans
var (
	globalWordfenceCache     []Vulnerability
	globalWordfenceCacheOnce sync.Once
	globalWordfenceCacheErr  error

	globalWPScanCache     []Vulnerability
	globalWPScanCacheOnce sync.Once
	globalWPScanCacheErr  error

	// Combined cache for all vulnerabilities (wordfence + wpscan)
	globalAllCache     []Vulnerability
	globalAllCacheOnce sync.Once
)

// DetermineSeverity determines severity by trying CVSS score first, then title fallback.
func DetermineSeverity(cvssScore float64, title string) string {
	if s := DetermineSeverityFromCVSS(cvssScore); s != "unknown" {
		return s
	}
	return DetermineSeverityFromTitle(title)
}

// DetermineAuthType determines auth type by trying CVSS vector first, then title fallback.
func DetermineAuthType(cvssVector, title string) string {
	if a := DetermineAuthTypeFromCVSS(cvssVector); a != "" {
		return a
	}
	return DetermineAuthTypeFromTitle(title)
}

// GetVulnerabilitiesForPlugin returns vulnerabilities for a specific plugin and version.
func GetVulnerabilitiesForPlugin(
	vulnerabilities []Vulnerability,
	plugin string,
	version string,
) []Vulnerability {
	result := []Vulnerability{}

	for _, vuln := range vulnerabilities {
		if vuln.Slug != plugin {
			continue
		}

		// Skip if no CVE and no title
		if vuln.CVE == "" && vuln.Title == "" {
			continue
		}

		// If version is unknown or empty, include all vulnerabilities for this plugin
		if version == "" || version == "unknown" {
			result = append(result, vuln)
			continue
		}

		// Filter by version
		if versionpkg.IsVersionVulnerable(version, vuln.FromVersion, vuln.ToVersion) {
			result = append(result, vuln)
		}
	}

	return result
}

// SaveVulnerabilitiesToFile saves vulnerabilities to a JSON file.
func SaveVulnerabilitiesToFile(
	vulnerabilities []Vulnerability,
	filename string,
	sourceName string,
) error {
	outputPath, err := file.GetStoragePath(filename)
	if err != nil {
		logger.DefaultLogger.Error("Error getting storage path: " + err.Error())
		return err
	}

	file, err := os.Create(outputPath)
	if err != nil {
		logger.DefaultLogger.Error("Error saving file: " + err.Error())
		return err
	}
	defer func() { _ = file.Close() }()

	encoder := json.NewEncoder(file)
	encoder.SetIndent("", "  ")

	if err := encoder.Encode(vulnerabilities); err != nil {
		logger.DefaultLogger.Error("Error encoding JSON: " + err.Error())
		return err
	}

	logger.DefaultLogger.Success(fmt.Sprintf("%s data saved in %s", sourceName, outputPath))
	return nil
}

// DetermineSeverityFromCVSS determines severity from CVSS score.
func DetermineSeverityFromCVSS(cvssScore float64) string {
	if cvssScore >= 9.0 {
		return "critical"
	}
	if cvssScore >= 7.0 {
		return "high"
	}
	if cvssScore >= 4.0 {
		return "medium"
	}
	if cvssScore > 0 {
		return "low"
	}
	return "unknown"
}

// DetermineSeverityFromTitle determines severity from title if CVSS is not available.
func DetermineSeverityFromTitle(title string) string {
	lowerTitle := strings.ToLower(title)
	if strings.Contains(lowerTitle, "critical") {
		return "critical"
	}
	if strings.Contains(lowerTitle, "high") {
		return "high"
	}
	if strings.Contains(lowerTitle, "medium") {
		return "medium"
	}
	if strings.Contains(lowerTitle, "low") {
		return "low"
	}
	return "unknown"
}

// DetermineAuthTypeFromCVSS determines authentication type from CVSS vector.
func DetermineAuthTypeFromCVSS(cvssVector string) string {
	if cvssVector == "" {
		return ""
	}
	if strings.Contains(cvssVector, "PR:N") {
		return "Unauth"
	}
	if strings.Contains(cvssVector, "PR:L") {
		return "Auth"
	}
	if strings.Contains(cvssVector, "PR:H") {
		return "Privileged"
	}
	return ""
}

// DetermineAuthTypeFromTitle determines authentication type from title if CVSS is not available.
func DetermineAuthTypeFromTitle(title string) string {
	lowerTitle := strings.ToLower(title)
	if strings.Contains(lowerTitle, "unauth") {
		return "Unauth"
	}
	if strings.Contains(lowerTitle, "auth") {
		return "Auth"
	}
	return "Unknown"
}

// BuildCVELink builds a CVE link from a CVE identifier.
func BuildCVELink(cve string) string {
	if cve == "" {
		return ""
	}
	return fmt.Sprintf("https://www.cve.org/CVERecord?id=%s", cve)
}

// LoadWordfenceVulnerabilities loads Wordfence vulnerabilities from file.
// Uses a global cache - data is loaded only once and shared across all scans.
func LoadWordfenceVulnerabilities() ([]Vulnerability, error) {
	globalWordfenceCacheOnce.Do(func() {
		globalWordfenceCache, globalWordfenceCacheErr = loadVulnerabilitiesFromFile("wordfence_vulnerabilities.json", "Wordfence")
	})
	return globalWordfenceCache, globalWordfenceCacheErr
}

// LoadWPScanVulnerabilities loads WPScan vulnerabilities from file.
// Uses a global cache - data is loaded only once and shared across all scans.
func LoadWPScanVulnerabilities() ([]Vulnerability, error) {
	globalWPScanCacheOnce.Do(func() {
		globalWPScanCache, globalWPScanCacheErr = loadVulnerabilitiesFromFile("wpscan_vulnerabilities.json", "WPScan")
	})
	return globalWPScanCache, globalWPScanCacheErr
}

// loadVulnerabilitiesFromFile loads vulnerabilities from a JSON file.
func loadVulnerabilitiesFromFile(filename string, sourceName string) ([]Vulnerability, error) {
	filePath, err := file.GetStoragePath(filename)
	if err != nil {
		logger.DefaultLogger.Warning("Failed to get storage path: " + err.Error())
		return nil, err
	}

	data, err := os.ReadFile(filePath)
	if err != nil {
		logger.DefaultLogger.Warning(fmt.Sprintf("Failed to read %s JSON: %s", sourceName, err.Error()))
		logger.DefaultLogger.Info("Run 'wpprobe update-db' to fetch the latest vulnerability database.")
		logger.DefaultLogger.Warning("The scan will proceed, but vulnerabilities will not be displayed.")
		return nil, err
	}

	var vulns []Vulnerability
	if err := json.Unmarshal(data, &vulns); err != nil {
		logger.DefaultLogger.Warning("JSON unmarshal error: " + err.Error())
		return nil, err
	}

	return vulns, nil
}

// ReloadVulnerabilityCache forces a reload of the vulnerability cache.
// Call this after updating the database files.
func ReloadVulnerabilityCache() {
	globalWordfenceCacheOnce = sync.Once{}
	globalWordfenceCache = nil
	globalWordfenceCacheErr = nil

	globalWPScanCacheOnce = sync.Once{}
	globalWPScanCache = nil
	globalWPScanCacheErr = nil

	globalAllCacheOnce = sync.Once{}
	globalAllCache = nil
}

// GetAllVulnerabilities returns all vulnerabilities (Wordfence + WPScan) from the global cache.
// This function returns a reference to the cached slice - do not modify the returned slice.
// The cache is loaded once and shared across all scanners to prevent memory bloat.
func GetAllVulnerabilities() []Vulnerability {
	globalAllCacheOnce.Do(func() {
		wordfence, _ := LoadWordfenceVulnerabilities()
		wpscan, _ := LoadWPScanVulnerabilities()

		// Pre-allocate with exact capacity to avoid reallocations
		globalAllCache = make([]Vulnerability, 0, len(wordfence)+len(wpscan))
		globalAllCache = append(globalAllCache, wordfence...)
		globalAllCache = append(globalAllCache, wpscan...)
	})
	return globalAllCache
}

// GetWordfenceVulnerabilitiesForPlugin returns Wordfence vulnerabilities for a specific plugin and version.
func GetWordfenceVulnerabilitiesForPlugin(plugin string, version string) []Vulnerability {
	data, err := LoadWordfenceVulnerabilities()
	if err != nil {
		return []Vulnerability{}
	}
	return GetVulnerabilitiesForPlugin(data, plugin, version)
}

// GetWPScanVulnerabilitiesForPlugin returns WPScan vulnerabilities for a specific plugin and version.
func GetWPScanVulnerabilitiesForPlugin(plugin string, version string) []Vulnerability {
	data, err := LoadWPScanVulnerabilities()
	if err != nil {
		return []Vulnerability{}
	}
	return GetVulnerabilitiesForPlugin(data, plugin, version)
}

