"""Tests for DURATION data types."""

import datetime

from pydantic import field_serializer
import pytest

from ical.component import ComponentModel
from ical.parsing.property import ParsedProperty
from ical.types.data_types import serialize_field


class FakeModel(ComponentModel):
    """Model under test."""

    duration: datetime.timedelta

    serialize_fields = field_serializer("*")(serialize_field)  # type: ignore[pydantic-field]


@pytest.mark.parametrize(
    "value,duration,encoded_value",
    [
        (
            "P15DT5H0M20S",
            datetime.timedelta(days=15, hours=5, seconds=20),
            "P2W1DT5H20S",
        ),
        ("P7W", datetime.timedelta(days=7 * 7), "P7W"),
        ("-P7W", datetime.timedelta(days=-7 * 7), "-P7W"),
        ("-P1W6DT15H", datetime.timedelta(days=-(7 + 6), hours=-15), "-P1W6DT15H"),
    ],
)
def test_duration(value: str, duration: datetime.timedelta, encoded_value: str) -> None:
    """Test for duration fields."""

    model = FakeModel.model_validate(
        {"duration": [ParsedProperty(name="duration", value=value)]}
    )
    assert model.duration == duration
    component = model.__encode_component_root__()
    assert component.name == "FakeModel"
    assert component.properties == [
        ParsedProperty(name="duration", value=encoded_value)
    ]


def test_duration_from_object() -> None:
    """Test for a duration field from a native object."""

    model = FakeModel(duration=datetime.timedelta(hours=1))
    assert model.duration == datetime.timedelta(hours=1)

    component = model.__encode_component_root__()
    assert component.name == "FakeModel"
    assert component.properties == [ParsedProperty(name="duration", value="PT1H")]
