<?php

declare(strict_types=1);

namespace Doctrine\Tests\Common\Collections;

use Doctrine\Common\Collections\Criteria;
use Doctrine\Common\Collections\Expr\Comparison;
use Doctrine\Common\Collections\Expr\CompositeExpression;
use Doctrine\Common\Collections\ExpressionBuilder;
use Doctrine\Common\Collections\Order;
use Doctrine\Deprecations\PHPUnit\VerifyDeprecations;
use PHPUnit\Framework\Attributes\IgnoreDeprecations;
use PHPUnit\Framework\TestCase;

class CriteriaTest extends TestCase
{
    use VerifyDeprecations;

    #[IgnoreDeprecations]
    public function testCreateLegacy(): void
    {
        $this->expectDeprecationWithIdentifier('https://github.com/doctrine/collections/pull/472');

        $criteria = Criteria::create();

        self::assertInstanceOf(Criteria::class, $criteria);
    }

    public function testCreate(): void
    {
        $criteria = Criteria::create(true);

        self::assertInstanceOf(Criteria::class, $criteria);
    }

    public function testConstructor(): void
    {
        $expr     = new Comparison('field', '=', 'value');
        $criteria = new Criteria($expr, ['foo' => Order::Ascending], 10, 20, accessRawFieldValues: true);

        self::assertSame($expr, $criteria->getWhereExpression());
        self::assertSame(['foo' => Order::Ascending], $criteria->orderings());
        self::assertSame(10, $criteria->getFirstResult());
        self::assertSame(20, $criteria->getMaxResults());
    }

    #[IgnoreDeprecations]
    public function testDeprecatedNullOffset(): void
    {
        $expr = new Comparison('field', '=', 'value');

        $this->expectDeprecationWithIdentifier('https://github.com/doctrine/collections/pull/311');
        $criteria = new Criteria($expr, ['foo' => Order::Ascending], null, 20);

        self::assertSame($expr, $criteria->getWhereExpression());
        self::assertSame(['foo' => Order::Ascending], $criteria->orderings());
        self::assertNull($criteria->getFirstResult());
        self::assertSame(20, $criteria->getMaxResults());
    }

    #[IgnoreDeprecations]
    public function testDefaultConstructor(): void
    {
        $this->expectDeprecationWithIdentifier('https://github.com/doctrine/collections/pull/472');
        $this->expectNoDeprecationWithIdentifier('https://github.com/doctrine/collections/pull/311');

        $criteria = new Criteria();

        self::assertNull($criteria->getWhereExpression());
        self::assertSame([], $criteria->orderings());
        self::assertNull($criteria->getFirstResult());
        self::assertNull($criteria->getMaxResults());
    }

    public function testNamedArgs(): void
    {
        $this->expectNoDeprecationWithIdentifier('https://github.com/doctrine/collections/pull/311');
        new Criteria(orderings: ['startDate' => Order::Descending], accessRawFieldValues: true);
    }

    public function testWhere(): void
    {
        $expr     = new Comparison('field', '=', 'value');
        $criteria = Criteria::create(true);

        $criteria->where($expr);

        self::assertSame($expr, $criteria->getWhereExpression());
    }

    public function testAndWhere(): void
    {
        $expr     = new Comparison('field', '=', 'value');
        $criteria = Criteria::create(true);

        $criteria->where($expr);
        $expr = $criteria->getWhereExpression();
        $criteria->andWhere($expr);

        $where = $criteria->getWhereExpression();
        self::assertInstanceOf(CompositeExpression::class, $where);

        self::assertEquals(CompositeExpression::TYPE_AND, $where->getType());
        self::assertSame([$expr, $expr], $where->getExpressionList());
    }

    public function testAndWhereWithoutWhere(): void
    {
        $expr     = new Comparison('field', '=', 'value');
        $criteria = Criteria::create(true);

        $criteria->andWhere($expr);

        self::assertSame($expr, $criteria->getWhereExpression());
    }

    public function testOrWhere(): void
    {
        $expr     = new Comparison('field', '=', 'value');
        $criteria = Criteria::create(true);

        $criteria->where($expr);
        $expr = $criteria->getWhereExpression();
        $criteria->orWhere($expr);

        $where = $criteria->getWhereExpression();
        self::assertInstanceOf(CompositeExpression::class, $where);

        self::assertEquals(CompositeExpression::TYPE_OR, $where->getType());
        self::assertSame([$expr, $expr], $where->getExpressionList());
    }

    public function testOrWhereWithoutWhere(): void
    {
        $expr     = new Comparison('field', '=', 'value');
        $criteria = Criteria::create(true);

        $criteria->orWhere($expr);

        self::assertSame($expr, $criteria->getWhereExpression());
    }

    public function testOrderings(): void
    {
        $criteria = Criteria::create(true)
            ->orderBy(['foo' => Order::Ascending]);

        self::assertEquals(['foo' => Order::Ascending], $criteria->orderings());
    }

    public function testExpr(): void
    {
        self::assertInstanceOf(ExpressionBuilder::class, Criteria::expr());
    }

    #[IgnoreDeprecations]
    public function testPassingNonOrderEnumToOrderByIsDeprecated(): void
    {
        $this->expectDeprecationWithIdentifier('https://github.com/doctrine/collections/pull/389');
        $criteria = Criteria::create(true)->orderBy(['foo' => 'ASC']);
    }

    #[IgnoreDeprecations]
    public function testConstructingCriteriaWithNonOrderEnumIsDeprecated(): void
    {
        $this->expectDeprecationWithIdentifier('https://github.com/doctrine/collections/pull/389');
        $criteria = new Criteria(null, ['foo' => 'ASC'], firstResult: 0, accessRawFieldValues: true);
    }

    public function testUsingOrderEnumIsTheRightWay(): void
    {
        $this->expectNoDeprecationWithIdentifier('https://github.com/doctrine/collections/pull/389');
        Criteria::create(true)->orderBy(['foo' => Order::Ascending]);
        new Criteria(null, ['foo' => Order::Ascending], firstResult: 0, accessRawFieldValues: true);
    }

    #[IgnoreDeprecations]
    public function testCallingGetOrderingsIsDeprecated(): void
    {
        $criteria = Criteria::create(true)->orderBy(['foo' => Order::Ascending]);
        $this->expectDeprecationWithIdentifier('https://github.com/doctrine/collections/pull/389');
        $criteria->getOrderings();
    }
}
