#!/usr/bin/env fbpython
# Copyright (c) Meta Platforms, Inc. and affiliates.

import os
import typing as t

import click

import numpy as np
import pandas as pd
import xgboost as xgb

from zstrong_ml import CoreModel, get_core_model, get_serialized_model

# Fixes incompatibility issue between the versions of pandas and XGBoost
setattr(pd, "Int64Index", pd.Index)
setattr(pd, "Float64Index", pd.Index)


def generate_binary_classification_data(size=100):
    """
    Generates random data features (x) and labels (y).
    The formula we are learning is: y = x['a'] & (!x['b']).
    """
    features_data = np.random.randint(0, 2, size * 2).reshape((size, 2))
    x = pd.DataFrame(features_data, columns=["a", "b"])
    y = x.a & (x.b ^ 1)
    return x, y


def train_binary_classification_model():
    x_train, y_train = generate_binary_classification_data(100)
    bst = xgb.XGBClassifier(
        n_estimators=2, max_depth=2, learning_rate=1, objective="binary:logistic"
    )
    bst.fit(x_train, y_train)
    x_test, y_test = generate_binary_classification_data(100)
    assert np.all(bst.predict(x_test) == y_test)
    return bst, x_test


def generate_binary_classification_model() -> str:
    """
    Returns a json representation of a binary classification model
    """
    bst, x_test = train_binary_classification_model()
    return get_serialized_model(
        bst,
        [str(c) for c in x_test.columns],
        ["zero", "one"],
    )


def generate_core_binary_classification_model(
    prefix: str,
) -> CoreModel:
    """
    Return a dictionary containing two keys: c_strings and header_strings

    The 'c_strings' key maps to a strings that contain the implementation details of
    nodes/trees/forests, the predictor, feature lists and class labels.

    The 'header_strings' key maps to a string that contains extern declarations
    that can be used to create a core binary classification model.
    """
    bst, x_test = train_binary_classification_model()

    return get_core_model(
        bst, [str(c) for c in x_test.columns], ["zero", "one"], prefix
    )


def generate_multiclass_classification_data(size=100):
    """
    Generates random data features (x) and labels (y).
    The formula we are learning is: y = (x['a'] + x['b'] + x['c']) % 3
    Where a,b,c are numbers between 0 and 2.
    """
    features_data = np.random.randint(0, 3, size * 3).reshape((size, 3))
    x = pd.DataFrame(features_data, columns=["a", "b", "c"])
    y = np.sum(x, axis=1) % 3
    return x, y


def train_multiclass_classification_model():
    x_train, y_train = generate_multiclass_classification_data(1000)
    bst = xgb.XGBClassifier(
        n_estimators=2, max_depth=6, learning_rate=1, objective="multi:softmax"
    )
    bst.fit(x_train, y_train)

    for a in range(2):
        for b in range(2):
            for c in range(2):
                x = pd.DataFrame([[a, b, c]], columns=x_train.columns)
                assert bst.predict(x) == ((a + b + c) % 3)
    return bst, x_train


def generate_multiclass_classification_model() -> str:
    """
    Returns a json representation of a multiclass classification model
    """
    bst, x_train = train_multiclass_classification_model()
    return get_serialized_model(
        bst, [str(c) for c in x_train.columns], ["zero", "one", "two"]
    )


def generate_core_multiclass_classification_model(
    prefix: str,
) -> CoreModel:
    """
    Return a dictionary containing two keys: c_strings and header_strings

    The 'c_strings' key maps to a strings that contain the implementation details of
    nodes/trees/forests, the predictor, feature lists and class labels.

    The 'header_strings' key maps to a string that contains extern declarations
    that can be used to create a core multiclass classification model.
    """
    bst, x_train = train_multiclass_classification_model()

    return get_core_model(
        bst, [str(c) for c in x_train.columns], ["zero", "one", "two"], prefix
    )


def write_core_model(path: str, model_strings: str):
    """
    Writes a .c file containing all the implementation details needed to
    initialize a GBTModel for a core model.
    """
    c_path = path + ".c"
    include_str = f'#include "{path}.h"' if path else ""

    output = f"""// Copyright (c) Meta Platforms, Inc. and affiliates.
// THIS FILE WAS AUTOMATICALLY GENERATED BY test_zstrong_ml_models_generator.py

{include_str}

// clang-format off

{model_strings}
"""
    if path:
        c_path = path + ".c"
        with open(c_path, "wt") as f:
            f.write(output)
    else:
        print(output)


def write_core_files(path: str, model_strings: t.Dict[str, CoreModel]):
    """
    Writes a header file containing extern declarations that can be used to
    implement a core GBTModel
    """
    c_strings = ""
    header_strings = ""
    for K, V in model_strings.items():
        c_strings += f"// GENERATED {K} MODEL MEMBER FIELDS\n"
        c_strings += V.c_strings + "\n"
        header_strings += V.header_strings
    output = f"""// Copyright (c) Meta Platforms, Inc. and affiliates.
// THIS FILE WAS AUTOMATICALLY GENERATED BY test_zstrong_ml_models_generator.py

#include "zstrong/compress/selectors/ml/gbt.h"

// clang-format off

#ifndef {path.replace("/", "_").replace(".", "_").upper()}
#define {path.replace("/", "_").replace(".", "_").upper()}

#ifdef __cplusplus
extern "C" {{
#endif
{header_strings}
#ifdef __cplusplus
}}
#endif

#endif
"""
    if path:
        header_path = path + ".h"
        with open(header_path, "wt") as f:
            f.write(output)
    else:
        print(output)
    write_core_model(path, c_strings)


def write_header(path: str, models: t.Dict[str, str]):
    model_strings = "\n\n".join(
        f'const std::string_view  {K} = R"~DELIM~({V})~DELIM~";'
        for K, V in models.items()
    )
    output = f"""// Copyright (c) Meta Platforms, Inc. and affiliates.
// THIS FILE WAS AUTOMATICALLY GENERATED BY test_zstrong_ml_models_generator.py

#include <string>

#pragma once

namespace {{

{model_strings}


}} //namespace
"""
    if path:
        with open(path, "wt") as f:
            f.write(output)
    else:
        print(output)


@click.command()
@click.option(
    "-p",
    "--path",
    help="""output path for the generated files - please do not include file extension in the path, if empty will print to stdout. """,
    default="",
    type=click.Path(),
)
@click.option(
    "-c",
    "--core",
    is_flag=True,
    help="generate a core Zstrong predictor instead of serialized json model.",
    default=False,
)
def main(path: str, core: bool):
    if not os.getcwd().endswith("zstrong"):
        print("Please run file in zstrong root directory")
        return
    if path.endswith(".h") or path.endswith(".c"):
        print("Please do not include file extension in the path")
        return

    if not core:
        print("Generating serialized binary classification model")
        binary = generate_binary_classification_model()
        print("Generating serialized multiclass classification model")
        multiclass = generate_multiclass_classification_model()
        print("Writing header file")
        write_header(
            path + ".h",
            {
                "GBT_BINARY_MODEL": binary,
                "GBT_MULTICLASS_MODEL": multiclass,
            },
        )
        print("Done")
    else:
        multiclass_prefix = "GBT_MULTICLASS_CORE"
        binary_prefix = "GBT_BINARY_CORE"
        print("Generating core binary classification model")
        binary_core = generate_core_binary_classification_model(binary_prefix)
        print("Generating core multiclass classification model")
        multiclass_core = generate_core_multiclass_classification_model(
            multiclass_prefix
        )
        print("Writing .h and .c file")
        write_core_files(
            path,
            {binary_prefix: binary_core, multiclass_prefix: multiclass_core},
        )
        print("Done")


if __name__ == "__main__":
    main()
