import { LabelName, Stage, Status } from "allure-js-commons";
import { getMessageAndTraceFromError, getStatusFromError } from "allure-js-commons/sdk";
import { ReporterRuntime, createDefaultWriter, getEnvironmentLabels, getFrameworkLabel, getHostLabel, getLanguageLabel, getSuiteLabels, getThreadLabel, } from "allure-js-commons/sdk/reporter";
import { getTestMetadata } from "./utils.js";
export default class AllureVitestReporter {
    constructor(config) {
        this.config = config;
    }
    onInit() {
        const { listeners, resultsDir, ...config } = this.config;
        this.allureReporterRuntime = new ReporterRuntime({
            ...config,
            writer: createDefaultWriter({ resultsDir }),
            listeners,
        });
        this.allureReporterRuntime.writeCategoriesDefinitions();
        this.allureReporterRuntime.writeEnvironmentInfo();
    }
    // eslint-disable-next-line @typescript-eslint/array-type
    onTestRunEnd(tests) {
        for (const test of tests) {
            // actually there's the task property in the test object
            // @ts-ignore
            if (!test?.task) {
                continue;
            }
            // @ts-ignore
            this.handleTask(test.task);
        }
    }
    handleTask(task) {
        // do not report skipped tests
        if (task.mode === "skip" && !task.result) {
            return;
        }
        if (task.type === "suite") {
            for (const innerTask of task.tasks) {
                this.handleTask(innerTask);
            }
            return;
        }
        const { allureRuntimeMessages = [], VITEST_POOL_ID, allureSkip = false, } = task.meta;
        // do not report tests skipped by test plan
        if (allureSkip) {
            return;
        }
        const { specPath, fullName, name, suitePath, labels: metadataLabels, links: metadataLinks } = getTestMetadata(task);
        const testUuid = this.allureReporterRuntime.startTest({
            name,
            start: task.result?.startTime,
        });
        this.allureReporterRuntime.updateTest(testUuid, (result) => {
            const suiteLabels = getSuiteLabels(suitePath);
            const fsPath = specPath.split("/");
            const titlePath = [...fsPath, ...suitePath];
            result.fullName = fullName;
            result.titlePath = titlePath;
            result.labels.push(getFrameworkLabel("vitest"));
            result.labels.push(getLanguageLabel());
            result.labels.push(...metadataLabels);
            result.labels.push(...suiteLabels);
            result.labels.push(...getEnvironmentLabels());
            result.labels.push(getHostLabel());
            result.labels.push(getThreadLabel(VITEST_POOL_ID && `vitest-worker-${VITEST_POOL_ID}`));
            result.links.push(...metadataLinks);
            if (specPath) {
                result.labels.push({
                    name: LabelName.PACKAGE,
                    value: specPath.replaceAll("/", "."),
                });
            }
            this.allureReporterRuntime.applyRuntimeMessages(testUuid, allureRuntimeMessages);
            switch (task.result?.state) {
                case "fail": {
                    const [error] = task.result.errors || [];
                    const status = getStatusFromError(error);
                    result.statusDetails = {
                        ...getMessageAndTraceFromError(error),
                    };
                    result.status = status;
                    result.stage = Stage.FINISHED;
                    break;
                }
                case "pass": {
                    result.status = Status.PASSED;
                    result.stage = Stage.FINISHED;
                    break;
                }
                case "skip": {
                    result.status = Status.SKIPPED;
                    result.stage = Stage.PENDING;
                    break;
                }
            }
        });
        this.allureReporterRuntime.stopTest(testUuid, { duration: task.result?.duration ?? 0 });
        this.allureReporterRuntime.writeTest(testUuid);
    }
}
//# sourceMappingURL=reporter.js.map