-- Copyright 2026 Specter Ops, Inc.
--
-- Licensed under the Apache License, Version 2.0
-- you may not use this file except in compliance with the License.
-- You may obtain a copy of the License at
--
--     http://www.apache.org/licenses/LICENSE-2.0
--
-- Unless required by applicable law or agreed to in writing, software
-- distributed under the License is distributed on an "AS IS" BASIS,
-- WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
-- See the License for the specific language governing permissions and
-- limitations under the License.
--
-- SPDX-License-Identifier: Apache-2.0

-- OpenGraph Extension Management feature flag
INSERT INTO feature_flags (created_at, updated_at, key, name, description, enabled, user_updatable)
VALUES (current_timestamp,
        current_timestamp,
        'opengraph_extension_management',
        'OpenGraph Extension Management',
        'Enable OpenGraph Extension Management',
        false,
        false)
ON CONFLICT DO NOTHING;

-- Scheduled Analysis Configuration feature flag
INSERT INTO feature_flags (created_at, updated_at, key, name, description, enabled, user_updatable)
VALUES (current_timestamp,
        current_timestamp,
        'scheduled_analysis_configuration',
        'Scheduled Analysis Configuration',
        'Enable Scheduled Analysis Configuration form in the UI',
        false,
        false)
ON CONFLICT DO NOTHING;

-- OpenGraph Collector Platform Support feature flag
INSERT INTO feature_flags (created_at, updated_at, key, name, description, enabled, user_updatable)
VALUES (current_timestamp,
        current_timestamp,
        'opengraph_collector_platform_support',
        'OpenGraph Collector Platform Support',
        'Enable creation and communication with the OpenGraph Collector platform.',
        false,
        false)
ON CONFLICT DO NOTHING;

-- Remove pathfinding feature flag. We are keying off of opengraph_extension_management instead
DELETE FROM feature_flags WHERE key = 'opengraph_pathfinding';

-- upsert_kind is a stop-gap function used in open graph schema node,
-- relationship and source kind creation.
-- This function is needed to stave off kind table id exhaustion while maintaining
-- an acceptable level of performance. An exception will be raised if
-- the function fails to insert the kind after 5 attempts.
--
-- Underlying Issues: The kind table's id column is a SMALLINT, Postgres will
-- increase a SERIAL PK on conflict even if DO NOTHING or DO UPDATE is used and
-- a table lock on the Kind table greatly decreases performance.
CREATE OR REPLACE FUNCTION upsert_kind(node_kind_name TEXT) RETURNS kind AS $$
DECLARE
    kind_row kind%rowtype;
BEGIN
    -- Try to find existing kind based on name
    SELECT * INTO kind_row FROM kind WHERE name = node_kind_name;
    IF kind_row IS NOT NULL THEN
        RETURN kind_row;
    END IF;
    -- Insert with retry, handles the race condition where two transactions try to add the same kind at the same time
    FOR i IN 1..5 LOOP
        BEGIN
            INSERT INTO kind (name)
            VALUES (node_kind_name)
            RETURNING * INTO kind_row;
            RETURN kind_row;
        EXCEPTION
            WHEN unique_violation THEN
                -- Check if the insert conflict was for same kind
                SELECT * INTO kind_row FROM kind WHERE name = node_kind_name;
                IF kind_row IS NOT NULL THEN
                    RETURN kind_row;
                END IF;
        END;
    END LOOP;
    -- failed to insert kind after 5 retries
    RAISE EXCEPTION 'failed to insert kind % after 5 retries', node_kind_name;
END;
$$ LANGUAGE plpgsql;

-- Add FK from source_kind to kind:
-- add kind_id column, add any missing source_kinds to the kind table,
-- fill new kind_id column, add not null and unique constraint,
-- add fk to kind table, drop name column
ALTER TABLE source_kinds ADD COLUMN IF NOT EXISTS kind_id SMALLINT;
DO $$
    BEGIN
        IF EXISTS (
            SELECT 1
            FROM information_schema.columns
            WHERE table_schema = 'public' AND column_name = 'name' AND table_name = 'source_kinds'
        ) THEN
            INSERT INTO kind (name)
            SELECT name
            FROM source_kinds sk
            WHERE NOT EXISTS (
                             SELECT 1
                             FROM kind k
                             WHERE k.name = sk.name) ON CONFLICT DO NOTHING;
            UPDATE source_kinds sk SET kind_id = k.id FROM kind k WHERE sk.name = k.name;
        END IF;
        ALTER TABLE source_kinds ALTER COLUMN kind_id SET NOT NULL;
        IF NOT EXISTS (
                      SELECT 1
                      FROM pg_constraint
                      WHERE conname = 'source_kinds_kind_id_unique'
        ) THEN
            ALTER TABLE source_kinds
                ADD CONSTRAINT source_kinds_kind_id_unique UNIQUE (kind_id);
        END IF;
        IF NOT EXISTS (
                      SELECT 1
                      FROM pg_constraint
                      WHERE conname = 'fk_source_kinds_kind_id_kind'
        ) THEN
            ALTER TABLE source_kinds
                ADD CONSTRAINT fk_source_kinds_kind_id_kind FOREIGN KEY (kind_id) REFERENCES kind (id) ON DELETE CASCADE;
        END IF;
    END
$$;
ALTER TABLE source_kinds DROP COLUMN IF EXISTS name;

-- OpenGraph schema_list_findings
CREATE TABLE IF NOT EXISTS schema_list_findings (
    id SERIAL,
    schema_extension_id INTEGER NOT NULL REFERENCES schema_extensions(id) ON DELETE CASCADE,
    node_kind_id SMALLINT NOT NULL REFERENCES kind(id),
    environment_id INTEGER NOT NULL REFERENCES schema_environments(id) ON DELETE CASCADE,
    name TEXT NOT NULL,
    display_name TEXT NOT NULL,
    created_at TIMESTAMP WITH TIME ZONE DEFAULT current_timestamp,
    PRIMARY KEY(id),
    UNIQUE(name)
);

CREATE INDEX IF NOT EXISTS idx_schema_list_findings_extension_id ON schema_list_findings (schema_extension_id);
CREATE INDEX IF NOT EXISTS idx_schema_list_findings_environment_id ON schema_list_findings(environment_id);

-- Drop unique name constraint before migrating to PZ in case AGT names are not unique
ALTER TABLE IF EXISTS asset_group_tag_selectors DROP CONSTRAINT IF EXISTS asset_group_tag_selectors_unique_name_asset_group_tag;

-- Remigrate old custom AGI selectors to PZ selectors for any instances without PZ feature flag enabled
DO $$
  BEGIN
		IF
      (SELECT enabled FROM feature_flags WHERE key  = 'tier_management_engine') = false
    THEN
       -- Delete custom selectors
       DELETE FROM asset_group_tag_selectors WHERE is_default = false AND asset_group_tag_id IN ((SELECT id FROM asset_group_tags WHERE position = 1), (SELECT id FROM asset_group_tags WHERE type = 3));

       -- Re-Migrate existing Tier Zero selectors
       WITH inserted_selector AS (
         INSERT INTO asset_group_tag_selectors (asset_group_tag_id, created_at, created_by, updated_at, updated_by, name, description, is_default, allow_disable, auto_certify)
         SELECT (SELECT id FROM asset_group_tags WHERE position = 1), current_timestamp, 'BloodHound', current_timestamp, 'BloodHound', s.name, s.selector, false, true, 2
         FROM asset_group_selectors s JOIN asset_groups ag ON ag.id = s.asset_group_id
         WHERE ag.tag = 'admin_tier_0' and NOT EXISTS(SELECT 1 FROM asset_group_tag_selectors WHERE name = s.name)
         RETURNING id, description
         )
       INSERT INTO asset_group_tag_selector_seeds (selector_id, type, value) SELECT id, 1, description FROM inserted_selector;

      -- Re-Migrate existing Owned selectors
      WITH inserted_selector AS (
        INSERT INTO asset_group_tag_selectors (asset_group_tag_id, created_at, created_by, updated_at, updated_by, name, description, is_default, allow_disable, auto_certify)
        SELECT (SELECT id FROM asset_group_tags WHERE type = 3), current_timestamp, 'BloodHound', current_timestamp, 'BloodHound', s.name, s.selector, false, true, 0
        FROM asset_group_selectors s JOIN asset_groups ag ON ag.id = s.asset_group_id
        WHERE ag.tag = 'owned' and NOT EXISTS(SELECT 1 FROM asset_group_tag_selectors WHERE name = s.name)
          RETURNING id, description
          )
      INSERT INTO asset_group_tag_selector_seeds (selector_id, type, value) SELECT id, 1, description FROM inserted_selector;
    END IF;
  END;
$$;

-- Before we add unique constraint, rename any duplicates with `_X` to prevent constraint failing
WITH duplicate_selectors AS (
  SELECT id, name, asset_group_tag_id, ROW_NUMBER() OVER (PARTITION BY name, asset_group_tag_id ORDER BY id) AS rowNumber
  FROM asset_group_tag_selectors
)
UPDATE asset_group_tag_selectors agts
SET name = agts.name || '_' || rowNumber FROM duplicate_selectors
WHERE agts.id = duplicate_selectors.id AND duplicate_selectors.rowNumber > 1;

-- Reinstate unique constraint for asset group tag selectors name per asset group tag
ALTER TABLE IF EXISTS asset_group_tag_selectors ADD CONSTRAINT asset_group_tag_selectors_unique_name_asset_group_tag UNIQUE ("name",asset_group_tag_id,is_default);