// Copyright 2025 Specter Ops, Inc.
//
// Licensed under the Apache License, Version 2.0
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//
// SPDX-License-Identifier: Apache-2.0

package api_test

import (
	"fmt"
	"net/url"
	"testing"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"

	"github.com/specterops/bloodhound/cmd/api/src/api"
	"github.com/specterops/bloodhound/cmd/api/src/model"
	"github.com/specterops/dawgs/cypher/models/cypher"
	"github.com/specterops/dawgs/query"
)

func Test_ParseGraphSortParameters_InvalidSortColumn(t *testing.T) {
	domains := model.EnvironmentSelectors{}
	params := url.Values{}
	params.Add("sort_by", "invalidColumn")

	_, err := api.ParseGraphSortParameters(domains, params)
	require.ErrorIs(t, err, api.ErrResponseDetailsCriteriaNotSortable)
}

func Test_ParseGraphSortParameters_Success(t *testing.T) {
	domains := model.EnvironmentSelectors{}
	params := url.Values{}
	params.Add("sort_by", "objectid")
	params.Add("sort_by", "-name")

	orderCriteria, err := api.ParseGraphSortParameters(domains, params)
	require.Nil(t, err)
	require.Equal(t, orderCriteria[0].SortCriteria.(*cypher.PropertyLookup), query.NodeProperty("objectid"))
	require.Equal(t, orderCriteria[0].Direction, query.SortDirectionAscending)

	require.Equal(t, orderCriteria[1].SortCriteria.(*cypher.PropertyLookup), query.NodeProperty("name"))
	require.Equal(t, orderCriteria[1].Direction, query.SortDirectionDescending)
}

func Test_ParseSortParameters(t *testing.T) {
	domains := model.EnvironmentSelectors{}

	t.Run("invalid column", func(t *testing.T) {
		params := url.Values{}
		params.Add("sort_by", "invalidColumn")

		_, err := api.ParseSortParameters(domains, params)
		require.ErrorIs(t, err, api.ErrResponseDetailsColumnNotSortable)
	})

	t.Run("successful sort ascending", func(t *testing.T) {
		params := url.Values{}
		params.Add("sort_by", "objectid")

		sortItems, err := api.ParseSortParameters(domains, params)
		require.Nil(t, err)
		require.Equal(t, sortItems[0].Direction, model.AscendingSortDirection)
		require.Equal(t, sortItems[0].Column, "objectid")
	})

	t.Run("successful sort descending", func(t *testing.T) {
		params := url.Values{}
		params.Add("sort_by", "-objectid")

		sortItems, err := api.ParseSortParameters(domains, params)
		require.Nil(t, err)
		require.Equal(t, sortItems[0].Direction, model.DescendingSortDirection)
		require.Equal(t, sortItems[0].Column, "objectid")
	})
}

func TestBuildSQLSort(t *testing.T) {
	t.Parallel()
	type args struct {
		sort             model.Sort
		identifierColumn model.SortItem
	}
	type want struct {
		sortByItems []string
		err         error
	}
	tests := []struct {
		name string
		args args
		want want
	}{
		{
			name: "success - append identifier column",
			args: args{
				sort: []model.SortItem{{Column: "column1", Direction: model.AscendingSortDirection}, {Column: "column2", Direction: model.DescendingSortDirection}},
				identifierColumn: model.SortItem{
					Column:    "id",
					Direction: model.DescendingSortDirection,
				},
			},
			want: want{[]string{"column1", "column2 desc", "id desc"}, nil},
		},
		{
			name: "success - no identifier column",
			args: args{
				sort: []model.SortItem{{Column: "column1", Direction: model.AscendingSortDirection}, {Column: "column2", Direction: model.DescendingSortDirection}},
			},
			want: want{[]string{"column1", "column2 desc"}, nil},
		},
		{
			name: "success - empty sort with identifier column",
			args: args{
				sort: []model.SortItem{},
				identifierColumn: model.SortItem{
					Column:    "id",
					Direction: model.DescendingSortDirection,
				},
			},
			want: want{[]string{"id desc"}, nil},
		},
		{
			name: "success - empty sort with empty identifier column",
			args: args{},
			want: want{[]string{}, nil},
		},
		{
			name: "fail - empty sort item",
			args: args{
				sort:             []model.SortItem{{Column: ""}},
				identifierColumn: model.SortItem{},
			},
			want: want{[]string{}, fmt.Errorf("the specified column cannot be sorted because it is empty: column index: 0")},
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			t.Parallel()
			got, err := api.BuildSQLSort(tt.args.sort, tt.args.identifierColumn)
			if tt.want.err != nil {
				assert.EqualError(t, err, tt.want.err.Error())
			} else {
				assert.Equal(t, tt.want.sortByItems, got)
			}
		})
	}
}
