#!/usr/bin/env python3
# -*- coding: utf-8 -*-

__author__ = "Peace Lee"
__copyright__ = "Copyright 2015-2024, Guider"
__module__ = "guider"
__credits__ = "Peace Lee"
__license__ = "GPLv2"
__version__ = "3.9.8"
__revision__ = "240423"
__maintainer__ = "Peace Lee"
__email__ = "iipeace5@gmail.com"
__repository__ = "https://github.com/iipeace/guider"


# import essential packages #
import sys

try:
    import os
    import re
    import gc
    import time
    import errno
    import signal
    import atexit
    import struct
    from copy import deepcopy

    # from ctypes import *  # for lint check
except ImportError:
    err = sys.exc_info()[1]
    sys.exit("[ERROR] failed to import essential package: %s" % err.args[0])

# convert an unsupported type #
try:
    long  # pylint: disable=used-before-assignment
except:
    long = int

# prevent MemoryError in python2 #
try:
    xrange  # pylint: disable=used-before-assignment
except:
    xrange = range

# enable JIT compiler #
try:
    if "JITCOMPILE" in os.environ:
        from numba import jit  # pylint: disable=import-error
    else:

        def jit(func):
            def decorated():
                func()

            return decorated

except:
    err = sys.exc_info()[1]
    sys.exit("[ERROR] failed to import numba package: %s" % err.args[0])


class ConfigMgr(object):
    """Manager for config"""

    # logo, made by http://www.figlet.org, consider also jp2a #
    logo = """
   _____       _     _
  / ____|     (_)   | |
 | |  __ _   _ _  __| | ___ _ __
 | | |_ | | | | |/ _` |/ _ \ '__|
 | |__| | |_| | | (_| |  __/ |
  \_____|\__,_|_|\__,_|\___|_|  ver_%s_%s on python_%s.%s
""" % (
        __version__,
        __revision__,
        sys.version_info[0],
        sys.version_info[1],
    )

    # color #
    if (
        sys.platform.startswith("linux")
        or sys.platform.startswith("darwin")
        or sys.platform.startswith("freebsd")
    ) and not "REMOTERUN" in os.environ:
        ENDC = "\033[0m"
    else:
        ENDC = ""

    # config info #
    confData = {}

    # support architecture #
    supportArch = ["arm", "aarch64", "riscv64", "x86", "x64"]

    # ANSI color #
    COLOR_LIST = {
        "DEFAULT": "\033[0m",
        "BOLD": "\033[1m",
        "ITALIC": "\033[3m",
        "UNDERLINE": "\033[4m",
        "LIGHT": "\033[5m",
        "BLINK": "\033[6m",
        "REVERSE": "\033[7m",
        "SCRATCH": "\033[9m",
        "BLACK": "\033[30m",
        "RED": "\033[31m",
        "GREEN": "\033[32m",
        "YELLOW": "\033[33m",
        "BLUE": "\033[34m",
        "PINK": "\033[35m",
        "CYAN": "\033[36m",
        "WHITE": "\033[37m",
        "DEFCOLOR": "\033[39m",
        "BGBLACK": "\033[40m",
        "BGRED": "\033[41m",
        "BGGREEN": "\033[42m",
        "BGYELLOW": "\033[43m",
        "BGBLUE": "\033[44m",
        "BGPINK": "\033[45m",
        "BGCYAN": "\033[46m",
        "BGWHITE": "\033[47m",
        "FAIL": "\033[91m",
        "OKGREEN": "\033[92m",
        "SPECIAL": "\033[93m",
        "OKBLUE": "\033[94m",
        "WARNING": "\033[95m",
        "LIGHTGREEN": "\033[1;32m",
    }

    # cgroup entity #
    CGROUP_VALUE = [
        "tasks",
        "cgroup.procs",
        "cpu.shares",
        "cpu.cfs_period_us",
        "cpu.cfs_quota_us",
        "cpu.rt_period_us",
        "cpu.rt_runtime_us",
        "cpu.stat",
        "cpu.uclamp.min",
        "cpu.uclamp.max",
        "cpu.uclamp.latency_sensitive",
        "cpuacct.usage",
        "cpuset.cpus",
        "cpuset.cpus.effective",
        "cpus",
        "memory.limit_in_bytes",
        "memory.memsw.limit_in_bytes",
        "blkio.weight",
        "blkio.bfq.weight",
        "blkio.weight_device",
        "blkio.io_wait_time",
        "blkio.throttle.io_service_bytes",
        "cpu.pressure",
        "memory.pressure",
        "io.pressure",
    ]

    # cgroup stat #
    CGROUP_STAT = {
        "blkio.throttle.io_service_bytes_recursive": None,
        "cgroup.procs": None,
        "cpu.stat": None,
        "cpuacct.usage": None,
        "io.stat": None,
        "memory.current": None,
        "memory.usage_in_bytes": None,
        "tasks": None,
        "cpu.pressure": None,
        "memory.pressure": None,
        "io.pressure": None,
    }

    # state of process #
    PROC_STAT_TYPE = {
        "R": "running",
        "S": "sleep",
        "D": "disk",
        "T": "stopped",
        "t": "traced",
        "X": "dead",
        "x": "dead",
        "Z": "zombie",
        "K": "wakekill",
        "W": "waking",
        "P": "parked",
        "I": "idle",
    }

    # diskstat statistics #
    DISKSTAT = [
        "major",
        "minor",
        "name",
        "readComplete",
        "readMerge",
        "sectorRead",
        "readTime",
        "writeComplete",
        "writeMerge",
        "sectorWrite",
        "writeTime",
        "currentIO",
        "ioTime",
        "ioWTime",
        "discComplete",
        "discMerged",
        "sectorDisc",
        "discTime",  # 4.18+
        "flushComplete",
        "flushTime",  # 5.5+
    ]

    BGRECLAIMSTAT = [
        "pgsteal_kswapd",
        "pgsteal_kswapd_normal",
        "pgsteal_kswapd_high",
        "pgsteal_kswapd_dma",
        "pgsteal_kswapd_dma32",
        "pgsteal_kswapd_movable",
    ]

    FGRECLAIMSTAT = [
        "pgsteal_direct",
        "pgsteal_direct_normal",
        "pgsteal_direct_high",
        "pgsteal_direct_dma",
        "pgsteal_direct_dma32",
        "pgsteal_direct_movable",
    ]

    # socketcall attributes #
    SOCKETCALL = {
        1: "socket",
        2: "bind",
        3: "connect",
        4: "listen",
        5: "accept",
        6: "getsockname",
        7: "getpeername",
        8: "socketpair",
        9: "send",
        10: "recv",
        11: "sendto",
        12: "recvfrom",
        13: "shutdown",
        14: "setsockopt",
        15: "getsockopt",
        16: "sendmsg",
        17: "recvmsg",
        18: "accept4",
        19: "recvmmsg",
        20: "sendmmsg",
    }

    # socket family #
    SOCKET_FAMILY = {
        0: "AF_UNSPEC",
        1: "AF_FILE",
        2: "AF_INET",
        3: "AF_AX25",
        4: "AF_IPX",
        5: "AF_APPLETALK",
        6: "AF_NETROM",
        7: "AF_BRIDGE",
        8: "AF_ATMPVC",
        9: "AF_X25",
        10: "AF_INET6",
        11: "AF_ROSE",
        12: "AF_DECnet",
        13: "AF_NETBEUI",
        14: "AF_SECURITY",
        15: "AF_KEY",
        16: "AF_NETLINK",
        17: "AF_PACKET",
        18: "AF_ASH",
        19: "AF_ECONET",
        20: "AF_ATMSVC",
        22: "AF_SNA",
        23: "AF_IRDA",
        24: "AF_PPPOX",
        25: "AF_WANPIPE",
        31: "AF_BLUETOOTH",
    }

    # DLOPEN type #
    DLOPEN_TYPE = {
        1: "RTLD_LAZY",
        2: "RTLD_NOW",
        3: "RTLD_BINDING_MASK",
        4: "RTLD_NOLOAD",
        8: "RTLD_DEEPBIND",
    }

    # socket type #
    SOCKET_TYPE = {
        1: "SOCK_STREAM",
        2: "SOCK_DGRAM",
        3: "SOCK_RAW",
        4: "SOCK_RDM",
        5: "SOCK_SEQPACKET",
        10: "SOCK_PACKET",
        2048: "SOCK_NONBLOCK",
        524288: "SOCK_CLOEXEC",
    }

    # log level #
    LOG_LEVEL = {
        0: "EMERG",
        1: "ALERT",
        2: "CRIT",
        3: "ERR",
        4: "WARNING",
        5: "NOTICE",
        6: "INFO",
        7: "DEBUG",
    }

    # MSG type #
    MSG_TYPE = {
        0x1: "MSG_OOB",
        0x2: "MSG_PEEK",
        0x4: "MSG_DONTROUTE|MSG_TRYHARD",
        0x8: "MSG_CTRUNC",
        0x10: "MSG_PROBE",
        0x20: "MSG_TRUNC",
        0x40: "MSG_DONTWAIT",
        0x80: "MSG_EOR",
        0x100: "MSG_WAITALL",
        0x200: "MSG_FIN",
        0x400: "MSG_SYN",
        0x800: "MSG_CONFIRM",
        0x1000: "MSG_RST",
        0x2000: "MSG_ERRQUEUE",
        0x4000: "MSG_NOSIGNAL",
        0x8000: "MSG_MORE",
        0x40000000: "MSG_CMSG_CLOEXEC",
        0x80000000: "MSG_CMSG_COMPAT",
    }

    # control message type #
    CMSG_TYPE = {
        0x01: "SCM_RIGHTS",  # rw: access rights (array of int)
        0x02: "SCM_CREDENTIALS",  # rw: struct ucred
        0x03: "SCM_SECURITY",
    }

    # mmap prot type #
    ARCH_CODE = {
        0x1001: "ARCH_SET_GS",
        0x1002: "ARCH_SET_FS",
        0x1003: "ARCH_GET_FS",
        0x1004: "ARCH_GET_GS",
        0x1011: "ARCH_GET_CPUID",
        0x1012: "ARCH_SET_CPUID",
        0x1021: "ARCH_GET_XCOMP_SUPP",
        0x1022: "ARCH_GET_XCOMP_PERM",
        0x1023: "ARCH_REQ_XCOMP_PERM",
        0x1024: "ARCH_GET_XCOMP_GUEST_PERM",
        0x1025: "ARCH_REQ_XCOMP_GUEST_PERM",
        0x2001: "ARCH_MAP_VDSO_X32",
        0x2002: "ARCH_MAP_VDSO_32",
        0x2003: "ARCH_MAP_VDSO_64",
        0x3001: "ARCH_CET_STATUS",
        0x3002: "ARCH_CET_DISABLE",
        0x3003: "ARCH_CET_LOCK",
        0x3004: "ARCH_CET_ALLOC_SHSTK",
    }

    # mmap prot type #
    MAP_TYPE = {
        0x0000: "MAP_FILE",
        0x0001: "MAP_SHARED",
        0x0002: "MAP_PRIVATE",
        # 0x0003: "MAP_SHARED_VALIDATE",
        # 0x000f: "MAP_TYPE",
        0x0010: "MAP_FIXED",
        0x0020: "MAP_ANONYMOUS",
        0x0100: "MAP_GROWSDOWN",
        0x0800: "MAP_DENYWRITE",
        0x1000: "MAP_EXECUTABLE",
        0x2000: "MAP_LOCKED",
        0x4000: "MAP_NORESERVE",
        0x8000: "MAP_POPULATE",
        0x10000: "MAP_NONBLOCK",
        0x20000: "MAP_STACK",
        0x40000: "MAP_HUGETLB",
        0x80000: "MAP_SYNC",
    }

    # FAT type #
    FAT_TYPE = {
        -100: "AT_FDCWD",
        0x100: "AT_SYMLINK_NOFOLLOW",
        0x200: "AT_REMOVEDIR",
        0x400: "AT_SYMLINK_FOLLOW",
        0x800: "AT_NO_AUTOMOUNT",
        0x1000: "AT_EMPTY_PATH",
    }
    FAT_TYPE_REVERSE = {}

    # mmap prot type #
    PROT_TYPE = {
        0x0: "PROT_NONE",  # Page can not be accessed
        0x1: "PROT_READ",  # Page can be read
        0x2: "PROT_WRITE",  # Page can be written
        0x4: "PROT_EXEC",  # Page can be executed
    }

    # perm type #
    PERM_TYPE = {0x0: "F_OK", 0x1: "X_OK", 0x2: "W_OK", 0x4: "R_OK"}

    # seek type #
    SEEK_TYPE = {0x0: "SEEK_SET", 0x1: "SEEK_CUR", 0x2: "SEEK_END"}

    # prctl flags type #
    PRCTL_TYPE = {
        0: "PR_SET_VMA_ANON_NAME",
        1: "PR_SET_PDEATHSIG",
        2: "PR_GET_PDEATHSIG",
        3: "PR_GET_DUMPABLE",
        4: "PR_SET_DUMPABLE",
        5: "PR_GET_UNALIGN",
        6: "PR_SET_UNALIGN",
        7: "PR_GET_KEEPCAPS",
        8: "PR_SET_KEEPCAPS",
        9: "PR_GET_FPEMU",
        10: "PR_SET_FPEMU",
        11: "PR_GET_FPEXC",
        12: "PR_SET_FPEXC",
        13: "PR_GET_TIMING",
        14: "PR_SET_TIMING",
        15: "PR_SET_NAME",
        16: "PR_GET_NAME",
        19: "PR_GET_ENDIAN",
        20: "PR_SET_ENDIAN",
        21: "PR_GET_SECCOMP",
        22: "PR_SET_SECCOMP",
        23: "PR_CAPBSET_READ",
        24: "PR_CAPBSET_DROP",
        25: "PR_GET_TSC 25",
        26: "PR_SET_TSC 26",
        27: "PR_GET_SECUREBITS",
        28: "PR_SET_SECUREBITS",
        29: "PR_SET_TIMERSLACK",
        30: "PR_GET_TIMERSLACK",
        31: "PR_TASK_PERF_EVENTS_DISABLE",
        32: "PR_TASK_PERF_EVENTS_ENABLE",
        33: "PR_MCE_KILL",
        0x53564D41: "PR_SET_VMA",
    }

    # mount flags type #
    MOUNT_TYPE = {
        "MS_RDONLY": 1,  # Mount read-only
        "MS_NOSUID": 2,  # Ignore suid and sgid bits
        "MS_NODEV": 4,  # Disallow access to device special files
        "MS_NOEXEC": 8,  # Disallow program execution
        "MS_SYNCHRONOUS": 16,  # Writes are synced at once
        "MS_REMOUNT": 32,  # Alter flags of a mounted FS
        "MS_MANDLOCK": 64,  # Allow mandatory locks on an FS
        "MS_WRITE": 128,  # Write on file/directory/symlink
        "MS_APPEND": 256,  # Append-only file
        "MS_IMMUTABLE": 512,  # Immutable file
        "MS_NOATIME": 1024,  # Do not update access times
        "MS_NODIRATIME": 2048,  # Do not update directory access times
        "MS_BIND": 4096,  # Bind directory at different place
        "MS_REC": 16384,
        "MS_VERBOSE": 32768,  # War is peace. Verbosity is silence
        "MS_SILENT": 32768,
        "MS_POSIXACL": (1 << 16),  # VFS does not apply the umask
        "MS_UNBINDABLE": (1 << 17),  # change to unbindable
        "MS_PRIVATE": (1 << 18),  # change to private
        "MS_SLAVE": (1 << 19),  # change to slave
        "MS_SHARED": (1 << 20),  # change to shared
        "MS_RELATIME": (1 << 21),  # Update atime relative to mtime/ctime
        "MS_KERNMOUNT": (1 << 22),  # this is a kern_mount call
        "MS_I_VERSION": (1 << 23),  # Update inode I_version field
        "MS_STRICTATIME": (1 << 24),  # Always perform atime updates
        "MS_LAZYTIME": (1 << 25),  # Update the on-disk [acm]times lazily
        "MS_NOSEC": (1 << 28),
        "MS_BORN": (1 << 29),
        "MS_ACTIVE": (1 << 30),
        "MS_NOUSER": (1 << 31),
    }
    MOUNT_TYPE_REVERSE = {}

    # umount flags type #
    UMOUNT_TYPE = {
        "MNT_FORCE": 0x00000001,  # Attempt to forcibily umount
        "MNT_DETACH": 0x00000002,  # Just detach from the tree
        "MNT_EXPIRE": 0x00000004,  # Mark for expiry
        "MNT_NOFOLLOW": 0x00000008,  # Don't follow symlink on umount
        "MNT_UNUSED": 0x80000000,  # Flag guaranteed to be unused
    }
    UMOUNT_TYPE_REVERSE = {}

    # clone flags type #
    CLONE_TYPE = {
        0x000000FF: "CSIGNAL",
        0x00000100: "CLONE_VM",
        0x00000200: "CLONE_FS",
        0x00000400: "CLONE_FILES",
        0x00000800: "CLONE_SIGHAND",
        0x00002000: "CLONE_PTRACE",
        0x00004000: "CLONE_VFORK",
        0x00008000: "CLONE_PARENT",
        0x00010000: "CLONE_THREAD",
        0x00020000: "CLONE_NEWNS",
        0x00040000: "CLONE_SYSVSEM",
        0x00080000: "CLONE_SETTLS",
        0x00100000: "CLONE_PARENT_SETTID",
        0x00200000: "CLONE_CHILD_CLEARTID",
        0x00400000: "CLONE_DETACHED",
        0x00800000: "CLONE_UNTRACED",
        0x01000000: "CLONE_CHILD_SETTID",
        0x02000000: "CLONE_STOPPED",
        0x04000000: "CLONE_NEWUTS",
        0x08000000: "CLONE_NEWIPC",
        0x10000000: "CLONE_NEWUSER",
        0x20000000: "CLONE_NEWPID",
        0x40000000: "CLONE_NEWNET",
        0x80000000: "CLONE_IO",
    }

    # open flags type #
    OPEN_TYPE = {
        0o0: "O_RDONLY",
        0o1: "O_WRONLY",
        0o2: "O_RDWR",
        0o100: "O_CREAT",
        0o200: "O_EXCL",
        0o400: "O_NOCTTY",
        0o1000: "O_TRUNC",
        0o2000: "O_APPEND",
        0o4000: "O_NONBLOCK",
        0o10000: "O_SYNC",
        0o20000: "O_ASYNC",
        0o40000: "O_DIRECT",
        0o100000: "O_LARGEFILE",
        0o200000: "O_DIRECTORY",
        0o400000: "O_NOFOLLOW",
        0o1000000: "O_NOATIME",
        0o2000000: "O_CLOEXEC",
        0o10000000: "O_PATH",
        0o20200000: "O_TMPFILE",
    }

    # epoll op type #
    EPOLL_CMD_TYPE = [
        "EPOLL_CTL_NONE",  # None
        "EPOLL_CTL_ADD",  # Add a file descriptor to the interface
        "EPOLL_CTL_DEL",  # Remove a file descriptor from the interface
        "EPOLL_CTL_MOD",  # Change file descriptor epoll_event structure
    ]

    # epoll event type #
    EPOLL_EVENT_TYPE = {
        0x001: "EPOLLIN",
        0x002: "EPOLLPRI",
        0x004: "EPOLLOUT",
        0x040: "EPOLLRDNORM",
        0x080: "EPOLLRDBAND",
        0x100: "EPOLLWRNORM",
        0x200: "EPOLLWRBAND",
        0x400: "EPOLLMSG",
        0x008: "EPOLLERR",
        0x010: "EPOLLHUP",
        0x2000: "EPOLLRDHUP",
        (1 << 29): "EPOLLWAKEUP",
        (1 << 30): "EPOLLONESHOT",
        (1 << 31): "EPOLLET",
    }

    # clock type #
    CLOCK_TYPE = {
        "CLOCK_REALTIME": 0,  # Identifier for system-wide realtime clock
        "CLOCK_MONOTONIC": 1,  # Monotonic system-wide clock
        "CLOCK_PROCESS_CPUTIME_ID": 2,  # High-resolution timer from the CPU
        "CLOCK_THREAD_CPUTIME_ID": 3,  # Thread-specific CPU-time clock
        "CLOCK_BOOTTIME": 7,  # Monotonic system-wide clock that includes time spent in suspension
        "CLOCK_REALTIME_ALARM": 8,  # Like CLOCK_REALTIME but also wakes suspended system
        "CLOCK_BOOTTIME_ALARM": 9,  # CLOCK_BOOTTIME but also wakes suspended system
        "CLOCK_TAI": 11,  # Like CLOCK_REALTIME but in International Atomic Time
    }

    # fadvise type #
    FADV_TYPE = {
        0: "FADV_NORMAL",  # No further special treatment
        1: "FADV_RANDOM",  # Expect random page references
        2: "FADV_SEQUENTIAL",  # Expect sequential page references
        3: "FADV_WILLNEED",  # Will need these pages
        4: "FADV_DONTNEED",  # Don't need these pages
        5: "FADV_NOREUSE",  # Data will be accessed once
        # 6: "FADV_DONTNEED",  # Don't need these pages
        # 7: "FADV_NOREUSE",  # Data will be accessed once
    }
    FADV_TYPE_REVERSE = {}

    # madvise type #
    MADV_TYPE = {
        0: "MADV_NORMAL",  # No further special treatment
        1: "MADV_RANDOM",  # Expect random page references
        2: "MADV_SEQUENTIAL",  # Expect sequential page references
        3: "MADV_WILLNEED",  # Will need these pages
        4: "MADV_DONTNEED",  # Don't need these pages
        8: "MADV_FREE",  # Free pages only if memory pressure
        9: "MADV_REMOVE",  # Remove these pages and resources
        10: "MADV_DONTFORK",  # Do not inherit across fork
        11: "MADV_DOFORK",  # Do inherit across fork
        12: "MADV_MERGEABLE",  # KSM may merge identical pages
        13: "MADV_UNMERGEABLE",  # KSM may not merge identical pages
        14: "MADV_HUGEPAGE",  # Worth backing with hugepages
        15: "MADV_NOHUGEPAGE",  # Not worth backing with hugepages
        16: "MADV_DONTDUMP",  # Explicitly exclude from the core dump,
        17: "MADV_DODUMP",  # Clear the MADV_DONTDUMP flag
        18: "MADV_WIPEONFORK",  # Zero memory on fork, child only
        19: "MADV_KEEPONFORK",  # Undo MADV_WIPEONFORK
        20: "MADV_COLD",  # Deactivatie these pages
        21: "MADV_PAGEOUT",  # Reclaim these pages
        100: "MADV_HWPOISON",  # Poison a page for testing
    }
    MADV_TYPE_REVERSE = {}

    # fanotify types #
    FAN_EVENT_TYPE = {
        0x00000001: "FAN_ACCESS",  # File was accessed #
        0x00000002: "FAN_MODIFY",  # File was modified #
        0x00000004: "FAN_ATTRIB",  # Metadata changed #
        0x00000008: "FAN_CLOSE_WRITE",  # Writtable file closed #
        0x00000010: "FAN_CLOSE_NOWRITE",  # Unwrittable file closed #
        0x00000020: "FAN_OPEN",  # File was opened #
        0x00000040: "FAN_MOVED_FROM",  # File was moved from X #
        0x00000080: "FAN_MOVED_TO",  # File was moved to Y #
        0x00000100: "FAN_CREATE",  # Subfile was created #
        0x00000200: "FAN_DELETE",  # Subfile was deleted #
        0x00000400: "FAN_DELETE_SELF",  # Self was deleted #
        0x00000800: "FAN_MOVE_SELF",  # Self was moved #
        0x00001000: "FAN_OPEN_EXEC",  # File was opened for exec #
        0x00004000: "FAN_Q_OVERFLOW",  # Event queued overflowed #
        0x00010000: "FAN_OPEN_PERM",  # File open in perm check #
        0x00020000: "FAN_ACCESS_PERM",  # File accessed in perm check #
        0x00040000: "FAN_OPEN_EXEC_PERM",  # File open/exec in perm check #
        0x40000000: "FAN_ONDIR",  # event occurred against dir #
        0x08000000: "FAN_EVENT_ON_CHILD",  # interested in child events #
    }
    FAN_EVENT_TYPE_REVERSE = {}

    FAN_INIT_TYPE = {
        0x00000000: "FAN_CLASS_NOTIF",
        0x00000001: "FAN_CLOEXEC",
        0x00000002: "FAN_NONBLOCK",
        0x00000004: "FAN_CLASS_CONTENT",
        0x00000008: "FAN_CLASS_PRE_CONTENT",
        0x00000010: "FAN_UNLIMITED_QUEUE",
        0x00000020: "FAN_UNLIMITED_MARKS",
        0x00000040: "FAN_ENABLE_AUDIT",
        0x00000100: "FAN_REPORT_TID",
        0x00000200: "FAN_REPORT_FID",
        0x00000800: "FAN_REPORT_NAME",
        0x00001000: "FAN_REPORT_TARGET_FID",
    }
    FAN_INIT_TYPE_REVERSE = {}
    """
    FAN_REPORT_DFID_NAME = FAN_REPORT_NAME | FAN_REPORT_DIR_FID
    FAN_REPORT_DFID_NAME_TARGET = \
        FAN_REPORT_DFID_NAME | FAN_REPORT_FID | FAN_REPORT_TARGETFID
    """

    FAN_MARK_TYPE = {
        0x00000000: "FAN_MARK_INODE",
        0x00000001: "FAN_MARK_ADD",
        0x00000002: "FAN_MARK_REMOVE",
        0x00000004: "FAN_MARK_DONT_FOLLOW",
        0x00000008: "FAN_MARK_ONLYDIR",
        0x00000010: "FAN_MARK_MOUNT",
        0x00000020: "FAN_MARK_IGNORED_MASK",
        0x00000040: "FAN_MARK_IGNORED_SURV_MODIFY",
        0x00000080: "FAN_MARK_FLUSH",
        0x00000100: "FAN_MARK_FILESYSTEM",
    }

    FAN_PERM_TYPE = {
        0x01: "FAN_ALLOW",
        0x02: "FAN_DENY",
        0x10: "FAN_AUDIT",
    }

    # netlink type #
    NETLINK_TYPE = {
        "NETLINK_ROUTE": 0,
        "NETLINK_UNUSED": 1,
        "NETLINK_USERSOCK": 2,
        "NETLINK_FIREWALL": 3,
        "NETLINK_SOCK_DIAG": 4,
        "NETLINK_NFLOG": 5,
        "NETLINK_XFRM": 6,
        "NETLINK_SELINUX": 7,
        "NETLINK_ISCSI": 8,
        "NETLINK_AUDIT": 9,
        "NETLINK_FIB_LOOKUP": 10,
        "NETLINK_CONNECTOR": 11,
        "NETLINK_NETFILTER": 12,
        "NETLINK_IP6_FW": 13,
        "NETLINK_DNRTMSG": 14,
        "NETLINK_KOBJECT_UEVENT": 15,
        "NETLINK_GENERIC": 16,
        "NETLINK_SCSITRANSPORT": 18,
        "NETLINK_ECRYPTFS": 19,
        "NETLINK_RDMA": 20,
        "NETLINK_CRYPTO": 21,
    }

    # entry type #
    INOTIFY_TYPE = {
        "IN_ACCESS": 0x00000001,  # File was accessed */
        "IN_MODIFY": 0x00000002,  # File was modified */
        "IN_ATTRIB": 0x00000004,  # Metadata changed */
        "IN_CLOSE_WRITE": 0x00000008,  # Writtable file was closed */
        "IN_CLOSE_NOWRITE": 0x00000010,  # Unwrittable file closed */
        "IN_CLOSE": 0x00000008 | 0x00000010,  # file closed */
        "IN_OPEN": 0x00000020,  # File was opened */
        "IN_MOVED_FROM": 0x00000040,  # File was moved from X */
        "IN_MOVED_TO": 0x00000080,  # File was moved to Y */
        "IN_MOVED": 0x00000040 | 0x00000080,  # File was moved */
        "IN_CREATE": 0x00000100,  # Subfile was created */
        "IN_DELETE": 0x00000200,  # Subfile was deleted */
        "IN_DELETE_SELF": 0x00000400,  # Self was deleted */
        "IN_MOVE_SELF": 0x00000800,  # Self was moved */
        "IN_UNMOUNT": 0x00002000,  # Backing fs was unmounted */
        "IN_Q_OVERFLOW": 0x00004000,  # Event queued overflowed */
        "IN_IGNORED": 0x00008000,  # File was ignored */
        "IN_ONLYDIR": 0x01000000,  # only watch the path if it is a directory */
        "IN_DONT_FOLLOW": 0x02000000,  # don't follow a sym link */
        "IN_EXCL_UNLINK": 0x04000000,  # exclude events on unlinked objects */
        "IN_MASK_CREATE": 0x10000000,  # only create watches */
        "IN_MASK_ADD": 0x20000000,  # add to the mask of an already existing watch */
        "IN_ISDIR": 0x40000000,  # event occurred against dir */
        "IN_ONESHOT": 0x80000000,  # only send event once */
    }

    # entry type #
    AT_TYPE = {
        "AT_IGNORE": 1,  # Entry should be ignored
        "AT_EXECFD": 2,  # File descriptor of program
        "AT_PHDR": 3,  # Program headers for program
        "AT_PHENT": 4,  # Size of program header entry
        "AT_PHNUM": 5,  # Number of program headers
        "AT_PAGESZ": 6,  # System page size
        "AT_BASE": 7,  # Base address of interpreter
        "AT_FLAGS": 8,  # Flags
        "AT_ENTRY": 9,  # Entry point of program
        "AT_NOTELF": 10,  # Program is not ELF
        "AT_UID": 11,  # Real uid
        "AT_EUID": 12,  # Effective uid
        "AT_GID": 13,  # Real gid
        "AT_EGID": 14,  # Effective gid
        "AT_CLKTCK": 17,  # Frequency of times()
        "AT_PLATFORM": 15,  # String identifying platform
        "AT_HWCAP": 16,  # Machine-dependent hints about
        "AT_FPUCW": 18,  # Used FPU control word
        "AT_DCACHEBSIZE": 19,  # Data cache block size
        "AT_ICACHEBSIZE": 20,  # Instruction cache block size
        "AT_UCACHEBSIZE": 21,  # Unified cache block size
        "AT_IGNOREPPC": 22,  # Entry should be ignored
        "AT_SECURE": 23,  # Boolean, was exec setuid-like?
        "AT_BASE_PLATFORM": 24,  # String identifying real platforms
        "AT_RANDOM": 25,  # Address of 16 random bytes
        "AT_HWCAP2": 26,  # More machine-dependent hints about
        "AT_EXECFN": 31,  # Filename of executable
        "AT_SYSINFO": 32,
        "AT_SYSINFO_EHDR": 33,
        "AT_L1I_CACHESHAPE": 34,
        "AT_L1D_CACHESHAPE": 35,
        "AT_L2_CACHESHAPE": 36,
        "AT_L3_CACHESHAPE": 37,
        "AT_L1I_CACHESIZE": 40,
        "AT_L1I_CACHEGEOMETRY": 41,
        "AT_L1D_CACHESIZE": 42,
        "AT_L1D_CACHEGEOMETRY": 43,
        "AT_L2_CACHESIZE": 44,
        "AT_L2_CACHEGEOMETRY": 45,
        "AT_L3_CACHESIZE": 46,
        "AT_L3_CACHEGEOMETRY": 47,
        "AT_MINSIGSTKSZ": 51,  # Stack needed for signal delivery
    }

    # syscall prototypes #
    SYSCALL_REFBUF = {
        "write": 0,
        "read": 0,
        "send": 0,
        "sendto": 0,
        "recv": 0,
        "recvfrom": 0,
        "pread": 0,
        "pread64": 0,
        "pwrite": 0,
        "pwrite64": 0,
    }

    # BPF syscall commands #
    BPF_CMD = [
        "BPF_MAP_CREATE",
        "BPF_MAP_LOOKUP_ELEM",
        "BPF_MAP_UPDATE_ELEM",
        "BPF_MAP_DELETE_ELEM",
        "BPF_MAP_GET_NEXT_KEY",
        "BPF_PROG_LOAD",
        "BPF_OBJ_PIN",
        "BPF_OBJ_GET",
        "BPF_PROG_ATTACH",
        "BPF_PROG_DETACH",
        "BPF_PROG_TEST_RUN",
        "BPF_PROG_GET_NEXT_ID",
        "BPF_MAP_GET_NEXT_ID",
        "BPF_PROG_GET_FD_BY_ID",
        "BPF_MAP_GET_FD_BY_ID",
        "BPF_OBJ_GET_INFO_BY_FD",
    ]

    BPF_MAP_TYPE = [
        "BPF_MAP_TYPE_UNSPEC",
        "BPF_MAP_TYPE_HASH",
        "BPF_MAP_TYPE_ARRAY",
        "BPF_MAP_TYPE_PROG_ARRAY",
        "BPF_MAP_TYPE_PERF_EVENT_ARRAY",
        "BPF_MAP_TYPE_PERCPU_HASH",
        "BPF_MAP_TYPE_PERCPU_ARRAY",
        "BPF_MAP_TYPE_STACK_TRACE",
        "BPF_MAP_TYPE_CGROUP_ARRAY",
        "BPF_MAP_TYPE_LRU_HASH",
        "BPF_MAP_TYPE_LRU_PERCPU_HASH",
        "BPF_MAP_TYPE_LPM_TRIE",
        "BPF_MAP_TYPE_ARRAY_OF_MAPS",
        "BPF_MAP_TYPE_HASH_OF_MAPS",
        "BPF_MAP_TYPE_DEVMAP",
        "BPF_MAP_TYPE_SOCKMAP",
    ]

    BPF_PROG_TYPE = [
        "BPF_PROG_TYPE_UNSPEC",
        "BPF_PROG_TYPE_SOCKET_FILTER",
        "BPF_PROG_TYPE_KPROBE",
        "BPF_PROG_TYPE_SCHED_CLS",
        "BPF_PROG_TYPE_SCHED_ACT",
        "BPF_PROG_TYPE_TRACEPOINT",
        "BPF_PROG_TYPE_XDP",
        "BPF_PROG_TYPE_PERF_EVENT",
        "BPF_PROG_TYPE_CGROUP_SKB",
        "BPF_PROG_TYPE_CGROUP_SOCK",
        "BPF_PROG_TYPE_LWT_IN",
        "BPF_PROG_TYPE_LWT_OUT",
        "BPF_PROG_TYPE_LWT_XMIT",
        "BPF_PROG_TYPE_SOCK_OPS",
        "BPF_PROG_TYPE_SK_SKB",
    ]

    BPF_ATTACH_TYPE = [
        "BPF_CGROUP_INET_INGRESS",
        "BPF_CGROUP_INET_EGRESS",
        "BPF_CGROUP_INET_SOCK_CREATE",
        "BPF_CGROUP_SOCK_OPS",
        "BPF_SK_SKB_STREAM_PARSER",
        "BPF_SK_SKB_STREAM_VERDICT",
        "__MAX_BPF_ATTACH_TYPE",
    ]

    # SAVE RESOURCE COMMANDS #
    RESSET = (
        "SAVECPU",
        "SAVETOTALCPU",
        "SAVEMEM",
        "SAVEMEMPROC",
        "SAVEBLOCK",
        "SAVETOTALBLOCK",
        "SAVEPID",
        "SAVERUNTIME",
        "SAVEPRIO",
        "SAVESWAP",
    )

    # regset types #
    NT_TYPE = {
        "NT_PRSTATUS": 1,
        "NT_PRFPREG": 2,
        "NT_PRPSINFO": 3,
        "NT_TASKSTRUCT": 4,
        "NT_AUXV": 6,
        "NT_SIGINFO": 0x53494749,
        "NT_FILE": 0x46494C45,
        "NT_PRXFPREG": 0x46E62B7F,
        "NT_ARM_VFP": 0x400,
        "NT_ARM_TLS": 0x401,
        "NT_ARM_HW_BREAK": 0x402,
        "NT_ARM_HW_WATCH": 0x403,
        "NT_ARM_SYSTEM_CALL": 0x404,
    }

    # unpack types #
    UNPACK_TYPE = {
        "U1": "B",
        "U2": "H",
        "U4": "I",
        "U8": "Q",
        "S1": "b",
        "S2": "h",
        "S4": "i",
        "S8": "q",
        "float": "f",
        "double": "d",
    }

    # capabiliities #
    capabilities = [
        "chown",
        "dac_override",
        "dac_read_search",
        "fowner",
        "fsetid",
        "kill",
        "setgid",
        "setuid",
        "setpcap",
        "linux_immutable",
        "net_bind_service",
        "net_broadcast",
        "net_admin",
        "net_raw",
        "ipc_lock",
        "ipc_owner",
        "sys_module",
        "sys_rawio",
        "sys_chroot",
        "sys_ptrace",
        "sys_pacct",
        "sys_admin",
        "sys_boot",
        "sys_nice",
        "sys_resource",
        "sys_time",
        "sys_tty_config",
        "mknod",
        "lease",
        "audit_write",
        "audit_control",
        "setfcap",
        "mac_override",
        "mac_admin",
        "syslog",
        "wake_alarm",
        "block_suspend",
        "audit_read",
    ]

    # defferable syscall prototypes #
    SYSCALL_DEFFERABLE = {
        "clock_gettime": 0,
        "clone": 0,
        "clone3": 0,
        "getgroups": 0,
        "getgroups16": 0,
        "gethostname": 0,
        "getitimer": 0,
        "getpeername": 0,
        "getsockname": 0,
        "gettimeofday": 0,
        "io_uring_setup": 0,
        "pread": 0,
        "pread64": 0,
        "process_vm_readv": 0,
        "read": 0,
        "readlink": 0,
        "readv": 0,
        "recv": 0,
        "recvfrom": 0,
        "recvmmsg": 0,
        "recvmsg": 0,
    }

    # defferable syscall return prototypes #
    SYSCALL_DEFFERABLE_RET = {
        "ppoll": 0,
    }

    # syscall prototypes #
    SYSCALL_PROTOTYPES = {
        "accept": (
            "long",
            (
                ("int", "sockfd"),
                ("struct sockaddr *", "addr"),
                ("int *", "addrlen"),
            ),
        ),
        "accept4": (
            "long",
            (
                ("int", "sockfd"),
                ("struct sockaddr *", "addr"),
                ("int *", "addrlen"),
                ("int", "flags"),
            ),
        ),
        "access": ("long", (("const char *", "filename"), ("int", "mode"))),
        "acct": ("long", (("const char *", "name"),)),
        "add_key": (
            "long",
            (
                ("const char *", "_type"),
                ("const char *", "_description"),
                ("const void *", "_payload"),
                ("size_t", "plen"),
                ("key_serial_t", "destringid"),
            ),
        ),
        "adjtimex": ("long", (("struct timex *", "txc_p"),)),
        "alarm": ("long", (("unsigned int", "seconds"),)),
        "bdflush": ("long", (("int", "func"), ("long", "data"))),
        "bind": (
            "long",
            (
                ("int", "sockfd"),
                ("struct sockaddr *", "addr"),
                ("int", "addrlen"),
            ),
        ),
        "bpf": (
            "long",
            (
                ("int", "cmd"),
                ("union bpf_attr *", "attr"),
                ("unsigned int", "size"),
            ),
        ),
        "brk": ("long", (("unsigned long", "brk"),)),
        "capget": (
            "long",
            (("cap_user_header_t", "header"), ("cap_user_data_t", "dataptr")),
        ),
        "capset": (
            "long",
            (
                ("cap_user_header_t", "header"),
                ("const cap_user_data_t", "data"),
            ),
        ),
        "chdir": ("long", (("const char *", "filename"),)),
        "chmod": ("long", (("const char *", "filename"), ("umode_t", "mode"))),
        "chown": (
            "long",
            (
                ("const char *", "filename"),
                ("uid_t", "user"),
                ("gid_t", "group"),
            ),
        ),
        "chown16": (
            "long",
            (
                ("const char *", "filename"),
                ("old_uid_t", "user"),
                ("old_gid_t", "group"),
            ),
        ),
        "chroot": ("long", (("const char *", "filename"),)),
        "clock_adjtime": (
            "long",
            (("clockid_t", "which_clock"), ("struct timex *", "tx")),
        ),
        "clock_getres": (
            "long",
            (("clockid_t", "which_clock"), ("struct timespec *", "tp")),
        ),
        "clock_gettime": (
            "long",
            (("clockid_t", "which_clock"), ("struct timespec *", "tp")),
        ),
        "clock_nanosleep": (
            "long",
            (
                ("clockid_t", "which_clock"),
                ("int", "flags"),
                ("const struct timespec *", "rqtp"),
                ("struct timespec *", "rmtp"),
            ),
        ),
        "clock_settime": (
            "long",
            (("clockid_t", "which_clock"), ("const struct timespec *", "tp")),
        ),
        "clone": (
            "long",
            (
                ("unsigned long", "flags"),
                ("unsigned long", "child_stack"),
                ("int *", "ptid"),
                ("int *", "ctid"),
                ("unsigned long", "regs"),
            ),
        ),
        "clone3": (
            "long",
            (("struct clone_args *", "uargs"), ("size_t", "size")),
        ),
        "close": ("long", (("unsigned int", "fd"),)),
        "close_range": (
            "long",
            (
                ("unsigned int", "fd"),
                ("unsigned int", "max_fd"),
                ("unsigned int", "flags"),
            ),
        ),
        "connect": (
            "long",
            (
                ("int", "sockfd"),
                ("struct sockaddr *", "addr"),
                ("int", "addrlen"),
            ),
        ),
        "copy_file_range": (
            "long",
            (
                ("int", "fd_in"),
                ("loff_t *", "off_in"),
                ("int", "fd_out"),
                ("loff_t *", "off_out"),
                ("size_t", "len"),
                ("unsigned int", "flags"),
            ),
        ),
        "creat": ("long", (("const char *", "pathname"), ("umode_t", "mode"))),
        "creat_module": ("long", (("const char *", "name"), ("int", "size"))),
        "delete_module": (
            "long",
            (("const char *", "name_user"), ("unsigned int", "flags")),
        ),
        "dup": ("long", (("unsigned int", "fildes"),)),
        "dup2": (
            "long",
            (("unsigned int", "oldfd"), ("unsigned int", "newfd")),
        ),
        "dup3": (
            "long",
            (
                ("unsigned int", "oldfd"),
                ("unsigned int", "newfd"),
                ("int", "flags"),
            ),
        ),
        "epoll_create": ("long", (("int", "size"),)),
        "epoll_create1": ("long", (("int", "flags"),)),
        "epoll_ctl": (
            "long",
            (
                ("int", "epfd"),
                ("int", "op"),
                ("int", "fd"),
                ("struct epoll_event *", "event"),
            ),
        ),
        "epoll_pwait": (
            "long",
            (
                ("int", "epfd"),
                ("struct epoll_event *", "events"),
                ("int", "maxevents"),
                ("int", "timeout"),
                ("const sigset_t *", "sigmask"),
                ("size_t", "sigsetsize"),
            ),
        ),
        "epoll_wait": (
            "long",
            (
                ("int", "epfd"),
                ("struct epoll_event *", "events"),
                ("int", "maxevents"),
                ("int", "timeout"),
            ),
        ),
        "eventfd": ("long", (("unsigned int", "count"),)),
        "eventfd2": ("long", (("unsigned int", "count"), ("int", "flags"))),
        "execve": (
            "long",
            (
                ("const char *", "filename"),
                ("const char * const *", "argv"),
                ("const char * const *", "envp"),
            ),
        ),
        "execveat": (
            "long",
            (
                ("int", "dfd"),
                ("const char *", "filename"),
                ("const char * const *", "argv"),
                ("const char * const *", "envp"),
                ("int", "flags"),
            ),
        ),
        "exit": ("long", (("int", "error_code"),)),
        "exit_group": ("long", (("int", "error_code"),)),
        "faccessat": (
            "long",
            (("int", "dfd"), ("const char *", "filename"), ("int", "mode")),
        ),
        "faccessat2": (
            "long",
            (
                ("int", "dfd"),
                ("const char *", "filename"),
                ("int", "mode"),
                ("int", "flags"),
            ),
        ),
        "fadvise64": (
            "long",
            (
                ("int", "fd"),
                ("loff_t", "offset"),
                ("size_t", "len"),
                ("int", "advice"),
            ),
        ),
        "fadvise64_64": (
            "long",
            (
                ("int", "fd"),
                ("loff_t", "offset"),
                ("loff_t", "len"),
                ("int", "advice"),
            ),
        ),
        "fallocate": (
            "long",
            (
                ("int", "fd"),
                ("int", "mode"),
                ("loff_t", "offset"),
                ("loff_t", "len"),
            ),
        ),
        "fanotify_init": (
            "long",
            (("unsigned int", "flags"), ("unsigned int", "event_f_flags")),
        ),
        "fanotify_mark": (
            "long",
            (
                ("int", "fanotify_fd"),
                ("unsigned int", "flags"),
                ("u64", "mask"),
                ("int", "dirfd"),
                ("const char *", "pathname"),
            ),
        ),
        "fchdir": ("long", (("unsigned int", "fd"),)),
        "fchmod": ("long", (("unsigned int", "fd"), ("umode_t", "mode"))),
        "fchmodat": (
            "long",
            (
                ("int", "dfd"),
                ("const char *", "filename"),
                ("umode_t", "mode"),
            ),
        ),
        "fchown": (
            "long",
            (("unsigned int", "fd"), ("uid_t", "user"), ("gid_t", "group")),
        ),
        "fchown16": (
            "long",
            (
                ("unsigned int", "fd"),
                ("old_uid_t", "user"),
                ("old_gid_t", "group"),
            ),
        ),
        "fchownat": (
            "long",
            (
                ("int", "dfd"),
                ("const char *", "filename"),
                ("uid_t", "user"),
                ("gid_t", "group"),
                ("int", "flag"),
            ),
        ),
        "fcntl": (
            "long",
            (
                ("unsigned int", "fd"),
                ("unsigned int", "cmd"),
                ("unsigned long", "arg"),
            ),
        ),
        "fcntl64": (
            "long",
            (
                ("unsigned int", "fd"),
                ("unsigned int", "cmd"),
                ("unsigned long", "arg"),
            ),
        ),
        "fdatasync": ("long", (("unsigned int", "fd"),)),
        "fgetxattr": (
            "long",
            (
                ("int", "fd"),
                ("const char *", "name"),
                ("void *", "value"),
                ("size_t", "size"),
            ),
        ),
        "finit_module": (
            "long",
            (("int", "fd"), ("const char *", "uargs"), ("int", "flags")),
        ),
        "flistxattr": (
            "long",
            (("int", "fd"), ("const char *", "list"), ("size_t", "size")),
        ),
        "flock": ("long", (("unsigned int", "fd"), ("unsigned int", "cmd"))),
        "fork": ("long", ()),
        "fremovexattr": ("long", (("int", "fd"), ("const char *", "name"))),
        "fsconfig": (
            "long",
            (
                ("int", "fs_fd"),
                ("unsigned int", "cmd"),
                ("const char *", "key"),
                ("const void *", "value"),
                ("int", "aux"),
            ),
        ),
        "fsetxattr": (
            "long",
            (
                ("int", "fd"),
                ("const char *", "name"),
                ("const void *", "value"),
                ("size_t", "size"),
                ("int", "flags"),
            ),
        ),
        "fsmount": (
            "long",
            (
                ("int", "fs_fd"),
                ("unsigned int", "flags"),
                ("unsigned int", "ms_flags"),
            ),
        ),
        "fsopen": (
            "long",
            (("const char *", "fs_name"), ("unsigned int", "flags")),
        ),
        "fspick": (
            "long",
            (
                ("int", "dfd"),
                ("const char *", "path"),
                ("unsigned int", "flags"),
            ),
        ),
        "fstat": (
            "long",
            (
                ("unsigned int", "fd"),
                ("struct __old_kernel_stat *", "statbuf"),
            ),
        ),
        "fstat64": (
            "long",
            (("unsigned long", "fd"), ("struct stat64 *", "statbuf")),
        ),
        "fstatat64": (
            "long",
            (
                ("int", "dfd"),
                ("const char *", "filename"),
                ("struct stat64 *", "statbuf"),
                ("int", "flag"),
            ),
        ),
        "fstatfs": (
            "long",
            (("unsigned int", "fd"), ("struct statfs *", "buf")),
        ),
        "fstatfs64": (
            "long",
            (
                ("unsigned int", "fd"),
                ("size_t", "sz"),
                ("struct statfs64 *", "buf"),
            ),
        ),
        "fsync": ("long", (("unsigned int", "fd"),)),
        "ftruncate": (
            "long",
            (("unsigned int", "fd"), ("unsigned long", "length")),
        ),
        "ftruncate64": (
            "long",
            (("unsigned int", "fd"), ("loff_t", "length")),
        ),
        "futex": (
            "long",
            (
                ("u32 *", "uaddr"),
                ("int", "op"),
                ("u32", "val"),
                ("struct timespec *", "utime"),
                ("u32 *", "uaddr2"),
                ("u32", "val3"),
            ),
        ),
        "futimesat": (
            "long",
            (
                ("int", "dfd"),
                ("const char *", "filename"),
                ("struct old_timeval *", "utimes"),
            ),
        ),
        "get_kernel_syms": ("long", (("struct kernel_sym *", "table"))),
        "get_mempolicy": (
            "long",
            (
                ("int *", "policy"),
                ("unsigned long *", "nmask"),
                ("unsigned long", "maxnode"),
                ("unsigned long", "addr"),
                ("unsigned long", "flags"),
            ),
        ),
        "get_robust_list": (
            "long",
            (
                ("int", "pid"),
                ("struct robust_list_head * *", "head_ptr"),
                ("size_t *", "len_ptr"),
            ),
        ),
        "get_thread_area": ("long", (("struct user_desc *", "u_info"))),
        "getcpu": (
            "long",
            (
                ("unsigned *", "cpu"),
                ("unsigned *", "node"),
                ("struct getcpu_cache *", "cache"),
            ),
        ),
        "getcwd": (
            "long",
            (("char *", "pathname"), ("unsigned long", "size")),
        ),
        "getdents": (
            "long",
            (
                ("unsigned int", "fd"),
                ("struct linux_dirent *", "dirent"),
                ("unsigned int", "count"),
            ),
        ),
        "getdents64": (
            "long",
            (
                ("unsigned int", "fd"),
                ("struct linux_dirent64 *", "dirent"),
                ("unsigned int", "count"),
            ),
        ),
        "getegid": ("long", ()),
        "getegid16": ("long", ()),
        "geteuid": ("long", ()),
        "geteuid16": ("long", ()),
        "getgid": ("long", ()),
        "getgid16": ("long", ()),
        "getgroups": (
            "long",
            (("int", "gidsetsize"), ("gid_t *", "grouplist")),
        ),
        "getgroups16": (
            "long",
            (("int", "gidsetsize"), ("old_gid_t *", "grouplist")),
        ),
        "gethostname": ("long", (("char *", "name"), ("int", "len"))),
        "getitimer": (
            "long",
            (("int", "which"), ("struct itimerval *", "value")),
        ),
        "getpeername": (
            "long",
            (
                ("int", "sockfd"),
                ("struct sockaddr *", "addr"),
                ("int *", "addrlen"),
            ),
        ),
        "getpgid": ("long", (("pid_t", "pid"),)),
        "getpgrp": ("long", ()),
        "getpid": ("long", ()),
        "getppid": ("long", ()),
        "getpriority": ("long", (("int", "which"), ("int", "who"))),
        "getrandom": (
            "long",
            (
                ("void *", "buf"),
                ("size_t", "count"),
                ("unsigned int", "flags"),
            ),
        ),
        "getresgid": (
            "long",
            (("gid_t *", "rgid"), ("gid_t *", "egid"), ("gid_t *", "sgid")),
        ),
        "getresgid16": (
            "long",
            (
                ("old_gid_t *", "rgid"),
                ("old_gid_t *", "egid"),
                ("old_gid_t *", "sgid"),
            ),
        ),
        "getresuid": (
            "long",
            (("uid_t *", "ruid"), ("uid_t *", "euid"), ("uid_t *", "suid")),
        ),
        "getresuid16": (
            "long",
            (
                ("old_uid_t *", "ruid"),
                ("old_uid_t *", "euid"),
                ("old_uid_t *", "suid"),
            ),
        ),
        "getrlimit": (
            "long",
            (("unsigned int", "resource"), ("struct rlimit *", "rlim")),
        ),
        "ugetrlimit": (
            "long",
            (("unsigned int", "resource"), ("struct rlimit *", "rlim")),
        ),
        "getrusage": ("long", (("int", "who"), ("struct rusage *", "ru"))),
        "getsid": ("long", (("pid_t", "pid"),)),
        "getsockname": (
            "long",
            (
                ("int", "sockfd"),
                ("struct sockaddr *", "addr"),
                ("int *", "addrlen"),
            ),
        ),
        "getsockopt": (
            "long",
            (
                ("int", "fd"),
                ("int", "level"),
                ("int", "optname"),
                ("void *", "optval"),
                ("int *", "optlen"),
            ),
        ),
        "gettid": ("long", ()),
        "gettimeofday": (
            "long",
            (("struct timeval *", "tv"), ("struct timezone *", "tz")),
        ),
        "getuid": ("long", ()),
        "getuid16": ("long", ()),
        "getxattr": (
            "long",
            (
                ("const char *", "path"),
                ("const char *", "name"),
                ("void *", "value"),
                ("size_t", "size"),
            ),
        ),
        "init_module": (
            "long",
            (
                ("void *", "umod"),
                ("unsigned long", "len"),
                ("const char *", "uargs"),
            ),
        ),
        "inotify_add_watch": (
            "long",
            (("int", "fd"), ("const char *", "path"), ("u32", "mask")),
        ),
        "inotify_init": ("long", ()),
        "inotify_init1": ("long", (("int", "flags"),)),
        "inotify_rm_watch": ("long", (("int", "fd"), ("__s32", "wd"))),
        "io_cancel": (
            "long",
            (
                ("aio_context_t", "ctx_id"),
                ("struct iocb *", "iocb"),
                ("struct io_event *", "result"),
            ),
        ),
        "io_destroy": ("long", (("aio_context_t", "ctx"),)),
        "io_getevents": (
            "long",
            (
                ("aio_context_t", "ctx_id"),
                ("long", "min_nr"),
                ("long", "nr"),
                ("struct io_event *", "events"),
                ("struct timespec *", "timeout"),
            ),
        ),
        "io_pgetevents": (
            "long",
            (
                ("aio_context_t", "ctx_id"),
                ("long", "min_nr"),
                ("long", "nr"),
                ("struct io_event *", "events"),
                ("struct timespec *", "timeout"),
                ("const struct __aio_sigset *", "sig"),
            ),
        ),
        "io_setup": (
            "long",
            (("unsigned", "nr_reqs"), ("aio_context_t *", "ctx")),
        ),
        "io_submit": (
            "long",
            (
                ("aio_context_t", "ctx_id"),
                ("long", "nr"),
                ("struct iocb * *", "iocbpp"),
            ),
        ),
        "io_uring_enter": (
            "long",
            (
                ("unsigned int", "fd"),
                ("u32", "to_submit"),
                ("u32", "min_complete"),
                ("u32", "flags"),
                ("const sigset_t *", "sig"),
                ("size_t", "sigsz"),
            ),
        ),
        "io_uring_register": (
            "long",
            (
                ("unsigned int", "fd"),
                ("unsigned int", "op"),
                ("void *", "arg"),
                ("unsigned int", "nr_args"),
            ),
        ),
        "io_uring_setup": (
            "long",
            (("u32", "entries"), ("struct io_uring_params *", "p")),
        ),
        "ioctl": (
            "long",
            (
                ("unsigned int", "fd"),
                ("unsigned int", "cmd"),
                ("unsigned long", "arg"),
            ),
        ),
        "ioperm": (
            "long",
            (
                ("unsigned long", "from"),
                ("unsigned long", "num"),
                ("int", "on"),
            ),
        ),
        "iopl": (
            "long",
            (("int", "level")),
        ),
        "ioprio_get": ("long", (("int", "which"), ("int", "who"))),
        "ioprio_set": (
            "long",
            (("int", "which"), ("int", "who"), ("int", "ioprio")),
        ),
        "ipc": (
            "long",
            (
                ("unsigned int", "call"),
                ("int", "first"),
                ("unsigned long", "second"),
                ("unsigned long", "third"),
                ("void *", "ptr"),
                ("long", "fifth"),
            ),
        ),
        "kcmp": (
            "long",
            (
                ("pid_t", "pid1"),
                ("pid_t", "pid2"),
                ("int", "type"),
                ("unsigned long", "idx1"),
                ("unsigned long", "idx2"),
            ),
        ),
        "kexec_file_load": (
            "long",
            (
                ("int", "kernel_fd"),
                ("int", "initrd_fd"),
                ("unsigned long", "cmdline_len"),
                ("const char *", "cmdline_ptr"),
                ("unsigned long", "flags"),
            ),
        ),
        "kexec_load": (
            "long",
            (
                ("unsigned long", "entry"),
                ("unsigned long", "nr_segments"),
                ("struct kexec_segment *", "segments"),
                ("unsigned long", "flags"),
            ),
        ),
        "keyctl": (
            "long",
            (
                ("int", "cmd"),
                ("unsigned long", "arg2"),
                ("unsigned long", "arg3"),
                ("unsigned long", "arg4"),
                ("unsigned long", "arg5"),
            ),
        ),
        "kill": ("long", (("int", "pid"), ("int", "sig"))),
        "lchown": (
            "long",
            (
                ("const char *", "filename"),
                ("uid_t", "user"),
                ("gid_t", "group"),
            ),
        ),
        "lchown16": (
            "long",
            (
                ("const char *", "filename"),
                ("old_uid_t", "user"),
                ("old_gid_t", "group"),
            ),
        ),
        "lgetxattr": (
            "long",
            (
                ("const char *", "path"),
                ("const char *", "name"),
                ("void *", "value"),
                ("size_t", "size"),
            ),
        ),
        "link": (
            "long",
            (("const char *", "oldname"), ("const char *", "newname")),
        ),
        "linkat": (
            "long",
            (
                ("int", "olddfd"),
                ("const char *", "oldname"),
                ("int", "newdfd"),
                ("const char *", "newname"),
                ("int", "flags"),
            ),
        ),
        "listen": ("long", (("int", "sockfd"), ("int", "backlog"))),
        "listxattr": (
            "long",
            (
                ("const char *", "path"),
                ("char char *", "list"),
                ("size_t", "size"),
            ),
        ),
        "llistxattr": (
            "long",
            (
                ("const char *", "path"),
                ("char char *", "list"),
                ("size_t", "size"),
            ),
        ),
        "llseek": (
            "long",
            (
                ("unsigned int", "fd"),
                ("unsigned long", "offset_high"),
                ("unsigned long", "offset_low"),
                ("loff_t *", "result"),
                ("unsigned int", "whence"),
            ),
        ),
        "lookup_dcookie": (
            "long",
            (("u64", "cookie64"), ("char *", "buf"), ("size_t", "len")),
        ),
        "lremovexattr": (
            "long",
            (("const char *", "path"), ("const char *", "name")),
        ),
        "lseek": (
            "long",
            (
                ("unsigned int", "fd"),
                ("off_t", "offset"),
                ("unsigned int", "whence"),
            ),
        ),
        "lsetxattr": (
            "long",
            (
                ("const char *", "path"),
                ("const char *", "name"),
                ("const void *", "value"),
                ("size_t", "size"),
                ("int", "flags"),
            ),
        ),
        "lstat": (
            "long",
            (
                ("const char *", "filename"),
                ("struct __old_kernel_stat *", "statbuf"),
            ),
        ),
        "lstat64": (
            "long",
            (("const char *", "filename"), ("struct stat64 *", "statbuf")),
        ),
        "madvise": (
            "long",
            (
                ("unsigned long", "addr"),
                ("size_t", "length"),
                ("int", "advice"),
            ),
        ),
        "mbind": (
            "long",
            (
                ("unsigned long", "start"),
                ("unsigned long", "len"),
                ("unsigned long", "mode"),
                ("const unsigned long *", "nmask"),
                ("unsigned long", "maxnode"),
                ("unsigned", "flags"),
            ),
        ),
        "membarrier": ("long", (("int", "cmd"), ("int", "flags"))),
        "memfd_create": (
            "long",
            (("const char *", "uname_ptr"), ("unsigned int", "flags")),
        ),
        "migrate_pages": (
            "long",
            (
                ("pid_t", "pid"),
                ("unsigned long", "maxnode"),
                ("const unsigned long *", "from"),
                ("const unsigned long *", "to"),
            ),
        ),
        "mincore": (
            "long",
            (
                ("unsigned long", "start"),
                ("size_t", "len"),
                ("unsigned char *", "vec"),
            ),
        ),
        "mkdir": ("long", (("const char *", "pathname"), ("umode_t", "mode"))),
        "mkdirat": (
            "long",
            (
                ("int", "dfd"),
                ("const char *", "pathname"),
                ("umode_t", "mode"),
            ),
        ),
        "mknod": (
            "long",
            (
                ("const char *", "filename"),
                ("umode_t", "mode"),
                ("unsigned", "dev"),
            ),
        ),
        "mknodat": (
            "long",
            (
                ("int", "dfd"),
                ("const char *", "filename"),
                ("umode_t", "mode"),
                ("unsigned", "dev"),
            ),
        ),
        "mlock": ("long", (("unsigned long", "start"), ("size_t", "len"))),
        "mlock2": (
            "long",
            (("unsigned long", "start"), ("size_t", "len"), ("int", "flags")),
        ),
        "mlockall": ("long", (("int", "flags"),)),
        "mmap_pgoff": (
            "long",
            (
                ("unsigned long", "addr"),
                ("unsigned long", "len"),
                ("unsigned long", "prot"),
                ("unsigned long", "flags"),
                ("unsigned long", "fd"),
                ("unsigned long", "pgoff"),
            ),
        ),
        "mmap": (
            "long",
            (
                ("unsigned long", "addr"),
                ("unsigned long", "len"),
                ("unsigned long", "prot"),
                ("unsigned long", "flags"),
                ("unsigned long", "fd"),
                ("unsigned long", "pgoff"),
            ),
        ),
        "mmap2": (
            "long",
            (
                ("unsigned long", "addr"),
                ("unsigned long", "len"),
                ("unsigned long", "prot"),
                ("unsigned long", "flags"),
                ("unsigned long", "fd"),
                ("unsigned long", "pgoff"),
            ),
        ),
        "move_mount": (
            "long",
            (
                ("int", "from_dfd"),
                ("const char *", "from_path"),
                ("int", "to_dfd"),
                ("const char *", "to_path"),
                ("unsigned int", "ms_flags"),
            ),
        ),
        "arch_prctl": ("int", (("int", "code"), ("unsigned long", "addr"))),
        "mount": (
            "long",
            (
                ("const char *", "dev_name"),
                ("const char *", "dir_name"),
                ("const char *", "type"),
                ("unsigned long", "flags"),
                ("void *", "data"),
            ),
        ),
        "move_pages": (
            "long",
            (
                ("pid_t", "pid"),
                ("unsigned long", "nr_pages"),
                ("const void * *", "pages"),
                ("const int *", "nodes"),
                ("int *", "status"),
                ("int", "flags"),
            ),
        ),
        "mprotect": (
            "long",
            (
                ("unsigned long", "start"),
                ("size_t", "len"),
                ("unsigned long", "prot"),
            ),
        ),
        "mq_getsetattr": (
            "long",
            (
                ("mqd_t", "mqdes"),
                ("const struct mq_attr *", "mqstat"),
                ("struct mq_attr *", "omqstat"),
            ),
        ),
        "mq_notify": (
            "long",
            (("mqd_t", "mqdes"), ("const struct sigevent *", "notification")),
        ),
        "mq_open": (
            "long",
            (
                ("const char *", "name"),
                ("int", "oflag"),
                ("umode_t", "mode"),
                ("struct mq_attr *", "attr"),
            ),
        ),
        "mq_timedreceive": (
            "long",
            (
                ("mqd_t", "mqdes"),
                ("char *", "msg_ptr"),
                ("size_t", "msg_len"),
                ("unsigned int *", "msg_prio"),
                ("const struct timespec *", "abs_timeout"),
            ),
        ),
        "mq_timedsend": (
            "long",
            (
                ("mqd_t", "mqdes"),
                ("const char *", "msg_ptr"),
                ("size_t", "msg_len"),
                ("unsigned int", "msg_prio"),
                ("const struct timespec *", "abs_timeout"),
            ),
        ),
        "mq_unlink": ("long", (("const char *", "name"),)),
        "mremap": (
            "long",
            (
                ("unsigned long", "addr"),
                ("unsigned long", "old_len"),
                ("unsigned long", "new_len"),
                ("unsigned long", "flags"),
                ("unsigned long", "new_addr"),
            ),
        ),
        "msgctl": (
            "long",
            (("int", "msqid"), ("int", "cmd"), ("struct msqid_ds *", "buf")),
        ),
        "msgget": ("long", (("key_t", "key"), ("int", "msgflg"))),
        "msgrcv": (
            "long",
            (
                ("int", "msqid"),
                ("struct msgbuf *", "msgp"),
                ("size_t", "msgsz"),
                ("long", "msgtyp"),
                ("int", "msgflg"),
            ),
        ),
        "msgsnd": (
            "long",
            (
                ("int", "msqid"),
                ("struct msgbuf *", "msgp"),
                ("size_t", "msgsz"),
                ("int", "msgflg"),
            ),
        ),
        "msync": (
            "long",
            (("unsigned long", "start"), ("size_t", "len"), ("int", "flags")),
        ),
        "munlock": ("long", (("unsigned long", "start"), ("size_t", "len"))),
        "munlockall": ("long", ()),
        "munmap": ("long", (("unsigned long", "addr"), ("size_t", "len"))),
        "name_to_handle_at": (
            "long",
            (
                ("int", "dfd"),
                ("const char *", "name"),
                ("struct file_handle *", "handle"),
                ("int *", "mnt_id"),
                ("int", "flag"),
            ),
        ),
        "nanosleep": (
            "long",
            (("struct timespec *", "rqtp"), ("struct timespec *", "rmtp")),
        ),
        "newfstat": (
            "long",
            (("unsigned int", "fd"), ("struct stat *", "statbuf")),
        ),
        "newfstatat": (
            "long",
            (
                ("int", "dfd"),
                ("const char *", "filename"),
                ("struct stat *", "statbuf"),
                ("int", "flag"),
            ),
        ),
        "newlstat": (
            "long",
            (("const char *", "filename"), ("struct stat *", "statbuf")),
        ),
        "newstat": (
            "long",
            (("const char *", "filename"), ("struct stat *", "statbuf")),
        ),
        "newuname": ("long", (("struct new_utsname *", "name"),)),
        "ni_syscall": ("long", ()),
        "nice": ("long", (("int", "increment"),)),
        "old_getrlimit": (
            "long",
            (("unsigned int", "resource"), ("struct rlimit *", "rlim")),
        ),
        "old_mmap": ("long", (("struct mmap_arg_struct *", "arg"),)),
        "old_msgctl": (
            "long",
            (("int", "msqid"), ("int", "cmd"), ("struct msqid_ds *", "buf")),
        ),
        "old_readdir": (
            "long",
            (
                ("unsigned int", "fd"),
                ("struct old_linux_dirent *", "dirp"),
                ("unsigned int", "count"),
            ),
        ),
        "old_select": ("long", (("struct sel_arg_struct *", "arg"),)),
        "old_semctl": (
            "long",
            (
                ("int", "semid"),
                ("int", "semnum"),
                ("int", "cmd"),
                ("unsigned long", "arg"),
            ),
        ),
        "old_shmctl": (
            "long",
            (("int", "shmid"), ("int", "cmd"), ("struct shmid_ds *", "buf")),
        ),
        "oldumount": ("long", (("char char *", "name"),)),
        "olduname": ("long", (("struct oldold_utsname *", "buf"),)),
        "open": (
            "long",
            (
                ("const char *", "filename"),
                ("int", "flags"),
                ("umode_t", "mode"),
            ),
        ),
        "open_by_handle_at": (
            "long",
            (
                ("int", "mountdirfd"),
                ("struct file_handle *", "handle"),
                ("int", "flags"),
            ),
        ),
        "open_tree": (
            "long",
            (("int", "dfd"), ("const char *", "path"), ("unsigned", "flags")),
        ),
        "openat": (
            "long",
            (
                ("int", "dfd"),
                ("const char *", "filename"),
                ("int", "flags"),
                ("umode_t", "mode"),
            ),
        ),
        "openat2": (
            "long",
            (
                ("int", "dfd"),
                ("const char *", "filename"),
                ("struct open_how *", "how"),
                ("size_t", "size"),
            ),
        ),
        "pause": ("long", ()),
        "pciconfig_iobase": (
            "long",
            (
                ("long", "which"),
                ("unsigned long", "bus"),
                ("unsigned long", "devfn"),
            ),
        ),
        "pciconfig_read": (
            "long",
            (
                ("unsigned long", "bus"),
                ("unsigned long", "dfn"),
                ("unsigned long", "off"),
                ("unsigned long", "len"),
                ("void *", "buf"),
            ),
        ),
        "pciconfig_write": (
            "long",
            (
                ("unsigned long", "bus"),
                ("unsigned long", "dfn"),
                ("unsigned long", "off"),
                ("unsigned long", "len"),
                ("void *", "buf"),
            ),
        ),
        "perf_event_open": (
            "long",
            (
                ("struct perf_event_attr *", "attr_uptr"),
                ("pid_t", "pid"),
                ("int", "cpu"),
                ("int", "group_fd"),
                ("unsigned long", "flags"),
            ),
        ),
        "personality": ("long", (("unsigned int", "personality"),)),
        "pidfd_getfd": (
            "long",
            (("int", "pidfd"), ("int", "fd"), ("unsigned int", "flags")),
        ),
        "pidfd_open": ("long", (("pid_t", "pid"), ("unsigned int", "flags"))),
        "pidfd_send_signal": (
            "long",
            (
                ("int", "pidfd"),
                ("int", "sig"),
                ("siginfo_t *", "info"),
                ("unsigned int", "flags"),
            ),
        ),
        "pipe": ("long", (("int *", "fildes"),)),
        "pipe2": ("long", (("int *", "fildes"), ("int", "flags"))),
        "pivot_root": (
            "long",
            (("const char *", "new_root"), ("const char *", "put_old")),
        ),
        "pkey_alloc": (
            "long",
            (("unsigned long", "flags"), ("unsigned long", "init_val")),
        ),
        "pkey_free": ("long", (("int", "pkey"),)),
        "pkey_mprotect": (
            "long",
            (
                ("unsigned long", "start"),
                ("size_t", "len"),
                ("unsigned long", "prot"),
                ("int", "pkey"),
            ),
        ),
        "poll": (
            "long",
            (
                ("struct pollfd *", "ufds"),
                ("unsigned int", "nfds"),
                ("int", "timeout"),
            ),
        ),
        "ppoll": (
            "long",
            (
                ("struct pollfd *", "fds"),
                ("unsigned int", "nfds"),
                ("struct timespec *", "tmo_p"),
                ("const sigset_t *", "sigmask"),
            ),
        ),
        "prctl": (
            "long",
            (
                ("int", "option"),
                ("unsigned long", "arg2"),
                ("unsigned long", "arg3"),
                ("unsigned long", "arg4"),
                ("unsigned long", "arg5"),
            ),
        ),
        "pread64": (
            "long",
            (
                ("unsigned int", "fd"),
                ("void *", "buf"),
                ("size_t", "count"),
                ("loff_t", "pos"),
            ),
        ),
        "preadv": (
            "long",
            (
                ("unsigned long", "fd"),
                ("const struct iovec *", "vec"),
                ("unsigned long", "vlen"),
                ("unsigned long", "pos_l"),
                ("unsigned long", "pos_h"),
            ),
        ),
        "preadv2": (
            "long",
            (
                ("unsigned long", "fd"),
                ("const struct iovec *", "vec"),
                ("unsigned long", "vlen"),
                ("unsigned long", "pos_l"),
                ("unsigned long", "pos_h"),
                ("rwf_t", "flags"),
            ),
        ),
        "prlimit64": (
            "long",
            (
                ("pid_t", "pid"),
                ("unsigned int", "resource"),
                ("const struct rlimit64 *", "new_rlim"),
                ("struct rlimit64 *", "old_rlim"),
            ),
        ),
        "process_vm_readv": (
            "long",
            (
                ("pid_t", "pid"),
                ("const struct iovec *", "lvec"),
                ("unsigned long", "liovcnt"),
                ("const struct iovec *", "rvec"),
                ("unsigned long", "riovcnt"),
                ("unsigned long", "flags"),
            ),
        ),
        "process_vm_writev": (
            "long",
            (
                ("pid_t", "pid"),
                ("const struct iovec *", "lvec"),
                ("unsigned long", "liovcnt"),
                ("const struct iovec *", "rvec"),
                ("unsigned long", "riovcnt"),
                ("unsigned long", "flags"),
            ),
        ),
        "pselect6": (
            "long",
            (
                ("int", "nfds"),
                ("fd_set *", "readfds"),
                ("fd_set *", "writefds"),
                ("fd_set *", "exceptfds"),
                ("struct timespec *", "timeout"),
                ("void *", "sigmask"),
            ),
        ),
        "ptrace": (
            "long",
            (
                ("long", "request"),
                ("long", "pid"),
                ("unsigned long", "addr"),
                ("unsigned long", "data"),
            ),
        ),
        "pwrite64": (
            "long",
            (
                ("unsigned int", "fd"),
                ("const char *", "buf"),
                ("size_t", "count"),
                ("loff_t", "pos"),
            ),
        ),
        "pwritev": (
            "long",
            (
                ("unsigned long", "fd"),
                ("const struct iovec *", "vec"),
                ("unsigned long", "vlen"),
                ("unsigned long", "pos_l"),
                ("unsigned long", "pos_h"),
            ),
        ),
        "pwritev2": (
            "long",
            (
                ("unsigned long", "fd"),
                ("const struct iovec *", "vec"),
                ("unsigned long", "vlen"),
                ("unsigned long", "pos_l"),
                ("unsigned long", "pos_h"),
                ("rwf_t", "flags"),
            ),
        ),
        "quotactl": (
            "long",
            (
                ("unsigned int", "cmd"),
                ("const char *", "special"),
                ("qid_t", "id"),
                ("void *", "addr"),
            ),
        ),
        "query_module": (
            "long",
            (
                ("const char *", "name"),
                ("int", "which"),
                ("void *", "buf"),
                ("int", "bufsize"),
                ("int *", "ret"),
            ),
        ),
        "read": (
            "long",
            (("unsigned int", "fd"), ("void *", "buf"), ("size_t", "count")),
        ),
        "readahead": (
            "long",
            (("int", "fd"), ("loff_t", "offset"), ("size_t", "count")),
        ),
        "readlink": (
            "long",
            (("const char *", "path"), ("char *", "buf"), ("int", "bufsiz")),
        ),
        "readlinkat": (
            "long",
            (
                ("int", "dfd"),
                ("const char *", "path"),
                ("char *", "buf"),
                ("int", "bufsiz"),
            ),
        ),
        "readv": (
            "long",
            (
                ("unsigned long", "fd"),
                ("const struct iovec *", "vec"),
                ("unsigned long", "vlen"),
            ),
        ),
        "reboot": (
            "long",
            (
                ("int", "magic1"),
                ("int", "magic2"),
                ("unsigned int", "cmd"),
                ("void *", "arg"),
            ),
        ),
        "recv": (
            "long",
            (
                ("int", "sockfd"),
                ("void *", "buf"),
                ("size_t", "len"),
                ("unsigned", "flags"),
            ),
        ),
        "recvfrom": (
            "long",
            (
                ("int", "sockfd"),
                ("void *", "buf"),
                ("size_t", "len"),
                ("unsigned", "flags"),
                ("struct sockaddr *", "src_addr"),
                ("int *", "addrlen"),
            ),
        ),
        "recvmmsg": (
            "long",
            (
                ("int", "fd"),
                ("struct mmsghdr *", "msg"),
                ("unsigned int", "vlen"),
                ("unsigned", "flags"),
                ("struct timespec *", "timeout"),
            ),
        ),
        "recvmsg": (
            "long",
            (
                ("int", "fd"),
                ("struct user_msghdr *", "msg"),
                ("unsigned", "flags"),
            ),
        ),
        "remap_file_pages": (
            "long",
            (
                ("unsigned long", "start"),
                ("unsigned long", "size"),
                ("unsigned long", "prot"),
                ("unsigned long", "pgoff"),
                ("unsigned long", "flags"),
            ),
        ),
        "removexattr": (
            "long",
            (("const char *", "path"), ("const char *", "name")),
        ),
        "rename": (
            "long",
            (("const char *", "oldname"), ("const char *", "newname")),
        ),
        "renameat": (
            "long",
            (
                ("int", "olddfd"),
                ("const char *", "oldname"),
                ("int", "newdfd"),
                ("const char *", "newname"),
            ),
        ),
        "renameat2": (
            "long",
            (
                ("int", "olddfd"),
                ("const char *", "oldname"),
                ("int", "newdfd"),
                ("const char *", "newname"),
                ("unsigned int", "flags"),
            ),
        ),
        "request_key": (
            "long",
            (
                ("const char *", "_type"),
                ("const char *", "_description"),
                ("const char *", "_callout_info"),
                ("key_serial_t", "destringid"),
            ),
        ),
        "restart_syscall": ("long", ()),
        "rmdir": ("long", (("const char *", "pathname"),)),
        "rseq": (
            "long",
            (
                ("struct rseq *", "rseq"),
                ("uint32_t", "rseq_len"),
                ("int", "flags"),
                ("uint32_t", "sig"),
            ),
        ),
        "rt_sigaction": (
            "long",
            (
                ("int", "signum"),
                ("const struct sigaction *", "act"),
                ("struct sigaction *", "oldact"),
                ("size_t", ""),
            ),
        ),
        "rt_sigpending": (
            "long",
            (("sigset_t *", "set"), ("size_t", "sigsetsize")),
        ),
        "rt_sigreturn": ("long", ()),
        "rt_sigprocmask": (
            "long",
            (
                ("int", "how"),
                ("sigset_t *", "set"),
                ("sigset_t *", "oset"),
                ("size_t", "sigsetsize"),
            ),
        ),
        "rt_sigqueueinfo": (
            "long",
            (("int", "pid"), ("int", "sig"), ("siginfo_t *", "uinfo")),
        ),
        "rt_sigsuspend": (
            "long",
            (("sigset_t *", "unewset"), ("size_t", "sigsetsize")),
        ),
        "rt_sigtimedwait": (
            "long",
            (
                ("const sigset_t *", "uthese"),
                ("siginfo_t *", "uinfo"),
                ("const struct timespec *", "uts"),
                ("size_t", "sigsetsize"),
            ),
        ),
        "rt_tgsigqueueinfo": (
            "long",
            (
                ("pid_t", "tgid"),
                ("pid_t", "pid"),
                ("int", "sig"),
                ("siginfo_t *", "uinfo"),
            ),
        ),
        "sched_get_priority_max": ("long", (("int", "policy"),)),
        "sched_get_priority_min": ("long", (("int", "policy"),)),
        "sched_getaffinity": (
            "long",
            (
                ("pid_t", "pid"),
                ("unsigned int", "len"),
                ("unsigned long *", "user_mask_ptr"),
            ),
        ),
        "sched_getattr": (
            "long",
            (
                ("pid_t", "pid"),
                ("struct sched_attr *", "attr"),
                ("unsigned int", "size"),
                ("unsigned int", "flags"),
            ),
        ),
        "sched_getparam": (
            "long",
            (("pid_t", "pid"), ("struct sched_param *", "param")),
        ),
        "sched_getscheduler": ("long", (("pid_t", "pid"),)),
        "sched_rr_get_interval": (
            "long",
            (("pid_t", "pid"), ("struct timespec *", "interval")),
        ),
        "sched_setaffinity": (
            "long",
            (
                ("pid_t", "pid"),
                ("unsigned int", "len"),
                ("unsigned long *", "user_mask_ptr"),
            ),
        ),
        "sched_setattr": (
            "long",
            (
                ("pid_t", "pid"),
                ("struct sched_attr *", "attr"),
                ("unsigned int", "flags"),
            ),
        ),
        "sched_setparam": (
            "long",
            (("pid_t", "pid"), ("struct sched_param *", "param")),
        ),
        "sched_setscheduler": (
            "long",
            (
                ("pid_t", "pid"),
                ("int", "policy"),
                ("struct sched_param *", "param"),
            ),
        ),
        "sched_yield": ("long", ()),
        "seccomp": (
            "long",
            (
                ("unsigned int", "op"),
                ("unsigned int", "flags"),
                ("const char *", "uargs"),
            ),
        ),
        "select": (
            "long",
            (
                ("int", "n"),
                ("fd_set *", "inp"),
                ("fd_set *", "outp"),
                ("fd_set *", "exp"),
                ("struct timeval *", "tvp"),
            ),
        ),
        "newselect": (
            "long",
            (
                ("int", "n"),
                ("fd_set *", "inp"),
                ("fd_set *", "outp"),
                ("fd_set *", "exp"),
                ("struct timeval *", "tvp"),
            ),
        ),
        "semctl": (
            "long",
            (
                ("int", "semid"),
                ("int", "semnum"),
                ("int", "cmd"),
                ("unsigned long", "arg"),
            ),
        ),
        "semget": (
            "long",
            (("key_t", "key"), ("int", "nsems"), ("int", "semflg")),
        ),
        "semop": (
            "long",
            (
                ("int", "semid"),
                ("struct sembuf *", "sops"),
                ("unsigned", "nsops"),
            ),
        ),
        "semtimedop": (
            "long",
            (
                ("int", "semid"),
                ("struct sembuf *", "sops"),
                ("unsigned", "nsops"),
                ("const struct timespec *", "timeout"),
            ),
        ),
        "send": (
            "long",
            (
                ("int", "sockfd"),
                ("void *", "buf"),
                ("size_t", "len"),
                ("unsigned", "flags"),
            ),
        ),
        "sendfile": (
            "long",
            (
                ("int", "out_fd"),
                ("int", "in_fd"),
                ("off_t *", "offset"),
                ("size_t", "count"),
            ),
        ),
        "sendfile64": (
            "long",
            (
                ("int", "out_fd"),
                ("int", "in_fd"),
                ("loff_t *", "offset"),
                ("size_t", "count"),
            ),
        ),
        "sendmmsg": (
            "long",
            (
                ("int", "fd"),
                ("struct mmsghdr *", "msg"),
                ("unsigned int", "vlen"),
                ("unsigned", "flags"),
            ),
        ),
        "sendmsg": (
            "long",
            (
                ("int", "fd"),
                ("struct user_msghdr *", "msg"),
                ("unsigned", "flags"),
            ),
        ),
        "sendto": (
            "long",
            (
                ("int", "sockfd"),
                ("void *", "buf"),
                ("size_t", "len"),
                ("unsigned", "flags"),
                ("struct sockaddr *", "destAddr"),
                ("int", "addrlen"),
            ),
        ),
        "set_mempolicy": (
            "long",
            (
                ("int", "mode"),
                ("const unsigned long *", "nmask"),
                ("unsigned long", "maxnode"),
            ),
        ),
        "set_robust_list": (
            "long",
            (("struct robust_list_head *", "head"), ("size_t", "len")),
        ),
        "set_tid_address": ("long", (("int *", "tidptr"),)),
        "setdomainname": ("long", (("const char *", "name"), ("int", "len"))),
        "setfsgid": ("long", (("gid_t", "gid"),)),
        "setfsgid16": ("long", (("old_gid_t", "gid"),)),
        "setfsuid": ("long", (("uid_t", "uid"),)),
        "setfsuid16": ("long", (("old_uid_t", "uid"),)),
        "setgid": ("long", (("gid_t", "gid"),)),
        "setgid16": ("long", (("old_gid_t", "gid"),)),
        "setgroups": (
            "long",
            (("int", "gidsetsize"), ("gid_t *", "grouplist")),
        ),
        "setgroups16": (
            "long",
            (("int", "gidsetsize"), ("old_gid_t *", "grouplist")),
        ),
        "sethostname": ("long", (("const char *", "name"), ("int", "len"))),
        "setitimer": (
            "long",
            (
                ("int", "which"),
                ("struct itimerval *", "value"),
                ("struct itimerval *", "ovalue"),
            ),
        ),
        "setns": ("long", (("int", "fd"), ("int", "nstype"))),
        "setpgid": ("long", (("pid_t", "pid"), ("pid_t", "pgid"))),
        "setpriority": (
            "long",
            (("int", "which"), ("int", "who"), ("int", "niceval")),
        ),
        "setregid": ("long", (("gid_t", "rgid"), ("gid_t", "egid"))),
        "setregid16": ("long", (("old_gid_t", "rgid"), ("old_gid_t", "egid"))),
        "setresgid": (
            "long",
            (("gid_t", "rgid"), ("gid_t", "egid"), ("gid_t", "sgid")),
        ),
        "setresgid16": (
            "long",
            (
                ("old_gid_t", "rgid"),
                ("old_gid_t", "egid"),
                ("old_gid_t", "sgid"),
            ),
        ),
        "setresuid": (
            "long",
            (("uid_t", "ruid"), ("uid_t", "euid"), ("uid_t", "suid")),
        ),
        "setresuid16": (
            "long",
            (
                ("old_uid_t", "ruid"),
                ("old_uid_t", "euid"),
                ("old_uid_t", "suid"),
            ),
        ),
        "setreuid": ("long", (("uid_t", "ruid"), ("uid_t", "euid"))),
        "setreuid16": ("long", (("old_uid_t", "ruid"), ("old_uid_t", "euid"))),
        "setrlimit": (
            "long",
            (("unsigned int", "resource"), ("struct rlimit *", "rlim")),
        ),
        "setsid": ("long", ()),
        "setsockopt": (
            "long",
            (
                ("int", "fd"),
                ("int", "level"),
                ("int", "optname"),
                ("const void *", "optval"),
                ("int", "optlen"),
            ),
        ),
        "settimeofday": (
            "long",
            (("struct timeval *", "tv"), ("struct timezone *", "tz")),
        ),
        "setuid": ("long", (("uid_t", "uid"),)),
        "setuid16": ("long", (("old_uid_t", "uid"),)),
        "setxattr": (
            "long",
            (
                ("const char *", "path"),
                ("const char *", "name"),
                ("const void *", "value"),
                ("size_t", "size"),
                ("int", "flags"),
            ),
        ),
        "sgetmask": ("long", ()),
        "shmat": (
            "long",
            (("int", "shmid"), ("const void *", "shmaddr"), ("int", "shmflg")),
        ),
        "shmctl": (
            "long",
            (("int", "shmid"), ("int", "cmd"), ("struct shmid_ds *", "buf")),
        ),
        "shmdt": ("long", (("const void *", "shmaddr"),)),
        "shmget": (
            "long",
            (("key_t", "key"), ("size_t", "size"), ("int", "flag")),
        ),
        "shutdown": ("long", (("int", "sockfd"), ("int", "how"))),
        "sigaction": (
            "long",
            (
                ("int", "signum"),
                ("const struct old_sigaction *", "act"),
                ("struct old_sigaction *", "oldact"),
            ),
        ),
        "sigaltstack": (
            "long",
            (
                ("const struct sigaltstack *", "uss"),
                ("struct sigaltstack *", "uoss"),
            ),
        ),
        "signal": ("long", (("int", "sig"), ("__sighandler_t", "handler"))),
        "signalfd": (
            "long",
            (
                ("int", "ufd"),
                ("sigset_t *", "user_mask"),
                ("size_t", "sizemask"),
            ),
        ),
        "signalfd4": (
            "long",
            (
                ("int", "ufd"),
                ("sigset_t *", "user_mask"),
                ("size_t", "sizemask"),
                ("int", "flags"),
            ),
        ),
        "sigpending": ("long", (("old_sigset_t *", "set"),)),
        "sigprocmask": (
            "long",
            (
                ("int", "how"),
                ("old_sigset_t *", "set"),
                ("old_sigset_t *", "oset"),
            ),
        ),
        "sigsuspend": (
            "long",
            (("int", "unused1"), ("int", "unused2"), ("old_sigset_t", "mask")),
        ),
        "socket": (
            "long",
            (("int", "domain"), ("int", "type"), ("int", "protocol")),
        ),
        "socketcall": ("long", (("int", "call"), ("unsigned long *", "args"))),
        "socketpair": (
            "long",
            (
                ("int", "domain"),
                ("int", "type"),
                ("int", "protocol"),
                ("int *", "sv"),
            ),
        ),
        "splice": (
            "long",
            (
                ("int", "fd_in"),
                ("loff_t *", "off_in"),
                ("int", "fd_out"),
                ("loff_t *", "off_out"),
                ("size_t", "len"),
                ("unsigned int", "flags"),
            ),
        ),
        "spu_create": (
            "long",
            (
                ("const char *", "name"),
                ("unsigned int", "flags"),
                ("umode_t", "mode"),
                ("int", "fd"),
            ),
        ),
        "spu_run": (
            "long",
            (("int", "fd"), ("__u32 *", "unpc"), ("__u32 *", "ustatus")),
        ),
        "ssetmask": ("long", (("int", "newmask"),)),
        "stat": (
            "long",
            (
                ("const char *", "filename"),
                ("struct __old_kernel_stat *", "statbuf"),
            ),
        ),
        "stat64": (
            "long",
            (("const char *", "filename"), ("struct stat64 *", "statbuf")),
        ),
        "statfs": (
            "long",
            (("const char *", "path"), ("struct statfs *", "buf")),
        ),
        "statfs64": (
            "long",
            (
                ("const char *", "path"),
                ("size_t", "sz"),
                ("struct statfs64 *", "buf"),
            ),
        ),
        "statx": (
            "long",
            (
                ("int", "dfd"),
                ("const char *", "path"),
                ("unsigned", "flags"),
                ("unsigned", "mask"),
                ("struct statx *", "statxbuf"),
            ),
        ),
        "stime": ("long", (("old_time_t *", "tptr"),)),
        "swapoff": ("long", (("const char *", "specialfile"),)),
        "swapon": (
            "long",
            (("const char *", "specialfile"), ("int", "swap_flags")),
        ),
        "symlink": (
            "long",
            (("const char *", "old"), ("const char *", "new")),
        ),
        "symlinkat": (
            "long",
            (
                ("const char *", "oldname"),
                ("int", "newdfd"),
                ("const char *", "newname"),
            ),
        ),
        "sync": ("long", ()),
        "sync_file_range": (
            "long",
            (
                ("int", "fd"),
                ("loff_t", "offset"),
                ("loff_t", "nbytes"),
                ("unsigned int", "flags"),
            ),
        ),
        "sync_file_range2": (
            "long",
            (
                ("int", "fd"),
                ("unsigned int", "flags"),
                ("loff_t", "offset"),
                ("loff_t", "nbytes"),
            ),
        ),
        "syncfs": ("long", (("int", "fd"),)),
        "sysctl": ("long", (("struct __sysctl_args *", "args"),)),
        "sysfs": (
            "long",
            (
                ("int", "option"),
                ("unsigned long", "arg1"),
                ("unsigned long", "arg2"),
            ),
        ),
        "sysinfo": ("long", (("struct sysinfo *", "info"),)),
        "syslog": (
            "long",
            (("int", "type"), ("char *", "buf"), ("int", "len")),
        ),
        "tee": (
            "long",
            (
                ("int", "fdin"),
                ("int", "fdout"),
                ("size_t", "len"),
                ("unsigned int", "flags"),
            ),
        ),
        "tgkill": ("long", (("int", "tgid"), ("int", "pid"), ("int", "sig"))),
        "time": ("long", (("old_time_t *", "tloc"),)),
        "timer_create": (
            "long",
            (
                ("clockid_t", "which_clock"),
                ("struct sigevent *", "timer_event_spec"),
                ("timer_t *", "created_timer_id"),
            ),
        ),
        "timer_delete": ("long", (("timer_t", "timer_id"),)),
        "timer_getoverrun": ("long", (("timer_t", "timer_id"),)),
        "timer_gettime": (
            "long",
            (("timer_t", "timer_id"), ("struct itimerspec *", "setting")),
        ),
        "timer_settime": (
            "long",
            (
                ("timer_t", "timer_id"),
                ("int", "flags"),
                ("const struct itimerspec *", "new_setting"),
                ("struct itimerspec *", "old_setting"),
            ),
        ),
        "timerfd_create": ("long", (("int", "clockid"), ("int", "flags"))),
        "timerfd_gettime": (
            "long",
            (("int", "ufd"), ("struct itimerspec *", "otmr")),
        ),
        "timerfd_settime": (
            "long",
            (
                ("int", "ufd"),
                ("int", "flags"),
                ("const struct itimerspec *", "utmr"),
                ("struct itimerspec *", "otmr"),
            ),
        ),
        "times": ("long", (("struct tms *", "tbuf"),)),
        "tkill": ("long", (("int", "pid"), ("int", "sig"))),
        "truncate": ("long", (("const char *", "path"), ("long", "length"))),
        "truncate64": (
            "long",
            (("const char *", "path"), ("loff_t", "length")),
        ),
        "umask": ("long", (("int", "mask"),)),
        "umount": ("long", (("const char *", "name"), ("int", "flags"))),
        "umount2": ("long", (("const char *", "name"), ("int", "flags"))),
        "uname": ("long", (("struct old_utsname *", "buf"),)),
        "unlink": ("long", (("const char *", "pathname"),)),
        "unlinkat": (
            "long",
            (("int", "dfd"), ("const char *", "pathname"), ("int", "flag")),
        ),
        "unshare": ("long", (("unsigned long", "unshare_flags"),)),
        "uselib": ("long", (("const char *", "library"),)),
        "userfaultfd": ("long", (("int", "flags"),)),
        "ustat": ("long", (("unsigned", "dev"), ("struct ustat *", "ubuf"))),
        "utime": (
            "long",
            (("const char *", "filename"), ("struct utimbuf *", "times")),
        ),
        "utimensat": (
            "long",
            (
                ("int", "dfd"),
                ("const char *", "filename"),
                ("struct timespec *", "utimes"),
                ("int", "flags"),
            ),
        ),
        "utimes": (
            "long",
            (("const char *", "filename"), ("struct old_timeval *", "utimes")),
        ),
        "vfork": ("long", ()),
        "vhangup": ("long", ()),
        "vmsplice": (
            "long",
            (
                ("int", "fd"),
                ("const struct iovec *", "iov"),
                ("unsigned long", "nr_segs"),
                ("unsigned int", "flags"),
            ),
        ),
        "wait4": (
            "long",
            (
                ("pid_t", "pid"),
                ("int *", "stat_addr"),
                ("int", "options"),
                ("struct rusage *", "ru"),
            ),
        ),
        "waitid": (
            "long",
            (
                ("int", "which"),
                ("pid_t", "pid"),
                ("struct siginfo *", "infop"),
                ("int", "options"),
                ("struct rusage *", "ru"),
            ),
        ),
        "waitpid": (
            "long",
            (("pid_t", "pid"), ("int *", "stat_addr"), ("int", "options")),
        ),
        "write": (
            "long",
            (
                ("unsigned int", "fd"),
                ("const char *", "buf"),
                ("size_t", "count"),
            ),
        ),
        "writev": (
            "long",
            (
                ("unsigned long", "fd"),
                ("const struct iovec *", "vec"),
                ("unsigned long", "vlen"),
            ),
        ),
    }

    """
    update syscalls from https://github.com/strace/src/linux/ARCH/syscallent.h
    1. %s/\[.*= //g
    2. %s/{.*),\s*"/\'sys_/g
    3. %s/".*,/\',/g
    4. paste the processed list
    4. call UtilMgr.printSyscalls(ConfigMgr.SYSCALL_X86)
    """

    # common 32bit syscalls from 403 ~ 423 #
    SYSCALL_COMMON32 = [
        "sys_clock_gettime64",
        "sys_clock_settime64",
        "sys_clock_adjtime64",
        "sys_clock_getres_time64",
        "sys_clock_nanosleep_time64",
        "sys_timer_gettime64",
        "sys_timer_settime64",
        "sys_timerfd_gettime64",
        "sys_timerfd_settime64",
        "sys_utimensat_time64",
        "sys_pselect6_time64",
        "sys_ppoll_time64",
        "sys_unused",
        "sys_io_pgetevents_time64",
        "sys_recvmmsg_time64",
        "sys_mq_timedsend_time64",
        "sys_mq_timedreceive_time64",
        "sys_semtimedop_time64",
        "sys_rt_sigtimedwait_time64",
        "sys_futex_time64",
        "sys_sched_rr_get_interval_time64",
    ]

    # common syscalls from 424 ~ 450 #
    SYSCALL_COMMON = [
        "sys_pidfd_send_signal",
        "sys_io_uring_setup",
        "sys_io_uring_enter",
        "sys_io_uring_register",
        "sys_open_tree",
        "sys_move_mount",
        "sys_fsopen",
        "sys_fsconfig",
        "sys_fsmount",
        "sys_fspick",
        "sys_pidfd_open",
        "sys_clone3",
        "sys_close_range",
        "sys_openat2",
        "sys_pidfd_getfd",
        "sys_faccessat2",
        "sys_process_madvise",
        "sys_epoll_pwait2",
        "sys_mount_setattr",
        "sys_quotactl_fd",
        "sys_landlock_create_ruleset",
        "sys_landlock_add_rule",
        "sys_landlock_restrict_self",
        "sys_memfd_secret",
        "sys_process_mrelease",
        "sys_futex_waitv",
        "sys_set_mempolicy_home_node",
    ]

    # syscall for ARM #
    SYSCALL_ARM = (
        [
            "sys_restart_syscall",
            "sys_exit",
            "sys_fork",
            "sys_read",
            "sys_write",
            "sys_open",
            "sys_close",
            "sys_waitpid",
            "sys_creat",
            "sys_link",
            "sys_unlink",
            "sys_execve",
            "sys_chdir",
            "sys_time",
            "sys_mknod",
            "sys_chmod",
            "sys_lchown",
            "sys_break",
            "sys_oldstat",
            "sys_lseek",
            "sys_getpid",
            "sys_mount",
            "sys_umount",
            "sys_setuid",
            "sys_getuid",
            "sys_stime",
            "sys_ptrace",
            "sys_alarm",
            "sys_oldfstat",
            "sys_pause",
            "sys_utime",
            "sys_stty",
            "sys_gtty",
            "sys_access",
            "sys_nice",
            "sys_ftime",
            "sys_sync",
            "sys_kill",
            "sys_rename",
            "sys_mkdir",
            "sys_rmdir",
            "sys_dup",
            "sys_pipe",
            "sys_times",
            "sys_prof",
            "sys_brk",
            "sys_setgid",
            "sys_getgid",
            "sys_signal",
            "sys_geteuid",
            "sys_getegid",
            "sys_acct",
            "sys_umount2",
            "sys_lock",
            "sys_ioctl",
            "sys_fcntl",
            "sys_mpx",
            "sys_setpgid",
            "sys_ulimit",
            "sys_oldolduname",
            "sys_umask",
            "sys_chroot",
            "sys_ustat",
            "sys_dup2",
            "sys_getppid",
            "sys_getpgrp",
            "sys_setsid",
            "sys_sigaction",
            "sys_sgetmask",
            "sys_ssetmask",
            "sys_setreuid",
            "sys_setregid",
            "sys_sigsuspend",
            "sys_sigpending",
            "sys_sethostname",
            "sys_setrlimit",
            "sys_getrlimit",
            "sys_getrusage",
            "sys_gettimeofday",
            "sys_settimeofday",
            "sys_getgroups",
            "sys_setgroups",
            "sys_select",
            "sys_symlink",
            "sys_oldlstat",
            "sys_readlink",
            "sys_uselib",
            "sys_swapon",
            "sys_reboot",
            "sys_readdir",
            "sys_mmap",
            "sys_munmap",
            "sys_truncate",
            "sys_ftruncate",
            "sys_fchmod",
            "sys_fchown",
            "sys_getpriority",
            "sys_setpriority",
            "sys_profil",
            "sys_statfs",
            "sys_fstatfs",
            "sys_ioperm",
            "sys_socketcall",
            "sys_syslog",
            "sys_setitimer",
            "sys_getitimer",
            "sys_stat",
            "sys_lstat",
            "sys_fstat",
            "sys_olduname",
            "sys_iopl",
            "sys_vhangup",
            "sys_idle",
            "sys_syscall",
            "sys_wait4",
            "sys_swapoff",
            "sys_sysinfo",
            "sys_ipc",
            "sys_fsync",
            "sys_sigreturn",
            "sys_clone",
            "sys_setdomainname",
            "sys_uname",
            "sys_modify_ldt",
            "sys_adjtimex",
            "sys_mprotect",
            "sys_sigprocmask",
            "sys_create_module",
            "sys_init_module",
            "sys_delete_module",
            "sys_get_kernel_syms",
            "sys_quotactl",
            "sys_getpgid",
            "sys_fchdir",
            "sys_bdflush",
            "sys_sysfs",
            "sys_personality",
            "sys_afs_syscall",
            "sys_setfsuid",
            "sys_setfsgid",
            "sys_llseek",
            "sys_getdents",
            "sys_newselect",
            "sys_flock",
            "sys_msync",
            "sys_readv",
            "sys_writev",
            "sys_getsid",
            "sys_fdatasync",
            "sys_sysctl",
            "sys_mlock",
            "sys_munlock",
            "sys_mlockall",
            "sys_munlockall",
            "sys_sched_setparam",
            "sys_sched_getparam",
            "sys_sched_setscheduler",
            "sys_sched_getscheduler",
            "sys_sched_yield",
            "sys_sched_get_priority_max",
            "sys_sched_get_priority_min",
            "sys_sched_rr_get_interval",
            "sys_nanosleep",
            "sys_mremap",
            "sys_setresuid",
            "sys_getresuid",
            "sys_vm86",
            "sys_query_module",
            "sys_poll",
            "sys_nfsservctl",
            "sys_setresgid",
            "sys_getresgid",
            "sys_prctl",
            "sys_rt_sigreturn",
            "sys_rt_sigaction",
            "sys_rt_sigprocmask",
            "sys_rt_sigpending",
            "sys_rt_sigtimedwait",
            "sys_rt_sigqueueinfo",
            "sys_rt_sigsuspend",
            "sys_pread64",
            "sys_pwrite64",
            "sys_chown",
            "sys_getcwd",
            "sys_capget",
            "sys_capset",
            "sys_sigaltstack",
            "sys_sendfile",
            "sys_getpmsg",
            "sys_putpmsg",
            "sys_vfork",
            "sys_ugetrlimit",
            "sys_mmap2",
            "sys_truncate64",
            "sys_ftruncate64",
            "sys_stat64",
            "sys_lstat64",
            "sys_fstat64",
            "sys_lchown32",
            "sys_getuid32",
            "sys_getgid32",
            "sys_geteuid32",
            "sys_getegid32",
            "sys_setreuid32",
            "sys_setregid32",
            "sys_getgroups32",
            "sys_setgroups32",
            "sys_fchown32",
            "sys_setresuid32",
            "sys_getresuid32",
            "sys_setresgid32",
            "sys_getresgid32",
            "sys_chown32",
            "sys_setuid32",
            "sys_setgid32",
            "sys_setfsuid32",
            "sys_setfsgid32",
            "sys_getdents64",
            "sys_pivot_root",
            "sys_mincore",
            "sys_madvise",
            "sys_fcntl64",
            "sys_tux",
            "sys_unused",
            "sys_gettid",
            "sys_readahead",
            "sys_setxattr",
            "sys_lsetxattr",
            "sys_fsetxattr",
            "sys_getxattr",
            "sys_lgetxattr",
            "sys_fgetxattr",
            "sys_listxattr",
            "sys_llistxattr",
            "sys_flistxattr",
            "sys_removexattr",
            "sys_lremovexattr",
            "sys_fremovexattr",
            "sys_tkill",
            "sys_sendfile64",
            "sys_futex",
            "sys_sched_setaffinity",
            "sys_sched_getaffinity",
            "sys_io_setup",
            "sys_io_destroy",
            "sys_io_getevents",
            "sys_io_submit",
            "sys_io_cancel",
            "sys_exit_group",
            "sys_lookup_dcookie",
            "sys_epoll_create",
            "sys_epoll_ctl",
            "sys_epoll_wait",
            "sys_remap_file_pages",
            "sys_set_thread_area",
            "sys_get_thread_area",
            "sys_set_tid_address",
            "sys_timer_create",
            "sys_timer_settime",
            "sys_timer_gettime",
            "sys_timer_getoverrun",
            "sys_timer_delete",
            "sys_clock_settime",
            "sys_clock_gettime",
            "sys_clock_getres",
            "sys_clock_nanosleep",
            "sys_statfs64",
            "sys_fstatfs64",
            "sys_tgkill",
            "sys_utimes",
            "sys_fadvise64_64",
            "sys_pciconfig_iobase",
            "sys_pciconfig_read",
            "sys_pciconfig_write",
            "sys_mq_open",
            "sys_mq_unlink",
            "sys_mq_timedsend",
            "sys_mq_timedreceive",
            "sys_mq_notify",
            "sys_mq_getsetattr",
            "sys_waitid",
            "sys_socket",
            "sys_bind",
            "sys_connect",
            "sys_listen",
            "sys_accept",
            "sys_getsockname",
            "sys_getpeername",
            "sys_socketpair",
            "sys_send",
            "sys_sendto",
            "sys_recv",
            "sys_recvfrom",
            "sys_shutdown",
            "sys_setsockopt",
            "sys_getsockopt",
            "sys_sendmsg",
            "sys_recvmsg",
            "sys_semop",
            "sys_semget",
            "sys_semctl",
            "sys_msgsnd",
            "sys_msgrcv",
            "sys_msgget",
            "sys_msgctl",
            "sys_shmat",
            "sys_shmdt",
            "sys_shmget",
            "sys_shmctl",
            "sys_add_key",
            "sys_request_key",
            "sys_keyctl",
            "sys_semtimedop",
            "sys_vserver",
            "sys_ioprio_set",
            "sys_ioprio_get",
            "sys_inotify_init",
            "sys_inotify_add_watch",
            "sys_inotify_rm_watch",
            "sys_mbind",
            "sys_get_mempolicy",
            "sys_set_mempolicy",
            "sys_openat",
            "sys_mkdirat",
            "sys_mknodat",
            "sys_fchownat",
            "sys_futimesat",
            "sys_fstatat64",
            "sys_unlinkat",
            "sys_renameat",
            "sys_linkat",
            "sys_symlinkat",
            "sys_readlinkat",
            "sys_fchmodat",
            "sys_faccessat",
            "sys_pselect6",
            "sys_ppoll",
            "sys_unshare",
            "sys_set_robust_list",
            "sys_get_robust_list",
            "sys_splice",
            "sys_sync_file_range2",
            "sys_tee",
            "sys_vmsplice",
            "sys_move_pages",
            "sys_getcpu",
            "sys_epoll_pwait",
            "sys_kexec_load",
            "sys_utimensat",
            "sys_signalfd",
            "sys_timerfd_create",
            "sys_eventfd",
            "sys_fallocate",
            "sys_timerfd_settime",
            "sys_timerfd_gettime",
            "sys_signalfd4",
            "sys_eventfd2",
            "sys_epoll_create1",
            "sys_dup3",
            "sys_pipe2",
            "sys_inotify_init1",
            "sys_preadv",
            "sys_pwritev",
            "sys_rt_tgsigqueueinfo",
            "sys_perf_event_open",
            "sys_recvmmsg",
            "sys_accept4",
            "sys_fanotify_init",
            "sys_fanotify_mark",
            "sys_prlimit64",
            "sys_name_to_handle_at",
            "sys_open_by_handle_at",
            "sys_clock_adjtime",
            "sys_syncfs",
            "sys_sendmmsg",
            "sys_setns",
            "sys_process_vm_readv",
            "sys_process_vm_writev",
            "sys_kcmp",
            "sys_finit_module",
            "sys_sched_setattr",
            "sys_sched_getattr",
            "sys_renameat2",
            "sys_seccomp",
            "sys_getrandom",
            "sys_memfd_create",
            "sys_bpf",
            "sys_execveat",
            "sys_userfaultfd",
            "sys_membarrier",
            "sys_mlock2",
            "sys_copy_file_range",
            "sys_preadv2",
            "sys_pwritev2",
            "sys_pkey_mprotect",
            "sys_pkey_alloc",
            "sys_pkey_free",
            "sys_statx",
            "sys_rseq",
            "sys_io_pgetevents",
            "sys_migrate_pages",
            "sys_kexec_file_load",
            "sys_unused",
        ]
        + SYSCALL_COMMON32
        + SYSCALL_COMMON
    )

    # syscall for AARCH64 #
    SYSCALL_AARCH64 = (
        [
            "sys_io_setup",
            "sys_io_destroy",
            "sys_io_submit",
            "sys_io_cancel",
            "sys_io_getevents",
            "sys_setxattr",
            "sys_lsetxattr",
            "sys_fsetxattr",
            "sys_getxattr",
            "sys_lgetxattr",
            "sys_fgetxattr",
            "sys_listxattr",
            "sys_llistxattr",
            "sys_flistxattr",
            "sys_removexattr",
            "sys_lremovexattr",
            "sys_fremovexattr",
            "sys_getcwd",
            "sys_lookup_dcookie",
            "sys_eventfd2",
            "sys_epoll_create1",
            "sys_epoll_ctl",
            "sys_epoll_pwait",
            "sys_dup",
            "sys_dup3",
            "sys_fcntl",
            "sys_inotify_init1",
            "sys_inotify_add_watch",
            "sys_inotify_rm_watch",
            "sys_ioctl",
            "sys_ioprio_set",
            "sys_ioprio_get",
            "sys_flock",
            "sys_mknodat",
            "sys_mkdirat",
            "sys_unlinkat",
            "sys_symlinkat",
            "sys_linkat",
            "sys_renameat",
            "sys_umount2",
            "sys_mount",
            "sys_pivot_root",
            "sys_nfsservctl",
            "sys_statfs",
            "sys_fstatfs",
            "sys_truncate",
            "sys_ftruncate",
            "sys_fallocate",
            "sys_faccessat",
            "sys_chdir",
            "sys_fchdir",
            "sys_chroot",
            "sys_fchmod",
            "sys_fchmodat",
            "sys_fchownat",
            "sys_fchown",
            "sys_openat",
            "sys_close",
            "sys_vhangup",
            "sys_pipe2",
            "sys_quotactl",
            "sys_getdents64",
            "sys_lseek",
            "sys_read",
            "sys_write",
            "sys_readv",
            "sys_writev",
            "sys_pread64",
            "sys_pwrite64",
            "sys_preadv",
            "sys_pwritev",
            "sys_sendfile",
            "sys_pselect6",
            "sys_ppoll",
            "sys_signalfd4",
            "sys_vmsplice",
            "sys_splice",
            "sys_tee",
            "sys_readlinkat",
            "sys_newfstatat",
            "sys_fstat",
            "sys_sync",
            "sys_fsync",
            "sys_fdatasync",
            "sys_sync_file_range",
            "sys_timerfd_create",
            "sys_timerfd_settime",
            "sys_timerfd_gettime",
            "sys_utimensat",
            "sys_acct",
            "sys_capget",
            "sys_capset",
            "sys_personality",
            "sys_exit",
            "sys_exit_group",
            "sys_waitid",
            "sys_set_tid_address",
            "sys_unshare",
            "sys_futex",
            "sys_set_robust_list",
            "sys_get_robust_list",
            "sys_nanosleep",
            "sys_getitimer",
            "sys_setitimer",
            "sys_kexec_load",
            "sys_init_module",
            "sys_delete_module",
            "sys_timer_create",
            "sys_timer_gettime",
            "sys_timer_getoverrun",
            "sys_timer_settime",
            "sys_timer_delete",
            "sys_clock_settime",
            "sys_clock_gettime",
            "sys_clock_getres",
            "sys_clock_nanosleep",
            "sys_syslog",
            "sys_ptrace",
            "sys_sched_setparam",
            "sys_sched_setscheduler",
            "sys_sched_getscheduler",
            "sys_sched_getparam",
            "sys_sched_setaffinity",
            "sys_sched_getaffinity",
            "sys_sched_yield",
            "sys_sched_get_priority_max",
            "sys_sched_get_priority_min",
            "sys_sched_rr_get_interval",
            "sys_restart_syscall",
            "sys_kill",
            "sys_tkill",
            "sys_tgkill",
            "sys_sigaltstack",
            "sys_rt_sigsuspend",
            "sys_rt_sigaction",
            "sys_rt_sigprocmask",
            "sys_rt_sigpending",
            "sys_rt_sigtimedwait",
            "sys_rt_sigqueueinfo",
            "sys_rt_sigreturn",
            "sys_setpriority",
            "sys_getpriority",
            "sys_reboot",
            "sys_setregid",
            "sys_setgid",
            "sys_setreuid",
            "sys_setuid",
            "sys_setresuid",
            "sys_getresuid",
            "sys_setresgid",
            "sys_getresgid",
            "sys_setfsuid",
            "sys_setfsgid",
            "sys_times",
            "sys_setpgid",
            "sys_getpgid",
            "sys_getsid",
            "sys_setsid",
            "sys_getgroups",
            "sys_setgroups",
            "sys_uname",
            "sys_sethostname",
            "sys_setdomainname",
            "sys_getrlimit",
            "sys_setrlimit",
            "sys_getrusage",
            "sys_umask",
            "sys_prctl",
            "sys_getcpu",
            "sys_gettimeofday",
            "sys_settimeofday",
            "sys_adjtimex",
            "sys_getpid",
            "sys_getppid",
            "sys_getuid",
            "sys_geteuid",
            "sys_getgid",
            "sys_getegid",
            "sys_gettid",
            "sys_sysinfo",
            "sys_mq_open",
            "sys_mq_unlink",
            "sys_mq_timedsend",
            "sys_mq_timedreceive",
            "sys_mq_notify",
            "sys_mq_getsetattr",
            "sys_msgget",
            "sys_msgctl",
            "sys_msgrcv",
            "sys_msgsnd",
            "sys_semget",
            "sys_semctl",
            "sys_semtimedop",
            "sys_semop",
            "sys_shmget",
            "sys_shmctl",
            "sys_shmat",
            "sys_shmdt",
            "sys_socket",
            "sys_socketpair",
            "sys_bind",
            "sys_listen",
            "sys_accept",
            "sys_connect",
            "sys_getsockname",
            "sys_getpeername",
            "sys_sendto",
            "sys_recvfrom",
            "sys_setsockopt",
            "sys_getsockopt",
            "sys_shutdown",
            "sys_sendmsg",
            "sys_recvmsg",
            "sys_readahead",
            "sys_brk",
            "sys_munmap",
            "sys_mremap",
            "sys_add_key",
            "sys_request_key",
            "sys_keyctl",
            "sys_clone",
            "sys_execve",
            "sys_mmap",
            "sys_fadvise64",
            "sys_swapon",
            "sys_swapoff",
            "sys_mprotect",
            "sys_msync",
            "sys_mlock",
            "sys_munlock",
            "sys_mlockall",
            "sys_munlockall",
            "sys_mincore",
            "sys_madvise",
            "sys_remap_file_pages",
            "sys_mbind",
            "sys_get_mempolicy",
            "sys_set_mempolicy",
            "sys_migrate_pages",
            "sys_move_pages",
            "sys_rt_tgsigqueueinfo",
            "sys_perf_event_open",
            "sys_accept4",
            "sys_recvmmsg",
            "sys_reserved",
            "sys_reserved",
            "sys_reserved",
            "sys_reserved",
            "sys_reserved",
            "sys_reserved",
            "sys_reserved",
            "sys_reserved",
            "sys_reserved",
            "sys_reserved",
            "sys_reserved",
            "sys_reserved",
            "sys_reserved",
            "sys_reserved",
            "sys_reserved",
            "sys_reserved",
            "sys_wait4",
            "sys_prlimit64",
            "sys_fanotify_init",
            "sys_fanotify_mark",
            "sys_name_to_handle_at",
            "sys_open_by_handle_at",
            "sys_clock_adjtime",
            "sys_syncfs",
            "sys_setns",
            "sys_sendmmsg",
            "sys_process_vm_readv",
            "sys_process_vm_writev",
            "sys_kcmp",
            "sys_finit_module",
            "sys_sched_setattr",
            "sys_sched_getattr",
            "sys_renameat2",
            "sys_seccomp",
            "sys_getrandom",
            "sys_memfd_create",
            "sys_bpf",
            "sys_execveat",
            "sys_userfaultfd",
            "sys_membarrier",
            "sys_mlock2",
            "sys_copy_file_range",
            "sys_preadv2",
            "sys_pwritev2",
            "sys_pkey_mprotect",
            "sys_pkey_alloc",
            "sys_pkey_free",
            "sys_statx",
            "sys_io_pgetevents",
            "sys_rseq",
            "sys_kexec_file_load",
        ]
        + ["sys_null" for idx in xrange(295, 424, 1)]
        + SYSCALL_COMMON
    )

    # syscall for x86 #
    SYSCALL_X86 = (
        [
            "sys_restart_syscall",
            "sys_exit",
            "sys_fork",
            "sys_read",
            "sys_write",
            "sys_open",
            "sys_close",
            "sys_waitpid",
            "sys_creat",
            "sys_link",
            "sys_unlink",
            "sys_execve",
            "sys_chdir",
            "sys_time",
            "sys_mknod",
            "sys_chmod",
            "sys_lchown",
            "sys_break",
            "sys_oldstat",
            "sys_lseek",
            "sys_getpid",
            "sys_mount",
            "sys_umount",
            "sys_setuid",
            "sys_getuid",
            "sys_stime",
            "sys_ptrace",
            "sys_alarm",
            "sys_oldfstat",
            "sys_pause",
            "sys_utime",
            "sys_stty",
            "sys_gtty",
            "sys_access",
            "sys_nice",
            "sys_ftime",
            "sys_sync",
            "sys_kill",
            "sys_rename",
            "sys_mkdir",
            "sys_rmdir",
            "sys_dup",
            "sys_pipe",
            "sys_times",
            "sys_prof",
            "sys_brk",
            "sys_setgid",
            "sys_getgid",
            "sys_signal",
            "sys_geteuid",
            "sys_getegid",
            "sys_acct",
            "sys_umount2",
            "sys_lock",
            "sys_ioctl",
            "sys_fcntl",
            "sys_mpx",
            "sys_setpgid",
            "sys_ulimit",
            "sys_oldolduname",
            "sys_umask",
            "sys_chroot",
            "sys_ustat",
            "sys_dup2",
            "sys_getppid",
            "sys_getpgrp",
            "sys_setsid",
            "sys_sigaction",
            "sys_sgetmask",
            "sys_ssetmask",
            "sys_setreuid",
            "sys_setregid",
            "sys_sigsuspend",
            "sys_sigpending",
            "sys_sethostname",
            "sys_setrlimit",
            "sys_getrlimit",
            "sys_getrusage",
            "sys_gettimeofday",
            "sys_settimeofday",
            "sys_getgroups",
            "sys_setgroups",
            "sys_select",
            "sys_symlink",
            "sys_oldlstat",
            "sys_readlink",
            "sys_uselib",
            "sys_swapon",
            "sys_reboot",
            "sys_readdir",
            "sys_mmap",
            "sys_munmap",
            "sys_truncate",
            "sys_ftruncate",
            "sys_fchmod",
            "sys_fchown",
            "sys_getpriority",
            "sys_setpriority",
            "sys_profil",
            "sys_statfs",
            "sys_fstatfs",
            "sys_ioperm",
            "sys_socketcall",
            "sys_syslog",
            "sys_setitimer",
            "sys_getitimer",
            "sys_stat",
            "sys_lstat",
            "sys_fstat",
            "sys_olduname",
            "sys_iopl",
            "sys_vhangup",
            "sys_idle",
            "sys_vm86old",
            "sys_wait4",
            "sys_swapoff",
            "sys_sysinfo",
            "sys_ipc",
            "sys_fsync",
            "sys_sigreturn",
            "sys_clone",
            "sys_setdomainname",
            "sys_uname",
            "sys_modify_ldt",
            "sys_adjtimex",
            "sys_mprotect",
            "sys_sigprocmask",
            "sys_create_module",
            "sys_init_module",
            "sys_delete_module",
            "sys_get_kernel_syms",
            "sys_quotactl",
            "sys_getpgid",
            "sys_fchdir",
            "sys_bdflush",
            "sys_sysfs",
            "sys_personality",
            "sys_afs_syscall",
            "sys_setfsuid",
            "sys_setfsgid",
            "sys_llseek",
            "sys_getdents",
            "sys_newselect",
            "sys_flock",
            "sys_msync",
            "sys_readv",
            "sys_writev",
            "sys_getsid",
            "sys_fdatasync",
            "sys_sysctl",
            "sys_mlock",
            "sys_munlock",
            "sys_mlockall",
            "sys_munlockall",
            "sys_sched_setparam",
            "sys_sched_getparam",
            "sys_sched_setscheduler",
            "sys_sched_getscheduler",
            "sys_sched_yield",
            "sys_sched_get_priority_max",
            "sys_sched_get_priority_min",
            "sys_sched_rr_get_interval",
            "sys_nanosleep",
            "sys_mremap",
            "sys_setresuid",
            "sys_getresuid",
            "sys_vm86",
            "sys_query_module",
            "sys_poll",
            "sys_nfsservctl",
            "sys_setresgid",
            "sys_getresgid",
            "sys_prctl",
            "sys_rt_sigreturn",
            "sys_rt_sigaction",
            "sys_rt_sigprocmask",
            "sys_rt_sigpending",
            "sys_rt_sigtimedwait",
            "sys_rt_sigqueueinfo",
            "sys_rt_sigsuspend",
            "sys_pread64",
            "sys_pwrite64",
            "sys_chown",
            "sys_getcwd",
            "sys_capget",
            "sys_capset",
            "sys_sigaltstack",
            "sys_sendfile",
            "sys_getpmsg",
            "sys_putpmsg",
            "sys_vfork",
            "sys_ugetrlimit",
            "sys_mmap2",
            "sys_truncate64",
            "sys_ftruncate64",
            "sys_stat64",
            "sys_lstat64",
            "sys_fstat64",
            "sys_lchown32",
            "sys_getuid32",
            "sys_getgid32",
            "sys_geteuid32",
            "sys_getegid32",
            "sys_setreuid32",
            "sys_setregid32",
            "sys_getgroups32",
            "sys_setgroups32",
            "sys_fchown32",
            "sys_setresuid32",
            "sys_getresuid32",
            "sys_setresgid32",
            "sys_getresgid32",
            "sys_chown32",
            "sys_setuid32",
            "sys_setgid32",
            "sys_setfsuid32",
            "sys_setfsgid32",
            "sys_pivot_root",
            "sys_mincore",
            "sys_madvise",
            "sys_getdents64",
            "sys_fcntl64",
            "sys_null",
            "sys_null",
            "sys_gettid",
            "sys_readahead",
            "sys_setxattr",
            "sys_lsetxattr",
            "sys_fsetxattr",
            "sys_getxattr",
            "sys_lgetxattr",
            "sys_fgetxattr",
            "sys_listxattr",
            "sys_llistxattr",
            "sys_flistxattr",
            "sys_removexattr",
            "sys_lremovexattr",
            "sys_fremovexattr",
            "sys_tkill",
            "sys_sendfile64",
            "sys_futex",
            "sys_sched_setaffinity",
            "sys_sched_getaffinity",
            "sys_set_thread_area",
            "sys_get_thread_area",
            "sys_io_setup",
            "sys_io_destroy",
            "sys_io_getevents",
            "sys_io_submit",
            "sys_io_cancel",
            "sys_fadvise64",
            "sys_set_zone_reclaim",
            "sys_exit_group",
            "sys_lookup_dcookie",
            "sys_epoll_create",
            "sys_epoll_ctl",
            "sys_epoll_wait",
            "sys_remap_file_pages",
            "sys_set_tid_address",
            "sys_timer_create",
            "sys_timer_settime",
            "sys_timer_gettime",
            "sys_timer_getoverrun",
            "sys_timer_delete",
            "sys_clock_settime",
            "sys_clock_gettime",
            "sys_clock_getres",
            "sys_clock_nanosleep",
            "sys_statfs64",
            "sys_fstatfs64",
            "sys_tgkill",
            "sys_utimes",
            "sys_fadvise64_64",
            "sys_vserver",
            "sys_mbind",
            "sys_get_mempolicy",
            "sys_set_mempolicy",
            "sys_mq_open",
            "sys_mq_unlink",
            "sys_mq_timedsend",
            "sys_mq_timedreceive",
            "sys_mq_notify",
            "sys_mq_getsetattr",
            "sys_kexec_load",
            "sys_waitid",
            "sys_setaltroot",
            "sys_add_key",
            "sys_request_key",
            "sys_keyctl",
            "sys_ioprio_set",
            "sys_ioprio_get",
            "sys_inotify_init",
            "sys_inotify_add_watch",
            "sys_inotify_rm_watch",
            "sys_migrate_pages",
            "sys_openat",
            "sys_mkdirat",
            "sys_mknodat",
            "sys_fchownat",
            "sys_futimesat",
            "sys_fstatat64",
            "sys_unlinkat",
            "sys_renameat",
            "sys_linkat",
            "sys_symlinkat",
            "sys_readlinkat",
            "sys_fchmodat",
            "sys_faccessat",
            "sys_pselect6",
            "sys_ppoll",
            "sys_unshare",
            "sys_set_robust_list",
            "sys_get_robust_list",
            "sys_splice",
            "sys_sync_file_range",
            "sys_tee",
            "sys_vmsplice",
            "sys_move_pages",
            "sys_getcpu",
            "sys_epoll_pwait",
            "sys_utimensat",
            "sys_signalfd",
            "sys_timerfd_create",
            "sys_eventfd",
            "sys_fallocate",
            "sys_timerfd_settime",
            "sys_timerfd_gettime",
            "sys_signalfd4",
            "sys_eventfd2",
            "sys_epoll_create1",
            "sys_dup3",
            "sys_pipe2",
            "sys_inotify_init1",
            "sys_preadv",
            "sys_pwritev",
            "sys_rt_tgsigqueueinfo",
            "sys_perf_event_open",
            "sys_recvmmsg",
            "sys_fanotify_init",
            "sys_fanotify_mark",
            "sys_prlimit64",
            "sys_name_to_handle_at",
            "sys_open_by_handle_at",
            "sys_clock_adjtime",
            "sys_syncfs",
            "sys_sendmmsg",
            "sys_setns",
            "sys_process_vm_readv",
            "sys_process_vm_writev",
            "sys_kcmp",
            "sys_finit_module",
            "sys_sched_setattr",
            "sys_sched_getattr",
            "sys_renameat2",
            "sys_seccomp",
            "sys_getrandom",
            "sys_memfd_create",
            "sys_bpf",
            "sys_execveat",
            "sys_socket",
            "sys_socketpair",
            "sys_bind",
            "sys_connect",
            "sys_listen",
            "sys_accept4",
            "sys_getsockopt",
            "sys_setsockopt",
            "sys_getsockname",
            "sys_getpeername",
            "sys_sendto",
            "sys_sendmsg",
            "sys_recvfrom",
            "sys_recvmsg",
            "sys_shutdown",
            "sys_userfaultfd",
            "sys_membarrier",
            "sys_mlock2",
            "sys_copy_file_range",
            "sys_preadv2",
            "sys_pwritev2",
            "sys_pkey_mprotect",
            "sys_pkey_alloc",
            "sys_pkey_free",
            "sys_statx",
            "sys_arch_prctl",
            "sys_io_pgetevents",
            "sys_rseq",
            "sys_reserved",
            "sys_reserved",
            "sys_reserved",
            "sys_reserved",
            "sys_reserved",
            "sys_reserved",
            "sys_semget",
            "sys_semctl",
            "sys_shmget",
            "sys_shmctl",
            "sys_shmat",
            "sys_shmdt",
            "sys_msgget",
            "sys_msgsnd",
            "sys_msgrcv",
            "sys_msgctl",
        ]
        + SYSCALL_COMMON32
        + SYSCALL_COMMON
    )

    # syscall for x86_64 #
    SYSCALL_X64 = (
        [
            "sys_read",
            "sys_write",
            "sys_open",
            "sys_close",
            "sys_stat",
            "sys_fstat",
            "sys_lstat",
            "sys_poll",
            "sys_lseek",
            "sys_mmap",
            "sys_mprotect",
            "sys_munmap",
            "sys_brk",
            "sys_rt_sigaction",
            "sys_rt_sigprocmask",
            "sys_rt_sigreturn",
            "sys_ioctl",
            "sys_pread64",
            "sys_pwrite64",
            "sys_readv",
            "sys_writev",
            "sys_access",
            "sys_pipe",
            "sys_select",
            "sys_sched_yield",
            "sys_mremap",
            "sys_msync",
            "sys_mincore",
            "sys_madvise",
            "sys_shmget",
            "sys_shmat",
            "sys_shmctl",
            "sys_dup",
            "sys_dup2",
            "sys_pause",
            "sys_nanosleep",
            "sys_getitimer",
            "sys_alarm",
            "sys_setitimer",
            "sys_getpid",
            "sys_sendfile",
            "sys_socket",
            "sys_connect",
            "sys_accept",
            "sys_sendto",
            "sys_recvfrom",
            "sys_sendmsg",
            "sys_recvmsg",
            "sys_shutdown",
            "sys_bind",
            "sys_listen",
            "sys_getsockname",
            "sys_getpeername",
            "sys_socketpair",
            "sys_setsockopt",
            "sys_getsockopt",
            "sys_clone",
            "sys_fork",
            "sys_vfork",
            "sys_execve",
            "sys_exit",
            "sys_wait4",
            "sys_kill",
            "sys_uname",
            "sys_semget",
            "sys_semop",
            "sys_semctl",
            "sys_shmdt",
            "sys_msgget",
            "sys_msgsnd",
            "sys_msgrcv",
            "sys_msgctl",
            "sys_fcntl",
            "sys_flock",
            "sys_fsync",
            "sys_fdatasync",
            "sys_truncate",
            "sys_ftruncate",
            "sys_getdents",
            "sys_getcwd",
            "sys_chdir",
            "sys_fchdir",
            "sys_rename",
            "sys_mkdir",
            "sys_rmdir",
            "sys_creat",
            "sys_link",
            "sys_unlink",
            "sys_symlink",
            "sys_readlink",
            "sys_chmod",
            "sys_fchmod",
            "sys_chown",
            "sys_fchown",
            "sys_lchown",
            "sys_umask",
            "sys_gettimeofday",
            "sys_getrlimit",
            "sys_getrusage",
            "sys_sysinfo",
            "sys_times",
            "sys_ptrace",
            "sys_getuid",
            "sys_syslog",
            "sys_getgid",
            "sys_setuid",
            "sys_setgid",
            "sys_geteuid",
            "sys_getegid",
            "sys_setpgid",
            "sys_getppid",
            "sys_getpgrp",
            "sys_setsid",
            "sys_setreuid",
            "sys_setregid",
            "sys_getgroups",
            "sys_setgroups",
            "sys_setresuid",
            "sys_getresuid",
            "sys_setresgid",
            "sys_getresgid",
            "sys_getpgid",
            "sys_setfsuid",
            "sys_setfsgid",
            "sys_getsid",
            "sys_capget",
            "sys_capset",
            "sys_rt_sigpending",
            "sys_rt_sigtimedwait",
            "sys_rt_sigqueueinfo",
            "sys_rt_sigsuspend",
            "sys_sigaltstack",
            "sys_utime",
            "sys_mknod",
            "sys_uselib",
            "sys_personality",
            "sys_ustat",
            "sys_statfs",
            "sys_fstatfs",
            "sys_sysfs",
            "sys_getpriority",
            "sys_setpriority",
            "sys_sched_setparam",
            "sys_sched_getparam",
            "sys_sched_setscheduler",
            "sys_sched_getscheduler",
            "sys_sched_get_priority_max",
            "sys_sched_get_priority_min",
            "sys_sched_rr_get_interval",
            "sys_mlock",
            "sys_munlock",
            "sys_mlockall",
            "sys_munlockall",
            "sys_vhangup",
            "sys_modify_ldt",
            "sys_pivot_root",
            "sys_sysctl",
            "sys_prctl",
            "sys_arch_prctl",
            "sys_adjtimex",
            "sys_setrlimit",
            "sys_chroot",
            "sys_sync",
            "sys_acct",
            "sys_settimeofday",
            "sys_mount",
            "sys_umount2",
            "sys_swapon",
            "sys_swapoff",
            "sys_reboot",
            "sys_sethostname",
            "sys_setdomainname",
            "sys_iopl",
            "sys_ioperm",
            "sys_create_module",
            "sys_init_module",
            "sys_delete_module",
            "sys_get_kernel_syms",
            "sys_query_module",
            "sys_quotactl",
            "sys_nfsservctl",
            "sys_getpmsg",
            "sys_putpmsg",
            "sys_afs_syscall",
            "sys_tuxcall",
            "sys_security",
            "sys_gettid",
            "sys_readahead",
            "sys_setxattr",
            "sys_lsetxattr",
            "sys_fsetxattr",
            "sys_getxattr",
            "sys_lgetxattr",
            "sys_fgetxattr",
            "sys_listxattr",
            "sys_llistxattr",
            "sys_flistxattr",
            "sys_removexattr",
            "sys_lremovexattr",
            "sys_fremovexattr",
            "sys_tkill",
            "sys_time",
            "sys_futex",
            "sys_sched_setaffinity",
            "sys_sched_getaffinity",
            "sys_set_thread_area",
            "sys_io_setup",
            "sys_io_destroy",
            "sys_io_getevents",
            "sys_io_submit",
            "sys_io_cancel",
            "sys_get_thread_area",
            "sys_lookup_dcookie",
            "sys_epoll_create",
            "sys_epoll_ctl_old",
            "sys_epoll_wait_old",
            "sys_remap_file_pages",
            "sys_getdents64",
            "sys_set_tid_address",
            "sys_restart_syscall",
            "sys_semtimedop",
            "sys_fadvise64",
            "sys_timer_create",
            "sys_timer_settime",
            "sys_timer_gettime",
            "sys_timer_getoverrun",
            "sys_timer_delete",
            "sys_clock_settime",
            "sys_clock_gettime",
            "sys_clock_getres",
            "sys_clock_nanosleep",
            "sys_exit_group",
            "sys_epoll_wait",
            "sys_epoll_ctl",
            "sys_tgkill",
            "sys_utimes",
            "sys_vserver",
            "sys_mbind",
            "sys_set_mempolicy",
            "sys_get_mempolicy",
            "sys_mq_open",
            "sys_mq_unlink",
            "sys_mq_timedsend",
            "sys_mq_timedreceive",
            "sys_mq_notify",
            "sys_mq_getsetattr",
            "sys_kexec_load",
            "sys_waitid",
            "sys_add_key",
            "sys_request_key",
            "sys_keyctl",
            "sys_ioprio_set",
            "sys_ioprio_get",
            "sys_inotify_init",
            "sys_inotify_add_watch",
            "sys_inotify_rm_watch",
            "sys_migrate_pages",
            "sys_openat",
            "sys_mkdirat",
            "sys_mknodat",
            "sys_fchownat",
            "sys_futimesat",
            "sys_newfstatat",
            "sys_unlinkat",
            "sys_renameat",
            "sys_linkat",
            "sys_symlinkat",
            "sys_readlinkat",
            "sys_fchmodat",
            "sys_faccessat",
            "sys_pselect6",
            "sys_ppoll",
            "sys_unshare",
            "sys_set_robust_list",
            "sys_get_robust_list",
            "sys_splice",
            "sys_tee",
            "sys_sync_file_range",
            "sys_vmsplice",
            "sys_move_pages",
            "sys_utimensat",
            "sys_epoll_pwait",
            "sys_signalfd",
            "sys_timerfd_create",
            "sys_eventfd",
            "sys_fallocate",
            "sys_timerfd_settime",
            "sys_timerfd_gettime",
            "sys_accept4",
            "sys_signalfd4",
            "sys_eventfd2",
            "sys_epoll_create1",
            "sys_dup3",
            "sys_pipe2",
            "sys_inotify_init1",
            "sys_preadv",
            "sys_pwritev",
            "sys_rt_tgsigqueueinfo",
            "sys_perf_event_open",
            "sys_recvmmsg",
            "sys_fanotify_init",
            "sys_fanotify_mark",
            "sys_prlimit64",
            "sys_name_to_handle_at",
            "sys_open_by_handle_at",
            "sys_clock_adjtime",
            "sys_syncfs",
            "sys_sendmmsg",
            "sys_setns",
            "sys_getcpu",
            "sys_process_vm_readv",
            "sys_process_vm_writev",
            "sys_kcmp",
            "sys_finit_module",
            "sys_sched_setattr",
            "sys_sched_getattr",
            "sys_renameat2",
            "sys_seccomp",
            "sys_getrandom",
            "sys_memfd_create",
            "sys_kexec_file_load",
            "sys_bpf",
            "sys_execveat",
            "sys_userfaultfd",
            "sys_membarrier",
            "sys_mlock2",
            "sys_copy_file_range",
            "sys_preadv2",
            "sys_pwritev2",
            "sys_pkey_mprotect",
            "sys_pkey_alloc",
            "sys_pkey_free",
            "sys_statx",
            "sys_io_pgetevents",
            "sys_rseq",
        ]
        + ["sys_null" for idx in xrange(335, 424, 1)]
        + SYSCALL_COMMON
    )

    # syscall for riscv64 #
    SYSCALL_RISCV64 = (
        [
            "sys_io_setup",
            "sys_io_destroy",
            "sys_io_submit",
            "sys_io_cancel",
            "sys_io_getevents",
            "sys_setxattr",
            "sys_lsetxattr",
            "sys_fsetxattr",
            "sys_getxattr",
            "sys_lgetxattr",
            "sys_fgetxattr",
            "sys_listxattr",
            "sys_llistxattr",
            "sys_flistxattr",
            "sys_removexattr",
            "sys_lremovexattr",
            "sys_fremovexattr",
            "sys_getcwd",
            "sys_lookup_dcookie",
            "sys_eventfd2",
            "sys_epoll_create1",
            "sys_epoll_ctl",
            "sys_epoll_pwait",
            "sys_dup",
            "sys_dup3",
            "sys_fcntl",
            "sys_inotify_init1",
            "sys_inotify_add_watch",
            "sys_inotify_rm_watch",
            "sys_ioctl",
            "sys_ioprio_set",
            "sys_ioprio_get",
            "sys_flock",
            "sys_mknodat",
            "sys_mkdirat",
            "sys_unlinkat",
            "sys_symlinkat",
            "sys_linkat",
            "sys_renameat",
            "sys_umount2",
            "sys_mount",
            "sys_pivot_root",
            "sys_nfsservctl",
            "sys_statfs",
            "sys_fstatfs",
            "sys_truncate",
            "sys_ftruncate",
            "sys_fallocate",
            "sys_faccessat",
            "sys_chdir",
            "sys_fchdir",
            "sys_chroot",
            "sys_fchmod",
            "sys_fchmodat",
            "sys_fchownat",
            "sys_fchown",
            "sys_openat",
            "sys_close",
            "sys_vhangup",
            "sys_pipe2",
            "sys_quotactl",
            "sys_getdents64",
            "sys_lseek",
            "sys_read",
            "sys_write",
            "sys_readv",
            "sys_writev",
            "sys_pread64",
            "sys_pwrite64",
            "sys_preadv",
            "sys_pwritev",
            "sys_sendfile",
            "sys_pselect6",
            "sys_ppoll",
            "sys_signalfd4",
            "sys_vmsplice",
            "sys_splice",
            "sys_tee",
            "sys_readlinkat",
            "sys_newfstatat",
            "sys_fstat",
            "sys_sync",
            "sys_fsync",
            "sys_fdatasync",
            "sys_sync_file_range",
            "sys_timerfd_create",
            "sys_timerfd_settime",
            "sys_timerfd_gettime",
            "sys_utimensat",
            "sys_acct",
            "sys_capget",
            "sys_capset",
            "sys_personality",
            "sys_exit",
            "sys_exit_group",
            "sys_waitid",
            "sys_set_tid_address",
            "sys_unshare",
            "sys_futex",
            "sys_set_robust_list",
            "sys_get_robust_list",
            "sys_nanosleep",
            "sys_getitimer",
            "sys_setitimer",
            "sys_kexec_load",
            "sys_init_module",
            "sys_delete_module",
            "sys_timer_create",
            "sys_timer_gettime",
            "sys_timer_getoverrun",
            "sys_timer_settime",
            "sys_timer_delete",
            "sys_clock_settime",
            "sys_clock_gettime",
            "sys_clock_getres",
            "sys_clock_nanosleep",
            "sys_syslog",
            "sys_ptrace",
            "sys_sched_setparam",
            "sys_sched_setscheduler",
            "sys_sched_getscheduler",
            "sys_sched_getparam",
            "sys_sched_setaffinity",
            "sys_sched_getaffinity",
            "sys_sched_yield",
            "sys_sched_get_priority_max",
            "sys_sched_get_priority_min",
            "sys_sched_rr_get_interval",
            "sys_restart_syscall",
            "sys_kill",
            "sys_tkill",
            "sys_tgkill",
            "sys_sigaltstack",
            "sys_rt_sigsuspend",
            "sys_rt_sigaction",
            "sys_rt_sigprocmask",
            "sys_rt_sigpending",
            "sys_rt_sigtimedwait",
            "sys_rt_sigqueueinfo",
            "sys_rt_sigreturn",
            "sys_setpriority",
            "sys_getpriority",
            "sys_reboot",
            "sys_setregid",
            "sys_setgid",
            "sys_setreuid",
            "sys_setuid",
            "sys_setresuid",
            "sys_getresuid",
            "sys_setresgid",
            "sys_getresgid",
            "sys_setfsuid",
            "sys_setfsgid",
            "sys_times",
            "sys_setpgid",
            "sys_getpgid",
            "sys_getsid",
            "sys_setsid",
            "sys_getgroups",
            "sys_setgroups",
            "sys_uname",
            "sys_sethostname",
            "sys_setdomainname",
            "sys_getrlimit",
            "sys_setrlimit",
            "sys_getrusage",
            "sys_umask",
            "sys_prctl",
            "sys_getcpu",
            "sys_gettimeofday",
            "sys_settimeofday",
            "sys_adjtimex",
            "sys_getpid",
            "sys_getppid",
            "sys_getuid",
            "sys_geteuid",
            "sys_getgid",
            "sys_getegid",
            "sys_gettid",
            "sys_sysinfo",
            "sys_mq_open",
            "sys_mq_unlink",
            "sys_mq_timedsend",
            "sys_mq_timedreceive",
            "sys_mq_notify",
            "sys_mq_getsetattr",
            "sys_msgget",
            "sys_msgctl",
            "sys_msgrcv",
            "sys_msgsnd",
            "sys_semget",
            "sys_semctl",
            "sys_semtimedop",
            "sys_semop",
            "sys_shmget",
            "sys_shmctl",
            "sys_shmat",
            "sys_shmdt",
            "sys_socket",
            "sys_socketpair",
            "sys_bind",
            "sys_listen",
            "sys_accept",
            "sys_connect",
            "sys_getsockname",
            "sys_getpeername",
            "sys_sendto",
            "sys_recvfrom",
            "sys_setsockopt",
            "sys_getsockopt",
            "sys_shutdown",
            "sys_sendmsg",
            "sys_recvmsg",
            "sys_readahead",
            "sys_brk",
            "sys_munmap",
            "sys_mremap",
            "sys_add_key",
            "sys_request_key",
            "sys_keyctl",
            "sys_clone",
            "sys_execve",
            "sys_mmap",
            "sys_fadvise64",
            "sys_swapon",
            "sys_swapoff",
            "sys_mprotect",
            "sys_msync",
            "sys_mlock",
            "sys_munlock",
            "sys_mlockall",
            "sys_munlockall",
            "sys_mincore",
            "sys_madvise",
            "sys_remap_file_pages",
            "sys_mbind",
            "sys_get_mempolicy",
            "sys_set_mempolicy",
            "sys_migrate_pages",
            "sys_move_pages",
            "sys_rt_tgsigqueueinfo",
            "sys_perf_event_open",
            "sys_accept4",
            "sys_riscv_flush_icache",
            "sys_recvmmsg",
            "sys_wait4",
            "sys_prlimit64",
            "sys_fanotify_init",
            "sys_fanotify_mark",
            "sys_name_to_handle_at",
            "sys_open_by_handle_at",
            "sys_clock_adjtime",
            "sys_syncfs",
            "sys_setns",
            "sys_sendmmsg",
            "sys_process_vm_readv",
            "sys_process_vm_writev",
            "sys_kcmp",
            "sys_finit_module",
            "sys_sched_setattr",
            "sys_sched_getattr",
            "sys_renameat2",
            "sys_seccomp",
            "sys_getrandom",
            "sys_memfd_create",
            "sys_bpf",
            "sys_execveat",
            "sys_userfaultfd",
            "sys_membarrier",
            "sys_mlock2",
            "sys_copy_file_range",
            "sys_preadv2",
            "sys_pwritev2",
            "sys_pkey_mprotect",
            "sys_pkey_alloc",
            "sys_pkey_free",
            "sys_statx",
            "sys_io_pgetevents",
            "sys_rseq",
            "sys_kexec_file_load",
        ]
        + ["sys_null" for idx in xrange(295, 424, 1)]
        + SYSCALL_COMMON
    )

    # default syscall list #
    sysList = []

    # default register list #
    regList = []

    # registers sorted by number #
    REGS_X86 = [
        "eax",
        "ecx",
        "edx",
        "ebx",
        "esp",
        "ebp",  # 5 #
        "esi",
        "edi",
        "eip",
        "eflags",
        "<none>",  # 10 #
        "st0",
        "st1",
        "st2",
        "st3",
        "st4",  # 15 #
        "st5",
        "st6",
        "st7",
        "<none>",
        "<none>",  # 20 #
        "xmm0",
        "xmm1",
        "xmm2",
        "xmm3",
        "xmm4",  # 25 #
        "xmm5",
        "xmm6",
        "xmm7",
        "mm0",
        "mm1",  # 30 #
        "mm2",
        "mm3",
        "mm4",
        "mm5",
        "mm6",  # 35 #
        "mm7",
        "fcw",
        "fsw",
        "mxcsr",
        "es",  # 40 #
        "cs",
        "ss",
        "ds",
        "fs",
        "gs",  # 45 #
        "<none>",
        "<none>",
        "tr",
        "ldtr",
    ]

    REGS_X64 = [
        "rax",
        "rdx",
        "rcx",
        "rbx",
        "rsi",
        "rdi",  # 5 #
        "rbp",
        "rsp",
        "r8",
        "r9",
        "r10",  # 10 #
        "r11",
        "r12",
        "r13",
        "r14",
        "r15",  # 15 #
        "rip",
        "xmm0",
        "xmm1",
        "xmm2",
        "xmm3",  # 20 #
        "xmm4",
        "xmm5",
        "xmm6",
        "xmm7",
        "xmm8",  # 25 #
        "xmm9",
        "xmm10",
        "xmm11",
        "xmm12",
        "xmm13",  # 30 #
        "xmm14",
        "xmm15",
        "st0",
        "st1",
        "st2",  # 35 #
        "st3",
        "st4",
        "st5",
        "st6",
        "st7",  # 40 #
        "mm0",
        "mm1",
        "mm2",
        "mm3",
        "mm4",  # 45 #
        "mm5",
        "mm6",
        "mm7",
        "rflags",
        "es",  # 50 #
        "cs",
        "ss",
        "ds",
        "fs",
        "gs",  # 55 #
        "<none>",
        "<none>",
        "fs.base",
        "gs.base",
        "<none>",  # 60 #
        "<none>",
        "tr",
        "ldtr",
        "mxcsr",
        "fcw",  # 65 #
        "fsw",
    ]

    REGS_RISCV64 = [
        "pc",
        "ra",
        "sp",
        "gp",
        "tp",
        "t0",
        "t1",
        "t2",
        "s0",
        "s1",
        "a0",
        "a1",
        "a2",
        "a3",
        "a4",
        "a5",
        "a6",
        "a7",
        "s2",
        "s3",
        "s4",
        "s5",
        "s6",
        "s7",
        "s8",
        "s9",
        "s10",
        "s11",
        "t3",
        "t4",
        "t5",
        "t6",
    ]

    REGS_ARM = ["r%d" % idx for idx in xrange(16)]

    REGS_AARCH64 = (
        ["x%d" % idx for idx in xrange(31)]
        + ["sp", "pc", "ELP_mode", "RA_SIGN_STATE", "<none>", "<none>"]
        + ["reserved" for idx in xrange(37, 46, 1)]
        + ["VG", "FFR"]
        + ["p%d" % idx for idx in xrange(16)]
        + ["v%d" % idx for idx in xrange(32)]
        + ["z%d" % idx for idx in xrange(32)]
    )

    pcRegIndex = {
        "arm": REGS_ARM.index("r14"),
        "aarch64": REGS_AARCH64.index("pc"),
        "x86": REGS_X86.index("eip"),
        "x64": REGS_X64.index("rip"),
        "riscv64": REGS_RISCV64.index("pc"),
    }

    # syscall register #
    SYSREG_LIST = {
        "powerpc": "gpr0",
        "arm": "r7",
        "aarch64": "x8",
        "x64": "orig_rax",
        "x86": "orig_eax",
        "riscv64": "a7",
    }

    # return register #
    RET_LIST = {
        "powerpc": "result",
        "arm": "r0",
        "aarch64": "x0",
        "x64": "rax",
        "x86": "eax",
        "riscv64": "a0",
    }

    # signal for Linux #
    SIG_LIST = [
        "0",
        "SIGHUP",
        "SIGINT",
        "SIGQUIT",
        "SIGILL",  # 4
        "SIGTRAP",
        "SIGABRT",
        "SIGBUS",
        "SIGFPE",  # 8
        "SIGKILL",
        "SIGUSR1",
        "SIGSEGV",
        "SIGUSR2",  # 12
        "SIGPIPE",
        "SIGALRM",
        "SIGTERM",
        "SIGSTKFLT",  # 16
        "SIGCHLD",
        "SIGCONT",
        "SIGSTOP",
        "SIGTSTP",  # 20
        "SIGTTIN",
        "SIGTTOU",
        "SIGURG",
        "SIGXCPU",  # 24
        "SIGXFSZ",
        "SIGVTALRM",
        "SIGPROF",
        "SIGWINCH",  # 28
        "SIGIO",
        "SIGPWR",
        "SIGSYS",
        "SIG32",  # 32
        "SIG33",
    ] + ["SIGRT%d" % idx for idx in xrange(32)]
    SIGKILL = SIG_LIST.index("SIGKILL")

    # signal for MacOS #
    SIG_LIST_MACOS = [
        "0",
        "SIGHUP",
        "SIGINT",
        "SIGQUIT",
        "SIGILL",  # 4
        "SIGTRAP",
        "SIGABRT",
        "SIGEMT",
        "SIGFPE",  # 8
        "SIGKILL",
        "SIGBUS",
        "SIGSEGV",
        "SIGSYS",  # 12
        "SIGPIPE",
        "SIGALRM",
        "SIGTERM",
        "SIGURG",  # 16
        "SIGSTOP",
        "SIGTSTP",
        "SIGCONT",
        "SIGCHLD",  # 20
        "SIGTTIN",
        "SIGTTOU",
        "SIGIO",
        "SIGXCPU",  # 24
        "SIGXFSZ",
        "SIGVTALRM",
        "SIGPROF",
        "SIGWINCH",  # 28
        "SIGINFO",
        "SIGUSR1",
        "SIGUSR2",  # 31
    ]

    # SIGCHLD si_codes #
    SIGCHLD_CODE = [
        "N/A",
        "CLD_EXITED",  # /* child has exited */
        "CLD_KILLED",  # /* child was killed */
        "CLD_DUMPED",  # /* child terminated abnormally */
        "CLD_TRAPPED",  # /* traced child has trapped */
        "CLD_STOPPED",  # /* child has stopped */
        "CLD_CONTINUED",  # /* stopped child has continued */
    ]

    # SIGSEGV si_codes #
    SIGSEGV_CODE = [
        "N/A",
        "SEGV_MAPERR",  # /* address not mapped to object */
        "SEGV_ACCERR",  # /* invalid permissions for mapped object */
        "SEGV_BNDERR",  # /* failed address bound checks */
        "SEGV_PSTKOVF",  # /* paragraph stack overflow */
        "SEGV_ACCADI",  # /* ADI not enabled for mapped object */
        "SEGV_ADIDERR",  # /* Disrupting MCD error */
        "SEGV_ADIPERR",  # /* Precise MCD exception */
        "SEGV_MTEAERR",  # /* Asynchronous ARM MTE error */
        "SEGV_MTESERR",  # /* Synchronous ARM MTE exception */
    ]

    # SIGILL si_codes #
    SIGILL_CODE = [
        "N/A",
        "ILL_ILLOPC",  # /* illegal opcode */
        "ILL_ILLOPN",  # /* illegal operand */
        "ILL_ILLADR",  # /* illegal addressing mode */
        "ILL_ILLTRP",  # /* illegal trap */
        "ILL_PRVOPC",  # /* privileged opcode */
        "ILL_PRVREG",  # /* privileged register */
        "ILL_COPROC",  # /* coprocessor error */
        "ILL_BADSTK",  # /* internal stack error */
        "ILL_BADIADDR",  # /* unimplemented instruction address */
        "__ILL_BREAK",  # /* illegal break */
        "__ILL_BNDMOD",  # /* bundle-update (modification) in progress */
    ]

    # SIGTRAP si_codes #
    SIGTRAP_CODE = [
        "N/A",
        "TRAP_BRKPT",  # /* process breakpoint */
        "TRAP_TRACE",  # /* process trace trap */
        "TRAP_BRANCH",  # /* process taken branch trap */
        "TRAP_HWBKPT",  # /* hardware breakpoint/watchpoint */
        "TRAP_UNK",  # /* undiagnosed trap */
    ]

    # SIGFPE si_codes #
    SIGFPE_CODE = [
        "N/A",
        "FPE_INTDIV",  # /* integer divide by zero */
        "FPE_INTOVF",  # /* integer overflow */
        "FPE_FLTDIV",  # /* floating point divide by zero */
        "FPE_FLTOVF",  # /* floating point overflow */
        "FPE_FLTUND",  # /* floating point underflow */
        "FPE_FLTRES",  # /* floating point inexact result */
        "FPE_FLTINV",  # /* floating point invalid operation */
        "FPE_FLTSUB",  # /* subscript out of range */
        "__FPE_DECOVF",  # /* decimal overflow */
        "__FPE_DECDIV",  # /* decimal division by zero */
        "__FPE_DECERR",  # /* packed decimal error */
        "__FPE_INVASC",  # /* invalid ASCII digit */
        "__FPE_INVDEC",  # /* invalid decimal digit */
        "FPE_FLTUNK",  # /* undiagnosed floating-point exception */
        "FPE_CONDTRAP",  # /* trap on condition */
    ]

    # si_codes #
    SI_CODE = {
        0: "SI_USER",  # /* sent by kill, sigsend, raise */
        0x80: "SI_KERNEL",  # /* sent by the kernel from somewhere */
        -1: "SI_QUEUE",  # /* sent by sigqueue */
        -2: "SI_TIMER",  # /* sent by timer expiration */
        -3: "SI_MESGQ",  # /* sent by real time mesq state change */
        -4: "SI_ASYNCIO",  # /* sent by AIO completion */
        -5: "SI_SIGIO",  # /* sent by queued SIGIO */
        -6: "SI_TKILL",  # /* sent by tkill system call */
        -7: "SI_DETHREAD",  # /* sent by execve() killing subsidiary threads */
        -60: "SI_ASYNCNL",  # /* sent by glibc async name lookup completion */
    }

    # stat fields from http://linux.die.net/man/5/proc #
    STAT_ATTR = [
        "CORE",
        "USER",
        "NICE",
        "SYSTEM",
        "IDLE",
        "IOWAIT",
        "IRQ",
        "SOFTIRQ",
        "STEAL",
        "GUEST",
        "GUESTNICE",
    ]

    # proc stat fields from http://linux.die.net/man/5/proc #
    PROCSTAT_ATTR = [
        "PID",
        "COMM",
        "STATE",
        "PPID",
        "PGRP",
        "SESSIONID",  # 5
        "NRTTY",
        "TPGID",
        "FLAGS",
        "MINFLT",
        "CMINFLT",  # 10
        "MAJFLT",
        "CMAJFLT",
        "UTIME",
        "STIME",
        "CUTIME",  # 15
        "CSTIME",
        "PRIORITY",
        "NICE",
        "NRTHREAD",
        "ITERALVAL",  # 20
        "STARTTIME",
        "VSIZE",
        "RSS",
        "RSSLIM",
        "STARTCODE",  # 25
        "ENDCODE",
        "STARTSTACK",
        "SP",
        "PC",
        "SIGNAL",  # 30
        "BLOCKED",
        "SIGIGNORE",
        "SIGCATCH",
        "WCHEN",
        "NSWAP",  # 35
        "CNSWAP",
        "EXITSIGNAL",
        "PROCESSOR",
        "RTPRIORITY",  # 39
        "POLICY",
        "DELAYBLKTICK",
        "GUESTTIME",
        "CGUESTTIME",
        "STARTDATA",  # 44
        "ENDDATA",
        "STARTBRK",
        "ARGSTART",
        "ARGEND",  # 48
        "ENVSTART",
        "ENVEND",
        "EXITCODE",  # 51
    ]

    # sched policy #
    SCHED_POLICY = [
        "C",  # 0: CFS #
        "F",  # 1: FIFO #
        "R",  # 2: RR #
        "B",  # 3: BATCH #
        "N",  # 4: NONE #
        "I",  # 5: IDLE #
        "D",  # 6: DEADLINE #
    ]

    # sched policy for Windows #
    SCHED_POLICY_WINDOWS = {
        "ABOVE_NORMAL_PRIORITY_CLASS": "ANOR",
        "BELOW_NORMAL_PRIORITY_CLASS": "BNOR",
        "HIGH_PRIORITY_CLASS": "HIGH",
        "IDLE_PRIORITY_CLASS": "IDLE",
        "NORMAL_PRIORITY_CLASS": "NORM",
        "PROCESS_MODE_BACKGROUND_BEGIN": "BGBE",
        "PROCESS_MODE_BACKGROUND_END": "BGEN",
        "REALTIME_PRIORITY_CLASS": "RT",
    }

    # I/O sched class #
    IOSCHED_CLASS = [
        "NONE",
        "IOPRIO_CLASS_RT",
        "IOPRIO_CLASS_BE",
        "IOPRIO_CLASS_IDLE",
    ]

    # I/O sched target #
    IOSCHED_TARGET = [
        "NONE",
        "IOPRIO_WHO_PROCESS",
        "IOPRIO_WHO_PGRP",
        "IOPRIO_WHO_USER",
    ]

    # sa_family_t #
    SA_FAMILY_TYPE = {
        "0": "AF_UNSPEC",
        "1": "AF_LOCAL",  # POSIX name for AF_UNIX
        "1": "AF_UNIX",  # Unix domain sockets
        "2": "AF_INET",  # Internet IP Protocol
        "3": "AF_AX25",  # Amateur Radio AX.25
        "4": "AF_IPX",  # Novell IPX
        "5": "AF_APPLETALK",  # AppleTalk DDP
        "6": "AF_NETROM",  # Amateur Radio NET/ROM
        "7": "AF_BRIDGE",  # Multiprotocol bridge
        "8": "AF_ATMPVC",  # ATM PVCs
        "9": "AF_X25",  # Reserved for X.25 project
        "10": "AF_INET6",  # IP version 6
        "11": "AF_ROSE",  # Amateur Radio X.25 PLP
        "12": "AF_DECnet",  # Reserved for DECnet project
        "13": "AF_NETBEUI",  # Reserved for 802.2LLC project
        "14": "AF_SECURITY",  # Security callback pseudo AF
        "15": "AF_KEY",  # PF_KEY key management API
        "16": "AF_ROUTE",  # Alias to emulate 4.4BSD
        "16": "AF_NETLINK",
        "17": "AF_PACKET",  # Packet family
        "18": "AF_ASH",  # Ash
        "19": "AF_ECONET",  # Acorn Econet
        "20": "AF_ATMSVC",  # ATM SVCs
        "21": "AF_RDS",  # RDS sockets
        "22": "AF_SNA",  # Linux SNA Project (nutters!)
        "23": "AF_IRDA",  # IRDA sockets
        "24": "AF_PPPOX",  # PPPoX sockets
        "25": "AF_WANPIPE",  # Wanpipe API Sockets
        "26": "AF_LLC",  # Linux LLC
        "29": "AF_CAN",  # Controller Area Network
        "30": "AF_TIPC",  # TIPC sockets
        "31": "AF_BLUETOOTH",  # Bluetooth sockets
        "32": "AF_IUCV",  # IUCV sockets
        "33": "AF_RXRPC",  # RxRPC sockets
        "34": "AF_ISDN",  # mISDN sockets
        "35": "AF_PHONET",  # Phonet sockets
        "36": "AF_IEEE802154",  # IEEE802154 sockets
        "37": "AF_MAX",  # For now..
    }

    # statm fields #
    STATM_TYPE = [
        "TOTAL",  # 0 #
        "RSS",  # 1 #
        "SHR",  # 2 #
        "TEXT",  # 3 #
        "DATA",  # 4 #
        "LIB",  # 5 #
        "DIRTY",  # 6 #
    ]

    # error types #
    ERR_TYPE = [
        "EPERM",  # Operation not permitted #
        "ENOENT",  # No such file or directory #
        "ESRCH",  # No such process #
        "EINTR",  # Interrupted system call #
        "EIO",  # I/O error #
        "ENXIO",  # No such device or address #
        "E2BIG",  # Argument list too long #
        "ENOEXEC",  # Exec format error #
        "EBADF",  # Bad file number #
        "ECHILD",  # No child processes #
        "EAGAIN",  # Try again #
        "ENOMEM",  # Out of memory #
        "EACCES",  # Permission denied #
        "EFAULT",  # Bad address #
        "ENOTBLK",  # Block device required #
        "EBUSY",  # Device or resource busy #
        "EEXIST",  # File exists #
        "EXDEV",  # Cross-device link #
        "ENODEV",  # No such device #
        "ENOTDIR",  # Not a directory #
        "EISDIR",  # Is a directory #
        "EINVAL",  # Invalid argument #
        "ENFILE",  # File table overflow #
        "EMFILE",  # Too many open files #
        "ENOTTY",  # Not a typewriter #
        "ETXTBSY",  # Text file busy #
        "EFBIG",  # File too large #
        "ENOSPC",  # No space left on device #
        "ESPIPE",  # Illegal seek #
        "EROFS",  # Read-only file system #
        "EMLINK",  # Too many links #
        "EPIPE",  # Broken pipe #
        "EDOM",  # Math argument out of domain of func #
        "ERANGE",  # Math result not representable #
        "EDEADLK",  # Resource deadlock would occur #
        "ENAMETOOLONG",  # File name too long #
        "ENOLCK",  # No record locks available #
        "ENOSYS",  # Function not implemented #
        "ENOTEMPTY",  # Directory not empty #
        "ELOOP",  # Too many symbolic links encountered #
        "EWOULDBLOCK",  # Operation would block #
        "ENOMSG",  # No message of desired type #
        "EIDRM",  # Identifier removed #
        "ECHRNG",  # Channel number out of range #
        "EL2NSYNC",  # Level 2 not synchronized #
        "EL3HLT",  # Level 3 halted #
        "EL3RST",  # Level 3 reset #
        "ELNRNG",  # Link number out of range #
        "EUNATCH",  # Protocol driver not attached #
        "ENOCSI",  # No CSI structure available #
        "EL2HLT",  # Level 2 halted #
        "EBADE",  # Invalid exchange #
        "EBADR",  # Invalid request descriptor #
        "EXFULL",  # Exchange full #
        "ENOANO",  # No anode #
        "EBADRQC",  # Invalid request code #
        "EBADSLT ",  # Invalid slot #
        "EDEADLOCK",
        "EBFONT",  # Bad font file format #
        "ENOSTR",  # Device not a stream #
        "ENODATA",  # No data available #
        "ETIME",  # Timer expired #
        "ENOSR",  # Out of streams resources #
        "ENONET",  # Machine is not on the network #
        "ENOPKG",  # Package not installed #
        "EREMOTE",  # Object is remote #
        "ENOLINK",  # Link has been severed #
        "EADV",  # Advertise error #
        "ESRMNT",  # Srmount error #
        "ECOMM",  # Communication error on send #
        "EPROTO",  # Protocol error #
        "EMULTIHOP",  # Multihop attempted #
        "EDOTDOT",  # RFS specific error #
        "EBADMSG",  # Not a data message #
        "EOVERFLOW",  # Value too large for defined data type #
        "ENOTUNIQ",  # Name not unique on network #
        "EBADFD",  # File descriptor in bad state #
        "EREMCHG",  # Remote address changed #
        "ELIBACC",  # Can not access a needed shared library #
        "ELIBBAD",  # Accessing a corrupted shared library #
        "ELIBSCN",  # .lib section in a.out corrupted #
        "ELIBMAX",  # Attempting to link in too many shared libraries #
        "ELIBEXEC",  # Cannot exec a shared library directly #
        "EILSEQ",  # Illegal byte sequence #
        "ERESTART",  # Interrupted system call should be restarted #
        "ESTRPIPE",  # Streams pipe error #
        "EUSERS",  # Too many users #
        "ENOTSOCK",  # Socket operation on non-socket #
        "EDESTADDRREQ",  # Destination address required #
        "EMSGSIZE",  # Message too long #
        "EPROTOTYPE",  # Protocol wrong type for socket #
        "EPROTOTYPE",  # Protocol wrong type for socket #
        "ENOPROTOOPT",  # Protocol not available #
        "EPROTONOSUPPORT",  # Protocol not supported #
        "ESOCKTNOSUPPORT",  # Socket type not supported #
        "EOPNOTSUPP",  # Operation not supported on transport endpoint #
        "EPFNOSUPPORT",  # Protocol family not supported #
        "EAFNOSUPPORT ",  # Address family not supported by protocol #
        "EADDRINUSE ",  # Address already in use #
        "EADDRNOTAVAIL",  # Cannot assign requested address #
        "ENETDOWN",  # Network is down #
        "ENETUNREACH",  # Network is unreachable #
        "ENETRESET",  # Network dropped connection because of reset #
        "ECONNABORTED",  # Software caused connection abort #
        "ECONNRESET",  # Connection reset by peer #
        "ENOBUFS",  # No buffer space available #
        "EISCONN",  # Transport endpoint is already connected #
        "ENOTCONN",  # Transport endpoint is not connected #
        "ESHUTDOWN",  # Cannot send after transport endpoint shutdown #
        "ETOOMANYREFS",  # Too many references: cannot splice #
        "ETIMEDOUT",  # Connection timed out #
        "ECONNREFUSED",  # Connection refused #
        "EHOSTDOWN",  # Host is down #
        "EHOSTUNREACH",  # No route to host #
        "EALREADY",  # Operation already in progress #
        "EINPROGRESS",  # Operation now in progress #
        "ESTALE",  # Stale NFS file handle #
        "EUCLEAN",  # Structure needs cleaning #
        "ENOTNAM",  # Not a XENIX named type file #
        "ENAVAIL",  # No XENIX semaphores available #
        "EISNAM",  # Is a named type file #
        "EREMOTEIO",  # Remote I/O error #
        "EDQUOT",  # Quota exceeded #
        "ENOMEDIUM",  # No medium found #
        "EMEDIUMTYPE",  # Wrong medium type #
        "ECANCELED",  # Operation Canceled #
        "ENOKEY",  # Required key not available #
        "EKEYEXPIRED",  # Key has expired #
        "EKEYREVOKED",  # Key has been revoked #
        "EKEYREJECTED",  # Key was rejected by service #
        "EOWNERDEAD",  # Owner died #
        "ENOTRECOVERABLE",  # State not recoverable #
        "ERFKILL",  # Operation not possible due to RF-kill #
        "EHWPOISON",  # Memory page has hardware error #
    ]

    # rlimit types #
    RLIMIT_TYPE = [
        "RLIMIT_CPU",
        "RLIMIT_FSIZE",
        "RLIMIT_DATA",
        "RLIMIT_STACK",
        "RLIMIT_CORE",
        "RLIMIT_RSS",
        "RLIMIT_NPROC",
        "RLIMIT_NOFILE",
        "RLIMIT_MEMLOCK",
        "RLIMIT_AS",
        "RLIMIT_LOCKS",
        "RLIMIT_SIGPENDING",
        "RLIMIT_MSGQUEUE",
        "RLIMIT_NICE",
        "RLIMIT_RTPRIO",
        "RLIMIT_RTTIME",
        "RLIMIT_NLIMITS",
    ]

    # UDP/TCP format #
    PROTOCOL_ATTR = [
        "sl",
        "local_address",
        "rem_address",
        "st",
        "tx_rx_queue",
        "tr_tm->when",
        "retrnsmt",
        "uid",
        "timeout",
        "inode",
        "ref",
        "pointer",
        "drops",
    ]

    # UDS format #
    UDS_ATTR = [
        "Num",
        "RefCount",
        "Protocol",
        "Flags",
        "Type",
        "St",
        "Inode",
        "Path",
    ]

    TCP_STAT = [
        "N/A",
        "ESTABLISHED",
        "SYN_SENT",
        "SYN_RECV",
        "FIN_WAIT1",
        "FIN_WAIT2",
        "TIME_WAIT",
        "CLOSE",
        "CLOSE_WAIT",
        "LAST_ACK",
        "LISTEN",
        "CLOSING",
    ]

    # futex operation flags #
    FUTEX_TYPE = [
        "FUTEX_WAIT",
        "FUTEX_WAKE",
        "FUTEX_FD",
        "FUTEX_REQUEUE",
        "FUTEX_CMP_REQUEUE",
        "FUTEX_WAKE_OP",
        "FUTEX_LOCK_PI",
        "FUTEX_UNLOCK_PI",
        "FUTEX_TRYLOCK_PI",
        "FUTEX_WAIT_BITSET",
        "FUTEX_WAKE_BITSET",
        "FUTEX_WAIT_REQUEUE_PI",
        "FUTEX_CMP_REQUEUE_PI",
    ]

    # fcntl command flags #
    FCNTL_TYPE = {
        0: "F_DUPFD",
        1: "F_GETFD",
        2: "F_SETFD",
        3: "F_GETFL",
        4: "F_SETFL",
        5: "F_GETOWN",
        6: "F_SETOWN",
        7: "F_GETLK",
        8: "F_SETLK",
        9: "F_SETLKW",
        10: "F_SETOWN",
        11: "F_GETOWN",
        12: "F_SETSIG",
        13: "F_GETSIG",
        14: "F_GETLK64",
        15: "F_SETLK64",
        16: "F_SETLKW64",
        17: "F_SETOWNEX",
        18: "F_GETOWNEX",
        19: "F_GETOWNERUIDS",
        1024: "F_SETLEASE",
        1025: "F_GETLEASE",
        1026: "F_NOTIFY",
        1029: "F_CANCELLK",
        1030: "F_DUPFD_CLOEXEC",
        1031: "F_SETPIPE_SZ",
        1032: "F_GETPIPE_SZ",
        1033: "F_ADD_SEALS",
        1034: "F_GET_SEALS",
        1035: "F_GET_RW_HINT",
        1036: "F_SET_RW_HINT",
        1037: "F_GET_FILE_RW_HINT",
        1038: "F_SET_FILE_RW_HINT",
    }

    # ipc call flags #
    IPC_TYPE = {
        1: "SEMOP",
        2: "SEMGET",
        3: "SEMCTL",
        4: "SEMTIMEDOP",
        11: "MSGSND",
        12: "MSGRCV",
        13: "MSGGET",
        14: "MSGCTL",
        21: "SHMAT",
        22: "SHMDT",
        23: "SHMGET",
        24: "SHMCTL",
    }

    # ioring_enter flags #
    IORING_ENTER_FLAG = {
        1: "IORING_ENTER_GETEVENTS",
        2: "IORING_ENTER_SQ_WAKEUP",
        4: "IORING_ENTER_SQ_WAIT",
        8: "IORING_ENTER_EXT_ARG",
        16: "IORING_ENTER_REGISTERED_RING",
    }

    # ioring_register opcode #
    IORING_REGISTER_OPCODE = [
        "IORING_REGISTER_BUFFERS",
        "IORING_UNREGISTER_BUFFERS",
        "IORING_REGISTER_FILES",
        "IORING_UNREGISTER_FILES",
        "IORING_REGISTER_EVENTFD",
        "IORING_UNREGISTER_EVENTFD",
        "IORING_REGISTER_FILES_UPDATE",
        "IORING_REGISTER_EVENTFD_ASYNC",
        "IORING_REGISTER_PROBE",
        "IORING_REGISTER_PERSONALITY",
        "IORING_UNREGISTER_PERSONALITY",
        "IORING_REGISTER_RESTRICTIONS",
        "IORING_REGISTER_ENABLE_RINGS",
        "IORING_REGISTER_FILES2",
        "IORING_REGISTER_FILES_UPDATE2",
        "IORING_REGISTER_BUFFERS2",
        "IORING_REGISTER_BUFFERS_UPDATE",
        "IORING_REGISTER_IOWQ_AFF",
        "IORING_UNREGISTER_IOWQ_AFF",
        "IORING_REGISTER_IOWQ_MAX_WORKERS",
        "IORING_REGISTER_RING_FDS",
        "IORING_UNREGISTER_RING_FDS",
        "IORING_REGISTER_PBUF_RING",
        "IORING_UNREGISTER_PBUF_RING",
        "IORING_REGISTER_SYNC_CANCEL",
        "IORING_REGISTER_FILE_ALLOC_RANGE",
    ]

    # ioring_param features #
    IORING_PARAM_FEATURES = {
        0x1: "IORING_FEAT_SINGLE_MMAP",
        0x2: "IORING_FEAT_NODROP",
        0x4: "IORING_FEAT_SUBMIT_STABLE",
        0x8: "IORING_FEAT_RW_CUR_POS",
        0x16: "IORING_FEAT_CUR_PERSONALITY",
        0x32: "IORING_FEAT_FAST_POLL",
        0x64: "IORING_FEAT_POLL_32BITS",
        0x128: "IORING_FEAT_SQPOLL_NONFIXED",
        0x256: "IORING_FEAT_EXT_ARG",
        0x512: "IORING_FEAT_NATIVE_WORKERS",
        0x1024: "IORING_FEAT_RSRC_TAGS",
        0x2048: "IORING_FEAT_CQE_SKIP",
        0x4096: "IORING_FEAT_LINKED_FILE",
    }

    # ioctl flags #
    IOCTL_TYPE = {
        "TCGETS": 0x5401,
        "TCSETS": 0x5402,
        "TCSETSW": 0x5403,
        "TCSETSF": 0x5404,
        "TCGETA": 0x5405,
        "TCSETA": 0x5406,
        "TCSETAW": 0x5407,
        "TCSETAF": 0x5408,
        "TCSBRK": 0x5409,
        "TCXONC": 0x540A,
        "TCFLSH": 0x540B,
        "TIOCEXCL": 0x540C,
        "TIOCNXCL": 0x540D,
        "TIOCSCTTY": 0x540E,
        "TIOCGPGRP": 0x540F,
        "TIOCSPGRP": 0x5410,
        "TIOCOUTQ": 0x5411,
        "TIOCSTI": 0x5412,
        "TIOCGWINSZ": 0x5413,
        "TIOCSWINSZ": 0x5414,
        "TIOCMGET": 0x5415,
        "TIOCMBIS": 0x5416,
        "TIOCMBIC": 0x5417,
        "TIOCMSET": 0x5418,
        "TIOCGSOFTCAR": 0x5419,
        "TIOCSSOFTCAR": 0x541A,
        "FIONREAD": 0x541B,
        "TIOCLINUX": 0x541C,
        "TIOCCONS": 0x541D,
        "TIOCGSERIAL": 0x541E,
        "TIOCSSERIAL": 0x541F,
        "TIOCPKT": 0x5420,
        "FIONBIO": 0x5421,
        "TIOCNOTTY": 0x5422,
        "TIOCSETD": 0x5423,
        "TIOCGETD": 0x5424,
        "TCSBRKP": 0x5425,
        "TIOCSBRK": 0x5427,
        "TIOCCBRK": 0x5428,
        "TIOCGSID": 0x5429,
        "TIOCGRS485": 0x542E,
        "TIOCSRS485": 0x542F,
        "TCGETX": 0x5432,
        "TCSETX": 0x5433,
        "TCSETXF": 0x5434,
        "TCSETXW": 0x5435,
        "FIONCLEX": 0x5450,
        "FIOCLEX": 0x5451,
        "FIOASYNC": 0x5452,
        "TIOCSERCONFIG": 0x5453,
        "TIOCSERGWILD": 0x5454,
        "TIOCSERSWILD": 0x5455,
        "TIOCGLCKTRMIOS": 0x5456,
        "TIOCSLCKTRMIOS": 0x5457,
        "TIOCSERGSTRUCT": 0x5458,
        "TIOCSERGETLSR": 0x5459,
        "TIOCSERGETMULTI": 0x545A,
        "TIOCSERSETMULTI": 0x545B,
        "TIOCMIWAIT": 0x545C,
        "TIOCGICOUNT": 0x545D,
    }
    IOCTL_TYPE_REVERSE = {}

    # ptrace request type #
    PTRACE_TYPE = [
        "PTRACE_TRACEME",  # 0
        "PTRACE_PEEKTEXT",
        "PTRACE_PEEKDATA",
        "PTRACE_PEEKUSR",
        "PTRACE_POKETEXT",
        "PTRACE_POKEDATA",
        "PTRACE_POKEUSR",
        "PTRACE_CONT",
        "PTRACE_KILL",
        "PTRACE_SINGLESTEP",  # 9
        "",
        "",
        "PTRACE_GETREGS",  # 12
        "PTRACE_SETREGS",  # 13
        "PTRACE_GETFPREGS",  # 14
        "PTRACE_SETFPREGS",  # 15
        "PTRACE_ATTACH",  # 16
        "PTRACE_DETACH",  # 17
        "PTRACE_GETFPXREGS",  # 18
        "PTRACE_SETFPXREGS",  # 19
        "",
        "",
        "",
        "PTRACE_SET_SYSCALL",  # 23
        "PTRACE_SYSCALL",  # 24
        "",
        "",
        "",
        "",
        "",
        "",
        "PTRACE_SYSEMU",  # 31
        "PTRACE_SYSEMU_SINGLESTEP",  # 32
    ]

    # ptrace event types #
    PTRACE_EVENT_TYPE = (
        [
            "PTRACE_EVENT_NONE",
            "PTRACE_EVENT_FORK",
            "PTRACE_EVENT_VFORK",
            "PTRACE_EVENT_CLONE",
            "PTRACE_EVENT_EXEC",
            "PTRACE_EVENT_VFORK_DONE",
            "PTRACE_EVENT_EXIT",
            "PTRACE_EVENT_SECCOMP",
        ]
        + ["NONE" for idx in xrange(120)]
        + ["PTRACE_EVENT_STOP"]
    )

    # perf event types #
    PERF_EVENT_TYPE = [
        "PERF_TYPE_HARDWARE",
        "PERF_TYPE_SOFTWARE",
        "PERF_TYPE_TRACEPOINT",
        "PERF_TYPE_HW_CACHE",
        "PERF_TYPE_RAW",
        "PERF_TYPE_BREAKPOINT",
    ]
    PERF_HW_EVENT_TYPE = [
        "PERF_COUNT_HW_CPU_CYCLES",
        "PERF_COUNT_HW_INSTRUCTIONS",
        "PERF_COUNT_HW_CACHE_REFERENCES",
        "PERF_COUNT_HW_CACHE_MISSES",
        "PERF_COUNT_HW_BRANCH_INSTRUCTIONS",
        "PERF_COUNT_HW_BRANCH_MISSES",
        "PERF_COUNT_HW_BUS_CYCLES",
        "PERF_COUNT_HW_STALLED_CYCLES_FRONTEND",
        "PERF_COUNT_HW_STALLED_CYCLES_BACKEND",
        "PERF_COUNT_HW_REF_CPU_CYCLES",
    ]
    PERF_SW_EVENT_TYPE = [
        "PERF_COUNT_SW_CPU_CLOCK",
        "PERF_COUNT_SW_TASK_CLOCK",
        "PERF_COUNT_SW_PAGE_FAULTS",
        "PERF_COUNT_SW_CONTEXT_SWITCHES",
        "PERF_COUNT_SW_CPU_MIGRATIONS",
        "PERF_COUNT_SW_PAGE_FAULTS_MIN",
        "PERF_COUNT_SW_PAGE_FAULTS_MAJ",
        "PERF_COUNT_SW_ALIGNMENT_FAULTS",
        "PERF_COUNT_SW_EMULATION_FAULTS",
        "PERF_COUNT_SW_DUMMY",
        "PERF_COUNT_SW_BPF_OUTPUT",
        "PERF_COUNT_SW_CGROUP_SWITCHES",
    ]
    PERF_CACHE_EVENT_TYPE = [
        "PERF_COUNT_HW_CACHE_L1D",
        "PERF_COUNT_HW_CACHE_L1I",
        "PERF_COUNT_HW_CACHE_LL",
        "PERF_COUNT_HW_CACHE_DTLB",
        "PERF_COUNT_HW_CACHE_ITLB",
        "PERF_COUNT_HW_CACHE_BPU",
        "PERF_COUNT_HW_CACHE_NODE",
    ]
    PERF_CACHE_EVENT_OP = [
        "PERF_COUNT_HW_CACHE_OP_READ",
        "PERF_COUNT_HW_CACHE_OP_WRITE",
        "PERF_COUNT_HW_CACHE_OP_PREFETCH",
    ]
    PERF_CACHE_EVENT_OPRES = [
        "PERF_COUNT_HW_CACHE_RESULT_ACCESS",
        "PERF_COUNT_HW_CACHE_RESULT_MISS",
    ]
    PERF_EVENT_SAMPLE = {
        "PERF_SAMPLE_IP": 1 << 0,
        "PERF_SAMPLE_TID": 1 << 1,
        "PERF_SAMPLE_TIME": 1 << 2,
        "PERF_SAMPLE_ADDR": 1 << 3,
        "PERF_SAMPLE_READ": 1 << 4,
        "PERF_SAMPLE_CALLCHAIN": 1 << 5,
        "PERF_SAMPLE_ID": 1 << 6,
        "PERF_SAMPLE_CPU": 1 << 7,
        "PERF_SAMPLE_PERIOD": 1 << 8,
        "PERF_SAMPLE_STREAM_ID": 1 << 9,
        "PERF_SAMPLE_RAW": 1 << 10,
        "PERF_SAMPLE_BRANCH_STACK": 1 << 11,
        "PERF_SAMPLE_REGS_USER": 1 << 12,
        "PERF_SAMPLE_STACK_USER": 1 << 13,
        "PERF_SAMPLE_WEIGHT": 1 << 14,
        "PERF_SAMPLE_DATA_SRC": 1 << 15,
        "PERF_SAMPLE_IDENTIFIER": 1 << 16,
        "PERF_SAMPLE_TRANSACTION": 1 << 17,
        "PERF_SAMPLE_REGS_INTR": 1 << 18,
        "PERF_SAMPLE_PHYS_ADDR": 1 << 19,
        "PERF_SAMPLE_AUX": 1 << 20,
        "PERF_SAMPLE_CGROUP": 1 << 21,
        "PERF_SAMPLE_DATA_PAGE_SIZE": 1 << 22,
        "PERF_SAMPLE_CODE_PAGE_SIZE": 1 << 23,
        "PERF_SAMPLE_WEIGHT_STRUCT": 1 << 24,
    }
    PERF_EVENT_READ_FORMAT = {
        "PERF_FORMAT_TOTAL_TIME_ENABLED": 1 << 0,
        "PERF_FORMAT_TOTAL_TIME_RUNNING": 1 << 1,
        "PERF_FORMAT_ID": 1 << 2,
        "PERF_FORMAT_GROUP": 1 << 3,
        "PERF_FORMAT_LOST": 1 << 4,
        "PERF_FORMAT_MAX": 1 << 5,
    }
    PERF_FLAG = {
        "PERF_FLAG_FD_NO_GROUP": 1 << 0,
        "PERF_FLAG_FD_OUTPUT": 1 << 1,
        "PERF_FLAG_PID_CGROUP": 1 << 2,
        "PERF_FLAG_FD_CLOEXEC": 1 << 3,
    }
    PERF_BRANCH_SAMPLE_SHIFT = [
        "PERF_SAMPLE_BRANCH_USER_SHIFT",  # user branches #
        "PERF_SAMPLE_BRANCH_KERNEL_SHIFT",  # kernel branches #
        "PERF_SAMPLE_BRANCH_HV_SHIFT",  # hypervisor branches #
        "PERF_SAMPLE_BRANCH_ANY_SHIFT",  # any branch types #
        "PERF_SAMPLE_BRANCH_ANY_CALL_SHIFT",  # any call branch #
        "PERF_SAMPLE_BRANCH_ANY_RETURN_SHIFT",  # any return branch #
        "PERF_SAMPLE_BRANCH_IND_CALL_SHIFT",  # indirect calls #
        "PERF_SAMPLE_BRANCH_ABORT_TX_SHIFT",  # transaction aborts #
        "PERF_SAMPLE_BRANCH_IN_TX_SHIFT",  # in transaction #
        "PERF_SAMPLE_BRANCH_NO_TX_SHIFT",  # not in transaction #
        "PERF_SAMPLE_BRANCH_COND_SHIFT",  # conditional branches #
        "PERF_SAMPLE_BRANCH_CALL_STACK_SHIFT",  # call/ret stack #
        "PERF_SAMPLE_BRANCH_IND_JUMP_SHIFT",  # indirect jumps #
        "PERF_SAMPLE_BRANCH_CALL_SHIFT",  # direct call #
        "PERF_SAMPLE_BRANCH_NO_FLAGS_SHIFT",  # no flags #
        "PERF_SAMPLE_BRANCH_NO_CYCLES_SHIFT",  # no cycles #
        "PERF_SAMPLE_BRANCH_TYPE_SAVE_SHIFT",  # save branch type #
        "PERF_SAMPLE_BRANCH_HW_INDEX_SHIFT",  # save low level index of raw branch records #
        "PERF_SAMPLE_BRANCH_PRIV_SAVE_SHIFT",  # save privilege mode #
    ]

    @staticmethod
    def getMmapId():
        if SysMgr.arch == "arm":
            return SysMgr.getNrSyscall("sys_mmap2")
        else:
            return SysMgr.getNrSyscall("sys_mmap")

    def __init__(self, mode):
        pass

    def __del__(self):
        pass


class UtilMgr(object):
    """Manager for utilities"""

    """
    [ TIPS ]
    - vim replacement
        - PROBLEM: replace all "type(???) is long" with "isinstance(???, (int, long))"
        - SOLUTION: %s/type(\(.*\)) is long/isinstance(\1, (int, long)/g
    """

    curTime = 0
    progressCnt = 0
    progressStr = 0
    progressChar = {0: "|", 1: "/", 2: "-", 3: "\\"}

    @staticmethod
    def startMemProf(reset=True):
        tracemalloc = SysMgr.getPkg("tracemalloc")
        if reset:
            tracemalloc.clear_traces()
        tracemalloc.start()

    @staticmethod
    def stopMemProf():
        SysMgr.getPkg("tracemalloc").stop()

    @staticmethod
    def getMemProf():
        return SysMgr.getPkg("tracemalloc").take_snapshot()

    @staticmethod
    def printMemProf(num=10, prev=None, backtrace=False):
        tracemalloc = SysMgr.getPkg("tracemalloc")
        snapshot = tracemalloc.take_snapshot()

        if prev:
            top_stats = snapshot.compare_to(prev, "lineno")
        else:
            top_stats = snapshot.statistics("lineno")

        print("[ Top %s%s ]" % ("Diff " if prev else "", num))
        for idx, stat in enumerate(top_stats[:num]):
            print("[%s] %s" % (idx, stat))
            if backtrace:
                for line in stat.traceback.format():
                    print(line)

    @staticmethod
    def printCtypeArray(addr, size, length=-1, isInt=True):
        # get ctypes object #
        SysMgr.importPkgItems("ctypes")

        # print as a bytearray #
        if isInt:
            res = list(map(long, string_at(addr, size)[:length]))
        else:
            res = string_at(addr, size)[:length]

        print(res)

    @staticmethod
    def convHtmlChar(string, skip=[]):
        chars = {"<": "&lt;", ">": "&gt;", "&": "&amp;"}

        for key, val in chars.items():
            if key in skip:
                continue
            string = string.replace(key, val)

        return string

    @staticmethod
    def makeReverseDict(orig, new):
        if not new:
            for name, value in orig.items():
                new[value] = name

    @staticmethod
    def getRange(start, end, step):
        def mrange(start, end, step):
            r = start
            while r < end:
                yield r
                r += step

        return [x for x in mrange(start, end, step)]

    @staticmethod
    def removeColor(string):
        if not SysMgr.ansiObj:
            ansi = r"(?:\x1B[@-_]|[\x80-\x9F])[0-?]*[ -/]*[@-~]"
            SysMgr.ansiObj = re.compile(ansi)
        return SysMgr.ansiObj.sub("", string)

    @staticmethod
    def convFloat2Str(val):
        val = ("%f" % val).rstrip("0")
        if val.endswith("."):
            val += "0"
        return val

    @staticmethod
    def convStr2Bytes(string):
        # build string #
        res = ""
        for byte in string:
            if isinstance(byte, str):
                byte = long(repr(struct.unpack("B", byte)[0]))
            bstr = "%x" % byte
            res = "{0:s}{1:0>2} ".format(res, bstr)
        return res

    @staticmethod
    def saveTime():
        UtilMgr.printTime(update=True, verb=False)

    @staticmethod
    def convLineStr(val, title, indent, maxLen, startIndent=True):
        resList = [val[i : i + maxLen] for i in xrange(0, len(val), maxLen)]

        res = ""
        cur = indent if startIndent else 1
        for string in resList:
            res += "{0:{cur}} {1:<1}\n".format(title, string, cur=cur)
            title = ""
            cur = indent

        return res.rstrip()

    @staticmethod
    def removeItems(target, val):
        while 1:
            try:
                if not val in target:
                    break
                target.remove(val)
            except SystemExit:
                sys.exit(0)
            except:
                pass

        return target

    @staticmethod
    def convStackStr(stack, indent, maxLen):
        newStack = ""
        nowLen = indent
        for call in stack.split(" <- "):
            if newStack and nowLen + len(call) + 4 > maxLen:
                newStack += "\n" + " " * indent
                nowLen = indent
            newStack += " <- " + call
            nowLen += 4 + len(call)
        return newStack

    @staticmethod
    def getInputNames(inputFile):
        if not inputFile:
            fileName = "guider"
            inputName = "N/A"
            inputList = []
        elif isinstance(inputFile, list):
            if SysMgr.outPath:
                fileName = SysMgr.outPath
            elif len(inputFile) == 1:
                fileName = inputFile[0]
            else:
                fileName = "merged"
            inputName = ", ".join(inputFile)
            inputList = inputFile
        else:
            fileName = inputFile
            inputName = inputFile
            inputList = [inputFile]

        return fileName, inputName, inputList

    @staticmethod
    def getSizeFilterFunc(var="SIZEFILTER"):
        # define size filter function #
        def _sizeChecker(val):
            return True

        # check size filter #
        if not var in SysMgr.environList:
            return _sizeChecker

        # get filter values #
        try:
            filters = SysMgr.environList[var][0]
            filterCode = filters[0]
            filterValue = UtilMgr.convUnit2Size(filters[1:])

            SysMgr.printInfo("applied %s '%s'" % (var, filters))
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printErr("failed to apply %s '%s'" % (var, filters), True)
            sys.exit(-1)

        # return function #
        if filterCode == "<":

            def _sizeChecker(val):
                if val < filterValue:
                    return True
                else:
                    return False

        elif filterCode == ">":

            def _sizeChecker(val):
                if val > filterValue:
                    return True
                else:
                    return False

        elif filterCode == "=":

            def _sizeChecker(val):
                if val == filterValue:
                    return True
                else:
                    return False

        else:
            SysMgr.printErr(
                "failed to apply %s '%s' because wrong operation"
                % (var, filters)
            )
            sys.exit(-1)

        return _sizeChecker

    @staticmethod
    def getPageAddrs(addr, length):
        PAGESIZE = SysMgr.PAGESIZE
        mod = addr % PAGESIZE
        start = addr - mod
        end = addr + length
        addr = start
        length = long((end - start + PAGESIZE - 1) / PAGESIZE) * PAGESIZE
        return start, end, addr, length

    @staticmethod
    def deepcopy(data):
        json = SysMgr.getPkg("marshal", False)
        if not json:
            json = SysMgr.getPkg("json", False)
        if json:
            return json.loads(json.dumps(data))
        else:
            return deepcopy(data)

    @staticmethod
    def printTime(name=None, update=True, verb=True):
        # get current time #
        now = time.time()
        if not UtilMgr.curTime and update:
            UtilMgr.curTime = now
            return

        diff = now - UtilMgr.curTime

        # update timestamp #
        if update:
            UtilMgr.curTime = now

        if not verb:
            return

        # add name #
        if name:
            prefix = "/ %s" % name
        else:
            prefix = ""

        # print time diff #
        string = "[ELAPSED] %f sec %s" % (diff, prefix)
        SysMgr.printWarn(string, True)

    @staticmethod
    def getCommonPart(strings):
        try:
            return os.path.commonprefix(strings)
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printErr("failed to get common part of %s" % strings)
            sys.exit(-1)

    @staticmethod
    def callPyFunc(path, fname, *args):
        try:
            # get function pointer #
            fullname = "%s_%s" % (path, fname)
            if fullname in SysMgr.externList:
                func = SysMgr.externList[fullname]
            elif sys.version_info < (3, 0, 0):
                execfile(path)  # pylint: disable=undefined-variable
                SysMgr.externList[fullname] = locals()[fname]
                func = SysMgr.externList[fullname]
            else:
                exec(open(path).read())
                SysMgr.externList[fullname] = locals()[fname]
                func = SysMgr.externList[fullname]

            # call the function and return #
            return func(args)
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printErr(
                "failed to call '%s()' from '%s'" % (func, path), True
            )
            sys.exit(-1)

    @staticmethod
    def printAllAttrs(objList, name, rep=None, nameFilter=[]):
        if not objList:
            return

        # get first item #
        obj = objList[0]

        # get attrs #
        attrs = sorted([attr for attr in dir(obj) if not attr.startswith("_")])

        # get extract flag #
        doExtract = "EXTRACT" in SysMgr.environList
        defPath = os.path.dirname(SysMgr.outPath) if SysMgr.outPath else "."

        # get printtree flag #
        printTree = "PRINTTREE" in SysMgr.environList
        treeList = {}

        def _applyTreeItem(treeList, items, val):
            if not isinstance(treeList, dict):
                return

            name = items.pop(0)

            if items:
                treeList.setdefault(name, {})
                _applyTreeItem(treeList[name], items, val)
            elif "file_size" in val:
                treeList[name] = val["file_size"]
            elif "size" in val:
                treeList[name] = val["size"]

        def _printTreeItems(data, indent=0):
            for key, value in data.items():
                if key:
                    if isinstance(value, long):
                        size = " (%s)" % UtilMgr.convSize2Unit(value)
                    else:
                        key = "[%s]" % key
                        size = ""

                    SysMgr.printPipe(
                        "  " * indent
                        + ("|-" if indent else "")
                        + str(key)
                        + size
                    )

                if isinstance(value, dict):
                    _printTreeItems(value, indent + 1)

        # print objects #
        for obj in objList:
            try:
                itemDict = {}
                for attr in attrs:
                    if not hasattr(obj, attr):
                        continue

                    val = getattr(obj, attr)
                    if not isinstance(val, (str, long, bytes)):
                        continue

                    itemDict[attr] = getattr(obj, attr)

                repName = (
                    getattr(obj, rep) if rep and hasattr(obj, rep) else ""
                )
                if nameFilter and not UtilMgr.isValidStr(repName, nameFilter):
                    continue

                if printTree:
                    _applyTreeItem(treeList, repName.split("/"), itemDict)
                    continue

                origRepName = repName
                if repName:
                    repName = "(%s)" % repName

                # print values #
                SysMgr.printPipe(
                    "\n[%s Info] %s\n%s" % (name, repName, twoLine)
                )
                for n, v in sorted(itemDict.items()):
                    if not v:
                        continue

                    try:
                        if isinstance(v, str):
                            v.decode("UTF-8")
                    except UnicodeDecodeError:
                        v = "BYTES"
                    except SystemExit:
                        sys.exit(0)
                    except:
                        pass

                    if isinstance(v, long):
                        hexstr = " (%s)" % hex(v)
                    else:
                        hexstr = ""

                    SysMgr.printPipe(
                        "{0:<20} {1:1}".format(n, str(v) + hexstr)
                    )
                SysMgr.printPipe(oneLine + "\n")

                # extract to file #
                if doExtract:
                    try:
                        SysMgr.doDecompress(args=[name], ffilter=[origRepName])
                    except SystemExit:
                        sys.exit(0)
                    except:
                        SysMgr.printErr(
                            "failed to extract '%s'(%s) to '%s'"
                            % (
                                origRepName,
                                name,
                                os.path.join(defPath, origRepName),
                            ),
                            True,
                        )
            except SystemExit:
                sys.exit(0)
            except:
                SysMgr.printErr("failed to print values for %s" % obj, True)

        _printTreeItems(treeList)

    @staticmethod
    def getTarFileList(path, perm="r|*", verb=True):
        tarfile = SysMgr.getPkg("tarfile", isExit=False)
        if not tarfile:
            return []

        try:
            with tarfile.open(path, perm) as fd:
                return fd.getmembers()
        except SystemExit:
            sys.exit(0)
        except:
            if not verb:
                return []
            SysMgr.printErr("failed to get tar info from '%s'" % path, True)
            return []

    @staticmethod
    def extractZipFile(path, target):
        zipfile = SysMgr.getPkg("zipfile", isExit=False)
        if not zipfile:
            return None

        try:
            with zipfile.ZipFile(path, "r") as fd:
                return fd.read(target)
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printErr("failed to get zip info from '%s'" % path, True)
            return None

    @staticmethod
    def getZipFileMap(path, onlyElf=False):
        # get zipfile object #
        try:
            zfd = SysMgr.getPkg("zipfile", False).ZipFile(path, "r")
            if not zfd:
                return {}
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printErr(
                "failed to get ZipFile object of '%s'" % path, True
            )
            return {}

        filemap = {}
        for item in zfd.filelist:
            fd = zfd.open(item.filename)

            if onlyElf:
                if not UtilMgr.isElfFile(fd=fd):
                    continue

            filemap[item.filename] = {
                "offset": fd._orig_compress_start,
                "size": item.file_size,
                "compsize": item.compress_size,
            }

        return filemap

    @staticmethod
    def getZipFileList(path):
        zipfile = SysMgr.getPkg("zipfile", isExit=False)
        if not zipfile:
            return []

        try:
            with zipfile.ZipFile(path, "r") as fd:
                return fd.infolist()
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printErr("failed to get zip info from '%s'" % path, True)
            return []

    @staticmethod
    def isElfFile(path=None, fd=None):
        try:
            if path:
                fd = open(path, "rb")
            return struct.unpack("4B", fd.read(4)) == (
                0x7F,
                ord("E"),
                ord("L"),
                ord("F"),
            )
        except IOError:
            SysMgr.printOpenErr(path)
            return False
        except SystemExit:
            sys.exit(0)
        except:
            return False

    @staticmethod
    def isArtFile(path):
        return SysMgr.readFile(path, size=3, verb=False, byte=True) == b"art"

    @staticmethod
    def isOatFile(path):
        # OAT header starts from oatdata symbol adderss (0x1000) #
        return (
            SysMgr.readFile(path, size=3, verb=False, byte=True, offset=0x1000)
            == b"oat"
        )

    @staticmethod
    def checkMagicNum(path, fd, _check):
        try:
            if fd:
                pos = fd.tell()
                ret = _check(fd)
                fd.seek(pos)
                return ret

            with open(path, "rb") as fd:
                return _check(fd)
        except SystemExit:
            sys.exit(0)
        except:
            return False

    @staticmethod
    def isBzipFile(path, fd=None):
        def _check(fd):
            return struct.unpack("BB", fd.read(2)) == (0x42, 0x5A)

        return UtilMgr.checkMagicNum(path, fd, _check)

    @staticmethod
    def isXzFile(path, fd=None):
        def _check(fd):
            # check .gz type #
            res = struct.unpack("6B", fd.read(6)) == (
                0xFD,
                0x37,
                0x7A,
                0x58,
                0x5A,
                0x00,
            )
            if res:
                return True
            fd.seek(0)
            # check .lzma type #
            return struct.unpack("5B", fd.read(5)) == (
                0x5D,
                0x00,
                0x00,
                0x80,
                0x00,
            )

        return UtilMgr.checkMagicNum(path, fd, _check)

    @staticmethod
    def isPKZipFile(path, fd=None):
        def _check(fd):
            return struct.unpack("4B", fd.read(4)) == (
                0x50,
                0x4B,
                0x07,
                0x08,
            )

        return UtilMgr.checkMagicNum(path, fd, _check)

    @staticmethod
    def isZipFile(path, fd=None):
        def _check(fd):
            return struct.unpack("4B", fd.read(4)) == (
                0x50,
                0x4B,
                0x03,
                0x04,
            )

        return UtilMgr.checkMagicNum(path, fd, _check)

    @staticmethod
    def isTarFile(path, fd=None):
        def _check(fd):
            fd.seek(257)
            return struct.unpack("5B", fd.read(5)) == (
                0x75,
                0x73,
                0x74,
                0x61,
                0x72,
            )

        return UtilMgr.checkMagicNum(path, fd, _check)

    @staticmethod
    def isZlibFile(path, fd=None):
        def _check(fd):
            magic = fd.read(4)
            return (magic == b"ZLIB") or (
                struct.unpack("BB", magic[:2]) == (0x78, 0x9C)
            )

        return UtilMgr.checkMagicNum(path, fd, _check)

    @staticmethod
    def isGzipFile(path, fd=None):
        def _check(fd):
            return struct.unpack("BB", fd.read(2)) == (0x1F, 0x8B)

        return UtilMgr.checkMagicNum(path, fd, _check)

    @staticmethod
    def sort(l, val=None, reverse=False):
        convert = lambda text: float(text) if text.isdigit() else text
        alphanum = lambda key: [
            convert(c) for c in re.split("([-+]?[0-9]+\.?[0-9]*)", key)
        ]

        if isinstance(l, list):
            l.sort(key=alphanum, reverse=reverse)
            return l
        elif isinstance(l, dict):
            return sorted(
                l.items(),
                key=lambda e: alphanum(e[1][val] if val else e[0]),
                reverse=reverse,
            )
        else:
            SysMgr.printWarn(
                "failed to sort items because '%s' is not supported" % type(l),
                True,
            )
            return l

    @staticmethod
    def compareSyscallSuperset():
        syscallList = (
            ConfigMgr.SYSCALL_COMMON
            + ConfigMgr.SYSCALL_COMMON32
            + ConfigMgr.SYSCALL_X86
            + ConfigMgr.SYSCALL_X64
            + ConfigMgr.SYSCALL_ARM
            + ConfigMgr.SYSCALL_AARCH64
            + ConfigMgr.SYSCALL_RISCV64
        )
        syscallList = set(syscallList)

        protoList = set(
            ["sys_%s" % name for name in ConfigMgr.SYSCALL_PROTOTYPES]
        )

        # print final diff list #
        SysMgr.printPipe("--- NO PROTOTYPE ---")
        for name in sorted(list(syscallList - protoList)):
            SysMgr.printPipe(name)

        SysMgr.printPipe("\n--- NO DEFINITION ---")
        for name in sorted(list(protoList - syscallList)):
            SysMgr.printPipe(name)

    @staticmethod
    def isBitEnabled(num, bits):
        if not bits:
            return None

        try:
            num = long(num)
        except SystemExit:
            sys.exit(0)
        except:
            num = long(num, 16)

        try:
            bits = long(bits)
        except SystemExit:
            sys.exit(0)
        except:
            bits = long(bits, 16)

        if bits & (1 << num - 1):
            return True
        else:
            return False

    @staticmethod
    def convxrange(targetList):
        result = []

        for item in targetList:
            try:
                nums = item.split(":")
                if len(nums) == 1:
                    result.append(item)
                elif len(nums) == 2:
                    for num in xrange(long(nums[0]), long(nums[1]) + 1):
                        result.append(str(num))
                else:
                    raise Exception()
            except SystemExit:
                sys.exit(0)
            except:
                SysMgr.printErr("wrong number range '%s'" % item, True)
                sys.exit(-1)

        return result

    @staticmethod
    def unionItem(targetList):
        result = []
        dictionary = {}

        for item in targetList:
            if item in dictionary:
                continue

            result.append(item)
            dictionary[item] = None

        return result

    @staticmethod
    def cleanItem(targetList, union=True):
        targetType = type(targetList)

        if targetType is str:
            targetStr = ""
            for val in targetList:
                if val:
                    targetStr += val
            return targetStr
        elif targetType is list:
            # remove redundant values #
            if union:
                targetList = UtilMgr.unionItem(targetList)

            # remove empty values #
            newList = []
            for val in targetList:
                val = val.strip()
                if val:
                    newList.append(val)

            return newList
        else:
            return targetList

    @staticmethod
    def genRangeDict(prefix, startIdx, endIdx, valStart):
        dictList = {}
        for index in xrange(startIdx, endIdx + 1):
            name = "%s%s" % (prefix, index)
            value = valStart + index - startIdx
            dictList[name] = value
        return dictList

    @staticmethod
    def printHist(table, title, unit, printer=None, newline=""):
        if not table:
            return

        convNum = UtilMgr.convNum
        if not printer:
            printer = SysMgr.printPipe

        # pop stats #
        try:
            statmin = table.pop("min", None)
            statmax = table.pop("max", None)
            statcnt = table.pop("cnt", None)
            statsum = table.pop("sum", None)
        except SystemExit:
            sys.exit(0)
        except:
            pass

        lineLen = 43
        printer(
            "\n[%s Histogram] (unit:%s)\n%s%s"
            % (title, unit, twoLine[:lineLen], newline[:lineLen])
        )
        printer(
            "{0:^21} | {1:^18}|\n{2:1}{3:1}".format(
                "Range", "Count", oneLine[:lineLen], newline[:lineLen]
            )
        )

        for digit, cnt in sorted(table.items()):
            srange = long(pow(2, digit - 1))

            if srange == 0:
                erange = 1
            else:
                erange = long((srange << 1) - 1)

            printer(
                "{0:10}-{1:>10} | {2:>10}({3:5.1f}%)|{4:1}".format(
                    convNum(srange),
                    convNum(erange),
                    convNum(cnt),
                    (cnt / float(statcnt)) * 100,
                    newline[:lineLen],
                )
            )

        try:
            printer(
                (
                    "{0:1}\n{1:^21} | {2:>18}|\n{3:^21} | {4:>18}|\n"
                    "{5:^21} | {6:>18}|\n{7:^21} | {8:>18}|\n{0:1}\n"
                ).format(
                    oneLine[:lineLen],
                    "Min",
                    convNum(statmin),
                    "Max",
                    convNum(statmax),
                    "Cnt",
                    convNum(statcnt),
                    "Sum",
                    convNum(statsum),
                )
            )
        except SystemExit:
            sys.exit(0)
        except:
            pass

    @staticmethod
    def ungzip(path):
        gzip = SysMgr.getPkg("gzip")
        return gzip.open(path, "rb").read()

    @staticmethod
    def unzip(fd, name):
        # get pkg #
        zlib = SysMgr.getPkg("zlib", False)
        if not zlib:
            return None, None

        # check format #
        magic = fd.read(4)
        if magic != b"ZLIB":
            SysMgr.printWarn(
                "wrong zlib magic number '%s' for %s section" % (magic, name)
            )
            return None, None

        compSize = struct.unpack(">Q", fd.read(8))[0]

        uncompBytes = b""
        decompressor = zlib.decompressobj()
        PAGESIZE = SysMgr.PAGESIZE

        # define getsize func #
        if hasattr(fd, "getsize"):

            def _getsize(fd):
                return fd.getsize()

        else:

            def _getsize(fd):
                return os.fstat(fd.fileno()).st_size

        while 1:
            remain = _getsize(fd) - fd.tell()
            size = PAGESIZE if remain >= PAGESIZE else remain
            chunk = fd.read(size)
            if not chunk:
                break
            uncompBytes += decompressor.decompress(chunk)

        uncompBytes += decompressor.flush()

        if len(uncompBytes) != compSize:
            SysMgr.printWarn(
                (
                    "failed to decompress %s section because "
                    "decompressed size is wrong [record: %s, actual: %s]"
                )
                % (name, compSize, len(uncompBytes))
            )

        return uncompBytes, len(uncompBytes)

    @staticmethod
    def convList2Histo(items, dtype="float", mult=1):
        def _getRangeIdx(value):
            if value <= 1:
                return 0

            digit = 0
            while 1:
                digit += 1
                value = value >> 1
                if value == 0:
                    return digit

        if not items:
            return None

        # convert type #
        if dtype == "float":
            if not isinstance(items[0], float):
                items = list(map(float, items))
        elif dtype == "long":
            if not isinstance(items[0], long):
                items = list(map(long, items))
        else:
            return None

        # convert unit #
        items = list(map(lambda x: long(x * mult), items))

        # get stats #
        histDict = {
            "max": max(items),
            "min": min(items),
            "cnt": len(items),
            "sum": sum(items),
        }

        for value in items:
            digit = _getRangeIdx(value)
            histDict.setdefault(digit, 0)
            histDict[digit] += 1

        return histDict

    @staticmethod
    def getSigList():
        sigList = dict(
            (k, v)
            for v, k in reversed(sorted(signal.__dict__.items()))
            if v.startswith("SIG") and not v.startswith("SIG_")
        )
        return sigList

    @staticmethod
    def getEnvironNum(
        name, isExit=True, default=None, verb=True, isInt=False, isFloat=False
    ):
        try:
            # get first value #
            value = SysMgr.environList[name][0]

            # return by type for compatibility #
            if isInt:
                return int(value)
            elif isFloat:
                return float(value)
            else:
                return long(value)
        except SystemExit:
            sys.exit(0)
        except:
            if verb:
                SysMgr.printErr(
                    "failed to get the number of %s variable" % name, True
                )

            if isExit:
                sys.exit(-1)

            return default

    @staticmethod
    def parseCommand(option):
        if not UtilMgr.isString(option):
            return option

        stringList = {}

        # process strings in "" #
        strings = re.findall('"(.*?)"', option)
        if strings:
            # create an dictionary for strings #
            for idx, item in enumerate(strings):
                if not item:
                    continue

                val = "#%s#" % idx
                stringList.setdefault(item.strip('"'), val)

            # replace strings #
            for string, value in stringList.items():
                option = option.replace('"%s"' % string, value)

        # split the option string #
        option = option.split(" ")
        for string, value in stringList.items():
            for idx, item in enumerate(deepcopy(option)):
                if value in item:
                    option[idx] = item.replace(value, string)

        return option

    @staticmethod
    def convList2Dict(optList, sep=":", cap=False):
        newDict = {}
        for item in optList:
            try:
                # get values #
                values = item.split(sep, 1)
                if len(values) == 1:
                    key = values[0]
                    value = "SET"
                else:
                    key, value = values

                # change to capital #
                if cap:
                    key = key.upper()

                # set values #
                if key in newDict:
                    newDict[key].append(value)
                else:
                    newDict[key] = [value]
            except SystemExit:
                sys.exit(0)
            except:
                SysMgr.printWarn(
                    "failed to parse %s by separator %s" % (item, sep)
                )
                continue

        return newDict

    @staticmethod
    def getStdev(data):
        def _variance(data, ddof=0):
            n = len(data)
            mean = sum(data) / n
            v = n - ddof
            diffs = [(x - mean) ** 2 for x in data]
            try:
                return __builtins__.sum(diffs) / v
            except SystemExit:
                sys.exit(0)
            except:
                return sum(diffs) / v

        var = _variance(data)
        math = SysMgr.getPkg("math")
        stdDev = math.sqrt(var)
        return stdDev

    @staticmethod
    def splitString(string):
        string = string.replace("\,", "$%")

        clist = string.split(",")

        for idx, item in enumerate(list(clist)):
            clist[idx] = item.replace("$%", ",").strip()

        return clist

    @staticmethod
    def getDrawOutputPath(inputPath, name, suffix=False):
        # set output path #
        if SysMgr.outPath:
            outputPath = SysMgr.outPath

            # check dir #
            if os.path.isdir(outputPath):
                outputFileName = (
                    "%s.svg" % os.path.splitext(os.path.basename(inputPath))[0]
                )
                outputPath = os.path.join(outputPath, outputFileName)
            elif suffix:
                dirName = os.path.dirname(os.path.realpath(inputPath))
                fileName = "%s_%s.svg" % (
                    os.path.splitext(os.path.basename(inputPath))[0],
                    name,
                )
                outputPath = os.path.join(dirName, fileName)
        else:
            outputPath = UtilMgr.prepareForImageFile(inputPath, name)

        return os.path.realpath(outputPath)

    @staticmethod
    def printSyscalls(systable):
        bufstring = ""
        for idx, syscall in enumerate(systable):
            if idx % 4 == 0:
                bufstring += "\n"
            bufstring = "%s'%s', " % (bufstring, syscall)

        sys.exit("%s\ntotal: %s" % (bufstring, len(systable)))

    @staticmethod
    def isValidStrAll(string, key=None, inc=False, ignCap=False):
        if not string or not key:
            return False

        final = True
        for item in key:
            if not UtilMgr.isValidStr(string, [item], inc, ignCap):
                final = False
                break

        return final

    @staticmethod
    def isValidStr(string, key=None, inc=False, ignCap=False):
        if not string:
            return False

        if not key:
            if not SysMgr.filterGroup:
                return True
            key = SysMgr.filterGroup

        if ignCap:
            string = string.lower()

        for cond in list(key):
            if not cond:
                continue

            cond = str(cond)

            if ignCap:
                cond = cond.lower()

            if inc:
                if cond in string:
                    return True
            else:
                if cond == "*":
                    return True
                elif (
                    cond[0] == "*"
                    and cond[-1] == "*"
                    and cond.strip("*") in string
                ):
                    return True
                elif cond[0] == "*" and string.endswith(cond[1:]):
                    return True
                elif cond[-1] == "*" and string.startswith(cond[:-1]):
                    return True
                elif cond == string:
                    return True

        return False

    @staticmethod
    def convWord2Str(word):
        try:
            return struct.pack("L", word)
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printErr("failed to convert word %s to string" % word, True)
            return None

    @staticmethod
    def lstrip(string, wordList):
        """
        str.lstrip() removes all heading characters including all characters
        in the string argument.
        ex) '12345'.lstrip('321') -> '45'
        ex) '12345'.lstrip('31') -> '2345'
        """

        if not isinstance(wordList, list):
            wordList = [wordList]

        for word in wordList:
            if string.startswith(word):
                string = string[len(word) :]

        return string

    @staticmethod
    def rstrip(string, wordList):
        """
        str.rstrip() removes all ending characters including all characters
        in the string argument.
        ex) '12345'.rstrip('543') -> '12'
        ex) '12345'.rstrip('53') -> '1234'
        """

        if not isinstance(wordList, list):
            wordList = [wordList]

        for word in wordList:
            if string.endswith(word):
                return string[: -len(word)]

        return string

    @staticmethod
    def strip(string, wordList):
        ret = UtilMgr.lstrip(string, wordList)
        ret = UtilMgr.rstrip(ret, wordList)
        return ret

    @staticmethod
    def getInodes(
        path,
        inodeFilter=[],
        nameFilter=[],
        exceptFilter=[],
        fileAttr=None,
        verb=True,
    ):
        inodeList = {}

        for r, d, f in os.walk(path):
            # get full path for upper dir #
            fdir = os.path.realpath(r)
            if not FileAnalyzer.isValidFile(fdir):
                continue

            # print progress #
            UtilMgr.printProgress()

            for name in f + d:
                # get full path for file ##
                fpath = os.path.join(fdir, name)

                # check name filter #
                if nameFilter and not UtilMgr.isValidStr(name, nameFilter):
                    continue
                elif exceptFilter and UtilMgr.isValidStr(fpath, exceptFilter):
                    continue

                # check inode filter #
                try:
                    fstat = os.stat(fpath)
                    if not fstat:
                        continue
                except SystemExit:
                    sys.exit(0)
                except:
                    SysMgr.printWarn(
                        "failed to get stat for %s" % fpath, reason=True
                    )
                    continue

                # get inode #
                inode = str(fstat.st_ino)
                if inodeFilter and not inode in inodeFilter:
                    continue

                # make device ID #
                major = os.major(fstat.st_dev)
                minor = os.minor(fstat.st_dev)
                devid = "%s:%s" % (major, minor)
                fpath = os.path.join(fdir, name)

                # register inode #
                inodeList.setdefault(devid, {})
                inodeList[devid][inode] = fpath

                # register file attribute #
                if isinstance(fileAttr, dict):
                    fileAttr[fpath] = fstat

        return inodeList

    @staticmethod
    def gerPermutation(inputList, union=False):
        visited = [0 for _ in xrange(len(inputList))]
        returnList = []

        def dfs(cnt, items):
            if cnt == len(inputList):
                returnList.append(items[:])
                return

            for i, val in enumerate(inputList):
                # check union #
                if union and visited[i] == 1:
                    continue

                # add item and check visit flag #
                items.append(val)
                visited[i] = 1

                dfs(cnt + 1, items)

                # remove item and uncheck visit flag #
                items.pop()
                visited[i] = 0

        dfs(0, [])

        return returnList

    @staticmethod
    def getPath(path):
        # get file info #
        filename = os.path.basename(path)
        dirname = os.path.dirname(path)
        if not dirname:
            dirname = "."
        return dirname, filename

    @staticmethod
    def getFiles(path, name=None, incFile=True, incDir=False, recursive=True):
        flist = []

        for r, d, f in os.walk(path):
            for inc, target in ((incFile, f), (incDir, d)):
                target = target if inc else []
                for s in target:
                    if name and not UtilMgr.isValidStr(s, name):
                        continue
                    flist.append(os.path.join(r, s))

            if not recursive:
                break

        return flist

    @staticmethod
    def convStr2Time(timestr, timefmt=None, isSec=True, verb=True):
        datetime = SysMgr.getPkg("datetime", False)
        if not datetime:
            return None

        if not timefmt:
            timefmt = "%Y-%m-%dT%H:%M:%SZ"

        try:
            timeObj = datetime.datetime.strptime(timestr, timefmt)
            if isSec:
                # seconds from UTC 1970-01-01 00:00:00 #
                return (
                    timeObj - datetime.datetime.fromtimestamp(0)
                ).total_seconds()
            else:
                return timeObj
        except SystemExit:
            sys.exit(0)
        except:
            if verb:
                SysMgr.printWarn(
                    "failed to convert '%s' to time" % timestr, True, True
                )
            return None

    @staticmethod
    def getEpoch2Start(utc=False):
        datetime = SysMgr.getPkg("datetime", False)
        if not datetime:
            return None

        # get epoch time #
        if utc:
            epoch = datetime.datetime.utcnow().timestamp()
        else:
            epoch = datetime.datetime.now().timestamp()

        return epoch - SysMgr.getUptime()

    @staticmethod
    def getClockTime(ctype="CLOCK_MONOTONIC", dlt=False):
        # get ctypes object #
        SysMgr.importPkgItems("ctypes")

        class timespec(Structure):
            _fields_ = [("tv_sec", c_long), ("tv_nsec", c_long)]

        # load libc #
        SysMgr.loadLibcObj(exit=True)

        # define arg types for free() #
        SysMgr.libcObj.clock_gettime.argtypes = [c_int, POINTER(timespec)]

        # create an object #
        t = timespec()

        # get time #
        ret = SysMgr.libcObj.clock_gettime(
            ConfigMgr.CLOCK_TYPE[ctype], pointer(t)
        )
        if ret != 0:
            SysMgr.printErr("failed to get clock %s time" % ctype, True)
            return -1

        # return time #
        if dlt:
            return (
                c_uint32(t.tv_sec * 10000).value / 10000
                + c_uint32(t.tv_nsec).value / 1000000000
            )
        else:
            return t.tv_sec + t.tv_nsec * 1e-9

    @staticmethod
    def getTime(utc=False):
        datetime = SysMgr.getPkg("datetime", False)
        if not datetime:
            return None

        if utc:
            timeobj = datetime.datetime.utcnow()
        else:
            timeobj = datetime.datetime.now()
        return timeobj.strftime("%Y-%m-%dT%H:%M:%SZ")

    @staticmethod
    def getFileList(flist, sort=False, exceptDir=False, verb=True):
        if not flist or not isinstance(flist, list):
            return []

        rlist = []
        for item in flist:
            item = item.strip()
            if item.startswith("-"):
                break

            # apply regex for path #
            ilist = UtilMgr.convPath(item)
            if not ilist:
                if verb:
                    SysMgr.printWarn(
                        "failed to find any file related to '%s'" % item, True
                    )
            elif UtilMgr.isString(ilist):
                rlist.append(ilist)
            elif isinstance(ilist, list):
                rlist += ilist

        if not rlist:
            return

        # check redundant files #
        if len(rlist) != len(set(rlist)):
            if verb:
                SysMgr.printWarn(
                    "detected redundant files in [ %s ]" % ", ".join(rlist),
                    True,
                )

        # exclude dir #
        if exceptDir:
            newlist = []
            for fname in rlist:
                if os.path.isdir(fname):
                    continue
                newlist.append(fname)
            rlist = newlist

        if not rlist:
            return

        # get filter list #
        targetList, exceptList = SysMgr.getTargetFileList()
        isValidStr = UtilMgr.isValidStr

        # apply filter #
        if targetList or exceptList:
            newlist = []
            for fname in rlist:
                if targetList and not isValidStr(fname, targetList):
                    continue
                elif exceptList and isValidStr(fname, exceptList):
                    continue
                newlist.append(fname)
            rlist = newlist

        # apply list filter #
        start = end = None
        ffilter = SysMgr.environList.get("FILELIST", [":"])[0].split(":", 1)
        try:
            if len(ffilter) == 1:
                ffilter.append(None)
            start, end = [long(f) if f else None for f in ffilter]
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printErr("failed to get FILELIST variables", True)
            sys.exit(0)

        # remove redundant files #
        if sort:
            return sorted(rlist)[start:end]
        else:
            return rlist[start:end]

    @staticmethod
    def decodeArg(value):
        try:
            text = repr(value.decode())
        except SystemExit:
            sys.exit(0)
        except:
            text = value

        # define start index by encoding type #
        if isinstance(text, bytes):
            start = 2
        else:
            start = 1

        return text[start:].rstrip("'")

    @staticmethod
    def convBin2Str(path, pos=False):
        try:
            if sys.version_info < (3, 0):
                fd = open(path, "rb")
            else:
                fd = open(path, encoding="latin-1")

            content = fd.read()
            strList = list(re.findall("[^\x00-\x1F\x7F-\xFF]{4,}", content))

            # add position #
            if pos:
                lastPos = 0
                dictList = {}
                for item in strList:
                    dictList.setdefault(item, content.find(item, lastPos))
                    lastPos = dictList[item]
                return dictList

            return strList
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printErr("failed to convert '%s' to strings" % path, True)
            return False

    @staticmethod
    def bisect_left(a, x, lo=0, hi=None):
        # copied from python standard library bisect.py #
        if lo < 0:
            raise ValueError("lo must be non-negative")
        if not hi:
            hi = len(a)
        while lo < hi:
            mid = (lo + hi) // 2
            if a[mid] <= x:
                lo = mid + 1
            else:
                hi = mid
        return lo

    @staticmethod
    def getFlagBit(vlist, flist):
        num = 0

        for flag in flist:
            try:
                num |= vlist[flag]
            except:
                SysMgr.printErr(
                    "failed to get %s in [%s]" % (flag, "|".join(list(vlist)))
                )
                sys.exit(-1)

        return num

    @staticmethod
    def getFlagList(value, flist, num="hex"):
        rlist = []
        numVal = long(value)

        for name, bits in flist.items():
            if numVal & bits:
                rlist.append(name)
        return rlist

    @staticmethod
    def getFlagString(value, flist, num="hex", zero=False):
        string = ""
        numVal = long(value)

        # get sorted dictionary vaue #
        if sys.version_info >= (3, 7, 0):
            targetList = flist
        else:
            targetList = sorted(list(flist))

        for bit in targetList:
            try:
                if numVal - bit < 0:
                    break
                elif numVal & bit:
                    string = "%s%s|" % (string, flist[bit])
            except SystemExit:
                sys.exit(0)
            except:
                SysMgr.printWarn(
                    "failed to get flag info for %s" % value, reason=True
                )

        # check value for 0 index in limited variable range #
        if 0 in flist:
            if zero:
                string = "%s|%s" % (flist[0], string)
            elif num == "hex" and numVal & 0xF == 0:
                string = "%s|%s" % (flist[0], string)
            elif num == "oct" and numVal & 0o7 == 0:
                string = "%s|%s" % (flist[0], string)
            elif num == "bin" and numVal & 1 == 0:
                string = "%s|%s" % (flist[0], string)

        if string:
            return string[:-1]
        else:
            return value

    @staticmethod
    def encodeBase64(value):
        base64 = SysMgr.getPkg("base64", False)

        try:
            return base64.b64encode(value)
        except SystemExit:
            sys.exit(0)
        except:
            return value

    @staticmethod
    def readLEB128(fd):
        data = None
        while 1:
            char = fd.read(1)
            if ord(char) & 0x80 == 0:
                break
            elif not data:
                data = char
            else:
                data += char
        return data

    @staticmethod
    def decodeSLEB128(obj):
        size = 1
        value = 0

        # get size #
        for i, b in enumerate(obj):
            b = b if isinstance(b, long) else ord(b)
            value += (b & 0x7F) << (i * 7)
            if (b & 0x80) == 0:
                break
            size += 1

        # decode data #
        obj = obj[:size]
        if ord(obj[-1]) & 0x40:
            # negative -> sign extend
            value |= -(1 << (7 * len(obj)))

        return value, size

    @staticmethod
    def decodeULEB128(obj):
        size = 1
        value = 0

        # get size #
        for b in obj:
            val = b if isinstance(b, long) else ord(b)
            if (val & 0x80) == 0:
                break
            size += 1

        # decode data #
        for b in reversed(obj[:size]):
            b = b if isinstance(b, long) else ord(b)
            value = (value << 7) + (b & 0x7F)

        return value, size

    @staticmethod
    def isCompressed(fname=None, fd=None):
        # file name #
        if fname:
            try:
                fd = open(fname, "rb")
            except SystemExit:
                sys.exit(0)
            except:
                SysMgr.printOpenErr(fname)
                sys.exit(-1)

        # file descriptor #
        if fd:
            data = fd.read(2)
            fd.seek(0, 0)
            if struct.unpack("BB", data) == (0x1F, 0x8B):
                return True
            else:
                return False

        # no value #
        return False

    @staticmethod
    def conv2BitStr(content):
        return bin(content)

    @staticmethod
    def decodeBase64(value):
        base64 = SysMgr.getPkg("base64", False)

        try:
            return base64.b64decode(value)
        except SystemExit:
            sys.exit(0)
        except:
            return value

    @staticmethod
    def encodeStr(value):
        try:
            return value.encode()
        except SystemExit:
            sys.exit(0)
        except:
            try:
                return value.encode("utf8", "surrogateescape")
            except SystemExit:
                sys.exit(0)
            except:
                return value

    @staticmethod
    def isString(value):
        if isinstance(value, str):
            return True
        elif sys.version_info >= (3, 0, 0):
            if isinstance(value, bytes):
                return True
        # pylint: disable-next=undefined-variable
        elif isinstance(value, unicode):
            return True
        return False

    @staticmethod
    def isFloat(value):
        if isinstance(value, float):
            return True
        elif isinstance(value, str):
            try:
                float(value)
                return True
            except SystemExit:
                sys.exit(0)
            except:
                return False
        else:
            return False

    @staticmethod
    def isNumber(value):
        if isinstance(value, (int, long)):
            return True
        elif isinstance(value, str):
            if value.isdigit():
                return True

            try:
                if value.startswith("0x") and long(value, 16):
                    return True
                else:
                    return False
            except SystemExit:
                sys.exit(0)
            except:
                return False
        else:
            return False

    @staticmethod
    def getTextLines(fname, verb=False, retfd=False, load=True):
        buf = []

        if verb:
            # get output size #
            fsize = UtilMgr.getFileSizeStr(fname)

            # set job type #
            if load:
                job = "loading"
            else:
                job = "checking"

            SysMgr.printStat(r"start %s '%s'%s..." % (job, fname, fsize))

        # open compressed file #
        fd = SysMgr.getCompFd(fname)

        # open normal file #
        try:
            if not fd:
                fd = open(fname, "r", encoding="utf-8")
        except SystemExit:
            sys.exit(0)
        except:
            fd = open(fname, "r")

        # just return fd #
        if retfd:
            return fd

        # get total size #
        try:
            totalSize = os.stat(fname).st_size
        except SystemExit:
            sys.exit(0)
        except:
            totalSize = 0

        # read data from a file #
        while 1:
            try:
                data = fd.readline()
                if not data:
                    break

                buf.append(data)

                if verb:
                    UtilMgr.printProgress(fd.tell(), totalSize)
            except SystemExit:
                sys.exit(0)
            except:
                break

        if verb:
            UtilMgr.deleteProgress()

        try:
            fd.close()
        except:
            pass

        return buf

    @staticmethod
    def convPath(value, retStr=False, isExit=False, separator=" ", warn=True):
        def _convType(retStr, res):
            # str #
            if retStr:
                if res:
                    return sorted(separator.join(res))
                else:
                    return ""
            # list #
            else:
                if res:
                    return sorted(res)
                else:
                    return []

        # strip path #
        value = value.strip()

        # get relative path option #
        isRelative = "RELPATH" in SysMgr.environList

        # all files including hidden files from specific directory recursively #
        if value.endswith("/**"):
            glob = None
            value = UtilMgr.rstrip(value, "/**")

            res = []
            for r, d, f in os.walk(value):
                if not isRelative:
                    r = os.path.realpath(r)
                res.append(r)
                for item in f:
                    res.append(os.path.join(r, item))

            return _convType(retStr, res)

        # all files including hidden files from specific directory recursively #
        glob = SysMgr.getPkg("glob", False)
        if glob:
            """
            # sort option #
            glob(value, key=os.path.getctime)
            glob(value, key=os.path.getatime)
            glob(value, key=os.path.getmtime)
            glob(value, key=os.path.getsize)
            """

            # check recursive path #
            if "**" in value:
                if sys.version_info >= (3, 11):
                    res = glob.glob(value, recursive=True, include_hidden=True)
                elif sys.version_info >= (3, 5):
                    res = glob.glob(value, recursive=True)
                else:
                    res = glob.glob(value)
            else:
                res = glob.glob(value)

            if not res and isExit:
                SysMgr.printErr("failed to find a file matching '%s'" % value)
                sys.exit(-1)

            return _convType(retStr, res)
        elif warn:
            if "*" in value:
                SysMgr.printWarn(
                    "failed to handle * character in the path "
                    "because there is no glob package",
                    True,
                )

        # str #
        if retStr:
            return value
        # list #
        else:
            return [value]

    @staticmethod
    def convStr2Word(bstring):
        try:
            return struct.unpack("L", bstring)[0]
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printErr("failed to convert string %s to word" % bstring)
            return None

    @staticmethod
    def convOverlayPath(path, overlayfsList):
        fileList = []
        itemList = ["lowerdir", "upperdir"]

        for pos, info in overlayfsList.items():
            if not pos in path:
                continue

            for target in itemList:
                for item in info[target]:
                    fullpath = item + path
                    if os.path.exists(fullpath):
                        fileList.append(fullpath)

        # return recent path #
        if fileList:
            return fileList[-1]
        else:
            return path

    @staticmethod
    def convStr2Num(string, verb=True, onlyHex=False):
        try:
            if isinstance(string, (int, long)):
                return string
            elif not onlyHex and string.isdigit():
                string = long(string)
            else:
                string = long(string, 16)
        except SystemExit:
            sys.exit(0)
        except:
            if verb:
                SysMgr.printErr(
                    "failed to convert %s to number" % string, True
                )
            return None

        return string

    @staticmethod
    def convNumIgn(number, isFloat=False, floatDigit=1):
        return str(number)

    @staticmethod
    def convNum(number, isFloat=False, floatDigit=1):
        try:
            if number == 0:
                return "0"
            elif isFloat:
                numstr = format(round(float(number), floatDigit), ",")
                first, second = numstr.split(".", 1)
                return first + "." + second[:floatDigit]
            else:
                return format(long(number), ",")
        except SystemExit:
            sys.exit(0)
        except:
            return number

    @staticmethod
    def convCpuColor(value, string=None, size=1, align="right"):
        if string is None:
            string = value

        if value >= SysMgr.cpuPerHighThr:
            return UtilMgr.convColor(string, "RED", size, align)
        elif value > 0:
            return UtilMgr.convColor(string, "YELLOW", size, align)
        else:
            return str(string)

    @staticmethod
    def convColor(string, color="LIGHT", size=1, align="right", force=False):
        # check skip condition #
        if not color:
            return str(string)
        elif force and not SysMgr.isWindows:
            pass
        elif not SysMgr.colorEnable:
            return str(string)
        elif SysMgr.outPath or SysMgr.jsonEnable:
            SysMgr.colorEnable = False
            return str(string)

        # direction #
        if align == "right":
            direct = ">"
        else:
            direct = "<"
        string = "{0:{direct}{size}}".format(
            str(string), direct=direct, size=size
        )

        # add color characters #
        try:
            return "%s%s%s" % (
                ConfigMgr.COLOR_LIST[color],
                string,
                ConfigMgr.ENDC,
            )
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printWarn(
                "failed to convert color for %s" % color, reason=True
            )
            return str(string)

    @staticmethod
    def convSize2UnitIgn(size, isInt=False, unit=None):
        return str(size)

    @staticmethod
    def convSize2Unit(size, isInt=False, unit=None):
        sizeKB = 1024
        sizeMB = 1048576
        sizeGB = 1073741824
        sizeTB = 1099511627776
        sizePB = 1125899906842624
        sizeEB = 1152921504606846976
        # sizeZB = 1180591620717411303424
        # sizeYB = 120892581961462917470617

        # convert to ABS value #
        try:
            sizeAbs = abs(size)
        except SystemExit:
            sys.exit(0)
        except:
            return "?"

        if unit:
            factor = 1 if isInt else 1.0

            upperUnit = unit.upper()
            if upperUnit == "K":
                val = size / sizeKB * factor
            elif upperUnit == "M":
                val = size / sizeMB * factor
            elif upperUnit == "G":
                val = size / sizeGB * factor
            elif upperUnit == "T":
                val = size / sizeTB * factor
            elif upperUnit == "P":
                val = size / sizePB * factor
            elif upperUnit == "E":
                val = size / sizeEB * factor
            else:
                SysMgr.printErr("no support size unit '%s'" % unit)
                sys.exit(-1)

            if isInt:
                num = UtilMgr.convNum(long(val))
            else:
                num = UtilMgr.convNum(round(float(val), 1), isFloat=True)

            return "%s%s" % (num, upperUnit)

        # Int type #
        if isInt:
            try:
                if sizeAbs >= sizeEB:
                    return "%dE" % (size >> 60)
                elif sizeAbs >= sizePB:
                    return "%dP" % (size >> 50)
                elif sizeAbs >= sizeTB:
                    return "%dT" % (size >> 40)
                elif sizeAbs >= sizeGB:
                    return "%dG" % (size >> 30)
                elif sizeAbs >= sizeMB:
                    return "%dM" % (size >> 20)
                elif sizeAbs >= sizeKB:
                    return "%dK" % (size >> 10)
                else:
                    return "%d" % size
            except SystemExit:
                sys.exit(0)
            except:
                return "?"
        # Float type #
        else:
            try:
                if sizeAbs >= sizeEB:
                    return "%.1fE" % ((size >> 50) / 1024.0)
                elif sizeAbs >= sizePB:
                    return "%.1fP" % ((size >> 40) / 1024.0)
                elif sizeAbs >= sizeTB:
                    return "%.1fT" % ((size >> 30) / 1024.0)
                elif sizeAbs >= sizeGB:
                    return "%.1fG" % ((size >> 20) / 1024.0)
                elif sizeAbs >= sizeMB:
                    return "%.1fM" % ((size >> 10) / 1024.0)
                elif sizeAbs >= sizeKB:
                    return "%.1fK" % (size / 1024.0)
                else:
                    return "%d" % (size)
            except SystemExit:
                sys.exit(0)
            except:
                return "?"

    @staticmethod
    def convTime2Unit(sec):
        sec = float(sec)
        conv = UtilMgr.convNum

        try:
            if sec == long(sec):
                return "%ssec" % conv(sec)

            msSec = round(sec * 1000, 3)
            msSecInt = long(msSec)
            if msSec and msSec == msSecInt:
                return "%sms" % msSecInt

            usSec = round(sec * 1000000, 6)
            usSecInt = long(usSec)
            if usSec and usSec == usSecInt:
                return "%sus" % usSecInt

            nsSec = round(sec * 1000000000, 9)
            nsSecInt = long(nsSec)
            if nsSec and nsSec == nsSecInt:
                return "%sns" % nsSecInt

            return sec
        except SystemExit:
            sys.exit(0)
        except:
            return "?"

    @staticmethod
    def convTime2Sec(time):
        # convert time to seconds #
        try:
            sec = 0

            nums = time.strip().split(":")
            if len(nums) == 3:
                y = d = 0
                times = nums
            elif len(nums) == 4:
                y = 0
                d = int(nums[0].rstrip("d"))
                times = nums[1:]
            elif len(nums) == 5:
                y = int(nums[0].rstrip("y"))
                d = int(nums[0].rstrip("d"))
                times = nums[1:]
            else:
                raise Exception()

            # convert type #
            h, m, s = list(map(int, times))

            # convert items to seconds #
            sec += s
            sec += m * 60
            sec += h * 3600
            if d:
                sec += d * 86400
            if y:
                sec += y * 31536000
        except SystemExit:
            sys.exit(0)
        except:
            pass

        return sec

    @staticmethod
    def convTime(time, remainder=False):
        try:
            if not remainder:
                raise Exception("integer")
            remain = ".%02d" % long(time * 100 % 100)
        except SystemExit:
            sys.exit(0)
        except:
            remain = ""

        # convert seconds to time #
        try:
            m, s = divmod(time, 60)
            h, m = divmod(m, 60)

            # hour #
            if h >= 24:
                d, h = divmod(h, 24)

                # year #
                if d >= 365:
                    y, d = divmod(d, 365)
                    d = "%dy:%dd:" % (y, d)
                else:
                    d = "%dd:" % d
            else:
                d = ""

            ctime = "%s%02d:%02d:%02d%s" % (d, h, m, s, remain)
        except SystemExit:
            sys.exit(0)
        except:
            ctime = "%s%02s:%02s:%02s%s" % ("", "?", "?", "?", remain)

        return ctime.strip()

    @staticmethod
    def prepareForImageFile(logFile, itype="", outFile=None):
        # build output file name #
        if outFile:
            outputFile = outFile
        else:
            # convert file object to string #
            if hasattr(logFile, "name"):
                logFile = logFile.name

            if SysMgr.outPath:
                outputFile = os.path.normpath(SysMgr.outPath)
            else:
                outputFile = os.path.normpath(logFile)

            # convert output path #
            if os.path.isdir(outputFile):
                filename = os.path.basename(logFile)
                filename = os.path.splitext(filename)[0]
                name = "%s/%s" % (outputFile, filename)
            else:
                name = os.path.splitext(outputFile)[0]

            outputFile = "%s%s.%s" % (
                name,
                ("_" + itype) if itype and not name.endswith(itype) else "",
                SysMgr.drawFormat,
            )

        try:
            # backup an exist image file #
            if os.path.isfile(outputFile):
                name, ext = os.path.splitext(outputFile)

                oldPath = "%s_old%s" % (name, ext)
                if os.path.isfile(oldPath):
                    os.remove(oldPath)

                os.rename(outputFile, oldPath)

                SysMgr.printInfo(
                    "renamed '%s' to '%s' for backup" % (outputFile, oldPath)
                )
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printErr(
                "failed to backup '%s' to '%s'" % (outputFile, oldPath), True
            )

        return outputFile

    @staticmethod
    def convUnit2Time(data, isFloat=False, verb=True):
        # check type #
        if isFloat:
            unit = float
        else:
            unit = long

        # check negative #
        if data.startswith("-"):
            negative = True
            data = data[1:]
        else:
            negative = False

        # convert to seconds #
        if str(data).isdigit():
            ret = unit(data)
        elif isFloat and str(data).replace(".", "").isdigit():
            ret = unit(data)
        elif data.upper().endswith("S"):
            ret = unit(data[:-1])
        elif data.upper().endswith("M"):
            ret = unit(data[:-1]) * 60
        elif data.upper().endswith("H"):
            ret = unit(data[:-1]) * 3600
        elif data.upper().endswith("D"):
            ret = unit(data[:-1]) * 86400
        elif data.upper().endswith("W"):
            ret = unit(data[:-1]) * 604800
        else:
            ret = 0
            if verb:
                SysMgr.printErr("failed to convert '%s' to seconds" % data)

        if negative:
            ret = -(ret)

        return ret

    @staticmethod
    def convUnit2Sec(data, verb=True):
        if str(data).isdigit():
            ret = long(data)
        elif data.upper().endswith("MS"):
            ret = long(data[:-2]) / 1000.0
        elif data.upper().endswith("US"):
            ret = long(data[:-2]) / 1000000.0
        elif data.upper().endswith("NS"):
            ret = long(data[:-2]) / 1000000000.0
        elif data.upper().endswith("PS"):
            ret = long(data[:-2]) / 1000000000000.0
        else:
            ret = 0
            if verb:
                SysMgr.printErr("failed to convert '%s' to seconds" % data)

        return ret

    @staticmethod
    def convUnit2Size(value, verb=True):
        sizeKB = 1024
        sizeMB = 1048576
        sizeGB = 1073741824
        sizeTB = 1099511627776
        sizePB = 1125899906842624
        sizeEB = 1152921504606846976
        # sizeZB = 1180591620717411303424
        # sizeYB = 120892581961462917470617

        if isinstance(value, (long, float)):
            return value

        value = str(value)

        # check type #
        origValue = value
        if value.startswith("-"):
            value = value.lstrip("-")
            factor = -1
        else:
            factor = 1

        if str(value).isdigit():
            return long(value) * factor

        # convert unit character to capital #
        value = value.upper()

        try:
            if value.endswith("K"):
                return long(float(value[:-1]) * sizeKB) * factor
            if value.endswith("KB"):
                return long(float(value[:-2]) * sizeKB) * factor

            if value.endswith("M"):
                return long(float(value[:-1]) * sizeMB) * factor
            if value.endswith("MB"):
                return long(float(value[:-2]) * sizeMB) * factor

            if value.endswith("G"):
                return long(float(value[:-1]) * sizeGB) * factor
            if value.endswith("GB"):
                return long(float(value[:-2]) * sizeGB) * factor

            if value.endswith("T"):
                return long(float(value[:-1]) * sizeTB) * factor
            if value.endswith("TB"):
                return long(float(value[:-2]) * sizeTB) * factor

            if value.endswith("P"):
                return long(float(value[:-1]) * sizePB) * factor
            if value.endswith("PB"):
                return long(float(value[:-2]) * sizePB) * factor

            if value.endswith("E"):
                return long(float(value[:-1]) * sizeEB) * factor
            if value.endswith("EB"):
                return long(float(value[:-2]) * sizeEB) * factor

            if verb:
                SysMgr.printErr("failed to convert '%s' to bytes" % origValue)

            raise ValueError()
        except SystemExit:
            sys.exit(0)
        except ValueError:
            raise Exception("wrong number unit")
        except:
            return origValue

    @staticmethod
    def saveStr2File(string, path):
        # open the file #
        try:
            with open(path, "w") as fd:
                fd.truncate()
                fd.write(string)
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printErr(
                "failed to write JSON format data to %s" % path, True
            )
            sys.exit(-1)

    @staticmethod
    def writeJsonObject(jsonObj, fd=None, trunc=False, path=None):
        if fd:
            try:
                if trunc:
                    fd.seek(0, 0)
                    fd.truncate()
            except SystemExit:
                sys.exit(0)
            except:
                pass

            try:
                fd.write(jsonObj)
                fd.flush()
            except SystemExit:
                sys.exit(0)
            except:
                SysMgr.printWarn(
                    "failed to write JSON format data", reason=True
                )
            return

        # check write option #
        if trunc:
            perm = "w"
        else:
            perm = "a"

        # open the file #
        try:
            with open(path, perm) as fd:
                fd.write(jsonObj)
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printErr(
                "failed to write JSON format data to %s" % path, True
            )
            sys.exit(-1)

    @staticmethod
    def saveObject2File(obj, path):
        if not obj:
            return False

        pickle = SysMgr.getPicklePkg(False)
        if not pickle:
            return False

        # compress by gzip #
        if "COMPCACHE" in SysMgr.environList:
            compressor = SysMgr.getPkg("gzip", False)
        else:
            compressor = False

        # original object #
        try:
            if compressor:
                with compressor.open(path, "wb") as fd:
                    pickle.dump(obj, fd, -1)
            else:
                with open(path, "wb") as fd:
                    pickle.dump(obj, fd, -1)
            os.chmod(path, 0o777)
            return True
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printWarn(
                "failed to save %s object to %s"
                % (obj.__class__.__name__, path),
                reason=True,
            )
            return False

    @staticmethod
    def loadObjectFromFile(path):
        # check object exists #
        if not os.path.isfile(path):
            return None

        # load pickle object #
        pickle = SysMgr.getPicklePkg(False)
        if not pickle:
            return None

        try:
            # try to load compressed cache #
            with SysMgr.getPkg("gzip", False).open(path, "rb") as fd:
                return pickle.load(fd)
        except SystemExit:
            sys.exit(0)
        except:
            try:
                with open(path, "rb") as fd:
                    return pickle.load(fd)
            except SystemExit:
                sys.exit(0)
            except:
                return None

    @staticmethod
    def printProgress(current=0, dest=0):
        if (
            not SysMgr.printEnable
            or not SysMgr.logEnable
            or dest == sys.maxsize
        ):
            return

        # just output #
        if not current and not dest:
            if UtilMgr.progressCnt >= len(UtilMgr.progressChar) - 1:
                UtilMgr.progressCnt = 0
            else:
                UtilMgr.progressCnt += 1

            mod = UtilMgr.progressCnt

            progressStr = ".... %s%s" % (UtilMgr.progressChar[mod], "\b" * 6)
        else:
            try:
                div = round((current / float(dest)) * 100, 1)
            except SystemExit:
                sys.exit(0)
            except:
                div = 0

            percent = long(div)

            mod = percent & 3

            progressStr = "%3d%% %s%s" % (
                percent,
                UtilMgr.progressChar[mod],
                "\b" * 6,
            )

        # handle reentrant call exception #
        try:
            if progressStr == UtilMgr.progressStr:
                return

            UtilMgr.progressStr = progressStr
            sys.stdout.write(progressStr)
            sys.stdout.flush()
        except SystemExit:
            sys.exit(0)
        except:
            return

    @staticmethod
    def writeFlamegraph(path, samples, title, depth=20):
        # flamegraph from https://github.com/rbspy/rbspy/tree/master/src/ui/flamegraph.rs #
        # fixed font size: 12, bar height: 15 #
        barHeight = 17
        titleHeight = (title.count("<tspan ") + 1) * barHeight
        height = barHeight * depth + 50 + titleHeight
        width = "1"
        flameCode = (
            (
                """<?xml version="1.0" standalone="no"?>
<!DOCTYPE svg PUBLIC "-//W3C//DTD SVG 1.1//EN" "http://www.w3.org/Graphics/SVG/1.1/DTD/svg11.dtd">
<svg version="1.1" width="%s" height="%s" onload="init(evt)" viewBox="0 0 %s %s"
        """
                % (width, height, width, height)
            )
            + """
        xmlns="http://www.w3.org/2000/svg"
        xmlns:xlink="http://www.w3.org/1999/xlink">
        <!--Flame graph stack visualization. See https://github.com/brendangregg/FlameGraph for latest version, and http://www.brendangregg.com/flamegraphs.html for examples.-->
        <!--NOTES: -->
        <defs>
                <linearGradient id="background" y1="0" y2="1" x1="0" x2="0">
                        <stop stop-color="#eeeeee" offset="5%"/>
                        <stop stop-color="#eeeeb0" offset="95%"/>
                </linearGradient>
        </defs>
        <style type="text/css">
            text { font-family:"Verdana"; font-size:12px; fill:rgb(0,0,0); }
            #title { text-anchor:middle; font-size:17px; }
            #search { opacity:0.1; cursor:pointer; }
            #search:hover, #search.show { opacity:1; }
            #subtitle { text-anchor:left; font-color:rgb(160,160,160); }
            #unzoom { cursor:pointer; }
            #frames > *:hover { stroke:black; stroke-width:0.5; cursor:pointer; }
            .hide { display:none; }
            .parent { opacity:0.5; }
        </style>
        <script type="text/ecmascript">
                <![CDATA[var nametype = 'Function:';
var fontsize = 12;
var fontwidth = 0.59;
var xpad = 10;
var inverted = true;
var searchcolor = 'rgb(255,0,0)';
var searchstroke = 'rgb(0,0,0)';
var fluiddrawing = true;
var truncate_text_right = true;]]>
                <![CDATA["use strict";
var details, searchbtn, unzoombtn, matchedtxt, svg, searching, frames;
function init(evt) {
    details = document.getElementById("details").firstChild;
    searchbtn = document.getElementById("search");
    unzoombtn = document.getElementById("unzoom");
    matchedtxt = document.getElementById("matched");
    svg = document.getElementsByTagName("svg")[0];
    frames = document.getElementById("frames");
    searching = 0;

    // Use GET parameters to restore a flamegraph's state.
    var restore_state = function() {
        var params = get_params();
        if (params.x && params.y)
            zoom(find_group(document.querySelector('[x="' + params.x + '"][y="' + params.y + '"]')));
        if (params.s)
            search(params.s);
    };

    if (fluiddrawing) {
        // Make width dynamic so the SVG fits its parent's width.
        svg.removeAttribute("width");
        // Edge requires us to have a viewBox that gets updated with size changes
        var isEdge = /Edge\/\d./i.test(navigator.userAgent);
        if (!isEdge) {
          svg.removeAttribute("viewBox");
        }
        var update_for_width_change = function() {
            if (isEdge) {
                svg.attributes.viewBox.value = "0 0 " + svg.width.baseVal.value + " " + svg.height.baseVal.value;
            }

            // Keep consistent padding on left and right of frames container.
            frames.attributes.width.value = svg.width.baseVal.value - xpad * 2;

            // Text truncation needs to be adjusted for the current width.
            var el = frames.children;
            for(var i = 0; i < el.length; i++) {
                update_text(el[i]);
            }

            // Keep search elements at a fixed distance from right edge.
            var svgWidth = svg.width.baseVal.value;
            searchbtn.attributes.x.value = svgWidth - xpad - 100;
            matchedtxt.attributes.x.value = svgWidth - xpad - 100;
        };
        window.addEventListener('resize', function() {
            update_for_width_change();
        });
        // This needs to be done asynchronously for Safari to work.
        setTimeout(function() {
            unzoom();
            update_for_width_change();
            restore_state();
        }, 0);
    } else {
        restore_state();
    }
}
// event listeners
window.addEventListener("click", function(e) {
    var target = find_group(e.target);
    if (target) {
        if (target.nodeName == "a") {
            if (e.ctrlKey === false) return;
            e.preventDefault();
        }
        if (target.classList.contains("parent")) unzoom();
        zoom(target);

        // set parameters for zoom state
        var el = target.querySelector("rect");
        if (el && el.attributes && el.attributes.y && el.attributes._orig_x) {
            var params = get_params()
            params.x = el.attributes._orig_x.value;
            params.y = el.attributes.y.value;
            history.replaceState(null, null, parse_params(params));
        }
    }
    else if (e.target.id == "unzoom") {
        unzoom();

        // remove zoom state
        var params = get_params();
        if (params.x) delete params.x;
        if (params.y) delete params.y;
        history.replaceState(null, null, parse_params(params));
    }
    else if (e.target.id == "search") search_prompt();
}, false)
// mouse-over for info
// show
window.addEventListener("mouseover", function(e) {
    var target = find_group(e.target);
    if (target && details) {
        details.nodeValue = nametype + " " + g_to_text(target);
    }
}, false)
// clear
window.addEventListener("mouseout", function(e) {
    var target = find_group(e.target);
    if (target && details) {
        details.nodeValue = ' ';
    }
}, false)
// F3 / ctrl-F for search, ESC for reset search or zoom
window.addEventListener("keydown",function (e) {
    if (e.keyCode === 114 || (e.ctrlKey && e.keyCode === 70)) {
        e.preventDefault();
        search_prompt();
    } else if (e.keyCode === 27 || (e.ctrlKey && e.keyCode === 90)) {
        e.preventDefault();
        reset_search();
        unzoom()
    }
}, false)
// functions
function get_params() {
    var params = {};
    var paramsarr = window.location.search.substr(1).split('&');
    for (var i = 0; i < paramsarr.length; ++i) {
        var tmp = paramsarr[i].split("=");
        if (!tmp[0] || !tmp[1]) continue;
        params[tmp[0]] = decodeURIComponent(tmp[1]);
    }
    return params;
}
function parse_params(params) {
    var uri = "?";
    for (var key in params) {
        uri += key + '=' + encodeURIComponent(params[key]) + '&';
    }
    if (uri.slice(-1) == "&")
        uri = uri.substring(0, uri.length - 1);
    if (uri == '?')
        uri = window.location.href.split('?')[0];
    return uri;
}
function find_child(node, selector) {
    var children = node.querySelectorAll(selector);
    if (children.length) return children[0];
    return;
}
function find_group(node) {
    var parent = node.parentElement;
    if (!parent) return;
    if (parent.id == "frames") return node;
    return find_group(parent);
}
function orig_save(e, attr, val) {
    if (e.attributes["_orig_" + attr] != undefined) return;
    if (e.attributes[attr] == undefined) return;
    if (val == undefined) val = e.attributes[attr].value;
    e.setAttribute("_orig_" + attr, val);
}
function orig_load(e, attr) {
    if (e.attributes["_orig_"+attr] == undefined) return;
    e.attributes[attr].value = e.attributes["_orig_" + attr].value;
    e.removeAttribute("_orig_" + attr);
}
function g_to_text(e) {
    var text = find_child(e, "title").firstChild.nodeValue;
    return (text)
}
function g_to_func(e) {
    var func = g_to_text(e);
    // if there's any manipulation we want to do to the function
    // name before it's searched, do it here before returning.
    return (func);
}
function update_text(e) {
    var r = find_child(e, "rect");
    var t = find_child(e, "text");
    var w = parseFloat(r.attributes.width.value) * frames.attributes.width.value / 100 - 3;
    var txt = find_child(e, "title").textContent.replace(/&lt;/g,'<').replace(/&gt;/g,'>').replace(/&amp;/g,'&');
    t.attributes.x.value = format_percent((parseFloat(r.attributes.x.value) + (100 * 3 / frames.attributes.width.value)));
    // Smaller than this size won't fit anything
    if (w < 2 * fontsize * fontwidth) {
        t.textContent = "";
        return;
    }
    t.textContent = txt;
    // Fit in full text width
    if (/^ *\$/.test(txt) || t.getComputedTextLength() < w)
        return;
    if (truncate_text_right) {
        // Truncate the right side of the text.
        for (var x = txt.length - 2; x > 0; x--) {
            if (t.getSubStringLength(0, x + 2) <= w) {
                t.textContent = txt.substring(0, x) + "..";
                return;
            }
        }
    } else {
        // Truncate the left side of the text.
        for (var x = 2; x < txt.length; x++) {
            if (t.getSubStringLength(x - 2, txt.length) <= w) {
                t.textContent = ".." + txt.substring(x, txt.length);
                return;
            }
        }
    }
    t.textContent = "";
}
// zoom
function zoom_reset(e) {
    if (e.attributes != undefined) {
        orig_load(e, "x");
        orig_load(e, "width");
    }
    if (e.childNodes == undefined) return;
    for(var i = 0, c = e.childNodes; i < c.length; i++) {
        zoom_reset(c[i]);
    }
}
function zoom_child(e, x, ratio) {
    if (e.attributes != undefined) {
        if (e.attributes.x != undefined) {
            orig_save(e, "x");
            e.attributes.x.value = format_percent((parseFloat(e.attributes.x.value) - x) * ratio);
            if (e.tagName == "text") {
                e.attributes.x.value = format_percent(parseFloat(find_child(e.parentNode, "rect[x]").attributes.x.value) + (100 * 3 / frames.attributes.width.value));
            }
        }
        if (e.attributes.width != undefined) {
            orig_save(e, "width");
            e.attributes.width.value = format_percent(parseFloat(e.attributes.width.value) * ratio);
        }
    }
    if (e.childNodes == undefined) return;
    for(var i = 0, c = e.childNodes; i < c.length; i++) {
        zoom_child(c[i], x, ratio);
    }
}
function zoom_parent(e) {
    if (e.attributes) {
        if (e.attributes.x != undefined) {
            orig_save(e, "x");
            e.attributes.x.value = "0.0%";
        }
        if (e.attributes.width != undefined) {
            orig_save(e, "width");
            e.attributes.width.value = "100.0%";
        }
    }
    if (e.childNodes == undefined) return;
    for(var i = 0, c = e.childNodes; i < c.length; i++) {
        zoom_parent(c[i]);
    }
}
function zoom(node) {
    var attr = find_child(node, "rect").attributes;
    var width = parseFloat(attr.width.value);
    var xmin = parseFloat(attr.x.value);
    var xmax = xmin + width;
    var ymin = parseFloat(attr.y.value);
    var ratio = 100 / width;
    // XXX: Workaround for JavaScript float issues (fix me)
    var fudge = 0.001;
    unzoombtn.classList.remove("hide");
    var el = frames.children;
    for (var i = 0; i < el.length; i++) {
        var e = el[i];
        var a = find_child(e, "rect").attributes;
        var ex = parseFloat(a.x.value);
        var ew = parseFloat(a.width.value);
        // Is it an ancestor
        if (!inverted) {
            var upstack = parseFloat(a.y.value) > ymin;
        } else {
            var upstack = parseFloat(a.y.value) < ymin;
        }
        if (upstack) {
            // Direct ancestor
            if (ex <= xmin && (ex+ew+fudge) >= xmax) {
                e.classList.add("parent");
                zoom_parent(e);
                update_text(e);
            }
            // not in current path
            else
                e.classList.add("hide");
        }
        // Children maybe
        else {
            // no common path
            if (ex < xmin || ex + fudge >= xmax) {
                e.classList.add("hide");
            }
            else {
                zoom_child(e, xmin, ratio);
                update_text(e);
            }
        }
    }
}
function unzoom() {
    unzoombtn.classList.add("hide");
    var el = frames.children;
    for(var i = 0; i < el.length; i++) {
        el[i].classList.remove("parent");
        el[i].classList.remove("hide");
        zoom_reset(el[i]);
        update_text(el[i]);
    }
}
// search
function reset_search() {
    var el = document.querySelectorAll("#frames rect");
    for (var i = 0; i < el.length; i++) {
        orig_load(el[i], "fill")
        el[i].setAttribute("stroke-width", "0")
    }
    var params = get_params();
    delete params.s;
    history.replaceState(null, null, parse_params(params));
}
function search_prompt() {
    if (!searching) {
        var term = prompt("Enter a search term (regexp " +
            "allowed, eg: ^ext4_)", "");
        if (term != null) {
            search(term)
        }
    } else {
        reset_search();
        searching = 0;
        searchbtn.classList.remove("show");
        if (searchbtn.firstChild) {
            searchbtn.firstChild.nodeValue = "Search (F3)"
        }
        matchedtxt.classList.add("hide");
        if (matchedtxt.firstChild) {
            matchedtxt.firstChild.nodeValue = ""
        }
    }
}
function search(term) {
    var re = new RegExp(term);
    var el = frames.children;
    var matches = new Object();
    var maxwidth = 0;
    for (var i = 0; i < el.length; i++) {
        var e = el[i];
        var func = g_to_func(e);
        var rect = find_child(e, "rect");
        if (func == null || rect == null)
            continue;
        // Save max width. Only works as we have a root frame
        var w = parseFloat(rect.attributes.width.value);
        if (w > maxwidth)
            maxwidth = w;
        if (func.match(re)) {
            // highlight
            var x = parseFloat(rect.attributes.x.value);
            orig_save(rect, "fill");
            rect.setAttribute("fill", searchcolor);
            rect.setAttribute("stroke", searchstroke);
            rect.setAttribute("stroke-width", "3");
            // remember matches
            if (matches[x] == undefined) {
                matches[x] = w;
            } else {
                if (w > matches[x]) {
                    // overwrite with parent
                    matches[x] = w;
                }
            }
            searching = 1;
        }
    }
    if (!searching)
        return;
    var params = get_params();
    params.s = term;
    history.replaceState(null, null, parse_params(params));

    searchbtn.classList.add("show");
    if (searchbtn.firstChild) {
        searchbtn.firstChild.nodeValue = "Reset Search (F3)";
    }
    // calculate percent matched, excluding vertical overlap
    var count = 0;
    var lastx = -1;
    var lastw = 0;
    var keys = Array();
    for (k in matches) {
        if (matches.hasOwnProperty(k))
            keys.push(k);
    }
    // sort the matched frames by their x location
    // ascending, then width descending
    keys.sort(function(a, b){
        return a - b;
    });
    // Step through frames saving only the biggest bottom-up frames
    // thanks to the sort order. This relies on the tree property
    // where children are always smaller than their parents.
    var fudge = 0.0001;    // JavaScript floating point
    for (var k in keys) {
        var x = parseFloat(keys[k]);
        var w = matches[keys[k]];
        if (x >= lastx + lastw - fudge) {
            count += w;
            lastx = x;
            lastw = w;
        }
    }
    // display matched percent
    matchedtxt.classList.remove("hide");
    var pct = 100 * count / maxwidth;
    if (pct != 100) pct = pct.toFixed(1);
    if (matchedtxt.firstChild) {
        matchedtxt.firstChild.nodeValue = "Matched: " + pct + "%";
    }
}
function format_percent(n) {
    return n.toFixed(4) + "%";
}
]]>
        </script>
"""
        )
        attrCode = """
    <rect x="0" y="0" width="100%%" height="%s" fill="url(#background)"/>
    <text id="title" x="50.0000%%" y="24.00">Guider Flame Graph</text>
    <text id="subtitle" x="0.0000%%" y="50.00">%s</text>
    <text id="details" x="10" y="213.00"></text>
    <text id="unzoom" class="hide" x="10" y="24.00">Reset Zoom (ESC)</text>
    <text id="search" x="1090" y="24.00">Search (F3)</text>
    <text id="matched" x="1090" y="213.00"></text>
    <svg id="frames" x="10" y="%s" width="%s">
""" % (
            height,
            "\r\n%s" % title if title else "",
            titleHeight,
            width,
        )

        # complete code for flamegraph #
        finalCode = flameCode + attrCode + samples + "\n</svg></svg>"

        # write flamegraph to file #
        try:
            with open(path, "w") as fd:
                fd.write(finalCode)

            os.chmod(path, 0o777)

            # get output size #
            fsize = UtilMgr.getFileSizeStr(path)

            SysMgr.printInfo(
                "saved a flame graph into '%s'%s successfully" % (path, fsize)
            )
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printErr("failed to write a flame graph to %s" % path, True)
            sys.exit(-1)

    @staticmethod
    def getFileSizeStr(path, string=True):
        fsize = UtilMgr.getFileSize(path)
        if fsize and fsize != "0":
            fsize = " [%s]" % fsize
        else:
            fsize = ""
        return fsize

    @staticmethod
    def getFileSize(path, string=True):
        try:
            fsize = long(os.path.getsize(path))
            if string:
                return UtilMgr.convSize2Unit(fsize)
            else:
                return fsize
        except SystemExit:
            sys.exit(0)
        except:
            if UtilMgr.isString(path) and os.path.exists(path):
                SysMgr.printWarn("failed to get file size for '%s'" % path)

            if string:
                return "?"
            else:
                return 0

    @staticmethod
    def printFile(path):
        try:
            with open(path, "r") as fd:
                for line in fd:
                    print(line.rstrip())
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printErr("failed to print %s" % path, reason=True)

    @staticmethod
    def deleteProgress():
        if not SysMgr.printEnable or not SysMgr.logEnable:
            return

        sys.stdout.write(" " * 6)
        sys.stdout.write("\b" * 6)

        # handle reentrant call exception #
        try:
            sys.stdout.flush()
        except SystemExit:
            sys.exit(0)
        except:
            return

    @staticmethod
    def which(cmd):
        if not "PATH" in os.environ:
            return None

        pathList = []
        for path in os.environ["PATH"].split(os.pathsep):
            if os.path.exists(os.path.join(path, cmd)):
                pathList.append(os.path.join(path, cmd))

        if pathList:
            return pathList
        else:
            return None

    @staticmethod
    def convDict2Str(dictObj, pretty=True, indent=2, ignore=False):
        try:
            jsonStr = SysMgr.getPkg("json").dumps(
                dictObj,
                indent=indent,
                ensure_ascii=False,
                default=(lambda o: "<not serializable>") if ignore else None,
            )

            if pretty:
                return jsonStr
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printWarn("failed to convert dict to string", reason=True)

            """
            # for debugging #
            SysMgr.printWarn(
                "failed to convert %s to string" % [dictObj], reason=True)
            """

            return None

        # when encode flag is disabled, remove whitespace [\t\n\r\f\v] #
        if not SysMgr.encodeEnable:
            jsonStr = re.sub("[\t\n\r\f\v]", "", jsonStr) + "\n"

        return jsonStr

    @staticmethod
    def convUlong2Long(retval):
        retval = retval & 0xFFFFFFFFFFFFFFFF
        if retval & 0x8000000000000000:
            retval = retval - 0x10000000000000000
        return retval

    @staticmethod
    def convStr2Dict(strObj, verb=False):
        try:
            return SysMgr.getPkg("json").loads(strObj)
        except SystemExit:
            sys.exit(0)
        except:
            try:
                strObj = strObj.replace("'", '"')
                return SysMgr.getPkg("json").loads(strObj)
            except SystemExit:
                sys.exit(0)
            except:
                SysMgr.printWarn(
                    "failed to convert %s to dict" % [strObj],
                    always=verb,
                    reason=True,
                )
                return None


class NetworkMgr(object):
    """Manager for remote communication"""

    # define valid request list #
    REQUEST_LIST = {
        "BROADCAST": None,
        "CLEAR": None,
        "DIR": None,
        "DOWNLOAD": None,
        "JOBS": None,
        "LIST": None,
        "NEW": None,
        "NOTIFY": None,
        "PING": None,
        "RESTART": None,
        "RUN": None,
        "UPLOAD": None,
        "UPSTREAM": None,
    }

    def __init__(
        self,
        mode,
        ip,
        port,
        tcp=False,
        uds=False,
        netlink=False,
        blocking=True,
        anyPort=False,
        bind=True,
        reuse=True,
    ):
        self.mode = mode
        self.ip = None
        self.port = None
        self.socket = None
        self.request = None
        self.status = None
        self.ignore = 0
        self.fileno = -1
        self.time = None
        self.sendSize = 32767
        self.recvSize = 32767
        self.tcp = tcp
        self.netlink = netlink
        self.connected = False

        # get socket object #
        socket = SysMgr.getPkg("socket")

        try:
            from socket import (
                socket,
                AF_INET,
                SOCK_DGRAM,
                SOCK_STREAM,
                SOL_SOCKET,
                SO_REUSEADDR,
                SO_SNDBUF,
                SO_RCVBUF,
                SOL_TCP,
                SO_RCVTIMEO,
                SO_SNDTIMEO,
                SOCK_RAW,
            )
        except:
            SysMgr.printWarn("failed to import socket", True, reason=True)
            return None

        try:
            # set socket type #
            if tcp:
                self.socket = socket(AF_INET, SOCK_STREAM)
            elif uds:
                # pylint: disable-next=no-name-in-module
                from socket import AF_UNIX

                self.socket = socket(AF_UNIX, SOCK_STREAM)
            elif netlink:
                try:
                    # pylint: disable-next=no-name-in-module
                    from socket import AF_NETLINK

                    self.socket = socket(
                        AF_NETLINK,
                        SOCK_RAW,
                        ConfigMgr.NETLINK_TYPE["NETLINK_GENERIC"],
                    )
                except:
                    SysMgr.printWarn(
                        "failed to create NETLINK socket", True, reason=True
                    )
                    return None
            else:
                self.socket = socket(AF_INET, SOCK_DGRAM)

            self.fileno = self.socket.fileno()

            # increase socket buffer size to 1MB #
            self.socket.setsockopt(SOL_SOCKET, SO_SNDBUF, 1 << 20)
            self.socket.setsockopt(SOL_SOCKET, SO_RCVBUF, 1 << 20)

            # get buffer size #
            self.sendSize = self.socket.getsockopt(SOL_SOCKET, SO_SNDBUF)
            self.recvSize = self.socket.getsockopt(SOL_SOCKET, SO_RCVBUF)

            # set REUSEADDR flag #
            if reuse:
                self.socket.setsockopt(SOL_SOCKET, SO_REUSEADDR, 1)

            # set REUSEPORT #
            """
            from socket import SO_REUSEPORT
            self.socket.setsockopt(SOL_SOCKET, SO_REUSEPORT, 0)
            """

            # set NODELAY #
            """
            self.setNoDelay()
            """

            # set SENDTIMEOUT #
            """
            sec = 1
            usec = 0
            timeval = struct.pack('ll', sec, usec)
            self.socket.setsockopt(SOL_SOCKET, SO_SNDTIMEO, timeval)
            """

            if uds:
                return None

            # convert IP #
            if ip == "*":
                ip = "0.0.0.0"
            elif str(ip).upper() == "LOCALHOST":
                ip = "127.0.0.1"

            # set IP & PORT #
            self.ip = ip
            self.port = port

            if mode == "server":
                # IP #
                if ip is None:
                    self.ip = "0.0.0.0"

                # PORT #
                if anyPort:
                    self.port = 0
                elif port is None:
                    self.port = SysMgr.defaultServPort

                # bind #
                if bind:
                    try:
                        self.socket.bind((self.ip, self.port))
                    except OSError as e:
                        if e.errno == errno.EADDRINUSE:
                            self.port = 0
                            self.socket.bind((self.ip, self.port))
                        else:
                            raise e
                    except SystemExit:
                        sys.exit(0)
                    except:
                        self.socket.bind((self.ip, self.port))

                    # get bind port #
                    self.port = self.socket.getsockname()[1]

            if not blocking:
                self.socket.setblocking(0)
        except:
            err = SysMgr.getErrMsg()
            if err.startswith("13") and not SysMgr.isRoot() and port < 1024:
                feedback = ", use port bigger than 1024"
            else:
                feedback = ""

            # check mode to print error message #
            if (
                SysMgr.warnEnable
                or SysMgr.checkMode("server")
                or SysMgr.checkMode("cli")
            ):
                SysMgr.printErr(
                    "failed to create a socket for %s:%s as server because %s%s"
                    % (self.ip, self.port, err, feedback)
                )

            """
            if error "99 Cannot assign requested address" occurs:
                add "net.ipv4.ip_nonlocal_bind = 1" in /etc/sysctl.conf
                execute sysctl -p /etc/sysctl.conf
            """

            self.ip = None
            self.port = None

            return None

    def __str__(self):
        return "%s object at %x, IP: %s, PORT: %s" % (
            self.__class__,
            id(self),
            self.ip,
            self.port,
        )

    def listen(self, nrQueue=5):
        return self.socket.listen(nrQueue)

    def accept(self):
        return self.socket.accept()

    def bind(self, ip, port):
        return self.socket.bind((ip, port))

    def write(self, message):
        return self.send(message, write=True)

    def close(self):
        if self.socket:
            ret = self.socket.close()
        else:
            ret = False

        self.socket = None

        return ret

    def flush(self):
        pass

    def timeout(self, sec=3):
        if "TIMEOUT" in SysMgr.environList:
            sec = UtilMgr.getEnvironNum("TIMEOUT")

        self.socket.settimeout(sec)

    @staticmethod
    def sendFile(sock, ip, port, src, des):
        # verify path #
        target = src
        sdir, name = UtilMgr.getPath(target)

        # get file list #
        targetList = UtilMgr.getFiles(sdir, [name])
        if not targetList:
            SysMgr.printWarn("failed to find %s to transfer" % target, True)

            return False

        for target in targetList:
            target = SysMgr.convFullPath(target)

            SysMgr.printInfo(
                "start uploading %s[%s] to %s:%s... "
                % (target, UtilMgr.getFileSize(target), sock.ip, sock.port),
                suffix=False,
            )

            # send file info #
            totalSize = os.stat(target).st_size
            fileInfo = "%s#%s" % (target, totalSize)
            sock.send(UtilMgr.encodeStr(fileInfo))

            # read for ACK #
            while 1:
                ret = sock.recv(3)
                if ret is None:
                    continue
                elif ret is False:
                    raise Exception("no response")
                else:
                    break

            # send a file #
            with open(target, "rb") as fd:
                buf = fd.read(sock.sendSize)
                curSize = 0
                while buf:
                    # send a chunk #
                    sock.send(buf)
                    curSize += len(buf)

                    # read for ACK #
                    while 1:
                        ret = sock.recv(3)
                        if ret is None:
                            continue
                        elif ret is False:
                            raise Exception("no response")
                        else:
                            break

                    # read a chunk #
                    buf = fd.read(sock.sendSize)

            # read for ACK #
            while 1:
                ret = sock.recv(3)
                if ret is None:
                    continue
                elif ret is False:
                    raise Exception("no response")
                else:
                    break

            SysMgr.printInfo(
                "uploaded %s[%s] to %s:%s successfully"
                % (target, UtilMgr.getFileSize(target), sock.ip, sock.port)
            )

        return True

    @staticmethod
    def recvData(sockObj, ip, port=0):
        connMan = NetworkMgr("server", ip, port, tcp=True, bind=True)
        if not connMan.ip:
            return

        # send connection info #
        try:
            sockObj.send(("%s:%s" % (connMan.ip, connMan.port)).encode())
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printErr(
                "failed to send connection info to %s:%s"
                % (sockObj.ip, sockObj.port),
                True,
            )
            return

        # listen #
        try:
            connMan.listen()
            connMan.timeout()
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printErr("failed to listen to prepare for connection", True)
            sys.exit(-1)

        # accept #
        try:
            sock, addr = connMan.accept()
        except SystemExit:
            sys.exit(0)
        except IOError as err:
            if err.errno != errno.EINTR:
                SysMgr.printWarn(
                    "failed to accept the connection request", reason=True
                )
                return
        except SysMgr.getPkg("socket").timeout:
            return
        except:
            SysMgr.printWarn(
                "failed to accept the connection request", reason=True
            )
            return

        SysMgr.printInfo(
            "connected to the new client (%s:%s)" % (addr[0], addr[1])
        )

        try:
            # get payload size #
            size = sock.recv(SysMgr.PAGESIZE)
            size = long(size.decode())

            # send ACK #
            sock.send("ACK".encode())
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printErr(
                "failed to get payload size from %s:%s" % (addr[0], addr[1]),
                True,
            )
            return

        # receive data #
        try:
            buf = b""
            while 1:
                data = sock.recv(SysMgr.PAGESIZE)
                if data:
                    buf += data
                else:
                    break
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printErr(
                "failed to receive data from %s:%s" % (addr[0], addr[1]), True
            )
            return

        return buf

    @staticmethod
    def recvFile(sock, ip, port, src, des):
        # get select object #
        selectObj = SysMgr.getPkg("select")

        while 1:
            curSize = 0
            totalSize = None

            # receive file info #
            while 1:
                fileInfo = sock.recv(sock.recvSize)
                if not fileInfo:
                    continue

                # check termination condition #
                name, size = fileInfo.decode().split("#", 1)
                if name == "@DONE":
                    return True
                elif not size:
                    continue

                # send ACK for file info #
                totalSize = long(size)
                sock.send("ACK".encode())
                break

            # set destination path #
            fname = os.path.basename(name)
            if des:
                if os.path.isdir(des):
                    target = os.path.join(des, fname)
                else:
                    target = des
            else:
                target = name

            # make dirs #
            dirPos = target.rfind("/")
            if dirPos >= 0:
                SysMgr.mkdirs(target[:dirPos])

            # convert path #
            target = SysMgr.convFullPath(target)
            if os.path.isdir(target):
                target = os.path.join(target, fname)

            SysMgr.printInfo(
                "start downloading %s[%s]@%s:%s to %s..."
                % (name, UtilMgr.convSize2Unit(totalSize), ip, port, target),
                suffix=False,
            )

            # receive a file #
            with open(target, "wb") as fd:
                # change permission #
                os.chmod(target, 0o777)

                sent = 0
                while 1:
                    selectObj.select([sock.socket], [], [], 3)

                    # receive a chunk #
                    buf = sock.recv(sock.recvSize)
                    if not buf:
                        break

                    # update stat and write a chunk to file #
                    curSize += len(buf)
                    sent += len(buf)
                    fd.write(buf)

                    # check received size #
                    if curSize >= totalSize:
                        sock.send("ACK".encode())
                        break

                    # send ACK for a complete chunk #
                    if sent == sock.recvSize:
                        sock.send("ACK".encode())
                        sent = 0

                    # print progress #
                    UtilMgr.printProgress(curSize, totalSize)

            # remove progress #
            UtilMgr.deleteProgress()

            # check received size #
            if curSize < totalSize:
                raise Exception("broken connection")

            SysMgr.printInfo(
                "downloaded %s[%s]@%s:%s to %s successfully\n"
                % (target, UtilMgr.getFileSize(target), ip, port, name),
                suffix=False,
            )

            # send ACK #
            sock.send("ACK".encode())

        return True

    def customBind(self):
        # get bind info #
        ipList = portList = []
        if "CLIIP" in SysMgr.environList:
            ipList = SysMgr.environList["CLIIP"]
        if "CLIPORT" in SysMgr.environList:
            portList = SysMgr.environList["CLIPORT"]

        # check bind address #
        if not ipList and not portList:
            return True

        # set default IP #
        if not ipList:
            ipList = ["0"]

        # convert PORT list #
        if portList:
            newPortList = []
            for item in portList:
                if not "-" in item:
                    newPortList.append(long(item))
                    continue

                try:
                    start, end = item.split("-")
                    if end == "":
                        end = 65535
                    for idx in xrange(long(start), long(end) + 1):
                        newPortList.append(idx)
                except SystemExit:
                    sys.exit(0)
                except:
                    SysMgr.printErr("failed to bind client socket", True)
                    return False

            portList = list(set(newPortList))
        else:
            portList = [0]

        # disable REUSEADDR flag #
        socket = SysMgr.getPkg("socket")
        self.socket.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 0)

        # bind #
        isBound = False
        for ip in ipList:
            for port in portList:
                try:
                    self.bind(ip, port)
                    SysMgr.printWarn(
                        "succeed to bind client socket to %s:%s" % (ip, port)
                    )
                    isBound = True
                    break
                except SystemExit:
                    sys.exit(0)
                except:
                    SysMgr.printWarn(
                        "failed to bind client socket to %s:%s" % (ip, port),
                        reason=True,
                    )

        # check result #
        if isBound:
            return True
        else:
            SysMgr.printErr("failed to bind local address for client socket")
            return False

    def connect(self, addr=None):
        if addr is None:
            addr = (self.ip, self.port)

        # bind to specific address #
        if not self.customBind():
            raise Exception("bind failure")

        self.socket.connect(addr)

        self.connected = True

    def handleServerRequest(self, req, onlySocket=False):
        def _onDownload(req):
            # parse path #
            reqList = req.split("|", 1)[1]
            path = UtilMgr.cleanItem(reqList.split("@"), False)
            if len(path) == 2:
                origPath, desPath = path
                if origPath and not desPath:
                    origPath = desPath
            else:
                origPath = path[0]
                desPath = None

            res = False

            # receive file #
            try:
                res = NetworkMgr.recvFile(
                    self, self.ip, self.port, origPath, desPath
                )
            except:
                res = False
                SysMgr.printErr(
                    "failed to download %s from %s in %s:%s"
                    % (origPath, desPath, self.ip, self.port),
                    True,
                )
            finally:
                try:
                    self.close()
                except:
                    pass
                finally:
                    return res

        def _onUpstream(req):
            pass

        def _onDir(req):
            SysMgr.printInfo(UtilMgr.lstrip(req, "DIR:"))
            return True

        def _onNotify(req):
            return True

        def _onUpload(req):
            # parse path #
            reqList = req.split("|", 1)[1]
            path = UtilMgr.cleanItem(reqList.split("@"), False)
            if len(path) == 2:
                origPath, targetPath = path
                if origPath and not targetPath:
                    origPath = targetPath
            else:
                origPath = targetPath = path[0]

            res = False

            # transfer file #
            try:
                res = NetworkMgr.sendFile(
                    self, self.ip, self.port, origPath, targetPath
                )
            except:
                res = False
                SysMgr.printErr(
                    "failed to upload %s to %s in %s:%s"
                    % (origPath, targetPath, self.ip, self.port),
                    True,
                )
            finally:
                try:
                    self.send("@DONE#0".encode())
                    self.close()
                except:
                    pass
                finally:
                    return res

        def _onList(req):
            SysMgr.printInfo(UtilMgr.lstrip(req, "LIST:"))
            return True

        def _onClear(req):
            SysMgr.printInfo(UtilMgr.lstrip(req, "CLEAR:"))
            return True

        def _onJobs(req):
            SysMgr.printInfo(UtilMgr.lstrip(req, "JOBS:"))
            return True

        def _onRun(req, onlySocket):
            # parse command #
            command = req.split("|", 1)[1]

            # parse addr #
            addr = "%s:%s" % (self.ip, self.port)

            if not onlySocket:
                SysMgr.printInfo("executed '%s' from %s\n" % (command, addr))

            # return just the connected socket #
            if onlySocket:
                return self

            # get select object #
            selectObj = SysMgr.getPkg("select")

            # set print flag #
            printFlag = SysMgr.getPrintFlag()

            if printFlag:
                print(oneLine)

            # run mainloop #
            isPrint = False
            while 1:
                try:
                    selectObj.select([self.socket], [], [])

                    # receive packet #
                    output = self.getData()
                    if not output:
                        break

                    if printFlag:
                        print(output[:-1])
                    isPrint = True
                except:
                    break

            # print output from server #
            if not isPrint:
                print("no response")

            if printFlag:
                print(oneLine)

            # close connection #
            try:
                self.close()
            except:
                pass

        # get select object to check reply #
        SysMgr.getPkg("select")

        # unmarshalling #
        if isinstance(req, tuple):
            try:
                req = req[0].decode()
            except:
                req = req[0]

            # check request #
            if not req:
                return

            # handle request #
            reqUpper = req.upper()

            if reqUpper.startswith("DOWNLOAD"):
                return _onDownload(req)

            elif reqUpper.startswith("UPLOAD"):
                return _onUpload(req)

            elif reqUpper.startswith("DIR"):
                return _onDir(req)

            elif reqUpper.startswith("RUN"):
                return _onRun(req, onlySocket)

            elif reqUpper.startswith("LIST:"):
                return _onList(req)

            elif reqUpper.startswith("CLEAR"):
                return _onClear(req)

            elif reqUpper.startswith("JOBS"):
                return _onJobs(req)

            elif reqUpper.startswith("NOTIFY"):
                return _onNotify(req)

            elif reqUpper.startswith("ERROR"):
                err = req.split("|", 1)[1]
                errToken = err.find("':")
                errMsg = "%s' from %s" % (err[:errToken], err[errToken + 2 :])
                SysMgr.printErr(errMsg)

            else:
                SysMgr.printErr("failed to recognize '%s' request" % req)

        elif not req:
            SysMgr.printErr("no response from server")

        else:
            SysMgr.printErr("received wrong reply '%s'" % req)

    def send(self, message, write=False):
        if None in (self.ip, self.port):
            SysMgr.printWarn(
                "failed to use IP address for client because it is not set"
            )
            return False
        elif not self.socket:
            SysMgr.printErr(
                "failed to use socket for client because it is not set"
            )
            return False

        # encode message #
        if UtilMgr.isString(message):
            message = UtilMgr.encodeStr(message)

        # convert list to bytes #
        if isinstance(message, list):
            if not message:
                message = "".encode()
            elif message[0][-1] == "\n":
                message = "".join(message).encode()
            else:
                message = "\n".join(message).encode()

        try:
            # send data #
            if self.tcp or self.netlink:
                ret = self.socket.send(message)
            elif not write and SysMgr.localServObj:
                ret = SysMgr.localServObj.socket.sendto(
                    message, (self.ip, self.port)
                )
            else:
                ret = self.socket.sendto(message, (self.ip, self.port))

            # check result #
            if ret < 0:
                raise Exception("send error")
            elif self.status != "ALWAYS":
                self.status = "SENT"
            return True
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printErr(
                "failed to send data to %s:%d as server"
                % (self.ip, self.port),
                True,
            )
            return False

    def sendto(self, message, ip, port):
        if not ip or not port:
            SysMgr.printWarn(
                "failed to use IP address for client because it is not set"
            )
            return False
        elif not self.socket:
            SysMgr.printErr(
                "failed to use socket for client because it is not set"
            )
            return False

        # encode message #
        if UtilMgr.isString(message):
            message = UtilMgr.encodeStr(message)

        try:
            self.socket.sendto(message, (ip, port))
            return True
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printErr(
                "failed to send data to %s:%d as client"
                % (self.ip, self.port),
                True,
            )
            return False

    def recv(self, size=0):
        if None in (self.ip, self.port):
            SysMgr.printWarn(
                "failed to use IP address for server because it is not set"
            )
            return False
        elif not self.socket:
            SysMgr.printErr(
                "failed to use socket for client because it is not set"
            )
            return False

        # set recv size #
        if size == 0:
            size = self.recvSize

        try:
            return self.socket.recv(size)
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printWarn(
                "failed to receive data from %s:%d as client"
                % (self.ip, self.port),
                reason=True,
            )
            return False

    def getData(self, noTimeout=True):
        try:
            data = b""

            # receive and composite packets #
            while 1:
                output = self.recvfrom(noTimeout=noTimeout)

                # handle error #
                if not output:
                    continue

                # handle timeout #
                if not noTimeout and (not output[0] and not output[1]):
                    if data:
                        return data
                    else:
                        return None

                # get only data #
                output = output[0]

                # composite packets #
                data += output

                if not output:
                    break

                # decode data #
                try:
                    output = output.decode()
                except:
                    pass

                if len(output) < self.recvSize and output[-1] == "\n":
                    break
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printErr(
                "failed to get data from %s:%d as client"
                % (self.ip, self.port),
                True,
            )
            return None

        # decode data #
        try:
            retstr = data.decode()
            return retstr
        except:
            return data

    def setNoDelay(self):
        from socket import socket, SOL_TCP, TCP_NODELAY

        # set NODELAY for NAGLE #
        self.socket.setsockopt(SOL_TCP, TCP_NODELAY, 1)

    def recvfrom(self, size=0, noTimeout=False, verb=True):
        if None in (self.ip, self.port):
            SysMgr.printWarn(
                "failed to use IP address for server because it is not set"
            )
            return False
        elif not self.socket:
            SysMgr.printErr(
                "failed to use socket for client because it is not set"
            )
            return False

        # get socket object #
        socket = SysMgr.getPkg("socket", False)
        if not socket:
            return

        # set recv size #
        if size == 0:
            size = self.recvSize

        while 1:
            try:
                message, address = self.socket.recvfrom(size)
                return (message, address)
            except socket.timeout:
                if noTimeout:
                    continue

                SysMgr.printWarn(
                    "failed to receive data from %s:%d as client because of %s"
                    % (self.ip, self.port, "timeout")
                )
                return None
            except KeyboardInterrupt:
                sys.exit(0)
            except SystemExit:
                sys.exit(0)
            except:
                if verb:
                    SysMgr.printWarn(
                        "failed to receive data from %s:%d as client"
                        % (self.ip, self.port),
                        reason=True,
                    )
                return None

    @staticmethod
    def getDataType(data):
        if not data:
            return "None"

        data = data.lstrip()

        if data.startswith("{"):
            return "JSON"
        elif (
            "[INFO" in data[:10]
            or "[ERROR" in data[:10]
            or "[WARN" in data[:10]
            or "[STEP" in data[:10]
        ):
            return "LOG"
        else:
            return "CONSOLE"

    @staticmethod
    def requestCmd(connObj, cmd):
        if not connObj:
            return

        # send request to server #
        connObj.send(cmd)

        # receive reply from server #
        reply = connObj.recvfrom()

        # handle reply from server #
        try:
            connObj.handleServerRequest(reply)
        except:
            return

    @staticmethod
    def requestPing(addr=None, verb=True, cmd=None):
        if cmd:
            cmd = cmd.upper()
            cmd = "%s:%s" % (cmd, cmd)
        else:
            cmd = "PING:PING"

        return NetworkMgr.execRemoteCmd(cmd, addr=addr, verb=verb)

    @staticmethod
    def getCmdPipe(connObj, cmd):
        if not cmd:
            return None

        # add command prefix #
        reqCmd = cmd.split(":", 1)[0].upper()
        if reqCmd in NetworkMgr.REQUEST_LIST:
            pass
        elif not cmd.startswith("run:"):
            cmd = "run:%s" % cmd

        # send request to server #
        connObj.send(cmd)

        # receive reply from server #
        reply = connObj.recvfrom()
        try:
            if not reply:
                raise Exception()

            msg = reply[0].decode()
            if not msg:
                pass
            elif msg == "PONG":
                return True
            elif msg == "NO_SERV_NODE":
                SysMgr.printErr(
                    "no service node to %s:%s" % (connObj.ip, connObj.port)
                )
                return True
            elif msg.startswith("MSG:"):
                # print message in the packet #
                SysMgr.printInfo(UtilMgr.lstrip(msg, "MSG:"))

                # send ACK to prevent receiving two packegs at once #
                connObj.send("ACK")

                # wait for a request again #
                reply = connObj.recvfrom()
            elif msg.startswith("LIST:"):
                reply = (msg,)
            elif msg.startswith("NOTIFY:"):
                return True
        except:
            pass

        # handle reply from server #
        try:
            return connObj.handleServerRequest(reply, onlySocket=True)
        except:
            return None

    @staticmethod
    def execRemoteCmd(command, addr=None, verb=True):
        # get new connection #
        connObj = NetworkMgr.getServerConn(addr, verb)
        if not connObj:
            return None

        # launch remote command #
        pipe = NetworkMgr.getCmdPipe(connObj, command)
        return pipe

    @staticmethod
    def getServerConn(addr=None, verb=True):
        def _printErr():
            SysMgr.printErr("no running server or wrong server address")

        # set server address in local #
        if addr:
            # classify IP and PORT #
            service, ip, port = NetworkMgr.parseAddr(addr)
            if service == ip == port == None:
                _printErr()
                return None
            else:
                NetworkMgr.setRemoteServer("%s:%s" % (ip, port), tcp=True)
        elif SysMgr.isLinux and not SysMgr.remoteServObj:
            try:
                addr = SysMgr.getProcNetAddrs(__module__)
            except:
                addr = None

            if not addr:
                return None

            # classify IP and PORT #
            service, ip, port = NetworkMgr.parseAddr(addr)
            if service == ip == port == None:
                _printErr()
                return None
            else:
                NetworkMgr.setRemoteServer(addr, tcp=True)
        # set server address again #
        elif SysMgr.remoteServObj:
            servObj = SysMgr.remoteServObj
            ip = servObj.ip
            port = servObj.port
            NetworkMgr.setRemoteServer("%s:%s" % (ip, port), tcp=True)

        # check server address #
        if not SysMgr.remoteServObj:
            _printErr()
            return None

        # bind local socket for UDP #
        try:
            if not SysMgr.remoteServObj.tcp and SysMgr.localServObj:
                lip = SysMgr.localServObj.ip
                lport = SysMgr.localServObj.port
                SysMgr.remoteServObj.socket.bind((lip, lport))
        except:
            SysMgr.printErr(
                "failed to bind socket to %s:%s for connection" % (lip, lport),
                True,
            )

        # do connect to server #
        try:
            connObj = SysMgr.remoteServObj

            if not "NOTIMEOUT" in SysMgr.environList:
                connObj.timeout()

            # set immediately exit flag to prevent hang on connect #
            origFlag = SysMgr.exitFlag
            SysMgr.exitFlag = True

            # connect with handling CLOSE_WAIT #
            while 1:
                try:
                    connObj.connect()
                    break
                except SystemExit:
                    sys.exit(0)
                except:
                    SysMgr.printWarn(
                        "failed to connect to %s:%s" % (ip, port),
                        reason=True,
                        always=verb,
                    )

                    # handle error #
                    _, err, to = sys.exc_info()
                    if err.args and err.args[0] == 99:
                        time.sleep(0.1)
                        continue
                    else:
                        raise Exception(err.args[0])
                finally:
                    SysMgr.exitFlag = origFlag

            return connObj
        except SystemExit:
            sys.exit(0)
        except:
            if verb:
                SysMgr.printErr("failed to set socket for connection", True)
            return None

    @staticmethod
    def parseAddr(value):
        service = None
        ip = None
        port = None

        if not UtilMgr.isString(value):
            return (service, ip, port)

        # get request and address #
        cmdList = value.split("@")
        if len(cmdList) >= 2:
            service = cmdList[0]
            addr = cmdList[1]
        else:
            addr = value

        # get IP and PORT #
        addrList = addr.split(":")
        arg = addrList[0]
        if len(addrList) >= 2:
            try:
                # set IP #
                if arg and str(arg).upper() == "LOCALHOST":
                    ip = "127.0.0.1"
                else:
                    ip = arg

                # set PORT #
                if addrList[1]:
                    port = long(addrList[1])
            except:
                pass
        else:
            try:
                # set IP #
                if "." in arg:
                    ip = arg
                elif str(arg).upper() == "LOCALHOST":
                    ip = "127.0.0.1"
                # set PORT #
                else:
                    port = long(arg)
            except:
                pass

        return (service, ip, port)

    @staticmethod
    def setRemoteServer(value, tcp=False, verb=True):
        # receive mode #
        if not value:
            SysMgr.remoteServObj = "NONE"
            return

        # request mode #
        service, ip, port = NetworkMgr.parseAddr(value)

        # set PRINT as default #
        if not service:
            service = "PRINT"

        if not ip:
            ip = NetworkMgr.getPublicIp()

        if port is None:
            port = SysMgr.defaultServPort

        # check server addresses #
        if (
            SysMgr.localServObj
            and SysMgr.localServObj.ip == ip
            and SysMgr.localServObj.port == port
        ):
            SysMgr.printErr(
                (
                    "wrong value for remote connection, "
                    "local address and remote address are same (%s:%s)"
                )
                % (ip, port)
            )
            sys.exit(-1)

        # check request #
        if not ip or not port or not SysMgr.isValidRequest(service):
            reqList = ""
            for req in TaskAnalyzer.requestType:
                reqList += req + "|"

            SysMgr.printErr(
                "wrong input address, input [%s]@IP:PORT as remote address"
                % reqList[:-1]
            )
            sys.exit(-1)

        # create a socket #
        networkObject = NetworkMgr("client", ip, port, tcp=tcp)
        if not networkObject.ip:
            sys.exit(-1)
        else:
            networkObject.request = service
            SysMgr.remoteServObj = networkObject

        # set protocol #
        if tcp:
            proto = "TCP"
        else:
            proto = "UDP"

        if verb:
            SysMgr.printInfo(
                "use %s:%d(%s) as remote address" % (ip, port, proto)
            )

        return SysMgr.remoteServObj

    @staticmethod
    def setRemoteNetwork(service, ip, port):
        # set default service #
        if not service:
            service = "PRINT"

        errMsg = (
            "wrong value for remote server, input in the format [%s]@IP:PORT"
        ) % "|".join(TaskAnalyzer.requestType)

        if not ip or not SysMgr.isValidRequest(service):
            SysMgr.printErr(errMsg)
            sys.exit(-1)

        if port is None:
            port = SysMgr.defaultServPort

        # create a new object #
        netObj = NetworkMgr("client", ip, port)
        if not netObj.ip:
            sys.exit(-1)
        else:
            netObj.status = "ALWAYS"
            netObj.request = service
            naddr = "%s:%s" % (ip, str(port))

            if service == "PRINT":
                SysMgr.addrListForPrint[naddr] = netObj
            elif service.startswith("REPORT"):
                SysMgr.reportEnable = True
                SysMgr.addrListForReport[naddr] = netObj
            else:
                SysMgr.printErr(errMsg)

        SysMgr.printInfo(
            "use %s:%d as remote address to request %s" % (ip, port, service)
        )

    @staticmethod
    def setServerNetwork(
        ip,
        port,
        force=False,
        blocking=False,
        tcp=False,
        anyPort=False,
        reuse=True,
        weakPort=False,
        verb=False,
    ):
        if SysMgr.localServObj and not force:
            SysMgr.printWarn(
                "ignored to set server network because its already set", verb
            )
            return

        # get internet available IP first #
        if not ip:
            ip = NetworkMgr.getPublicIp()

        # set default port #
        if port is None:
            if SysMgr.checkMode("cli"):
                port = SysMgr.defaultCliPort
            else:
                port = SysMgr.defaultServPort

        # print available IP list #
        try:
            iplist = sorted(NetworkMgr.getUsingIps())
            if iplist:
                SysMgr.printWarn(
                    "available IP list [ %s ]" % ", ".join(iplist), verb
                )
        except:
            pass

        # check server setting #
        localObj = SysMgr.localServObj
        if (
            localObj
            and localObj.socket
            and localObj.ip == ip
            and localObj.port == port
        ):
            if blocking:
                localObj.socket.setblocking(1)
            else:
                localObj.socket.setblocking(0)
            return

        # create a new server setting #
        networkObject = NetworkMgr(
            "server",
            ip,
            port,
            tcp=tcp,
            blocking=blocking,
            anyPort=anyPort,
            reuse=reuse,
        )
        if not networkObject.ip and weakPort:
            networkObject = NetworkMgr(
                "server",
                ip,
                port,
                tcp=tcp,
                blocking=blocking,
                anyPort=True,
                reuse=reuse,
            )
        if not networkObject.ip:
            SysMgr.printWarn("failed to set server IP", verb)
            return

        # set protocol #
        if tcp:
            proto = "TCP"
        else:
            proto = "UDP"

        localObj = SysMgr.localServObj = networkObject

        # print adderss info #
        if not SysMgr.masterPid:
            if SysMgr.parentPid:
                printer = SysMgr.printWarn
            else:
                printer = SysMgr.printInfo
            printer(
                "use %s:%d(%s) as local address"
                % (localObj.ip, localObj.port, proto)
            )

        return networkObject

    @staticmethod
    def prepareServerConn(cliAddr, servAddr):
        # set local address #
        if not cliAddr:
            NetworkMgr.setServerNetwork(None, None, anyPort=True)
        else:
            service, ip, port = NetworkMgr.parseAddr(cliAddr)

            NetworkMgr.setServerNetwork(ip, port)

        # set remote address #
        if servAddr:
            NetworkMgr.setRemoteServer(servAddr)

        # set client address #
        if SysMgr.localServObj:
            cliIp = SysMgr.localServObj.ip
            cliPort = SysMgr.localServObj.port
        else:
            cliIp = None
            cliPort = None

        # set server address #
        if SysMgr.remoteServObj.ip:
            servIp = SysMgr.remoteServObj.ip
            servPort = SysMgr.remoteServObj.port
        else:
            servIp = None
            servPort = None

        return (cliIp, cliPort), (servIp, servPort)

    @staticmethod
    def getDevByIp(ip):
        # get device data #
        data = SysMgr.getNetDevData()
        if not data:
            return

        # create a new socket #
        try:
            fcntl = SysMgr.getPkg("fcntl", False)
            socket = SysMgr.getPkg("socket", False)
            sobj = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
        except SystemExit:
            sys.exit(0)
        except:
            return

        # scan IP device and address #
        for line in data:
            try:
                dev, stats = line.split(":")

                dev = dev.strip()

                res = fcntl.ioctl(
                    sobj.fileno(),
                    0x8915,  # SIOCGIFADDR
                    struct.pack("256s", dev[:15].encode("utf-8")),
                )

                ipaddr = socket.inet_ntoa(res[20:24])
                if ipaddr == ip:
                    return dev
            except SystemExit:
                sys.exit(0)
            except:
                pass

    @staticmethod
    def getMainMacAddr():
        dirPath = "/sys/class/net"

        try:
            devices = os.listdir(dirPath)
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printOpenErr(dirPath)
            return

        # return main IP and device #
        try:
            ip = NetworkMgr.getPublicIp()
            if ip:
                dev = NetworkMgr.getDevByIp(ip)
                if dev:
                    return (dev, ip)
        except SystemExit:
            sys.exit(0)
        except:
            pass

        for dev in devices:
            # skip virtual device #
            if dev == "lo" or dev.startswith("docker"):
                continue

            target = "%s/%s/address" % (dirPath, dev)
            try:
                with open(target, "r") as fd:
                    addr = fd.readline()[:-1]
                    return (dev, addr)
            except SystemExit:
                sys.exit(0)
            except:
                SysMgr.printOpenErr(target)

        return ("None", "None")

    @staticmethod
    def getUsingIps():
        if not SysMgr.isLinux:
            return

        connPaths = [
            "%s/net/udp" % SysMgr.procPath,
            "%s/net/tcp" % SysMgr.procPath,
        ]

        validList = {}
        cacheList = {}

        for path in connPaths:
            try:
                with open(path, "r") as fd:
                    ipList = fd.read().split("\n")

                # remove title #
                ipList.pop(0)

                for line in ipList:
                    if not line:
                        continue
                    items = line.split(None, 2)
                    addr = items[1].split(":")[0]
                    if addr in cacheList:
                        continue

                    cacheList[addr] = None
                    ip = SysMgr.convCIDR(addr)
                    validList[ip] = None
            except SystemExit:
                sys.exit(0)
            except:
                SysMgr.printOpenWarn(path)

        return list(validList)

    @staticmethod
    def getGateways():
        gateways = {}

        ips = NetworkMgr.getRoutedIps()

        for item in ips:
            try:
                ip = item[1]
                if ip in ("0.0.0.0", "127.0.0.1") or not ip.endswith(".1"):
                    continue

                gw = "%s.1" % ip[: ip.rfind(".")]
                gateways[gw] = None
            except SystemExit:
                sys.exit(0)
            except:
                pass

        return list(gateways)

    @staticmethod
    def getMainIp():
        ipList = NetworkMgr.getUsingIps()

        # remove IP for all IPv4 addresses #
        try:
            ipList.remove("0.0.0.0")
        except SystemExit:
            sys.exit(0)
        except:
            pass

        # return main IP #
        if not ipList:
            return None
        elif "127.0.0.1" in ipList:
            return "127.0.0.1"
        else:
            return list(sorted(ipList, reverse=True))[0]

    @staticmethod
    def getRoutedIps():
        validList = []
        routePath = "%s/net/route" % SysMgr.procPath
        try:
            with open(routePath, "r") as fd:
                ipList = fd.readlines()

            # remove title #
            ipList.pop(0)

            for line in ipList:
                items = line.split()
                validList.append([items[0], SysMgr.convCIDR(items[1])])

            return validList
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printOpenWarn(routePath)
            return validList

    @staticmethod
    def getHostName():
        try:
            return SysMgr.getPkg("socket").gethostname()
        except:
            return None

    @staticmethod
    def getPublicIp(force=False):
        if SysMgr.ipAddr and not force:
            return SysMgr.ipAddr

        # get socket object #
        socket = SysMgr.getPkg("socket", False)
        if not socket:
            return

        from socket import socket, AF_INET, SOCK_DGRAM, SOCK_STREAM

        ret = None

        try:
            s = socket(AF_INET, SOCK_STREAM)
            s.settimeout(0.3)

            # connect to google public IP #
            s.connect(("8.8.8.8", 53))

            ret = s.getsockname()[0]
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printWarn("failed to get public IP address")

        if not ret:
            ret = NetworkMgr.getMainIp()

        SysMgr.ipAddr = ret

        return ret

    def __del__(self):
        try:
            self.close()
        except:
            pass


class Timeline(object):
    """SVG viewer for timeline segments"""

    class Segment(object):
        def __init__(self, group, time_start, time_end, extra):
            self.group = group
            self.time_start = time_start
            self.time_end = time_end
            self.text = None
            self.id = None
            self.color = None
            self.state = None
            self.info = None
            self._init_extra(extra)

        def _init_extra(self, extra):
            if "text" in extra:
                self.text = extra["text"]

            if "id" in extra:
                self.id = extra["id"]

            if "color" in extra:
                self.color = extra["color"]

            if "state" in extra:
                self.state = extra["state"]

            if "info" in extra:
                self.info = extra["info"]

    class Config(object):
        def _conv_palette(self, palette):
            plist = []
            for palette_entry in palette:
                """
                rgb = [
                    int(rgb_value)
                    for rgb_value in re.findall("\d+", palette_entry)
                ]
                """
                color = "rgb%s" % palette_entry
                plist.append(color)
            return plist

        def __init__(self):
            self.WIDTH = 1500
            self.HEIGHT = 770
            self.TIME_AXIS_HEIGHT = 1
            self.TICKS = 100

            # set item filter #
            self.FILTER = SysMgr.environList.get("FILTER", [])
            if self.FILTER:
                SysMgr.printInfo(
                    "only specific items [ %s ] are drawn"
                    % ", ".join(self.FILTER)
                )

            # set label filter #
            if "LABELMIN" in SysMgr.environList:
                self.LABEL_SIZE_MIN = UtilMgr.getEnvironNum(
                    "LABELMIN", isInt=True
                )
                SysMgr.printInfo(
                    "only time segments bigger than %s are printed"
                    % UtilMgr.convNum(self.LABEL_SIZE_MIN)
                )
            else:
                self.LABEL_SIZE_MIN = 0

            # set font size #
            self.FONT_SIZE = UtilMgr.getEnvironNum(
                "FONTSIZE", False, 3, False, isInt=True
            )

            # set group font size #
            self.GROUP_FONT_SIZE = UtilMgr.getEnvironNum(
                "GROUPFONTSIZE", False, 0, False, isInt=True
            )

            palette = [
                "(0,150,136)",
                "(0,188,212)",
                "(0,0,128)",
                "(0,0,139)",
                "(0,0,205)",
                "(0,0,255)",
                "(0,100,0)",
                "(0,128,0)",
                "(0,128,128)",
                "(0,139,139)",
                "(0,191,255)",
                "(0,206,209)",
                "(0,250,154)",
                "(0,255,0)",
                "(0,255,127)",
                "(0,255,255)",
                "(100,149,237)",
                "(102,205,170)",
                "(103,58,183)",
                "(106,90,205)",
                "(107,142,35)",
                "(121,85,72)",
                "(123,104,238)",
                "(124,252,0)",
                "(127,255,0)",
                "(127,255,212)",
                "(128,0,0)",
                "(128,0,128)",
                "(128,128,0)",
                "(128,128,128)",
                "(135,206,235)",
                "(135,206,250)",
                "(138,43,226)",
                "(139,195,74)",
                "(139,0,0)",
                "(139,0,139)",
                "(139,69,19)",
                "(143,188,143)",
                "(144,238,144)",
                "(147,112,219)",
                "(148,0,211)",
                "(152,251,152)",
                "(153,50,204)",
                "(154,205,50)",
                "(156,39,176)",
                "(158,158,158)",
                "(160,82,45)",
                "(165,42,42)",
                "(173,216,230)",
                "(173,255,47)",
                "(175,238,238)",
                "(176,224,230)",
                "(178,34,34)",
                "(184,134,11)",
                "(186,85,211)",
                "(188,143,143)",
                "(189,183,107)",
                "(192,192,192)",
                "(199,21,133)",
                "(205,133,63)",
                "(205,220,57)",
                "(205,92,92)",
                "(210,105,30)",
                "(210,180,140)",
                "(216,191,216)",
                "(218,112,214)",
                "(218,165,32)",
                "(219,112,147)",
                "(220,20,60)",
                "(221,160,221)",
                "(222,184,135)",
                "(224,255,255)",
                "(233,150,122)",
                "(233,30,99)",
                "(238,130,238)",
                "(238,232,170)",
                "(240,128,128)",
                "(240,230,140)",
                "(244,67,54)",
                "(244,164,96)",
                "(245,222,179)",
                "(245,245,220)",
                "(25,25,112)",
                "(250,128,114)",
                "(250,235,215)",
                "(250,250,210)",
                "(255,0,0)",
                "(255,0,255)",
                "(255,105,180)",
                "(255,127,80)",
                "(255,140,0)",
                "(255,152,0)",
                "(255,160,122)",
                "(255,165,0)",
                "(255,182,193)",
                "(255,192,203)",
                "(255,193,7)",
                "(255,20,147)",
                "(255,215,0)",
                "(255,228,196)",
                "(255,235,205)",
                "(255,235,59)",
                "(255,248,220)",
                "(255,250,205)",
                "(255,255,0)",
                "(255,255,224)",
                "(255,69,0)",
                "(255,87,34)",
                "(255,99,71)",
                "(3,169,244)",
                "(30,144,255)",
                "(32,178,170)",
                "(33,150,243)",
                "(34,139,34)",
                "(46,139,87)",
                "(47,79,79)",
                "(50,205,50)",
                "(60,179,113)",
                "(63,81,181)",
                "(64,224,208)",
                "(65,105,225)",
                "(70,130,180)",
                "(72,209,204)",
                "(72,61,139)",
                "(75,0,130)",
                "(76,175,80)",
                "(85,107,47)",
                "(95,158,160)",
                "(96,125,139)",
            ]

            self.PALETTE = self._conv_palette(palette)

            # shuffle list #
            random = SysMgr.getPkg("random", False)
            if random:
                random.shuffle(self.PALETTE)

        @staticmethod
        def _load(fileName=None, data=None):
            if fileName:
                with open(fileName) as fd:
                    data = fd.read()
                    data = UtilMgr.convStr2Dict(data)
            elif not data:
                SysMgr.printErr("no path or data for timeline config")
                sys.exit(-1)

            config = Timeline.Config()
            config.WIDTH = data.get("width", 20000)
            config.HEIGHT = data.get("height", 4000)
            config.FONT_SIZE = data.get("font_size", 3)
            config.TICKS = data.get("time_ticks", 5)
            config.TIME_AXIS_HEIGHT = data.get("time_axis_height", 5)
            config.PALETTE = config._conv_palette(data.get("palette", []))
            config.TIMEUNIT = data.get("time_unit")
            config.TIMEFACTOR = data.get("time_factor", 1)
            if config.LABEL_SIZE_MIN == 0:
                config.LABEL_SIZE_MIN = data.get("label_size_min", 5)

            return config

    def __init__(self, title, segments, time_unit, fontsize, config, tasks=[]):
        self.title = title
        self.segments = segments
        self.time_unit = time_unit
        self.config = config

        self.time_start = min(
            segments, key=lambda segment: segment.time_start
        ).time_start
        self.time_end = max(
            segments, key=lambda segment: segment.time_end
        ).time_end
        self.segment_groups = sorted(
            list(map(long, set(s.group for s in self.segments)))
        )
        self.segment_groups = list(map(str, self.segment_groups))
        self.groups = len(self.segment_groups)
        self.scaled_height = self.config.HEIGHT / self.groups
        self.group_list = list(self.segment_groups)
        self.group_stat = [
            {
                "cnt": 0,
                "du_min": 0,
                "du_max": 0,
                "du_list": [],
                "inter_min": 0,
                "inter_max": 0,
                "inter_list": [],
                "prev": 0,
            }
            for _ in xrange(len(self.group_list))
        ]

        try:
            self.ratio = self.config.WIDTH / float(
                self.time_end - self.time_start
            )
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printErr(
                "failed to recognize timeline because start and end are same"
            )
            sys.exit(-1)

        self.last_group_segment = {}
        self.last_group_time = {}
        self.height_group_pos = {}
        self.last_iogroup_segment = {}
        self.last_iogroup_time = {}
        self.height_iogroup_pos = {}

        # apply task converter #
        try:
            self.tasks = {}
            if tasks:
                for task in tasks:
                    if task in Timeline.conv_table:
                        new = Timeline.conv_table[task]
                        self.tasks[new] = tasks[task]
                    elif isinstance(tasks, dict):
                        self.tasks[task] = tasks[task]
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printErr("failed to change tasks in timeline", True)
            sys.exit(-1)

        # time factor #
        if hasattr(self.config, "TIMEFACTOR"):
            self.time_factor = self.config.TIMEFACTOR
        else:
            self.time_factor = 1

        # task color #
        if self.tasks:
            self.color_map = self._build_task_color_map()
        else:
            self.color_map = self._build_color_map()

        # update font size #
        if fontsize and self.config:
            self.config.FONT_SIZE = fontsize

    def _build_task_color_map(self):
        color_map = {}
        palette = self.config.PALETTE
        for i, group in enumerate(self.tasks):
            color_map[group] = palette[i % len(palette)]
        return color_map

    def _build_color_map(self):
        color_map = {}
        palette = self.config.PALETTE
        segment_groups = self.segment_groups
        for i, group in enumerate(segment_groups):
            color_map[group] = palette[i % len(palette)]
        return color_map

    def _draw_grid(self, dwg):
        for x in xrange(0, self.config.WIDTH, self.config.TICKS):
            dwg.add(
                dwg.line(
                    (x, 0),
                    (x, self.config.HEIGHT),
                    stroke="black",
                    stroke_width=0.1,
                )
            )

    def _draw_group_stat(self, dwg):
        # set group font size #
        if self.config.GROUP_FONT_SIZE > 0:
            fontsize = self.config.GROUP_FONT_SIZE
        else:
            fontsize = self.scaled_height / 4

        idx = 0
        convNum = UtilMgr.convNum
        for y_tick in xrange(self.config.HEIGHT):
            try:
                stats = self.group_stat[idx]
                idx += 1

                # count #
                count = stats["cnt"]
                if count == 0:
                    continue

                # build string #
                statstr = "Cnt: %s" % convNum(count)

                # interval #
                if count > 1:
                    try:
                        inter = stats["inter_list"]
                        inter_avg = long(sum(inter) / (len(inter)))
                    except SystemExit:
                        sys.exit(0)
                    except:
                        inter_avg = 0
                    inter_std = UtilMgr.getStdev(inter)

                    statstr += (
                        ", IntMin: %s, IntMax: %s, IntAvg: %s, IntStd: %s"
                        % (
                            convNum(stats["inter_min"]),
                            convNum(stats["inter_max"]),
                            convNum(inter_avg),
                            convNum(inter_std),
                        )
                    )

                # duration #
                if stats["du_min"] > 0:
                    try:
                        du = stats["du_list"]
                        du_avg = long(sum(du) / (len(du)))
                    except SystemExit:
                        sys.exit(0)
                    except:
                        du_avg = 0
                    du_std = UtilMgr.getStdev(du)

                    statstr += (
                        ", DuMin: %s, DuMax: %s, DuAvg: %s, DuStd: %s"
                        % (
                            convNum(stats["du_min"]),
                            convNum(stats["du_max"]),
                            convNum(du_avg),
                            convNum(du_std),
                        )
                    )
            except SystemExit:
                sys.exit(0)
            except:
                continue

            y_tick *= self.scaled_height
            dwg.add(
                dwg.text(
                    statstr,
                    (2, y_tick + self.scaled_height / 2),
                    font_size=fontsize / 2,
                    fill="rgb(200,200,200)",
                )
            )

    def _draw_group_axis(self, dwg, yval):
        dwg.add(
            dwg.rect(
                (0, 0),
                (self.config.TIME_AXIS_HEIGHT, self.config.HEIGHT),
                fill="black",
            )
        )

        # set group font size #
        if self.config.GROUP_FONT_SIZE > 0:
            fontsize = self.config.GROUP_FONT_SIZE
        else:
            fontsize = self.scaled_height / 2

        idx = 0
        groupList = list(self.segment_groups)
        for y_tick in xrange(self.config.HEIGHT):
            try:
                name = groupList[idx]
                idx += 1
            except:
                continue

            y_tick *= self.scaled_height

            dwg.add(
                dwg.line(
                    (0, y_tick),
                    (self.config.WIDTH, y_tick),
                    stroke="black",
                    stroke_width=2,
                )
            )

            dwg.add(
                dwg.text(
                    name,
                    (2, y_tick + self.scaled_height),
                    font_size=fontsize,
                    fill="rgb(200,200,200)",
                )
            )

            # add info #
            addinfo = yval[name] if yval and name in yval else ""
            name = str(name)
            if not addinfo and self.tasks and name in self.tasks:
                addinfo = self.tasks[name]
            if addinfo:
                x = 2 * self.scaled_height * len(name) / 5
                dwg.add(
                    dwg.text(
                        addinfo,
                        (x, y_tick + self.scaled_height),
                        font_size=fontsize / 2,
                        fill="rgb(200,200,200)",
                    )
                )

    def _draw_time_axis(self, dwg, start=0, annotation=None):
        dwg.add(
            dwg.rect(
                (0, self.config.HEIGHT),
                (self.config.WIDTH, self.config.TIME_AXIS_HEIGHT),
                fill="black",
            )
        )

        y_time_tick = self.config.HEIGHT + self.config.TIME_AXIS_HEIGHT / 2
        ratio = 1 / self.ratio

        # set time delta #
        start = float(start) / self.time_factor

        # create the drawing group #
        g = dwg.add(dwg.g())
        if annotation:
            g.set_desc(annotation)

        for x_tick_time in xrange(0, self.config.WIDTH, self.config.TICKS):
            abs_x_tick_time = (x_tick_time * ratio) + start
            tick_time = "{:10}".format(long(abs_x_tick_time))
            g.add(
                dwg.text(
                    "%s %s" % (UtilMgr.convNum(tick_time), self.time_unit),
                    (x_tick_time, y_time_tick + self.config.FONT_SIZE * 5),
                    font_size=self.config.FONT_SIZE * 5,
                    color="rgb(255,255,255)",
                )
            )

    def _draw_background(self, dwg):
        dwg.add(
            dwg.rect(
                (0, 0),
                (self.config.WIDTH, self.config.HEIGHT),
                fill="rgb(245,245,245)",
            )
        )

        # set title #
        if self.title:
            title = self.title
        else:
            title = "Guider Timeline"

        # set font size for title #
        fontsize = self.config.FONT_SIZE * 10

        dwg.add(
            dwg.text(
                title,
                (
                    (self.config.WIDTH / 2) - (len(title) * fontsize / 4),
                    fontsize,
                ),
                font_size=fontsize,
                font_weight="bolder",
                fill="rgb(200,200,200)",
            )
        )

    def _draw_segments(self, dwg, start=0):
        # set time delta #
        start = float(start) / self.time_factor

        for idx, segment in enumerate(
            sorted(self.segments, key=lambda e: len(e.state))
        ):
            UtilMgr.printProgress(idx, len(self.segments))
            self._draw_segment(segment, dwg, start)

        if self.segments:
            UtilMgr.deleteProgress()

    def _draw_segment(self, segment, dwg, start=0):
        x0 = float(segment.time_start - self.time_start) * self.ratio
        x1 = float(segment.time_end - self.time_start) * self.ratio
        group_idx = self.group_list.index(segment.group)
        y0 = self.scaled_height * float(group_idx % self.groups)
        y1 = self.scaled_height * float((group_idx % self.groups) + 1)
        scaled_width = x1 - x0
        scaled_top_height = y0 + (self.scaled_height / 7)
        scaled_bottom_height = y1 - (self.scaled_height * 0.25)
        duration = segment.time_end - segment.time_start
        time_end = segment.time_end + start - self.time_start
        strokeSize = 3.5

        # check filter #
        if self.config.FILTER and not UtilMgr.isValidStr(
            segment.text, self.config.FILTER
        ):
            return

        # get color ID #
        if segment.color:
            color = segment.color
        else:
            if segment.id:
                colorid = segment.id
            else:
                colorid = group_idx

            # get real color via ID #
            try:
                color = self.color_map[str(colorid)]
            except SystemExit:
                sys.exit(0)
            except:
                try:
                    cidx = long(colorid) % len(self.color_map)
                    color = self.color_map[list(self.color_map)[cidx]]
                    self.color_map[str(colorid)] = color
                except SystemExit:
                    sys.exit(0)
                except:
                    color = self.color_map[list(self.color_map)[0]]

        # check stroke option #
        if self.stroke_text and UtilMgr.isValidStr(
            segment.text, self.stroke_text
        ):
            stroke = "rgb(255,0,0)"
            stroke_width = 1
        else:
            stroke = "none"
            stroke_width = 0

        # update count stat #
        stats = self.group_stat[group_idx]
        stats["cnt"] += 1

        # update interval stats #
        if stats["prev"] > 0:
            interval = segment.time_start - stats["prev"]
        else:
            interval = 0
        stats["prev"] = segment.time_start
        if stats["inter_min"] == 0 or stats["inter_min"] > interval:
            stats["inter_min"] = interval
        if stats["inter_max"] == 0 or stats["inter_max"] < interval:
            stats["inter_max"] = interval
        stats["inter_list"].append(interval)

        # apply stroke by interval #
        if self.stroke_interval > 0 and self.stroke_interval <= interval:
            stroke = "rgb(255,0,0)"
            stroke_width = 1

        # update duration stats #
        if stats["du_min"] == 0 or stats["du_min"] > duration:
            stats["du_min"] = duration
        if stats["du_max"] == 0 or stats["du_max"] < duration:
            stats["du_max"] = duration
        stats["du_list"].append(duration)

        # apply stroke by duration #
        if self.stroke_duration > 0 and self.stroke_duration <= duration:
            stroke = "rgb(255,0,0)"
            stroke_width = 1

        # draw bold line for core off #
        if segment.state == "OFF":
            dwg.add(
                dwg.line(
                    (x0, y0), (x1, y0), stroke="gray", stroke_width=strokeSize
                )
            )
            return

        # draw circle and text for event #
        if segment.state and segment.state.startswith("EVENT"):
            if segment.state == "EVENT_MARK":
                font_size = self.config.FONT_SIZE
            elif segment.state == "EVENT_USER":
                font_size = self.config.FONT_SIZE * 3
            else:
                font_size = self.config.FONT_SIZE

            dwg.add(
                dwg.circle(
                    center=(x0, y0),
                    r=font_size / 2,
                    stroke="darkgreen",
                    fill="darkgreen",
                )
            )

            # draw text #
            dwg.add(
                dwg.text(
                    segment.text,
                    (x0 + font_size, y0),
                    fill="darkgreen",
                    stroke="none",
                    font_size=font_size,
                    font_weight="bolder",
                )
            )

            return

        # define shortcut and duration string #
        convNum = UtilMgr.convNum
        durationstr = convNum(duration)

        if hasattr(segment, "info") and segment.info:
            info = "\n%s" % segment.info
        else:
            info = ""

        # create the drawing group #
        g = dwg.add(dwg.g())
        g.set_desc(
            "%s %s / ~%s (%s)%s"
            % (
                segment.text,
                durationstr,
                convNum(time_end),
                self.time_unit,
                info,
            )
        )

        # draw line for block_read status #
        if segment.state == "RD":
            g.add(
                dwg.rect(
                    (x0, scaled_bottom_height),
                    (scaled_width, self.scaled_height * 0.25),
                    rx=1,
                    ry=1,
                    fill="purple",
                    fill_opacity=0.5,
                    stroke=stroke,
                    stroke_width=stroke_width,
                )
            )
        # draw line for block_write status #
        elif segment.state == "WR":
            g.add(
                dwg.rect(
                    (x0, scaled_bottom_height),
                    (scaled_width, self.scaled_height * 0.25),
                    rx=1,
                    ry=1,
                    fill="darkcyan",
                    fill_opacity=0.5,
                    stroke=stroke,
                    stroke_width=stroke_width,
                )
            )
        # draw line for syscall status #
        elif segment.state == "SYSCALL":
            g.add(
                dwg.rect(
                    (x0, y0),
                    (scaled_width, self.scaled_height * 0.5),
                    rx=1,
                    ry=1,
                    fill=color,
                    fill_opacity=0.5,
                    stroke=stroke,
                    stroke_width=stroke_width,
                )
            )
        # draw line for sched status #
        else:
            # draw CPU timeslice #
            g.add(
                dwg.rect(
                    (x0, y0),
                    (scaled_width, self.scaled_height),
                    rx=1,
                    ry=1,
                    fill=color,
                    fill_opacity=0.5,
                    stroke=stroke,
                    stroke_width=stroke_width,
                )
            )

            # draw preempted status #
            if segment.state == "R":
                g.add(
                    dwg.line(
                        (x1, y0),
                        (x1, scaled_top_height),
                        stroke="red",
                        stroke_width=strokeSize,
                    )
                )
            # draw wait status #
            elif segment.state == "D":
                g.add(
                    dwg.line(
                        (x1, y0),
                        (x1, scaled_top_height),
                        stroke="black",
                        stroke_width=strokeSize,
                    )
                )

        # check label flag #
        if (
            not SysMgr.showAll
            and not "LABEL" in SysMgr.environList
            and not "LABELMIN" in SysMgr.environList
        ):
            return
        # check duration #
        elif not duration or duration < self.config.LABEL_SIZE_MIN:
            return

        # convert duration to text #
        duration = "~%s" % convNum(durationstr)

        # I/O #
        if segment.state in ("RD", "WR"):
            # initialize group data #
            self.last_iogroup_segment.setdefault(group_idx, None)
            self.last_iogroup_time.setdefault(group_idx, x0)

            # define data #
            last_iogroup_segment = self.last_iogroup_segment[group_idx]
            last_iogroup_time = self.last_iogroup_time[group_idx]

            # set text attributes for block_read #
            if segment.state == "RD":
                if (
                    last_iogroup_segment == segment.text
                    and x0 - last_iogroup_time < self.config.TICKS
                ):
                    segment_label = durationstr
                    font_size = self.config.FONT_SIZE - 1
                else:
                    segment_label = "[R] %s %s" % (segment.text, durationstr)
                    font_size = self.config.FONT_SIZE - 0.7
                color = "rgb(128,0,128)"
            # set text attributes for block_write #
            elif segment.state == "WR":
                if (
                    last_iogroup_segment == segment.text
                    and x0 - last_iogroup_time < self.config.TICKS
                ):
                    segment_label = durationstr
                    font_size = self.config.FONT_SIZE - 1
                else:
                    segment_label = "[W] %s %s" % (segment.text, durationstr)
                    font_size = self.config.FONT_SIZE - 0.7
                color = "rgb(0,139,139)"

            # update group info #
            self.last_iogroup_segment[group_idx] = segment.text
            self.last_iogroup_time[group_idx] = x0
            self.height_iogroup_pos.setdefault(group_idx, 0)

            # set text position #
            scaled_pos = self.scaled_height * 0.75
            self.height_iogroup_pos[group_idx] += self.config.FONT_SIZE
            height_pos = self.height_iogroup_pos[group_idx]
            if height_pos + scaled_pos >= self.scaled_height:
                height_pos = self.height_iogroup_pos[
                    group_idx
                ] = self.config.FONT_SIZE
        # CPU & SYSCALL #
        else:
            # SYSCALL #
            if segment.state == "SYSCALL":
                prefix = "[S] "
            else:
                prefix = ""

            # initialize group data #
            self.last_group_segment.setdefault(group_idx, None)
            self.last_group_time.setdefault(group_idx, x0)

            # set text attributes for same task #
            if (
                self.last_group_segment[group_idx] == segment.text
                and x0 - self.last_group_time[group_idx] < self.config.TICKS
            ):
                segment_label = durationstr
                color = "rgb(50,50,50)"
                font_size = self.config.FONT_SIZE - 1
            # set text attributes for new task #
            else:
                segment_label = "%s%s %s" % (prefix, segment.text, durationstr)
                color = "rgb(255,0,0)"
                font_size = self.config.FONT_SIZE - 0.5

            # update group info #
            self.last_group_segment[group_idx] = segment.text
            self.last_group_time[group_idx] = x0
            self.height_group_pos.setdefault(group_idx, 0)

            # set text position #
            scaled_pos = self.scaled_height * 0.15
            self.height_group_pos[group_idx] += self.config.FONT_SIZE
            height_pos = self.height_group_pos[group_idx]
            if height_pos + scaled_pos * 2.5 >= self.scaled_height:
                height_pos = self.height_group_pos[group_idx] = 0

        # set text position #
        xpos = x0
        ypos = y0 + scaled_pos + height_pos

        # draw text #
        dwg.add(
            dwg.text(
                segment_label,
                (xpos, ypos),
                fill=color,
                stroke="none",
                font_size=font_size,
                font_weight="normal",
                transform="rotate(0,%s,%s)" % (xpos, ypos),
            )
        )

    def draw(self, dwg, start=0, annotation=None, yval=None):
        self._draw_background(dwg)
        self._draw_grid(dwg)
        self._draw_group_axis(dwg, yval)
        self._draw_time_axis(dwg, start, annotation=annotation)
        self._draw_segments(dwg, start)
        self._draw_group_stat(dwg)

    @staticmethod
    def load(
        fileName=None,
        data=None,
        config=None,
        tasks=None,
        begin=0,
        end=0,
        duration=0,
        durationMin=0,
    ):
        if fileName:
            with open(fileName) as json_file:
                # get json object #
                json = SysMgr.getPkg("json")

                data = json.load(json_file)
        elif not data:
            SysMgr.printErr("no path or data for timeline input")
            sys.exit(-1)

        # get default timeunit #
        time_unit = ""
        if "time_unit" in data:
            time_unit = data["time_unit"]
            time_unit = time_unit.lower()

        # get title #
        if "title" in data:
            title = data["title"]
        else:
            title = ""

        # get font size #
        if "names" in data and data["names"]:
            if tasks:
                tasks.update(data["names"])
            else:
                tasks = data["names"]

        # get font size #
        if "font_size" in data:
            fontsize = data["font_size"]
        else:
            fontsize = None

        # get configured timeunit #
        time_factor = 1.0
        if hasattr(config, "TIMEUNIT") and config.TIMEUNIT:
            new_time_unit = config.TIMEUNIT.lower()

            if time_unit == new_time_unit:
                pass
            elif time_unit == "us":
                if new_time_unit == "ms":
                    time_factor = 1 / 1000.0
                elif new_time_unit == "sec":
                    time_factor = 1 / 1000000.0
                else:
                    SysMgr.printErr(
                        "no support '%s' unit for timeline" % new_time_unit
                    )
            elif time_unit == "ms":
                if new_time_unit == "us":
                    time_factor = 1000.0
                elif new_time_unit == "sec":
                    time_factor = 1 / 1000.0
                else:
                    SysMgr.printErr(
                        "no support '%s' unit for timeline" % new_time_unit
                    )
            else:
                SysMgr.printErr(
                    "no support '%s' unit for timeline" % new_time_unit
                )

            time_unit = new_time_unit
            config.TIMEFACTOR = time_factor

        # print time unit #
        if time_unit:
            SysMgr.printInfo("apply '%s' in time unit" % time_unit)

        # load segments #
        segments = Timeline._load_segments(
            data, time_factor, begin, end, duration, durationMin
        )
        if not segments:
            return None

        return Timeline(title, segments, time_unit, fontsize, config, tasks)

    @staticmethod
    def _load_segments(
        data, time_factor=1, begin=0, end=0, duration=0, durationMin=0
    ):
        segments = []
        for segment_data in sorted(
            data["segments"], key=lambda e: e["time_start"]
        ):
            # convert group #
            group = str(segment_data["group"])
            group_num = long(group)
            if group in Timeline.conv_table:
                group = Timeline.conv_table[group]

            # apply core filter ;) #
            if SysMgr.perCoreDrawList:
                if not group_num in SysMgr.perCoreDrawList:
                    continue

            # verify time #
            if segment_data["time_start"] > segment_data["time_end"]:
                SysMgr.printWarn(
                    "time_start is bigger than time_end for %s" % segment_data
                )
                continue

            # apply common duration #
            time_start = long(segment_data["time_start"] * time_factor)
            if duration > 0:
                time_end = time_start + duration
            else:
                time_end = long(segment_data["time_end"] * time_factor)

            # apply minimum duration #
            if durationMin > 0:
                diff = time_end - time_start
                if durationMin > diff:
                    time_end = time_start + durationMin

            # check time #
            if 0 < end < time_end:
                time_end = end
                if time_start > end:
                    continue
            if 0 < begin < time_start:
                if time_end < begin:
                    continue
                time_start = begin

            # add segment #
            segments.append(
                Timeline.Segment(group, time_start, time_end, segment_data)
            )

        return segments


class Ext4Analyzer(object):
    """Analyzer for ext4"""

    FILE_TYPE = {
        0: "unkn",
        1: "file",
        2: "dir",
        3: "chr",
        4: "blk",
        5: "fifo",
        6: "sock",
        7: "sym",
    }

    def __init__(self, path, verb=False):
        # get ctypes object #
        SysMgr.importPkgItems("ctypes")

        # init variables #
        self.volume = None
        self.inodeList = {}
        self.failDirList = {}

        # start loading classes #
        # refer to https://github.com/cubinator/ext4 #
        class Ext4Error(Exception):
            """
            Base class for all custom errors
            """

            pass

        class BlockMapError(Ext4Error):
            # a requested file_block is not mapped to disk
            pass

        class EndOfStreamError(Ext4Error):
            # BlockReader reads beyond the end of the volume's underlying stream
            pass

        class MagicError(Ext4Error):
            # a structures magic value is wrong and ignoreMagic is False
            pass

        class ext4_struct(LittleEndianStructure):
            """
            Simplifies access to *_lo and *_hi fields
            """

            def __getattr__(self, name):
                """
                Enables reading *_lo and *_hi fields together.
                """
                try:
                    # Combining *_lo and *_hi fields
                    lo_field = LittleEndianStructure.__getattribute__(
                        type(self), name + "_lo"
                    )
                    size = lo_field.size

                    lo = lo_field.__get__(self)
                    hi = LittleEndianStructure.__getattribute__(
                        self, name + "_hi"
                    )

                    return (hi << (8 * size)) | lo
                except AttributeError:
                    return LittleEndianStructure.__getattribute__(self, name)

            def __setattr__(self, name, value):
                """
                Enables setting *_lo and *_hi fields together.
                """
                try:
                    # Combining *_lo and *_hi fields
                    lo_field = (
                        lo_field
                    ) = LittleEndianStructure.__getattribute__(
                        type(self), name + "_lo"
                    )
                    size = lo_field.size

                    lo_field.__set__(self, value & ((1 << (8 * size)) - 1))
                    LittleEndianStructure.__setattr__(
                        self, name + "_hi", value >> (8 * size)
                    )
                except AttributeError:
                    LittleEndianStructure.__setattr__(self, name, value)

        class ext4_dir_entry_2(ext4_struct):
            _fields_ = [
                ("inode", c_uint),  # 0x0
                ("rec_len", c_ushort),  # 0x4
                ("name_len", c_ubyte),  # 0x6
                ("file_type", c_ubyte)  # 0x7
                # Variable length field "name" missing at 0x8
            ]

            @classmethod
            def _from_buffer_copy(cls, raw, offset=0, platform64=True):
                # pylint: disable=no-self-argument
                struct = ext4_dir_entry_2.from_buffer_copy(raw, offset)
                struct.name = raw[
                    offset + 0x8 : offset + 0x8 + struct.name_len
                ]
                return struct

        class ext4_extent(ext4_struct):
            _fields_ = [
                ("ee_block", c_uint),  # 0x0000
                ("ee_len", c_ushort),  # 0x0004
                ("ee_start_hi", c_ushort),  # 0x0006
                ("ee_start_lo", c_uint),  # 0x0008
            ]

        class ext4_extent_header(ext4_struct):
            _fields_ = [
                ("eh_magic", c_ushort),  # 0x0000, Must be 0xF30A
                ("eh_entries", c_ushort),  # 0x0002
                ("eh_max", c_ushort),  # 0x0004
                ("eh_depth", c_ushort),  # 0x0006
                ("eh_generation", c_uint),  # 0x0008
            ]

        class ext4_extent_idx(ext4_struct):
            _fields_ = [
                ("ei_block", c_uint),  # 0x0000
                ("ei_leaf_lo", c_uint),  # 0x0004
                ("ei_leaf_hi", c_ushort),  # 0x0008
                ("ei_unused", c_ushort),  # 0x000A
            ]

        class ext4_group_descriptor(ext4_struct):
            _fields_ = [
                ("bg_block_bitmap_lo", c_uint),  # 0x0000
                ("bg_inode_bitmap_lo", c_uint),  # 0x0004
                ("bg_inode_table_lo", c_uint),  # 0x0008
                ("bg_free_blocks_count_lo", c_ushort),  # 0x000C
                ("bg_free_inodes_count_lo", c_ushort),  # 0x000E
                ("bg_used_dirs_count_lo", c_ushort),  # 0x0010
                ("bg_flags", c_ushort),  # 0x0012
                ("bg_exclude_bitmap_lo", c_uint),  # 0x0014
                ("bg_block_bitmap_csum_lo", c_ushort),  # 0x0018
                ("bg_inode_bitmap_csum_lo", c_ushort),  # 0x001A
                ("bg_itable_unused_lo", c_ushort),  # 0x001C
                ("bg_checksum", c_ushort),  # 0x001E
                # 64-bit fields
                ("bg_block_bitmap_hi", c_uint),  # 0x0020
                ("bg_inode_bitmap_hi", c_uint),  # 0x0024
                ("bg_inode_table_hi", c_uint),  # 0x0028
                ("bg_free_blocks_count_hi", c_ushort),  # 0x002C
                ("bg_free_inodes_count_hi", c_ushort),  # 0x002E
                ("bg_used_dirs_count_hi", c_ushort),  # 0x0030
                ("bg_itable_unused_hi", c_ushort),  # 0x0032
                ("bg_exclude_bitmap_hi", c_uint),  # 0x0034
                ("bg_block_bitmap_csum_hi", c_ushort),  # 0x0038
                ("bg_inode_bitmap_csum_hi", c_ushort),  # 0x003A
                ("bg_reserved", c_uint),  # 0x003C
            ]

            @classmethod
            def _from_buffer_copy(cls, raw, offset=0, platform64=True):
                # pylint: disable=no-self-argument
                struct = ext4_group_descriptor.from_buffer_copy(raw, offset)

                if not platform64:
                    struct.bg_block_bitmap_hi = 0
                    struct.bg_inode_bitmap_hi = 0
                    struct.bg_inode_table_hi = 0
                    struct.bg_free_blocks_count_hi = 0
                    struct.bg_free_inodes_count_hi = 0
                    struct.bg_used_dirs_count_hi = 0
                    struct.bg_itable_unused_hi = 0
                    struct.bg_exclude_bitmap_hi = 0
                    struct.bg_block_bitmap_csum_hi = 0
                    struct.bg_inode_bitmap_csum_hi = 0
                    struct.bg_reserved = 0

                return struct

        class ext4_inode(ext4_struct):
            """
            Every field passing 128 bytes is "additional data",
            whose size is specified by i_extra_isize.
            """

            EXT2_GOOD_OLD_INODE_SIZE = 128

            # i_mode
            S_IXOTH = 0x1  # Others can execute
            S_IWOTH = 0x2  # Others can write
            S_IROTH = 0x4  # Others can read
            S_IXGRP = 0x8  # Group can execute
            S_IWGRP = 0x10  # Group can write
            S_IRGRP = 0x20  # Group can read
            S_IXUSR = 0x40  # Owner can execute
            S_IWUSR = 0x80  # Owner can write
            S_IRUSR = 0x100  # Owner can read
            S_ISVTX = 0x200  # Sticky bit (only owner can delete)
            S_ISGID = 0x400  # Set GID
            S_ISUID = 0x800  # Set UID
            S_IFIFO = 0x1000  # FIFO device (named pipe)
            S_IFCHR = 0x2000  # Character device
            S_IFDIR = 0x4000  # Directory
            S_IFBLK = 0x6000  # Block device
            S_IFREG = 0x8000  # Regular file
            S_IFLNK = 0xA000  # Symbolic link
            S_IFSOCK = 0xC000  # Socket

            # i_flags
            EXT4_INDEX_FL = 0x1000  # Uses hash trees
            EXT4_EXTENTS_FL = 0x80000  # Uses extents
            EXT4_EA_INODE_FL = 0x200000  # Inode stores large xattr
            EXT4_INLINE_DATA_FL = 0x10000000  # Has inline data

            _fields_ = [
                ("i_mode", c_ushort),  # 0x0000
                ("i_uid_lo", c_ushort),  # 0x0002
                ("i_size_lo", c_uint),  # 0x0004
                ("i_atime", c_uint),  # 0x0008
                ("i_ctime", c_uint),  # 0x000C
                ("i_mtime", c_uint),  # 0x0010
                ("i_dtime", c_uint),  # 0x0014
                ("i_gid_lo", c_ushort),  # 0x0018
                ("i_links_count", c_ushort),  # 0x001A
                ("i_blocks_lo", c_uint),  # 0x001C
                ("i_flags", c_uint),  # 0x0020
                ("osd1", c_uint),  # 0x0024
                ("i_block", c_uint * 15),  # 0x0028
                ("i_generation", c_uint),  # 0x0064
                ("i_file_acl_lo", c_uint),  # 0x0068
                ("i_size_hi", c_uint),  # 0x006C
                ("i_obso_faddr", c_uint),  # 0x0070
                ("i_osd2_blocks_high", c_ushort),  # 0x0074
                ("i_file_acl_hi", c_ushort),  # 0x0076
                ("i_uid_hi", c_ushort),  # 0x0078
                ("i_gid_hi", c_ushort),  # 0x007A
                ("i_osd2_checksum_lo", c_ushort),  # 0x007C
                ("i_osd2_reserved", c_ushort),  # 0x007E
                ("i_extra_isize", c_ushort),  # 0x0080
                ("i_checksum_hi", c_ushort),  # 0x0082
                ("i_ctime_extra", c_uint),  # 0x0084
                ("i_mtime_extra", c_uint),  # 0x0088
                ("i_atime_extra", c_uint),  # 0x008C
                ("i_crtime", c_uint),  # 0x0090
                ("i_crtime_extra", c_uint),  # 0x0094
                ("i_version_hi", c_uint),  # 0x0098
                ("i_projid", c_uint),  # 0x009C
            ]

        class ext4_superblock(ext4_struct):
            # Default value for s_desc_size, if INCOMPAT_64BIT is not set
            EXT2_MIN_DESC_SIZE = 0x20
            # Default value for s_desc_size, if INCOMPAT_64BIT is set
            EXT2_MIN_DESC_SIZE_64BIT = 0x40

            # s_feature_incompat
            # Uses 64-bit features
            INCOMPAT_64BIT = 0x80
            # Directory entries record file type (instead of inode flags)
            INCOMPAT_FILETYPE = 0x2

            _fields_ = [
                ("s_inodes_count", c_uint),  # 0x0000
                ("s_blocks_count_lo", c_uint),  # 0x0004
                ("s_r_blocks_count_lo", c_uint),  # 0x0008
                ("s_free_blocks_count_lo", c_uint),  # 0x000C
                ("s_free_inodes_count", c_uint),  # 0x0010
                ("s_first_data_block", c_uint),  # 0x0014
                ("s_log_block_size", c_uint),  # 0x0018
                ("s_log_cluster_size", c_uint),  # 0x001C
                ("s_blocks_per_group", c_uint),  # 0x0020
                ("s_clusters_per_group", c_uint),  # 0x0024
                ("s_inodes_per_group", c_uint),  # 0x0028
                ("s_mtime", c_uint),  # 0x002C
                ("s_wtime", c_uint),  # 0x0030
                ("s_mnt_count", c_ushort),  # 0x0034
                ("s_max_mnt_count", c_ushort),  # 0x0036
                ("s_magic", c_ushort),  # 0x0038
                ("s_state", c_ushort),  # 0x003A
                ("s_errors", c_ushort),  # 0x003C
                ("s_minor_rev_level", c_ushort),  # 0x003E
                ("s_lastcheck", c_uint),  # 0x0040
                ("s_checkinterval", c_uint),  # 0x0044
                ("s_creator_os", c_uint),  # 0x0048
                ("s_rev_level", c_uint),  # 0x004C
                ("s_def_resuid", c_ushort),  # 0x0050
                ("s_def_resgid", c_ushort),  # 0x0052
                ("s_first_ino", c_uint),  # 0x0054
                ("s_inode_size", c_ushort),  # 0x0058
                ("s_block_group_nr", c_ushort),  # 0x005A
                ("s_feature_compat", c_uint),  # 0x005C
                ("s_feature_incompat", c_uint),  # 0x0060
                ("s_feature_ro_compat", c_uint),  # 0x0064
                ("s_uuid", c_ubyte * 16),  # 0x0068
                ("s_volume_name", c_char * 16),  # 0x0078
                ("s_last_mounted", c_char * 64),  # 0x0088
                ("s_algorithm_usage_bitmap", c_uint),  # 0x00C8
                ("s_prealloc_blocks", c_ubyte),  # 0x00CC
                ("s_prealloc_dir_blocks", c_ubyte),  # 0x00CD
                ("s_reserved_gdt_blocks", c_ushort),  # 0x00CE
                ("s_journal_uuid", c_ubyte * 16),  # 0x00D0
                ("s_journal_inum", c_uint),  # 0x00E0
                ("s_journal_dev", c_uint),  # 0x00E4
                ("s_last_orphan", c_uint),  # 0x00E8
                ("s_hash_seed", c_uint * 4),  # 0x00EC
                ("s_def_hash_version", c_ubyte),  # 0x00FC
                ("s_jnl_backup_type", c_ubyte),  # 0x00FD
                ("s_desc_size", c_ushort),  # 0x00FE
                ("s_default_mount_opts", c_uint),  # 0x0100
                ("s_first_meta_bg", c_uint),  # 0x0104
                ("s_mkfs_time", c_uint),  # 0x0108
                ("s_jnl_blocks", c_uint * 17),  # 0x010C
                # 64-bit fields
                ("s_blocks_count_hi", c_uint),  # 0x0150
                ("s_r_blocks_count_hi", c_uint),  # 0x0154
                ("s_free_blocks_count_hi", c_uint),  # 0x0158
                ("s_min_extra_isize", c_ushort),  # 0x015C
                ("s_want_extra_isize", c_ushort),  # 0x015E
                ("s_flags", c_uint),  # 0x0160
                ("s_raid_stride", c_ushort),  # 0x0164
                ("s_mmp_interval", c_ushort),  # 0x0166
                ("s_mmp_block", c_ulonglong),  # 0x0168
                ("s_raid_stripe_width", c_uint),  # 0x0170
                ("s_log_groups_per_flex", c_ubyte),  # 0x0174
                ("s_checksum_type", c_ubyte),  # 0x0175
                ("s_reserved_pad", c_ushort),  # 0x0176
                ("s_kbytes_written", c_ulonglong),  # 0x0178
                ("s_snapshot_inum", c_uint),  # 0x0180
                ("s_snapshot_id", c_uint),  # 0x0184
                ("s_snapshot_r_blocks_count", c_ulonglong),  # 0x0188
                ("s_snapshot_list", c_uint),  # 0x0190
                ("s_error_count", c_uint),  # 0x0194
                ("s_first_error_time", c_uint),  # 0x0198
                ("s_first_error_ino", c_uint),  # 0x019C
                ("s_first_error_block", c_ulonglong),  # 0x01A0
                ("s_first_error_func", c_ubyte * 32),  # 0x01A8
                ("s_first_error_line", c_uint),  # 0x01C8
                ("s_last_error_time", c_uint),  # 0x01CC
                ("s_last_error_ino", c_uint),  # 0x01D0
                ("s_last_error_line", c_uint),  # 0x01D4
                ("s_last_error_block", c_ulonglong),  # 0x01D8
                ("s_last_error_func", c_ubyte * 32),  # 0x01E0
                ("s_mount_opts", c_ubyte * 64),  # 0x0200
                ("s_usr_quota_inum", c_uint),  # 0x0240
                ("s_grp_quota_inum", c_uint),  # 0x0244
                ("s_overhead_blocks", c_uint),  # 0x0248
                ("s_backup_bgs", c_uint * 2),  # 0x024C
                ("s_encrypt_algos", c_ubyte * 4),  # 0x0254
                ("s_encrypt_pw_salt", c_ubyte * 16),  # 0x0258
                ("s_lpf_ino", c_uint),  # 0x0268
                ("s_prj_quota_inum", c_uint),  # 0x026C
                ("s_checksum_seed", c_uint),  # 0x0270
                ("s_reserved", c_uint * 98),  # 0x0274
                ("s_checksum", c_uint),  # 0x03FC
            ]

            @classmethod
            def _from_buffer_copy(cls, raw, platform64=True):
                # pylint: disable=no-self-argument
                struct = ext4_superblock.from_buffer_copy(raw)

                if not platform64:
                    struct.s_blocks_count_hi = 0
                    struct.s_r_blocks_count_hi = 0
                    struct.s_free_blocks_count_hi = 0
                    struct.s_min_extra_isize = 0
                    struct.s_want_extra_isize = 0
                    struct.s_flags = 0
                    struct.s_raid_stride = 0
                    struct.s_mmp_interval = 0
                    struct.s_mmp_block = 0
                    struct.s_raid_stripe_width = 0
                    struct.s_log_groups_per_flex = 0
                    struct.s_checksum_type = 0
                    struct.s_reserved_pad = 0
                    struct.s_kbytes_written = 0
                    struct.s_snapshot_inum = 0
                    struct.s_snapshot_id = 0
                    struct.s_snapshot_r_blocks_count = 0
                    struct.s_snapshot_list = 0
                    struct.s_error_count = 0
                    struct.s_first_error_time = 0
                    struct.s_first_error_ino = 0
                    struct.s_first_error_block = 0
                    struct.s_first_error_func = 0
                    struct.s_first_error_line = 0
                    struct.s_last_error_time = 0
                    struct.s_last_error_ino = 0
                    struct.s_last_error_line = 0
                    struct.s_last_error_block = 0
                    struct.s_last_error_func = 0
                    struct.s_mount_opts = 0
                    struct.s_usr_quota_inum = 0
                    struct.s_grp_quota_inum = 0
                    struct.s_overhead_blocks = 0
                    struct.s_backup_bgs = 0
                    struct.s_encrypt_algos = 0
                    struct.s_encrypt_pw_salt = 0
                    struct.s_lpf_ino = 0
                    struct.s_prj_quota_inum = 0
                    struct.s_checksum_seed = 0
                    struct.s_reserved = 0
                    struct.s_checksum = 0

                if struct.s_desc_size != 0:
                    return struct

                if (
                    struct.s_feature_incompat & ext4_superblock.INCOMPAT_64BIT
                ) == 0:
                    struct.s_desc_size = ext4_superblock.EXT2_MIN_DESC_SIZE
                else:
                    struct.s_desc_size = (
                        ext4_superblock.EXT2_MIN_DESC_SIZE_64BIT
                    )

                return struct

        class ext4_xattr_entry(ext4_struct):
            _fields_ = [
                ("e_name_len", c_ubyte),  # 0x00
                ("e_name_index", c_ubyte),  # 0x01
                ("e_value_offs", c_ushort),  # 0x02
                ("e_value_inum", c_uint),  # 0x04
                ("e_value_size", c_uint),  # 0x08
                ("e_hash", c_uint)  # 0x0C
                # Variable length field "e_name" missing at 0x10
            ]

            @classmethod
            def _from_buffer_copy(cls, raw, offset=0, platform64=True):
                # pylint: disable=no-self-argument
                struct = ext4_xattr_entry.from_buffer_copy(raw, offset)
                struct.e_name = raw[
                    offset + 0x10 : offset + 0x10 + struct.e_name_len
                ]
                return struct

            @property
            def _size(self):
                # 4-byte alignment
                return 4 * ((sizeof(type(self)) + self.e_name_len + 3) // 4)

        class ext4_xattr_header(ext4_struct):
            _fields_ = [
                ("h_magic", c_uint),  # 0x0, Must be 0xEA020000
                ("h_refcount", c_uint),  # 0x4
                ("h_blocks", c_uint),  # 0x8
                ("h_hash", c_uint),  # 0xC
                ("h_checksum", c_uint),  # 0x10
                ("h_reserved", c_uint * 3),  # 0x14
            ]

        class ext4_xattr_ibody_header(ext4_struct):
            _fields_ = [("h_magic", c_uint)]  # 0x0, Must be 0xEA020000

        class InodeType:
            UNKNOWN = 0x0  # Unknown file type
            FILE = 0x1  # Regular file
            DIRECTORY = 0x2  # Directory
            CHARACTER_DEVICE = 0x3  # Character device
            BLOCK_DEVICE = 0x4  # Block device
            FIFO = 0x5  # FIFO
            SOCKET = 0x6  # Socket
            SYMBOLIC_LINK = 0x7  # Symbolic link
            CHECKSUM = 0xDE  # Checksum entry

        class MappingEntry:
            """
            Helper class: This class maps blkCnt file blocks
            indexed by fileBlkIdx to the associated disk blocks indexed
            by diskBlkIdx.
            """

            def __init__(self, fileBlkIdx, diskBlkIdx, blkCnt=1):
                """
                Initialize a MappingEntry instance with given fileBlkIdx,
                diskBlkIdx and blkCnt.
                """
                self.fileBlkIdx = fileBlkIdx
                self.diskBlkIdx = diskBlkIdx
                self.blkCnt = blkCnt

            def __iter__(self):
                """
                Can be used to convert an MappingEntry into a tuple
                (fileBlkIdx, diskBlkIdx, blkCnt).
                """
                yield self.fileBlkIdx
                yield self.diskBlkIdx
                yield self.blkCnt

            def __repr__(self):
                return "%s(%s, %s, %s)" % (
                    type(self).__name__,
                    self.fileBlkIdx,
                    self.diskBlkIdx,
                    self.blkCnt,
                )

            def copy(self):
                return MappingEntry(
                    self.fileBlkIdx, self.diskBlkIdx, self.blkCnt
                )

            def create_mapping(*entries):
                # pylint: disable=no-method-argument,no-self-argument
                """
                Converts a list of 2-tuples
                (diskBlkIdx, blkCnt) into a list of MappingEntry instances
                """
                fileBlkIdx = 0
                result = [None] * len(entries)

                for i, entry in enumerate(entries):
                    diskBlkIdx, blkCnt = entry
                    result[i] = MappingEntry(fileBlkIdx, diskBlkIdx, blkCnt)
                    fileBlkIdx += blkCnt

                return result

            @classmethod
            def optimize(cls, entries):
                # pylint: disable=no-self-argument
                """
                Sorts and stiches together a list of MappingEntry instances
                """
                entries = list(entries)
                entries.sort(key=lambda entry: entry.fileBlkIdx)

                idx = 0
                while idx < len(entries):
                    while (
                        (idx + 1 < len(entries))
                        and (
                            entries[idx].fileBlkIdx + entries[idx].blkCnt
                            == entries[idx + 1].fileBlkIdx
                        )
                        and (
                            entries[idx].diskBlkIdx + entries[idx].blkCnt
                            == entries[idx + 1].diskBlkIdx
                        )
                    ):
                        tmp = entries.pop(idx + 1)
                        entries[idx].blkCnt += tmp.blkCnt

                    idx += 1

        class Volume:
            """
            Provides functionality for reading ext4 volumes
            """

            ROOT_INODE = 2

            def __init__(
                self, stream, offset=0, ignoreFlag=False, ignoreMagic=False
            ):
                """
                Initializes a new ext4 reader at a given offset in stream.
                If ignoreMagic is True, no exception will be thrown,
                when a structure with wrong magic number is found.
                Analogously passing True to ignoreFlag suppresses Exception
                caused by wrong flags.
                """
                self.ignoreFlag = ignoreFlag
                self.ignoreMagic = ignoreMagic
                self.offset = offset
                # Initial value needed for Volume.read_struct
                self.platform64 = True
                self.stream = stream

                # Superblock
                self.superblock = self.read_struct(ext4_superblock, 0x400)
                self.platform64 = (
                    self.superblock.s_feature_incompat
                    & ext4_superblock.INCOMPAT_64BIT
                ) != 0

                if not ignoreMagic and self.superblock.s_magic != 0xEF53:
                    raise MagicError(
                        (
                            "Invalid magic value in superblock: "
                            "0x%04X (expected 0xEF53)"
                        )
                        % self.superblock.s_magic
                    )

                # Group descriptors
                self.group_descriptors = [None] * (
                    self.superblock.s_inodes_count
                    // self.superblock.s_inodes_per_group
                )

                # First block after superblock
                group_desc_table_offset = (
                    0x400 // self.block_size + 1
                ) * self.block_size
                for group_desc_idx in xrange(len(self.group_descriptors)):
                    group_desc_offset = (
                        group_desc_table_offset
                        + group_desc_idx * self.superblock.s_desc_size
                    )
                    self.group_descriptors[group_desc_idx] = self.read_struct(
                        ext4_group_descriptor, group_desc_offset
                    )

            def __repr__(self):
                return "%s(volume_name = %s, uuid = %s, last_mounted = %s)" % (
                    type(self).__name__,
                    self.superblock.s_volume_name,
                    self.uuid,
                    self.superblock.s_last_mounted,
                )

            @property
            def block_size(self):
                """
                Returns the volume's block size in bytes.
                """
                return 1 << (10 + self.superblock.s_log_block_size)

            def get_inode(self, inode_idx):
                """
                Returns an Inode instance representing the inode specified
                by its index inode_idx.
                """
                group_idx, inode_table_entry_idx = self.get_inode_group(
                    inode_idx
                )

                inode_table_offset = (
                    self.group_descriptors[group_idx].bg_inode_table
                    * self.block_size
                )
                inode_offset = (
                    inode_table_offset
                    + inode_table_entry_idx * self.superblock.s_inode_size
                )

                return Inode(self, inode_offset, inode_idx)

            def get_inode_group(self, inode_idx):
                """
                Returns a tuple (group_idx, inode_table_entry_idx)
                """
                group_idx = (
                    inode_idx - 1
                ) // self.superblock.s_inodes_per_group
                inode_table_entry_idx = (
                    inode_idx - 1
                ) % self.superblock.s_inodes_per_group
                return (group_idx, inode_table_entry_idx)

            def read(self, offset, byte_len):
                """
                Returns byte_len bytes at offset within this volume.
                """
                if self.offset + offset != self.stream.tell():
                    self.stream.seek(self.offset + offset, os.SEEK_SET)

                return self.stream.read(byte_len)

            def read_struct(self, structure, offset, platform64=None):
                """
                Interprets the bytes at offset as structure and returns
                the interpretd instance
                """
                raw = self.read(offset, sizeof(structure))

                if hasattr(structure, "_from_buffer_copy"):
                    return structure._from_buffer_copy(
                        raw,
                        platform64=platform64
                        if platform64 != None
                        else self.platform64,
                    )
                else:
                    return structure.from_buffer_copy(raw)

            @property
            def root(self):
                """
                Returns the volume's root inode
                """
                return self.get_inode(Volume.ROOT_INODE)

            @property
            def uuid(self):
                """
                Returns the volume's UUID in the format
                XXXXXXXX-XXXX-XXXX-XXXX-XXXXXXXXXXXX.
                """
                uuid = self.superblock.s_uuid
                uuid = [uuid[:4], uuid[4:6], uuid[6:8], uuid[8:10], uuid[10:]]
                return "-".join(
                    "".join("%02X" % c for c in part) for part in uuid
                )

        class Inode:
            """
            Provides functionality for parsing inodes and accessing raw data
            """

            def __init__(self, volume, offset, inode_idx):
                """
                Initializes a new inode parser at the specified offset
                within the specified volume. file_type is the file type
                of the inode as given by the directory entry
                referring to this inode.
                """
                self.inode_idx = inode_idx
                self.offset = offset
                self.volume = volume
                self.inode = volume.read_struct(ext4_inode, offset)

            def __len__(self):
                """
                Returns the length in bytes of the content
                referenced by this inode.
                """
                return int(self.inode.i_size)

            def __repr__(self):
                if self.inode_idx != None:
                    return (
                        "%s(inode_idx = %s, offset = 0x%X, volume_uuid = %s"
                    ) % (
                        type(self).__name__,
                        self.inode_idx,
                        self.offset,
                        self.volume.uuid,
                    )
                else:
                    return "%s(offset = 0x%X, volume_uuid = %s)" % (
                        type(self).__name__,
                        self.offset,
                        self.volume.uuid,
                    )

            def _parse_xattrs(self, raw_data, offset, prefix_override={}):
                """
                Generator: Parses raw_data (bytes) as ext4_xattr_entry
                structures and their referenced xattr values and yields
                tuples (xattr_name, xattr_value) where xattr_name (str)
                is the attribute name including its prefix and
                xattr_value (bytes) is the raw attribute value.
                raw_data must start with the first ext4_xattr_entry
                structure and offset specifies the offset to the "block start"
                for ext4_xattr_entry.e_value_offs.
                prefix_overrides allows specifying attributes apart from
                the default prefixes. The default prefix dictionary is
                updated with prefix_overrides.
                """
                prefixes = {
                    0: "",
                    1: "user.",
                    2: "system.posix_acl_access",
                    3: "system.posix_acl_default",
                    4: "trusted.",
                    6: "security.",
                    7: "system.",
                    8: "system.richacl",
                }
                prefixes.update(prefixes)

                # Iterator over ext4_xattr_entry structures
                i = 0
                while i < len(raw_data):
                    xattr_entry = ext4_xattr_entry._from_buffer_copy(
                        raw_data, i, platform64=self.volume.platform64
                    )

                    if (
                        xattr_entry.e_name_len
                        | xattr_entry.e_name_index
                        | xattr_entry.e_value_offs
                        | xattr_entry.e_value_inum
                    ) == 0:
                        # End of ext4_xattr_entry list
                        break

                    if not xattr_entry.e_name_index in prefixes:
                        raise Ext4Error(
                            "Unknown attribute prefix %d in inode %d"
                            % (xattr_entry.e_name_idx, self.inode_idx)
                        )

                    xattr_name = prefixes[
                        xattr_entry.e_name_index
                    ] + xattr_entry.e_name.decode("iso-8859-2")

                    if xattr_entry.e_value_inum != 0:
                        # external xattr
                        xattr_inode = self.volume.get_inode(
                            xattr_entry.e_value_inum
                        )

                        if (
                            not self.volume.ignoreFlag
                            and (
                                xattr_inode.inode.i_flags
                                & ext4_inode.EXT4_EA_INODE_FL
                            )
                            != 0
                        ):
                            raise Ext4Error(
                                (
                                    "Inode %d associated with the extended "
                                    "attribute %s of inode %d is not marked as "
                                    "large extended attribute value."
                                )
                                % (
                                    xattr_inode.inode_idx,
                                    xattr_name,
                                    self.inode_idx,
                                )
                            )

                        # TODO Use xattr_entry.e_value_size or xattr_inode.inode.i_size?
                        xattr_value = xattr_inode.open_read().read()
                    else:
                        # internal xattr
                        start = xattr_entry.e_value_offs + offset
                        end = (
                            xattr_entry.e_value_offs
                            + offset
                            + xattr_entry.e_value_size
                        )
                        xattr_value = raw_data[start:end]

                    yield (xattr_name, xattr_value)

                    i += xattr_entry._size

            def get_inode(self, *relative_path):
                """
                Returns the inode specified by the path relative_path
                (list of entry names) relative to this inode. "." and ".."
                usually are supported too, however in special cases
                (e.g. manually crafted volumes) they might not be supported
                due to them being real on-disk directory entries that
                might be missing or pointing somewhere else.
                decode_name is directly passed to open_dir.
                NOTE: Whitespaces will not be trimmed off the path's parts
                and "\\0" and "\\0\\0" as well as b"\\0" and b"\\0\\0" are
                seen as different names
                (unless decode_name actually trims the name).
                NOTE: Along the path file_type != FILETYPE_DIR will be
                ignored, however i_flags will not be ignored.
                """
                if not self.is_dir:
                    raise Ext4Error(
                        "Inode %d is not a directory" % self.inode_idx
                    )

                current_inode = self
                decode_name = None

                for i, part in enumerate(relative_path):
                    if not self.volume.ignoreFlag and not current_inode.is_dir:
                        current_path = "/".join(relative_path[:i])
                        raise Ext4Error(
                            "%s (Inode %d) is not a directory."
                            % (current_path, self.inode_idx)
                        )

                    file_name, inode_idx, file_type = next(
                        filter(
                            lambda entry: entry[0] == part,
                            current_inode.open_dir(decode_name),
                        ),
                        (None, None, None),
                    )

                    if inode_idx is None:
                        current_path = "/".join(relative_path[:i])
                        raise FileNotFoundError(
                            "%s not found in %s (Inode %d)"
                            % (part, current_path, current_inode.inode_idx)
                        )

                    current_inode = current_inode.volume.get_inode(inode_idx)

                return current_inode

            @property
            def is_dir(self):
                """
                Indicates whether the inode is marked as a directory.
                """
                return (self.inode.i_mode & ext4_inode.S_IFDIR) != 0

            @property
            def is_file(self):
                """
                Indicates whether the inode is marker as a regular file.
                """
                return (self.inode.i_mode & ext4_inode.S_IFREG) != 0

            @property
            def is_in_use(self):
                """
                Indicates whether the inode's associated bit
                in the inode bitmap is set.
                """
                group_idx, bitmap_bit = self.volume.get_inode_group(
                    self.inode_idx
                )

                inode_usage_bitmap_offset = (
                    self.volume.group_descriptors[group_idx].bg_inode_bitmap
                    * self.volume.block_size
                )
                inode_usage_byte = self.volume.read(
                    inode_usage_bitmap_offset + bitmap_bit // 8, 1
                )[0]

                return ((inode_usage_byte >> (7 - bitmap_bit % 8)) & 1) != 0

            @property
            def mode_str(self):
                """
                Returns the inode's permissions in form of a unix string
                (e.g. "-rwxrw-rw" or "drwxr-xr--").
                """
                special_flag = lambda letter, execute, special: {
                    (False, False): "-",
                    (False, True): letter.upper(),
                    (True, False): "x",
                    (True, True): letter.lower(),
                }[(execute, special)]

                try:
                    device_type = {
                        ext4_inode.S_IFIFO: "p",
                        ext4_inode.S_IFCHR: "c",
                        ext4_inode.S_IFDIR: "d",
                        ext4_inode.S_IFBLK: "b",
                        ext4_inode.S_IFREG: "-",
                        ext4_inode.S_IFLNK: "l",
                        ext4_inode.S_IFSOCK: "s",
                    }[self.inode.i_mode & 0xF000]
                except KeyError:
                    device_type = "?"

                return "".join(
                    [
                        device_type,
                        "r"
                        if (self.inode.i_mode & ext4_inode.S_IRUSR) != 0
                        else "-",
                        "w"
                        if (self.inode.i_mode & ext4_inode.S_IWUSR) != 0
                        else "-",
                        special_flag(
                            "s",
                            (self.inode.i_mode & ext4_inode.S_IXUSR) != 0,
                            (self.inode.i_mode & ext4_inode.S_ISUID) != 0,
                        ),
                        "r"
                        if (self.inode.i_mode & ext4_inode.S_IRGRP) != 0
                        else "-",
                        "w"
                        if (self.inode.i_mode & ext4_inode.S_IWGRP) != 0
                        else "-",
                        special_flag(
                            "s",
                            (self.inode.i_mode & ext4_inode.S_IXGRP) != 0,
                            (self.inode.i_mode & ext4_inode.S_ISGID) != 0,
                        ),
                        "r"
                        if (self.inode.i_mode & ext4_inode.S_IROTH) != 0
                        else "-",
                        "w"
                        if (self.inode.i_mode & ext4_inode.S_IWOTH) != 0
                        else "-",
                        special_flag(
                            "t",
                            (self.inode.i_mode & ext4_inode.S_IXOTH) != 0,
                            (self.inode.i_mode & ext4_inode.S_ISVTX) != 0,
                        ),
                    ]
                )

            def open_dir(self, decode_name=None, path=None):
                """
                Generator: Yields the directory entries as tuples
                (decode_name(name), inode, file_type) in their on-disk order,
                where name is the raw on-disk directory entry name (bytes).
                file_type is one of the Inode.IT_* constants. For
                special cases (e.g. invalid utf8 characters in entry names)
                you can try a different decoder
                (e.g. decode_name = lambda raw: raw).
                Default of decode_name = lambda raw: raw.decode("utf8")
                """
                # Parse args
                if decode_name is None:
                    # decode_name = lambda raw: raw.decode("utf8")
                    decode_name = lambda raw: raw.decode("latin-1")

                if not self.volume.ignoreFlag and not self.is_dir:
                    raise Ext4Error(
                        "Inode (%d) is not a directory (%s)"
                        % (self.inode_idx, path)
                    )

                # Hash trees are compatible with linear arrays
                if (self.inode.i_flags & ext4_inode.EXT4_INDEX_FL) != 0:
                    SysMgr.printWarn(
                        "hash trees are not implemented yet for %s" % path
                    )
                    return
                    # TODO: implement hash tree parser #
                    raise NotImplementedError(
                        "Hash trees are not implemented yet for %s" % path
                    )

                # Read raw directory content
                raw_data = self.open_read().read()
                offset = 0

                while offset < len(raw_data):
                    dirent = ext4_dir_entry_2._from_buffer_copy(
                        raw_data, offset, platform64=self.volume.platform64
                    )

                    if dirent.file_type != InodeType.CHECKSUM:
                        yield (
                            decode_name(dirent.name),
                            dirent.inode,
                            dirent.file_type,
                        )

                    offset += dirent.rec_len

            def open_read(self):
                """
                Returns an BlockReader instance for reading this inode's
                raw content.
                """
                if (self.inode.i_flags & ext4_inode.EXT4_EXTENTS_FL) != 0:
                    # Obtain mapping from extents
                    mapping = []  # List of MappingEntry instances

                    nodes = []
                    nodes.append(self.offset + ext4_inode.i_block.offset)

                    while nodes:
                        header_offset = nodes.pop(0)
                        header = self.volume.read_struct(
                            ext4_extent_header, header_offset
                        )

                        if (
                            not self.volume.ignoreMagic
                            and header.eh_magic != 0xF30A
                        ):
                            raise MagicError(
                                (
                                    "Invalid magic value in extent header at "
                                    "offset 0x%X of inode %d: 0x%04X "
                                    "(expected 0xF30A)"
                                )
                                % (
                                    header_offset,
                                    self.inode_idx,
                                    header.eh_magic,
                                )
                            )

                        if header.eh_depth != 0:
                            indices = self.volume.read_struct(
                                ext4_extent_idx * header.eh_entries,
                                header_offset + sizeof(ext4_extent_header),
                            )
                            for idx in indices:
                                nodes.append(
                                    idx.ei_leaf * self.volume.block_size
                                )
                        else:
                            extents = self.volume.read_struct(
                                ext4_extent * header.eh_entries,
                                header_offset + sizeof(ext4_extent_header),
                            )
                            for extent in extents:
                                mapping.append(
                                    MappingEntry(
                                        extent.ee_block,
                                        extent.ee_start,
                                        extent.ee_len,
                                    )
                                )

                    MappingEntry.optimize(mapping)
                    return BlockReader(self.volume, len(self), mapping)
                else:
                    # Inode uses inline data
                    i_block = self.volume.read(
                        self.offset + ext4_inode.i_block.offset,
                        ext4_inode.i_block.size,
                    )
                    return SysMgr.getPkg("io").BytesIO(
                        i_block[: self.inode.i_size]
                    )

            def xattrs(
                self,
                check_inline=True,
                check_block=True,
                force_inline=False,
                prefix_override={},
            ):
                """
                Generator: Yields the inode's extended attributes as tuples
                (name, value) in their on-disk order, where name (str)
                is the on-disk attribute name including its resolved name
                prefix and value (bytes) is the raw attribute value.
                check_inline and check_block control where to read attributes
                (the inode's inline data and/or the external data block
                pointed to by i_file_acl) and if check_inline as well as
                force_inline are set to True, the inode's inline data
                will not be verified to contain actual extended attributes
                and instead is just interpretd as such. prefix_overrides
                is directly passed to Inode._parse_xattrs.
                """
                # Inline xattrs
                inline_data_offset = (
                    self.offset
                    + ext4_inode.EXT2_GOOD_OLD_INODE_SIZE
                    + self.inode.i_extra_isize
                )
                inline_data_length = (
                    self.offset
                    + self.volume.superblock.s_inode_size
                    - inline_data_offset
                )

                if check_inline and inline_data_length > sizeof(
                    ext4_xattr_ibody_header
                ):
                    inline_data = self.volume.read(
                        inline_data_offset, inline_data_length
                    )
                    xattrs_header = ext4_xattr_ibody_header.from_buffer_copy(
                        inline_data
                    )

                    """
                    TODO: Find way to detect inline xattrs without checking
                    the h_magic field to enable error detection with
                    the h_magic field.
                    """
                    if force_inline or xattrs_header.h_magic == 0xEA020000:
                        # The ext4_xattr_entry following the header is aligned on a 4-byte boundary
                        offset = 4 * (
                            (sizeof(ext4_xattr_ibody_header) + 3) // 4
                        )
                        for xattr_name, xattr_value in self._parse_xattrs(
                            inline_data[offset:],
                            0,
                            prefix_override=prefix_override,
                        ):
                            yield (xattr_name, xattr_value)

                # xattr block(s)
                if check_block and self.inode.i_file_acl != 0:
                    xattrs_block_start = (
                        self.inode.i_file_acl * self.volume.block_size
                    )
                    xattrs_block = self.volume.read(
                        xattrs_block_start, self.volume.block_size
                    )

                    xattrs_header = ext4_xattr_header.from_buffer_copy(
                        xattrs_block
                    )
                    if (
                        not self.volume.ignoreMagic
                        and xattrs_header.h_magic != 0xEA020000
                    ):
                        raise MagicError(
                            (
                                "Invalid magic value in xattrs block header at "
                                "offset 0x%X of inode %d: 0x%X "
                                "(expected 0xEA020000)"
                            )
                            % (
                                xattrs_block_start,
                                self.inode_idx,
                                xattrs_header.h_magic,
                            )
                        )

                    if xattrs_header.h_blocks != 1:
                        raise Ext4Error(
                            "Invalid number of xattr blocks at offset "
                            "0x%X of inode %d: %d (expected 1)"
                            % (
                                xattrs_block_start,
                                self.inode_idx,
                                xattrs_header.h_blocks,
                            )
                        )

                    # The ext4_xattr_entry following the header is aligned on a 4-byte boundary
                    offset = 4 * ((sizeof(ext4_xattr_header) + 3) // 4)
                    for xattr_name, xattr_value in self._parse_xattrs(
                        xattrs_block[offset:],
                        -offset,
                        prefix_override=prefix_override,
                    ):
                        yield (xattr_name, xattr_value)

        class BlockReader:
            """
            Maps disk blocks into a linear byte stream.
            NOTE: This class does not implement buffering or caching.
            """

            # OSError
            EINVAL = 22

            def __init__(self, volume, byte_size, block_map):
                """
                Initializes a new block reader on the specified volume.
                mapping must be a list of MappingEntry instances. If
                you prefer a way to use 2-tuples (diskBlkIdx, blkCnt)
                with inferred file_block_index entries, see
                MappingEntry.create_mapping.
                """
                self.byte_size = byte_size
                self.volume = volume
                self.cursor = 0
                block_map = list(map(MappingEntry.copy, block_map))

                # Optimize mapping (stich together)
                MappingEntry.optimize(block_map)
                self.block_map = block_map

            def __repr__(self):
                return (
                    "%s(byte_size = %s, block_map = %s, volume_uuid = %s)"
                    % (
                        type(self).__name__,
                        self.byte_size,
                        self.block_map,
                        self.volume.uuid,
                    )
                )

            def get_block_mapping(self, fileBlkIdx):
                """
                Returns the disk block index of the file block specified
                by fileBlkIdx.
                """
                diskBlkIdx = None

                # Find disk block
                for entry in self.block_map:
                    if (
                        entry.fileBlkIdx
                        <= fileBlkIdx
                        < entry.fileBlkIdx + entry.blkCnt
                    ):
                        block_diff = fileBlkIdx - entry.fileBlkIdx
                        diskBlkIdx = entry.diskBlkIdx + block_diff
                        break

                return diskBlkIdx

            def read(self, byte_len=-1):
                """
                Reades up to byte_len bytes from the block device beginning
                at the cursor's current position. This operation will
                not exceed the inode's size. If -1 is passed for byte_len,
                the inode is read to the end.
                """
                # Parse args
                if byte_len < -1:
                    raise ValueError("byte_len must be non-negative or -1")

                bytes_remaining = self.byte_size - self.cursor
                byte_len = (
                    bytes_remaining
                    if byte_len == -1
                    else max(0, min(byte_len, bytes_remaining))
                )

                if byte_len == 0:
                    return b""

                # Reading blocks
                start_block_idx = self.cursor // self.volume.block_size
                end_block_idx = (
                    self.cursor + byte_len - 1
                ) // self.volume.block_size
                end_of_stream_check = byte_len

                blocks = [
                    self.read_block(i)
                    for i in xrange(
                        start_block_idx, end_block_idx - start_block_idx + 1
                    )
                ]

                startOffset = self.cursor % self.volume.block_size
                if startOffset != 0:
                    blocks[0] = blocks[0][startOffset:]
                byte_len = (
                    byte_len + startOffset - self.volume.block_size - 1
                ) % self.volume.block_size + 1
                blocks[-1] = blocks[-1][:byte_len]

                result = b"".join(blocks)

                # Check read
                if len(result) != end_of_stream_check:
                    raise EndOfStreamError(
                        (
                            "The volume's underlying stream ended "
                            "%s bytes before EOF."
                        )
                        % (UtilMgr.convNum(byte_len - len(result)))
                    )

                self.cursor += len(result)
                return result

            def read_block(self, fileBlkIdx):
                """
                Reads one block from disk
                (return a zero-block if the file block is not mapped)
                """
                diskBlkIdx = self.get_block_mapping(fileBlkIdx)

                if diskBlkIdx != None:
                    return self.volume.read(
                        diskBlkIdx * self.volume.block_size,
                        self.volume.block_size,
                    )
                else:
                    return bytes([0] * self.volume.block_size)

            def seek(self, seek, seek_mode=os.SEEK_SET):
                """
                Moves the internal cursor along the file
                (not the disk) and behaves like BufferedReader.seek
                """
                if seek_mode == os.SEEK_CUR:
                    seek += self.cursor
                elif seek_mode == os.SEEK_END:
                    seek += self.byte_size
                elif seek_mode == os.SEEK_SET:
                    seek += 0

                if seek < 0:
                    # Exception behavior copied from IOBase.seek
                    raise OSError(BlockReader.EINVAL, "Invalid argument")

                self.cursor = seek
                return seek

            def tell(self):
                """
                Returns the internal cursor's current file offset.
                """
                return self.cursor

        # open target file #
        try:
            self.fd = open(path, "rb")
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printWarn("failed to open %s" % path, reason=True)
            raise Exception("open failure")

        # init volume object #
        try:
            self.volume = Volume(self.fd)
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printWarn(
                "failed to init EXT4 object for %s'" % path, reason=True
            )

        # check volume object #
        if not self.volume:
            raise Exception("volume failure")

    def getInodeInfo(self, inode=None, path=None):
        if not inode and not path:
            SysMgr.printErr("no input for inode or path to get file meta info")
            return None

        if inode:
            inode = long(inode)
            inodeObj = self.volume.get_inode(inode)
            return inodeObj

        if path:
            parent = self.volume.root
            for item in path.split("/"):
                if not item:
                    continue
                parent = parent.get_inode(item)
            return parent

    def getInodeList(self, start=None, path=None, filters=[], verb=False):
        # define traverse function #
        def _traverseItems(self, start, path, filters, verb):
            if start:
                if isinstance(start, str):
                    path = start.strip()
                    parent = self.volume.root
                    for item in start.split("/"):
                        if not item:
                            continue
                        parent = parent.get_inode(item)
                    start = parent
            else:
                start = self.volume.root
                path = "/"

            # print progress #
            UtilMgr.printProgress()

            # open directory #
            FILE_TYPE = Ext4Analyzer.FILE_TYPE
            try:
                dirnode = start.open_dir(path=path)
                if not dirnode:
                    self.failDirList.setdefault(path, None)
            except SystemExit:
                sys.exit(0)
            except:
                self.failDirList.setdefault(path, None)
                SysMgr.printWarn("failed to open '%s' directory" % None, True)
                return

            # traverse all items #
            for fname, inode, ftype in dirnode:
                if fname in (".", ".."):
                    continue
                elif inode in self.inodeList:
                    continue

                # define attribute #
                ftype = FILE_TYPE[ftype] if ftype in FILE_TYPE else "?"
                inodeObj = self.volume.get_inode(inode)
                fpath = os.path.join(path, fname)

                # check condition #
                if not UtilMgr.isValidStr(
                    str(inode), filters
                ) and not UtilMgr.isValidStr(fpath, filters):
                    pass
                else:
                    self.inodeList[inode] = {
                        "name": fname,
                        "type": ftype,
                        "size": len(inodeObj),
                        "path": path,
                    }

                    if verb:
                        SysMgr.printWarn(
                            "%s [inode: %s] [type: %s] [size: %s]"
                            % (
                                fpath,
                                inode,
                                ftype,
                                UtilMgr.convSize2Unit(len(inodeObj)),
                            ),
                            always=True,
                            newline=False,
                        )

                if ftype == "dir":
                    _traverseItems(self, inodeObj, fpath, filters, verb)
                    continue

        # start traversing all items #
        _traverseItems(self, start, path, filters, verb)

        # return inode list #
        return self.inodeList


'''
class GlMgr(object):
    """Manager for GL"""

    instance = None

    @staticmethod
    def init():
        SysMgr.importPkgItems("ctypes")

        try:
            # load libglesobj library #
            if not SysMgr.libglesObj:
                SysMgr.libglesObj = SysMgr.loadLib(SysMgr.libglesPath)
        except:
            SysMgr.printErr("failed to load GLES object")
            sys.exit(-1)

        gl = GlMgr.instance = SysMgr.libglesObj
        gl.glActiveShaderProgram.argtypes = [c_uint32, c_uint32]
        gl.glActiveShaderProgram.restype = None
        gl.glActiveTexture.argtypes = [c_uint]
        gl.glActiveTexture.restype = None
        gl.glAttachShader.argtypes = [c_uint32, c_uint32]
        gl.glAttachShader.restype = None
        gl.glBeginQuery.argtypes = [c_uint, c_uint32]
        gl.glBeginQuery.restype = None
        gl.glBeginTransformFeedback.argtypes = [c_uint]
        gl.glBeginTransformFeedback.restype = None
        gl.glBindAttribLocation.argtypes = [
            c_uint32,
            c_uint32,
            POINTER(c_char),
        ]
        gl.glBindAttribLocation.restype = None
        gl.glBindBuffer.argtypes = [c_uint, c_uint32]
        gl.glBindBuffer.restype = None
        gl.glBindBufferBase.argtypes = [c_uint, c_uint32, c_uint32]
        gl.glBindBufferBase.restype = None
        gl.glBindBufferRange.argtypes = [
            c_uint,
            c_uint32,
            c_uint32,
            c_size_t,
            c_ssize_t,
        ]
        gl.glBindBufferRange.restype = None
        gl.glBindFramebuffer.argtypes = [c_uint, c_uint32]
        gl.glBindFramebuffer.restype = None
        gl.glBindImageTexture.argtypes = [
            c_uint32,
            c_uint32,
            c_int32,
            c_ubyte,
            c_int32,
            c_uint,
            c_uint,
        ]
        gl.glBindImageTexture.restype = None
        gl.glBindProgramPipeline.argtypes = [c_uint32]
        gl.glBindProgramPipeline.restype = None
        gl.glBindRenderbuffer.argtypes = [c_uint, c_uint32]
        gl.glBindRenderbuffer.restype = None
        gl.glBindSampler.argtypes = [c_uint32, c_uint32]
        gl.glBindSampler.restype = None
        gl.glBindTexture.argtypes = [c_uint, c_uint]
        gl.glBindTexture.restype = None
        gl.glBindTransformFeedback.argtypes = [c_uint, c_uint32]
        gl.glBindTransformFeedback.restype = None
        gl.glBindVertexArray.argtypes = [c_uint32]
        gl.glBindVertexArray.restype = None
        gl.glBindVertexBuffer.argtypes = [
            c_uint32,
            c_uint32,
            c_size_t,
            c_size_t,
        ]
        gl.glBindVertexBuffer.restype = None
        gl.glBlendColor.argtypes = [c_float, c_float, c_float, c_float]
        gl.glBlendColor.restype = None
        gl.glBlendEquation.argtypes = [c_uint]
        gl.glBlendEquation.restype = None
        gl.glBlendEquationSeparate.argtypes = [c_uint, c_uint]
        gl.glBlendEquationSeparate.restype = None
        gl.glBlendEquationSeparatei.argtypes = [c_uint32, c_uint, c_uint]
        gl.glBlendEquationSeparatei.restype = None
        gl.glBlendEquationi.argtypes = [c_uint32, c_uint]
        gl.glBlendEquationi.restype = None
        gl.glBlendFunc.argtypes = [c_uint, c_uint]
        gl.glBlendFunc.restype = None
        gl.glBlendFuncSeparate.argtypes = [c_uint, c_uint, c_uint, c_uint]
        gl.glBlendFuncSeparate.restype = None
        gl.glBlendFuncSeparatei.argtypes = [
            c_uint32,
            c_uint,
            c_uint,
            c_uint,
            c_uint,
        ]
        gl.glBlendFuncSeparatei.restype = None
        gl.glBlendFunci.argtypes = [c_uint32, c_uint, c_uint]
        gl.glBlendFunci.restype = None
        gl.glBlitFramebuffer.argtypes = [
            c_int32,
            c_int32,
            c_int32,
            c_int32,
            c_int32,
            c_int32,
            c_int32,
            c_int32,
            c_uint,
            c_uint,
        ]
        gl.glBlitFramebuffer.restype = None
        gl.glBufferData.argtypes = [c_uint, c_ssize_t, c_void_p, c_uint]
        gl.glBufferData.restype = None
        gl.glBufferSubData.argtypes = [c_uint, c_size_t, c_ssize_t, c_void_p]
        gl.glBufferSubData.restype = None
        gl.glCheckFramebufferStatus.argtypes = [c_uint]
        gl.glCheckFramebufferStatus.restype = c_uint
        gl.glClear.argtypes = [c_uint]
        gl.glClear.restype = None
        gl.glClearBufferfi.argtypes = [c_uint, c_int, c_float, c_int32]
        gl.glClearBufferfi.restype = None
        gl.glClearBufferfv.argtypes = [c_uint, c_int, POINTER(c_float)]
        gl.glClearBufferfv.restype = None
        gl.glClearBufferiv.argtypes = [c_uint, c_int, POINTER(c_int32)]
        gl.glClearBufferiv.restype = None
        gl.glClearBufferuiv.argtypes = [c_uint, c_int, POINTER(c_uint32)]
        gl.glClearBufferuiv.restype = None
        gl.glClearColor.argtypes = [c_float, c_float, c_float, c_float]
        gl.glClearColor.restype = None
        gl.glClearDepthf.argtypes = [c_float]
        gl.glClearDepthf.restype = None
        gl.glClearStencil.argtypes = [c_int]
        gl.glClearStencil.restype = None
        gl.glColorMask.argtypes = [c_ubyte, c_ubyte, c_ubyte, c_ubyte]
        gl.glColorMask.restype = None
        gl.glColorMaski.argtypes = [
            c_uint32,
            c_ubyte,
            c_ubyte,
            c_ubyte,
            c_ubyte,
        ]
        gl.glColorMaski.restype = None
        gl.glCompileShader.argtypes = [c_uint32]
        gl.glCompileShader.restype = None
        gl.glCompressedTexImage2D.argtypes = [
            c_uint,
            c_long,
            c_uint,
            c_size_t,
            c_size_t,
            c_long,
            c_size_t,
            c_void_p,
        ]
        gl.glCompressedTexImage2D.restype = None
        gl.glCompressedTexImage3D.argtypes = [
            c_uint,
            c_long,
            c_uint,
            c_size_t,
            c_size_t,
            c_size_t,
            c_long,
            c_size_t,
            c_void_p,
        ]
        gl.glCompressedTexImage3D.restype = None
        gl.glCompressedTexSubImage2D.argtypes = [
            c_uint,
            c_long,
            c_long,
            c_long,
            c_size_t,
            c_size_t,
            c_uint,
            c_size_t,
            c_void_p,
        ]
        gl.glCompressedTexSubImage2D.restype = None
        gl.glCompressedTexSubImage3D.argtypes = [
            c_uint,
            c_long,
            c_long,
            c_long,
            c_long,
            c_size_t,
            c_size_t,
            c_size_t,
            c_uint,
            c_size_t,
            c_void_p,
        ]
        gl.glCompressedTexSubImage3D.restype = None
        gl.glCopyBufferSubData.argtypes = [
            c_uint,
            c_uint,
            c_size_t,
            c_size_t,
            c_ssize_t,
        ]
        gl.glCopyBufferSubData.restype = None
        gl.glCopyImageSubData.argtypes = [
            c_uint32,
            c_uint,
            c_int32,
            c_int32,
            c_int32,
            c_int32,
            c_uint32,
            c_uint,
            c_int32,
            c_int32,
            c_int32,
            c_int32,
            c_size_t,
            c_size_t,
            c_size_t,
        ]
        gl.glCopyImageSubData.restype = None
        gl.glCopyTexImage2D.argtypes = [
            c_uint,
            c_long,
            c_uint,
            c_int,
            c_int,
            c_size_t,
            c_size_t,
            c_long,
        ]
        gl.glCopyTexImage2D.restype = None
        gl.glCopyTexSubImage2D.argtypes = [
            c_uint,
            c_long,
            c_long,
            c_long,
            c_int,
            c_int,
            c_size_t,
            c_size_t,
        ]
        gl.glCopyTexSubImage2D.restype = None
        gl.glCopyTexSubImage3D.argtypes = [
            c_uint,
            c_long,
            c_long,
            c_long,
            c_long,
            c_int,
            c_int,
            c_size_t,
            c_size_t,
        ]
        gl.glCopyTexSubImage3D.restype = None
        gl.glCreateProgram.argtypes = None
        gl.glCreateProgram.restype = c_uint32
        gl.glCreateShader.argtypes = [c_uint]
        gl.glCreateShader.restype = c_uint32
        gl.glCreateShaderProgramv.argtypes = [
            c_uint,
            c_size_t,
            POINTER(c_char_p),
        ]
        gl.glCreateShaderProgramv.restype = c_uint32
        gl.glCullFace.argtypes = [c_uint]
        gl.glCullFace.restype = None
        gl.glDebugMessageControl.argtypes = [
            c_uint,
            c_uint,
            c_uint,
            c_size_t,
            POINTER(c_uint32),
            c_ubyte,
        ]
        gl.glDebugMessageControl.restype = None
        gl.glDebugMessageInsert.argtypes = [
            c_uint,
            c_uint,
            c_uint32,
            c_uint,
            c_size_t,
            POINTER(c_char),
        ]
        gl.glDebugMessageInsert.restype = None
        gl.glDeleteBuffers.argtypes = [c_size_t, POINTER(c_ulong)]
        gl.glDeleteBuffers.restype = None
        gl.glDeleteFramebuffers.argtypes = [c_size_t, POINTER(c_uint32)]
        gl.glDeleteFramebuffers.restype = None
        gl.glDeleteProgram.argtypes = [c_uint32]
        gl.glDeleteProgram.restype = None
        gl.glDeleteProgramPipelines.argtypes = [c_size_t, POINTER(c_uint32)]
        gl.glDeleteProgramPipelines.restype = None
        gl.glDeleteQueries.argtypes = [c_size_t, POINTER(c_uint32)]
        gl.glDeleteQueries.restype = None
        gl.glDeleteRenderbuffers.argtypes = [c_size_t, POINTER(c_uint32)]
        gl.glDeleteRenderbuffers.restype = None
        gl.glDeleteSamplers.argtypes = [c_size_t, POINTER(c_uint32)]
        gl.glDeleteSamplers.restype = None
        gl.glDeleteShader.argtypes = [c_uint32]
        gl.glDeleteShader.restype = None
        gl.glDeleteTextures.argtypes = [c_size_t, POINTER(c_uint)]
        gl.glDeleteTextures.restype = None
        gl.glDeleteTransformFeedbacks.argtypes = [c_size_t, POINTER(c_uint32)]
        gl.glDeleteTransformFeedbacks.restype = None
        gl.glDeleteVertexArrays.argtypes = [c_size_t, POINTER(c_uint32)]
        gl.glDeleteVertexArrays.restype = None
        gl.glDepthFunc.argtypes = [c_uint]
        gl.glDepthFunc.restype = None
        gl.glDepthMask.argtypes = [c_ubyte]
        gl.glDepthMask.restype = None
        gl.glDepthRangef.argtypes = [c_float, c_float]
        gl.glDepthRangef.restype = None
        gl.glDetachShader.argtypes = [c_uint32, c_uint32]
        gl.glDetachShader.restype = None
        gl.glDisable.argtypes = [c_uint]
        gl.glDisable.restype = None
        gl.glDisableVertexAttribArray.argtypes = [c_uint32]
        gl.glDisableVertexAttribArray.restype = None
        gl.glDisablei.argtypes = [c_uint, c_uint32]
        gl.glDisablei.restype = None
        gl.glDispatchCompute.argtypes = [c_uint32, c_uint32, c_uint32]
        gl.glDispatchCompute.restype = None
        gl.glDispatchComputeIndirect.argtypes = [c_size_t]
        gl.glDispatchComputeIndirect.restype = None
        gl.glDrawArrays.argtypes = [c_uint, c_int32, c_size_t]
        gl.glDrawArrays.restype = None
        gl.glDrawArraysIndirect.argtypes = [c_uint, c_void_p]
        gl.glDrawArraysIndirect.restype = None
        gl.glDrawArraysInstanced.argtypes = [
            c_uint,
            c_int32,
            c_size_t,
            c_size_t,
        ]
        gl.glDrawArraysInstanced.restype = None
        gl.glDrawBuffers.argtypes = [c_size_t, POINTER(c_uint)]
        gl.glDrawBuffers.restype = None
        gl.glDrawElements.argtypes = [c_uint, c_size_t, c_uint, c_void_p]
        gl.glDrawElements.restype = None
        gl.glDrawElementsBaseVertex.argtypes = [
            c_uint,
            c_size_t,
            c_uint,
            c_void_p,
            c_int32,
        ]
        gl.glDrawElementsBaseVertex.restype = None
        gl.glDrawElementsIndirect.argtypes = [c_uint, c_uint, c_void_p]
        gl.glDrawElementsIndirect.restype = None
        gl.glDrawElementsInstanced.argtypes = [
            c_uint,
            c_size_t,
            c_uint,
            c_void_p,
            c_size_t,
        ]
        gl.glDrawElementsInstanced.restype = None
        gl.glDrawElementsInstancedBaseVertex.argtypes = [
            c_uint,
            c_size_t,
            c_uint,
            c_void_p,
            c_size_t,
            c_int32,
        ]
        gl.glDrawElementsInstancedBaseVertex.restype = None
        gl.glDrawRangeElements.argtypes = [
            c_uint,
            c_uint32,
            c_uint32,
            c_size_t,
            c_uint,
            c_void_p,
        ]
        gl.glDrawRangeElements.restype = None
        gl.glDrawRangeElementsBaseVertex.argtypes = [
            c_uint,
            c_uint32,
            c_uint32,
            c_size_t,
            c_uint,
            c_void_p,
            c_int32,
        ]
        gl.glDrawRangeElementsBaseVertex.restype = None
        gl.glEnable.argtypes = [c_uint]
        gl.glEnable.restype = None
        gl.glEnableVertexAttribArray.argtypes = [c_uint32]
        gl.glEnableVertexAttribArray.restype = None
        gl.glEnablei.argtypes = [c_uint, c_uint32]
        gl.glEnablei.restype = None
        gl.glEndQuery.argtypes = [c_uint]
        gl.glEndQuery.restype = None
        gl.glEndTransformFeedback.argtypes = None
        gl.glEndTransformFeedback.restype = None
        gl.glFinish.argtypes = None
        gl.glFinish.restype = None
        gl.glFlush.argtypes = None
        gl.glFlush.restype = None
        gl.glFlushMappedBufferRange.argtypes = [c_uint, c_size_t, c_ssize_t]
        gl.glFlushMappedBufferRange.restype = None
        gl.glFramebufferParameteri.argtypes = [c_uint, c_uint, c_int32]
        gl.glFramebufferParameteri.restype = None
        gl.glFramebufferRenderbuffer.argtypes = [
            c_uint,
            c_uint,
            c_uint,
            c_uint32,
        ]
        gl.glFramebufferRenderbuffer.restype = None
        gl.glFramebufferTexture.argtypes = [c_uint, c_uint, c_uint32, c_int32]
        gl.glFramebufferTexture.restype = None
        gl.glFramebufferTexture2D.argtypes = [
            c_uint,
            c_uint,
            c_uint,
            c_uint32,
            c_int32,
        ]
        gl.glFramebufferTexture2D.restype = None
        gl.glFramebufferTextureLayer.argtypes = [
            c_uint,
            c_uint,
            c_uint,
            c_long,
            c_long,
        ]
        gl.glFramebufferTextureLayer.restype = None
        gl.glFrontFace.argtypes = [c_uint]
        gl.glFrontFace.restype = None
        gl.glGenBuffers.argtypes = [c_size_t, POINTER(c_uint32)]
        gl.glGenBuffers.restype = None
        gl.glGenFramebuffers.argtypes = [c_size_t, POINTER(c_uint32)]
        gl.glGenFramebuffers.restype = None
        gl.glGenProgramPipelines.argtypes = [c_size_t, POINTER(c_uint32)]
        gl.glGenProgramPipelines.restype = None
        gl.glGenQueries.argtypes = [c_size_t, POINTER(c_uint32)]
        gl.glGenQueries.restype = None
        gl.glGenRenderbuffers.argtypes = [c_size_t, POINTER(c_uint32)]
        gl.glGenRenderbuffers.restype = None
        gl.glGenSamplers.argtypes = [c_size_t, POINTER(c_uint32)]
        gl.glGenSamplers.restype = None
        gl.glGenTextures.argtypes = [c_size_t, POINTER(c_uint)]
        gl.glGenTextures.restype = None
        gl.glGenTransformFeedbacks.argtypes = [c_size_t, POINTER(c_uint32)]
        gl.glGenTransformFeedbacks.restype = None
        gl.glGenVertexArrays.argtypes = [c_size_t, POINTER(c_uint32)]
        gl.glGenVertexArrays.restype = None
        gl.glGenerateMipmap.argtypes = [c_uint]
        gl.glGenerateMipmap.restype = None
        gl.glGetActiveAttrib.argtypes = [
            c_uint32,
            c_uint32,
            c_size_t,
            POINTER(c_size_t),
            POINTER(c_int32),
            POINTER(c_uint),
            POINTER(c_char),
        ]
        gl.glGetActiveAttrib.restype = None
        gl.glGetActiveUniform.argtypes = [
            c_uint32,
            c_uint32,
            c_size_t,
            POINTER(c_size_t),
            POINTER(c_int32),
            POINTER(c_uint),
            POINTER(c_char),
        ]
        gl.glGetActiveUniform.restype = None
        gl.glGetActiveUniformBlockName.argtypes = [
            c_uint32,
            c_uint32,
            c_size_t,
            POINTER(c_size_t),
            POINTER(c_char),
        ]
        gl.glGetActiveUniformBlockName.restype = None
        gl.glGetActiveUniformBlockiv.argtypes = [
            c_uint32,
            c_uint32,
            c_uint,
            POINTER(c_int32),
        ]
        gl.glGetActiveUniformBlockiv.restype = None
        gl.glGetActiveUniformsiv.argtypes = [
            c_uint32,
            c_size_t,
            POINTER(c_uint32),
            c_uint,
            POINTER(c_int32),
        ]
        gl.glGetActiveUniformsiv.restype = None
        gl.glGetAttachedShaders.argtypes = [
            c_uint32,
            c_size_t,
            POINTER(c_size_t),
            POINTER(c_uint32),
        ]
        gl.glGetAttachedShaders.restype = None
        gl.glGetAttribLocation.argtypes = [c_uint32, POINTER(c_char)]
        gl.glGetAttribLocation.restype = c_int32
        gl.glGetBooleanv.argtypes = [c_uint, POINTER(c_ubyte)]
        gl.glGetBooleanv.restype = None
        gl.glGetBufferParameteri64v.argtypes = [
            c_uint,
            c_uint,
            POINTER(c_int64),
        ]
        gl.glGetBufferParameteri64v.restype = None
        gl.glGetBufferParameteriv.argtypes = [c_uint, c_uint, POINTER(c_int32)]
        gl.glGetBufferParameteriv.restype = None
        gl.glGetBufferPointerv.argtypes = [c_uint, c_uint, POINTER(c_void_p)]
        gl.glGetBufferPointerv.restype = None
        gl.glGetDebugMessageLog.argtypes = [
            c_uint32,
            c_size_t,
            POINTER(c_uint),
            POINTER(c_uint),
            POINTER(c_uint32),
            POINTER(c_uint),
            POINTER(c_size_t),
            POINTER(c_char),
        ]
        gl.glGetDebugMessageLog.restype = c_uint32
        gl.glGetError.argtypes = None
        gl.glGetError.restype = c_uint
        gl.glGetFloatv.argtypes = [c_uint, POINTER(c_float)]
        gl.glGetFloatv.restype = None
        gl.glGetFragDataLocation.argtypes = [c_uint32, POINTER(c_char)]
        gl.glGetFragDataLocation.restype = c_int32
        gl.glGetFramebufferAttachmentParameteriv.argtypes = [
            c_uint,
            c_uint,
            c_uint,
            POINTER(c_int32),
        ]
        gl.glGetFramebufferAttachmentParameteriv.restype = None
        gl.glGetFramebufferParameteriv.argtypes = [
            c_uint,
            c_uint,
            POINTER(c_int32),
        ]
        gl.glGetFramebufferParameteriv.restype = None
        gl.glGetInteger64v.argtypes = [c_uint, POINTER(c_int64)]
        gl.glGetInteger64v.restype = None
        gl.glGetIntegerv.argtypes = [c_uint, POINTER(c_int32)]
        gl.glGetIntegerv.restype = None
        gl.glGetInternalformativ.argtypes = [
            c_uint,
            c_uint,
            c_uint,
            c_size_t,
            POINTER(c_int32),
        ]
        gl.glGetInternalformativ.restype = None
        gl.glGetMultisamplefv.argtypes = [c_uint, c_uint32, POINTER(c_float)]
        gl.glGetMultisamplefv.restype = None
        gl.glGetObjectLabel.argtypes = [
            c_uint,
            c_uint32,
            c_size_t,
            POINTER(c_size_t),
            POINTER(c_char),
        ]
        gl.glGetObjectLabel.restype = None
        gl.glGetObjectPtrLabel.argtypes = [
            c_void_p,
            c_size_t,
            POINTER(c_size_t),
            POINTER(c_char),
        ]
        gl.glGetObjectPtrLabel.restype = None
        gl.glGetPointerv.argtypes = [c_uint, POINTER(c_void_p)]
        gl.glGetPointerv.restype = None
        gl.glGetProgramBinary.argtypes = [
            c_uint32,
            c_size_t,
            POINTER(c_size_t),
            POINTER(c_uint),
            c_void_p,
        ]
        gl.glGetProgramBinary.restype = None
        gl.glGetProgramInfoLog.argtypes = [
            c_uint32,
            c_size_t,
            POINTER(c_size_t),
            POINTER(c_char),
        ]
        gl.glGetProgramInfoLog.restype = None
        gl.glGetProgramInterfaceiv.argtypes = [
            c_uint32,
            c_uint,
            c_uint,
            POINTER(c_int32),
        ]
        gl.glGetProgramInterfaceiv.restype = None
        gl.glGetProgramPipelineInfoLog.argtypes = [
            c_uint32,
            c_size_t,
            POINTER(c_size_t),
            POINTER(c_char),
        ]
        gl.glGetProgramPipelineInfoLog.restype = None
        gl.glGetProgramPipelineiv.argtypes = [
            c_uint32,
            c_uint,
            POINTER(c_int32),
        ]
        gl.glGetProgramPipelineiv.restype = None
        gl.glGetProgramResourceIndex.argtypes = [
            c_uint32,
            c_uint,
            POINTER(c_char),
        ]
        gl.glGetProgramResourceIndex.restype = c_uint32
        gl.glGetProgramResourceLocation.argtypes = [
            c_uint32,
            c_uint,
            POINTER(c_char),
        ]
        gl.glGetProgramResourceLocation.restype = c_int32
        gl.glGetProgramResourceName.argtypes = [
            c_uint32,
            c_uint,
            c_uint32,
            c_size_t,
            POINTER(c_size_t),
            POINTER(c_char),
        ]
        gl.glGetProgramResourceName.restype = None
        gl.glGetProgramResourceiv.argtypes = [
            c_uint32,
            c_uint,
            c_uint32,
            c_size_t,
            POINTER(c_uint),
            c_size_t,
            POINTER(c_size_t),
            POINTER(c_int32),
        ]
        gl.glGetProgramResourceiv.restype = None
        gl.glGetProgramiv.argtypes = [c_uint32, c_uint, POINTER(c_int32)]
        gl.glGetProgramiv.restype = None
        gl.glGetQueryObjectuiv.argtypes = [c_uint32, c_uint, POINTER(c_uint32)]
        gl.glGetQueryObjectuiv.restype = None
        gl.glGetQueryiv.argtypes = [c_uint, c_uint, POINTER(c_int32)]
        gl.glGetQueryiv.restype = None
        gl.glGetRenderbufferParameteriv.argtypes = [
            c_uint,
            c_uint,
            POINTER(c_int32),
        ]
        gl.glGetRenderbufferParameteriv.restype = None
        gl.glGetSamplerParameterIiv.argtypes = [
            c_uint32,
            c_uint,
            POINTER(c_int32),
        ]
        gl.glGetSamplerParameterIiv.restype = None
        gl.glGetSamplerParameterIuiv.argtypes = [
            c_uint32,
            c_uint,
            POINTER(c_uint32),
        ]
        gl.glGetSamplerParameterIuiv.restype = None
        gl.glGetSamplerParameterfv.argtypes = [
            c_uint32,
            c_uint,
            POINTER(c_float),
        ]
        gl.glGetSamplerParameterfv.restype = None
        gl.glGetSamplerParameteriv.argtypes = [
            c_uint32,
            c_uint,
            POINTER(c_int32),
        ]
        gl.glGetSamplerParameteriv.restype = None
        gl.glGetShaderInfoLog.argtypes = [
            c_uint32,
            c_size_t,
            POINTER(c_size_t),
            POINTER(c_char),
        ]
        gl.glGetShaderInfoLog.restype = None
        gl.glGetShaderPrecisionFormat.argtypes = [
            c_uint,
            c_uint,
            POINTER(c_int32),
            POINTER(c_int32),
        ]
        gl.glGetShaderPrecisionFormat.restype = None
        gl.glGetShaderSource.argtypes = [
            c_uint32,
            c_size_t,
            POINTER(c_size_t),
            POINTER(c_char),
        ]
        gl.glGetShaderSource.restype = None
        gl.glGetShaderiv.argtypes = [c_uint32, c_uint, POINTER(c_int32)]
        gl.glGetShaderiv.restype = None
        gl.glGetString.argtypes = [c_uint]
        gl.glGetString.restype = c_char_p
        gl.glGetStringi.argtypes = [c_uint, c_uint32]
        gl.glGetStringi.restype = c_char_p
        gl.glGetTexLevelParameterfv.argtypes = [
            c_uint,
            c_long,
            c_uint,
            POINTER(c_float),
        ]
        gl.glGetTexLevelParameterfv.restype = None
        gl.glGetTexLevelParameteriv.argtypes = [
            c_uint,
            c_long,
            c_uint,
            POINTER(c_int32),
        ]
        gl.glGetTexLevelParameteriv.restype = None
        gl.glGetTexParameterIiv.argtypes = [c_uint, c_uint, POINTER(c_int32)]
        gl.glGetTexParameterIiv.restype = None
        gl.glGetTexParameterIuiv.argtypes = [c_uint, c_uint, POINTER(c_uint32)]
        gl.glGetTexParameterIuiv.restype = None
        gl.glGetTexParameterfv.argtypes = [c_uint, c_uint, POINTER(c_float)]
        gl.glGetTexParameterfv.restype = None
        gl.glGetTexParameteriv.argtypes = [c_uint, c_uint, POINTER(c_int32)]
        gl.glGetTexParameteriv.restype = None
        gl.glGetTransformFeedbackVarying.argtypes = [
            c_uint32,
            c_uint32,
            c_size_t,
            POINTER(c_size_t),
            POINTER(c_size_t),
            POINTER(c_uint),
            POINTER(c_char),
        ]
        gl.glGetTransformFeedbackVarying.restype = None
        gl.glGetUniformBlockIndex.argtypes = [c_uint32, POINTER(c_char)]
        gl.glGetUniformBlockIndex.restype = c_uint32
        gl.glGetUniformIndices.argtypes = [
            c_uint32,
            c_size_t,
            POINTER(c_char_p),
            POINTER(c_uint32),
        ]
        gl.glGetUniformIndices.restype = None
        gl.glGetUniformLocation.argtypes = [c_uint32, POINTER(c_char)]
        gl.glGetUniformLocation.restype = c_int32
        gl.glGetUniformfv.argtypes = [c_uint32, c_int32, POINTER(c_float)]
        gl.glGetUniformfv.restype = None
        gl.glGetUniformiv.argtypes = [c_uint32, c_int32, POINTER(c_int32)]
        gl.glGetUniformiv.restype = None
        gl.glGetUniformuiv.argtypes = [c_uint32, c_int32, POINTER(c_uint32)]
        gl.glGetUniformuiv.restype = None
        gl.glGetVertexAttribIiv.argtypes = [c_uint32, c_uint, POINTER(c_int32)]
        gl.glGetVertexAttribIiv.restype = None
        gl.glGetVertexAttribIuiv.argtypes = [
            c_uint32,
            c_uint,
            POINTER(c_uint32),
        ]
        gl.glGetVertexAttribIuiv.restype = None
        gl.glGetVertexAttribPointerv.argtypes = [
            c_uint32,
            c_uint,
            POINTER(c_void_p),
        ]
        gl.glGetVertexAttribPointerv.restype = None
        gl.glGetVertexAttribfv.argtypes = [c_uint32, c_uint, POINTER(c_float)]
        gl.glGetVertexAttribfv.restype = None
        gl.glGetVertexAttribiv.argtypes = [c_uint32, c_uint, POINTER(c_int32)]
        gl.glGetVertexAttribiv.restype = None
        gl.glHint.argtypes = [c_uint, c_uint]
        gl.glHint.restype = None
        gl.glInvalidateFramebuffer.argtypes = [
            c_uint,
            c_size_t,
            POINTER(c_uint),
        ]
        gl.glInvalidateFramebuffer.restype = None
        gl.glInvalidateSubFramebuffer.argtypes = [
            c_uint,
            c_size_t,
            POINTER(c_uint),
            c_int32,
            c_int32,
            c_size_t,
            c_size_t,
        ]
        gl.glInvalidateSubFramebuffer.restype = None
        gl.glIsBuffer.argtypes = [c_uint32]
        gl.glIsBuffer.restype = c_ubyte
        gl.glIsEnabled.argtypes = [c_uint]
        gl.glIsEnabled.restype = c_ubyte
        gl.glIsEnabledi.argtypes = [c_uint, c_uint32]
        gl.glIsEnabledi.restype = c_ubyte
        gl.glIsFramebuffer.argtypes = [c_uint32]
        gl.glIsFramebuffer.restype = c_ubyte
        gl.glIsProgram.argtypes = [c_uint32]
        gl.glIsProgram.restype = c_ubyte
        gl.glIsProgramPipeline.argtypes = [c_uint32]
        gl.glIsProgramPipeline.restype = c_ubyte
        gl.glIsQuery.argtypes = [c_uint32]
        gl.glIsQuery.restype = c_ubyte
        gl.glIsRenderbuffer.argtypes = [c_uint32]
        gl.glIsRenderbuffer.restype = c_ubyte
        gl.glIsSampler.argtypes = [c_uint32]
        gl.glIsSampler.restype = c_ubyte
        gl.glIsShader.argtypes = [c_uint32]
        gl.glIsShader.restype = c_ubyte
        gl.glIsTexture.argtypes = [c_uint]
        gl.glIsTexture.restype = c_ubyte
        gl.glIsTransformFeedback.argtypes = [c_uint32]
        gl.glIsTransformFeedback.restype = c_ubyte
        gl.glIsVertexArray.argtypes = [c_uint32]
        gl.glIsVertexArray.restype = c_ubyte
        gl.glLineWidth.argtypes = [c_float]
        gl.glLineWidth.restype = None
        gl.glLinkProgram.argtypes = [c_uint32]
        gl.glLinkProgram.restype = None
        gl.glMapBufferRange.argtypes = [c_uint, c_size_t, c_ssize_t, c_uint]
        gl.glMapBufferRange.restype = c_void_p
        gl.glMemoryBarrier.argtypes = [c_uint]
        gl.glMemoryBarrier.restype = None
        gl.glMinSampleShading.argtypes = [c_float]
        gl.glMinSampleShading.restype = None
        gl.glObjectLabel.argtypes = [
            c_uint,
            c_uint32,
            c_size_t,
            POINTER(c_char),
        ]
        gl.glObjectLabel.restype = None
        gl.glObjectPtrLabel.argtypes = [c_void_p, c_size_t, POINTER(c_char)]
        gl.glObjectPtrLabel.restype = None
        gl.glPatchParameteri.argtypes = [c_uint, c_int32]
        gl.glPatchParameteri.restype = None
        gl.glPauseTransformFeedback.argtypes = None
        gl.glPauseTransformFeedback.restype = None
        gl.glPixelStorei.argtypes = [c_uint, c_long]
        gl.glPixelStorei.restype = None
        gl.glPolygonOffset.argtypes = [c_float, c_float]
        gl.glPolygonOffset.restype = None
        gl.glPopDebugGroup.argtypes = None
        gl.glPopDebugGroup.restype = None
        gl.glProgramBinary.argtypes = [c_uint32, c_uint, c_void_p, c_size_t]
        gl.glProgramBinary.restype = None
        gl.glProgramParameteri.argtypes = [c_uint32, c_uint, c_int32]
        gl.glProgramParameteri.restype = None
        gl.glProgramUniform1f.argtypes = [c_uint32, c_int32, c_float]
        gl.glProgramUniform1f.restype = None
        gl.glProgramUniform1fv.argtypes = [
            c_uint32,
            c_int32,
            c_size_t,
            POINTER(c_float),
        ]
        gl.glProgramUniform1fv.restype = None
        gl.glProgramUniform1i.argtypes = [c_uint32, c_int32, c_int32]
        gl.glProgramUniform1i.restype = None
        gl.glProgramUniform1iv.argtypes = [
            c_uint32,
            c_int32,
            c_size_t,
            POINTER(c_int32),
        ]
        gl.glProgramUniform1iv.restype = None
        gl.glProgramUniform1ui.argtypes = [c_uint32, c_int32, c_uint32]
        gl.glProgramUniform1ui.restype = None
        gl.glProgramUniform1uiv.argtypes = [
            c_uint32,
            c_int32,
            c_size_t,
            POINTER(c_uint32),
        ]
        gl.glProgramUniform1uiv.restype = None
        gl.glProgramUniform2f.argtypes = [c_uint32, c_int32, c_float, c_float]
        gl.glProgramUniform2f.restype = None
        gl.glProgramUniform2fv.argtypes = [
            c_uint32,
            c_int32,
            c_size_t,
            POINTER(c_float),
        ]
        gl.glProgramUniform2fv.restype = None
        gl.glProgramUniform2i.argtypes = [c_uint32, c_int32, c_int32, c_int32]
        gl.glProgramUniform2i.restype = None
        gl.glProgramUniform2iv.argtypes = [
            c_uint32,
            c_int32,
            c_size_t,
            POINTER(c_int32),
        ]
        gl.glProgramUniform2iv.restype = None
        gl.glProgramUniform2ui.argtypes = [
            c_uint32,
            c_int32,
            c_uint32,
            c_uint32,
        ]
        gl.glProgramUniform2ui.restype = None
        gl.glProgramUniform2uiv.argtypes = [
            c_uint32,
            c_int32,
            c_size_t,
            POINTER(c_uint32),
        ]
        gl.glProgramUniform2uiv.restype = None
        gl.glProgramUniform3f.argtypes = [
            c_uint32,
            c_int32,
            c_float,
            c_float,
            c_float,
        ]
        gl.glProgramUniform3f.restype = None
        gl.glProgramUniform3fv.argtypes = [
            c_uint32,
            c_int32,
            c_size_t,
            POINTER(c_float),
        ]
        gl.glProgramUniform3fv.restype = None
        gl.glProgramUniform3i.argtypes = [
            c_uint32,
            c_int32,
            c_int32,
            c_int32,
            c_int32,
        ]
        gl.glProgramUniform3i.restype = None
        gl.glProgramUniform3iv.argtypes = [
            c_uint32,
            c_int32,
            c_size_t,
            POINTER(c_int32),
        ]
        gl.glProgramUniform3iv.restype = None
        gl.glProgramUniform3ui.argtypes = [
            c_uint32,
            c_int32,
            c_uint32,
            c_uint32,
            c_uint32,
        ]
        gl.glProgramUniform3ui.restype = None
        gl.glProgramUniform3uiv.argtypes = [
            c_uint32,
            c_int32,
            c_size_t,
            POINTER(c_uint32),
        ]
        gl.glProgramUniform3uiv.restype = None
        gl.glProgramUniform4f.argtypes = [
            c_uint32,
            c_int32,
            c_float,
            c_float,
            c_float,
            c_float,
        ]
        gl.glProgramUniform4f.restype = None
        gl.glProgramUniform4fv.argtypes = [
            c_uint32,
            c_int32,
            c_size_t,
            POINTER(c_float),
        ]
        gl.glProgramUniform4fv.restype = None
        gl.glProgramUniform4i.argtypes = [
            c_uint32,
            c_int32,
            c_int32,
            c_int32,
            c_int32,
            c_int32,
        ]
        gl.glProgramUniform4i.restype = None
        gl.glProgramUniform4iv.argtypes = [
            c_uint32,
            c_int32,
            c_size_t,
            POINTER(c_int32),
        ]
        gl.glProgramUniform4iv.restype = None
        gl.glProgramUniform4ui.argtypes = [
            c_uint32,
            c_int32,
            c_uint32,
            c_uint32,
            c_uint32,
            c_uint32,
        ]
        gl.glProgramUniform4ui.restype = None
        gl.glProgramUniform4uiv.argtypes = [
            c_uint32,
            c_int32,
            c_size_t,
            POINTER(c_uint32),
        ]
        gl.glProgramUniform4uiv.restype = None
        gl.glProgramUniformMatrix2fv.argtypes = [
            c_uint32,
            c_int32,
            c_size_t,
            c_ubyte,
            POINTER(c_float),
        ]
        gl.glProgramUniformMatrix2fv.restype = None
        gl.glProgramUniformMatrix2x3fv.argtypes = [
            c_uint32,
            c_int32,
            c_size_t,
            c_ubyte,
            POINTER(c_float),
        ]
        gl.glProgramUniformMatrix2x3fv.restype = None
        gl.glProgramUniformMatrix2x4fv.argtypes = [
            c_uint32,
            c_int32,
            c_size_t,
            c_ubyte,
            POINTER(c_float),
        ]
        gl.glProgramUniformMatrix2x4fv.restype = None
        gl.glProgramUniformMatrix3fv.argtypes = [
            c_uint32,
            c_int32,
            c_size_t,
            c_ubyte,
            POINTER(c_float),
        ]
        gl.glProgramUniformMatrix3fv.restype = None
        gl.glProgramUniformMatrix3x2fv.argtypes = [
            c_uint32,
            c_int32,
            c_size_t,
            c_ubyte,
            POINTER(c_float),
        ]
        gl.glProgramUniformMatrix3x2fv.restype = None
        gl.glProgramUniformMatrix3x4fv.argtypes = [
            c_uint32,
            c_int32,
            c_size_t,
            c_ubyte,
            POINTER(c_float),
        ]
        gl.glProgramUniformMatrix3x4fv.restype = None
        gl.glProgramUniformMatrix4fv.argtypes = [
            c_uint32,
            c_int32,
            c_size_t,
            c_ubyte,
            POINTER(c_float),
        ]
        gl.glProgramUniformMatrix4fv.restype = None
        gl.glProgramUniformMatrix4x2fv.argtypes = [
            c_uint32,
            c_int32,
            c_size_t,
            c_ubyte,
            POINTER(c_float),
        ]
        gl.glProgramUniformMatrix4x2fv.restype = None
        gl.glProgramUniformMatrix4x3fv.argtypes = [
            c_uint32,
            c_int32,
            c_size_t,
            c_ubyte,
            POINTER(c_float),
        ]
        gl.glProgramUniformMatrix4x3fv.restype = None
        gl.glPushDebugGroup.argtypes = [
            c_uint,
            c_uint32,
            c_size_t,
            POINTER(c_char),
        ]
        gl.glPushDebugGroup.restype = None
        gl.glReadBuffer.argtypes = [c_uint]
        gl.glReadBuffer.restype = None
        gl.glReadPixels.argtypes = [
            c_int,
            c_int,
            c_size_t,
            c_size_t,
            c_uint,
            c_uint,
            c_void_p,
        ]
        gl.glReadPixels.restype = None
        gl.glReleaseShaderCompiler.argtypes = None
        gl.glReleaseShaderCompiler.restype = None
        gl.glRenderbufferStorage.argtypes = [
            c_uint,
            c_uint,
            c_size_t,
            c_size_t,
        ]
        gl.glRenderbufferStorage.restype = None
        gl.glRenderbufferStorageMultisample.argtypes = [
            c_uint,
            c_size_t,
            c_uint,
            c_size_t,
            c_size_t,
        ]
        gl.glRenderbufferStorageMultisample.restype = None
        gl.glResumeTransformFeedback.argtypes = None
        gl.glResumeTransformFeedback.restype = None
        gl.glSampleCoverage.argtypes = [c_float, c_ubyte]
        gl.glSampleCoverage.restype = None
        gl.glSampleMaski.argtypes = [c_uint32, c_uint]
        gl.glSampleMaski.restype = None
        gl.glSamplerParameterIiv.argtypes = [
            c_uint32,
            c_uint,
            POINTER(c_int32),
        ]
        gl.glSamplerParameterIiv.restype = None
        gl.glSamplerParameterIuiv.argtypes = [
            c_uint32,
            c_uint,
            POINTER(c_uint32),
        ]
        gl.glSamplerParameterIuiv.restype = None
        gl.glSamplerParameterf.argtypes = [c_uint32, c_uint, c_float]
        gl.glSamplerParameterf.restype = None
        gl.glSamplerParameterfv.argtypes = [c_uint32, c_uint, POINTER(c_float)]
        gl.glSamplerParameterfv.restype = None
        gl.glSamplerParameteri.argtypes = [c_uint32, c_uint, c_int32]
        gl.glSamplerParameteri.restype = None
        gl.glSamplerParameteriv.argtypes = [c_uint32, c_uint, POINTER(c_int32)]
        gl.glSamplerParameteriv.restype = None
        gl.glScissor.argtypes = [c_int, c_int, c_size_t, c_size_t]
        gl.glScissor.restype = None
        gl.glShaderBinary.argtypes = [
            c_size_t,
            POINTER(c_uint32),
            c_uint,
            c_void_p,
            c_size_t,
        ]
        gl.glShaderBinary.restype = None
        gl.glShaderSource.argtypes = [
            c_uint32,
            c_size_t,
            POINTER(c_char_p),
            POINTER(c_int32),
        ]
        gl.glShaderSource.restype = None
        gl.glStencilFunc.argtypes = [c_uint, c_int, c_uint]
        gl.glStencilFunc.restype = None
        gl.glStencilFuncSeparate.argtypes = [c_uint, c_uint, c_int, c_uint]
        gl.glStencilFuncSeparate.restype = None
        gl.glStencilMask.argtypes = [c_uint]
        gl.glStencilMask.restype = None
        gl.glStencilMaskSeparate.argtypes = [c_uint, c_uint]
        gl.glStencilMaskSeparate.restype = None
        gl.glStencilOp.argtypes = [c_uint, c_uint, c_uint]
        gl.glStencilOp.restype = None
        gl.glStencilOpSeparate.argtypes = [c_uint, c_uint, c_uint, c_uint]
        gl.glStencilOpSeparate.restype = None
        gl.glTexBuffer.argtypes = [c_uint, c_uint, c_uint32]
        gl.glTexBuffer.restype = None
        gl.glTexBufferRange.argtypes = [
            c_uint,
            c_uint,
            c_uint32,
            c_size_t,
            c_ssize_t,
        ]
        gl.glTexBufferRange.restype = None
        gl.glTexImage2D.argtypes = [
            c_uint,
            c_long,
            c_int,
            c_size_t,
            c_size_t,
            c_long,
            c_uint,
            c_uint,
            c_void_p,
        ]
        gl.glTexImage2D.restype = None
        gl.glTexParameterIiv.argtypes = [c_uint, c_uint, POINTER(c_int32)]
        gl.glTexParameterIiv.restype = None
        gl.glTexParameterIuiv.argtypes = [c_uint, c_uint, POINTER(c_uint32)]
        gl.glTexParameterIuiv.restype = None
        gl.glTexParameterf.argtypes = [c_uint, c_uint, c_float]
        gl.glTexParameterf.restype = None
        gl.glTexParameterfv.argtypes = [c_uint, c_uint, POINTER(c_float)]
        gl.glTexParameterfv.restype = None
        gl.glTexParameteri.argtypes = [c_uint, c_uint, c_long]
        gl.glTexParameteri.restype = None
        gl.glTexParameteriv.argtypes = [c_uint, c_uint, POINTER(c_long)]
        gl.glTexParameteriv.restype = None
        gl.glTexStorage2D.argtypes = [
            c_uint,
            c_size_t,
            c_uint,
            c_size_t,
            c_size_t,
        ]
        gl.glTexStorage2D.restype = None
        gl.glTexStorage2DMultisample.argtypes = [
            c_uint,
            c_size_t,
            c_uint,
            c_size_t,
            c_size_t,
            c_ubyte,
        ]
        gl.glTexStorage2DMultisample.restype = None
        gl.glTexStorage3D.argtypes = [
            c_uint,
            c_size_t,
            c_uint,
            c_size_t,
            c_size_t,
            c_size_t,
        ]
        gl.glTexStorage3D.restype = None
        gl.glTexStorage3DMultisample.argtypes = [
            c_uint,
            c_size_t,
            c_uint,
            c_size_t,
            c_size_t,
            c_size_t,
            c_ubyte,
        ]
        gl.glTexStorage3DMultisample.restype = None
        gl.glTexSubImage2D.argtypes = [
            c_uint,
            c_long,
            c_long,
            c_long,
            c_size_t,
            c_size_t,
            c_uint,
            c_uint,
            c_void_p,
        ]
        gl.glTexSubImage2D.restype = None
        gl.glTexSubImage3D.argtypes = [
            c_uint,
            c_long,
            c_long,
            c_long,
            c_long,
            c_size_t,
            c_size_t,
            c_size_t,
            c_uint,
            c_uint,
            c_void_p,
        ]
        gl.glTexSubImage3D.restype = None
        gl.glTransformFeedbackVaryings.argtypes = [
            c_uint32,
            c_size_t,
            POINTER(c_char_p),
            c_uint,
        ]
        gl.glTransformFeedbackVaryings.restype = None
        gl.glUniform1f.argtypes = [c_int32, c_float]
        gl.glUniform1f.restype = None
        gl.glUniform1fv.argtypes = [c_int32, c_size_t, POINTER(c_float)]
        gl.glUniform1fv.restype = None
        gl.glUniform1i.argtypes = [c_int32, c_int32]
        gl.glUniform1i.restype = None
        gl.glUniform1iv.argtypes = [c_int32, c_size_t, POINTER(c_int32)]
        gl.glUniform1iv.restype = None
        gl.glUniform1ui.argtypes = [c_int32, c_uint32]
        gl.glUniform1ui.restype = None
        gl.glUniform1uiv.argtypes = [c_int32, c_size_t, POINTER(c_uint32)]
        gl.glUniform1uiv.restype = None
        gl.glUniform2f.argtypes = [c_int32, c_float, c_float]
        gl.glUniform2f.restype = None
        gl.glUniform2fv.argtypes = [c_int32, c_size_t, POINTER(c_float)]
        gl.glUniform2fv.restype = None
        gl.glUniform2i.argtypes = [c_int32, c_int32, c_int32]
        gl.glUniform2i.restype = None
        gl.glUniform2iv.argtypes = [c_int32, c_size_t, POINTER(c_int32)]
        gl.glUniform2iv.restype = None
        gl.glUniform2ui.argtypes = [c_int32, c_uint32, c_uint32]
        gl.glUniform2ui.restype = None
        gl.glUniform2uiv.argtypes = [c_int32, c_size_t, POINTER(c_uint32)]
        gl.glUniform2uiv.restype = None
        gl.glUniform3f.argtypes = [c_int32, c_float, c_float, c_float]
        gl.glUniform3f.restype = None
        gl.glUniform3fv.argtypes = [c_int32, c_size_t, POINTER(c_float)]
        gl.glUniform3fv.restype = None
        gl.glUniform3i.argtypes = [c_int32, c_int32, c_int32, c_int32]
        gl.glUniform3i.restype = None
        gl.glUniform3iv.argtypes = [c_int32, c_size_t, POINTER(c_int32)]
        gl.glUniform3iv.restype = None
        gl.glUniform3ui.argtypes = [c_int32, c_uint32, c_uint32, c_uint32]
        gl.glUniform3ui.restype = None
        gl.glUniform3uiv.argtypes = [c_int32, c_size_t, POINTER(c_uint32)]
        gl.glUniform3uiv.restype = None
        gl.glUniform4f.argtypes = [c_int32, c_float, c_float, c_float, c_float]
        gl.glUniform4f.restype = None
        gl.glUniform4fv.argtypes = [c_int32, c_size_t, POINTER(c_float)]
        gl.glUniform4fv.restype = None
        gl.glUniform4i.argtypes = [c_int32, c_int32, c_int32, c_int32, c_int32]
        gl.glUniform4i.restype = None
        gl.glUniform4iv.argtypes = [c_int32, c_size_t, POINTER(c_int32)]
        gl.glUniform4iv.restype = None
        gl.glUniform4ui.argtypes = [
            c_int32,
            c_uint32,
            c_uint32,
            c_uint32,
            c_uint32,
        ]
        gl.glUniform4ui.restype = None
        gl.glUniform4uiv.argtypes = [c_int32, c_size_t, POINTER(c_uint32)]
        gl.glUniform4uiv.restype = None
        gl.glUniformBlockBinding.argtypes = [c_uint32, c_uint32, c_uint32]
        gl.glUniformBlockBinding.restype = None
        gl.glUniformMatrix2fv.argtypes = [
            c_int32,
            c_size_t,
            c_ubyte,
            POINTER(c_float),
        ]
        gl.glUniformMatrix2fv.restype = None
        gl.glUniformMatrix2x3fv.argtypes = [
            c_int32,
            c_size_t,
            c_ubyte,
            POINTER(c_float),
        ]
        gl.glUniformMatrix2x3fv.restype = None
        gl.glUniformMatrix2x4fv.argtypes = [
            c_int32,
            c_size_t,
            c_ubyte,
            POINTER(c_float),
        ]
        gl.glUniformMatrix2x4fv.restype = None
        gl.glUniformMatrix3fv.argtypes = [
            c_int32,
            c_size_t,
            c_ubyte,
            POINTER(c_float),
        ]
        gl.glUniformMatrix3fv.restype = None
        gl.glUniformMatrix3x2fv.argtypes = [
            c_int32,
            c_size_t,
            c_ubyte,
            POINTER(c_float),
        ]
        gl.glUniformMatrix3x2fv.restype = None
        gl.glUniformMatrix3x4fv.argtypes = [
            c_int32,
            c_size_t,
            c_ubyte,
            POINTER(c_float),
        ]
        gl.glUniformMatrix3x4fv.restype = None
        gl.glUniformMatrix4fv.argtypes = [
            c_int32,
            c_size_t,
            c_ubyte,
            POINTER(c_float),
        ]
        gl.glUniformMatrix4fv.restype = None
        gl.glUniformMatrix4x2fv.argtypes = [
            c_int32,
            c_size_t,
            c_ubyte,
            POINTER(c_float),
        ]
        gl.glUniformMatrix4x2fv.restype = None
        gl.glUniformMatrix4x3fv.argtypes = [
            c_int32,
            c_size_t,
            c_ubyte,
            POINTER(c_float),
        ]
        gl.glUniformMatrix4x3fv.restype = None
        gl.glUnmapBuffer.argtypes = [c_uint]
        gl.glUnmapBuffer.restype = c_ubyte
        gl.glUseProgram.argtypes = [c_uint32]
        gl.glUseProgram.restype = None
        gl.glUseProgramStages.argtypes = [c_uint32, c_uint, c_uint32]
        gl.glUseProgramStages.restype = None
        gl.glValidateProgram.argtypes = [c_uint32]
        gl.glValidateProgram.restype = None
        gl.glValidateProgramPipeline.argtypes = [c_uint32]
        gl.glValidateProgramPipeline.restype = None
        gl.glVertexAttribBinding.argtypes = [c_uint32, c_uint32]
        gl.glVertexAttribBinding.restype = None
        gl.glVertexAttribDivisor.argtypes = [c_uint32, c_uint32]
        gl.glVertexAttribDivisor.restype = None
        gl.glVertexAttribFormat.argtypes = [
            c_uint32,
            c_int32,
            c_uint,
            c_ubyte,
            c_uint32,
        ]
        gl.glVertexAttribFormat.restype = None
        gl.glVertexAttribIFormat.argtypes = [
            c_uint32,
            c_int32,
            c_uint,
            c_uint32,
        ]
        gl.glVertexAttribIFormat.restype = None
        gl.glVertexAttribIPointer.argtypes = [
            c_uint32,
            c_int32,
            c_uint,
            c_size_t,
            c_void_p,
        ]
        gl.glVertexAttribIPointer.restype = None
        gl.glVertexAttribPointer.argtypes = [
            c_uint32,
            c_int32,
            c_uint,
            c_ubyte,
            c_size_t,
            c_void_p,
        ]
        gl.glVertexAttribPointer.restype = None
        gl.glVertexBindingDivisor.argtypes = [c_uint32, c_uint32]
        gl.glVertexBindingDivisor.restype = None
        gl.glViewport.argtypes = [c_int, c_int, c_size_t, c_size_t]
        gl.glViewport.restype = None
'''


class PageAnalyzer(object):
    """Analyzer for kernel page"""

    # page flags from kernel/include/uapi/linux/kernel-page-flags.h #
    flagList = [
        "KPF_LOCKED",  # 0
        "KPF_ERROR",  # 1
        "KPF_REFERENCED",  # 2
        "KPF_UPTODATE",  # 3
        "KPF_DIRTY",  # 4
        "KPF_LRU",  # 5
        "KPF_ACTIVE",  # 6
        "KPF_SLAB",  # 7
        "KPF_WRITEBACK",  # 8
        "KPF_RECLAIM",  # 9
        "KPF_BUDDY",  # 10
        "KPF_MMAP",  # 11
        "KPF_ANON",  # 12
        "KPF_SWAPCACHE",  # 13
        "KPF_SWAPBACKED",  # 14
        "KPF_COMPOUND_HEAD",  # 15
        "KPF_COMPOUND_TAIL",  # 16
        "KPF_HUGE",  # 17
        "KPF_UNEVICTABLE",  # 18
        "KPF_HWPOISON",  # 19
        "KPF_NOPAGE",  # 20
        "KPF_KSM",  # 21
        "KPF_THP",  # 22
        "KPF_BALLOON",  # 23
        "KPF_ZERO_PAGE",  # 24
        "KPF_IDLE",  # 25
        "KPF_PAGETABLE",  # 26
        "",  # 27
        "",  # 28
        "",  # 29
        "",  # 30
        "",  # 31
        "KPF_RESERVED",  # 32
        "KPF_MLOCKED",  # 33
        "KPF_MAPPEDTODISK",  # 34
        "KPF_PRIVATE",  # 35
        "KPF_PRIVATE_2",  # 36
        "KPF_OWNER_PRIVATE",  # 37
        "KPF_ARCH",  # 38
        "KPF_UNCACHED",  # 39
        "KPF_SOFTDIRTY",  # 40
        "KPF_ARCH_2",  # 41
        "KPF_ARCH_3",  # 42
    ]

    mapFileCache = {}

    @staticmethod
    def getPageInfo(
        pid,
        vaddr,
        markIdle=False,
        checkIdle=False,
        retList=False,
        verb=True,
        progress=False,
        cmdset=[],
        showAttr=True,
    ):
        try:
            if not pid:
                raise Exception("no pid")
            elif not isinstance(pid, list):
                raise Exception("wrong pid")

            pids = []
            for item in pid:
                ret = SysMgr.getTids(item, isThread=False)
                if ret:
                    pids += ret
            if not pids:
                raise Exception("no task")
        except SystemExit:
            sys.exit(0)
        except:
            if pid:
                targetStr = " for '%s'" % ", ".join(pid)
            else:
                targetStr = ""

            SysMgr.printErr("failed to find task%s" % targetStr, reason=True)

            sys.exit(-1)

        # check root permission #
        SysMgr.checkRootPerm()

        # get target addresses #
        if vaddr:
            vaddrOrig = UtilMgr.cleanItem(vaddr.split(","), False)
        else:
            vaddrOrig = None

        # update verb #
        if verb and not SysMgr.printEnable:
            verb = False

        # set print function #
        if verb:
            _printPipe = SysMgr.printPipe
        else:
            _printPipe = lambda _: True

        # check idle page marking #
        if "MARKIDLE" in SysMgr.environList:
            markIdle = True
            if not SysMgr.checkIdlePageCond():
                sys.exit(-1)

        # check idle page checking #
        checkIdle = (
            "CHECKIDLE" in SysMgr.environList if not checkIdle else checkIdle
        )

        # check bitmap saving #
        saveBitmapFlag = "SAVEBITMAP" in SysMgr.environList

        PAGESIZE = SysMgr.PAGESIZE
        skipAttr = "SKIPATTR" in SysMgr.environList
        if "SKIPATTRSTR" in SysMgr.environList:
            showAttr = False

        # define shortcut variables #
        convNum = UtilMgr.convNum
        convSize = UtilMgr.convSize2Unit
        errMsg = (
            "failed to recognize addresses %s, "
            "input the address such as 102400 or 0x1234a-0x123ff"
        )

        idlePageList = {}

        for pidx, pid in enumerate(sorted(pids)):
            # read whole bitmap for idle pages #
            if checkIdle:
                bitmap = SysMgr.getIdleMap()
            else:
                bitmap = None

            # get process name #
            comm = SysMgr.getComm(pid)
            vaddrs = vaddrOrig
            if not vaddrs:
                PageAnalyzer.printMemoryArea(
                    pid,
                    comm=comm,
                    showall=SysMgr.showAll,
                    lastLine=True,
                    cache=True,
                )
                continue

            if retList:
                idlePageList.setdefault(pid, [])

            # get target address ranges #
            targetList = []
            for vaddr in vaddrs:
                if not vaddr:
                    continue

                # get address range #
                if vaddr in ("heap", "stack"):
                    vaddr = "[%s]" % vaddr

                    # get addresses by name #
                    vrange = FileAnalyzer.getMapAddr(pid, vaddr)
                    if not vrange:
                        SysMgr.printErr(
                            "failed to find memory address for '%s'" % vaddr
                        )
                        continue

                    targetList.append(vrange)
                else:
                    # get addresses by file #
                    vranges = FileAnalyzer.getMapAddr(pid, vaddr, retList=True)
                    if vranges:
                        targetList += vranges
                    else:
                        vrange = UtilMgr.cleanItem(vaddr.split("-"), False)
                        if len(vrange) > 1 and vrange[1].startswith("+"):
                            try:
                                start = long(vrange[0])
                            except SystemExit:
                                sys.exit(0)
                            except:
                                start = long(vrange[0], 16)

                            try:
                                var = UtilMgr.convUnit2Size(vrange[1])
                            except SystemExit:
                                sys.exit(0)
                            except:
                                var = long(vrange[1], 16)
                            vrange[1] = hex(start + var)

                        targetList.append(vrange)

            procPresent = 0
            procSwapped = 0
            procSoftdirty = 0
            procExmapped = 0
            procFile = 0
            procRef = 0
            procIdle = 0
            procTotalFlags = 0
            pageTable = []

            if cmdset:
                # create a debugger object #
                dbgObj = Debugger(pid)
                dbgObj.initValues()

            # print page info in target address ranges #
            for vcnt, vrange in enumerate(targetList):
                # print progress #
                if progress:
                    UtilMgr.printProgress(vcnt, len(targetList))

                rangeCnt = len(vrange)

                # check input count #
                if rangeCnt > 2:
                    SysMgr.printErr(errMsg % vrange, True)
                    sys.exit(-1)

                # check input format #
                try:
                    addrs = long(vrange[0], base=16)
                    addre = addrs
                except SystemExit:
                    sys.exit(0)
                except:
                    SysMgr.printErr(errMsg % vrange, True)
                    sys.exit(-1)

                try:
                    if rangeCnt == 2:
                        addre = long(vrange[1], base=16)
                        offset = 0
                    else:
                        offset = PAGESIZE

                    # check range #
                    if addrs > addre:
                        if verb:
                            # print memory area #
                            PageAnalyzer.printMemoryArea(
                                pid,
                                comm=comm,
                                showall=True,
                                lastLine=True,
                                cache=True,
                            )

                        # print error message #
                        SysMgr.printErr(
                            (
                                "failed to recognize addresses, "
                                "the first address (%s) is bigger than "
                                "the second address (%s)"
                            )
                            % (hex(addrs), hex(addre))
                        )
                        sys.exit(-1)
                except SystemExit:
                    sys.exit(0)
                except:
                    SysMgr.printErr(errMsg % vrange, True)
                    sys.exit(-1)

                size = addre - addrs

                _printPipe(
                    "\n[Mem Info] [Proc: %s(%s)] [AREA: %s] [HELP: %s]"
                    % (comm, pid, vaddr, "kernel/Documentation/vm/pagemap.txt")
                )

                # print memory area info #
                if verb:
                    PageAnalyzer.printMemoryArea(
                        pid, addrs, addre, lastLine=True, cache=True
                    )

                # execute commands #
                if cmdset:
                    cmds = []
                    # convert START and SIZE values #
                    for item in cmdset:
                        item = item.replace("START", str(addrs))
                        item = item.replace("END", str(addre))
                        item = item.replace("SIZE", str(size))
                        cmds.append(item)

                    # execute remote commands #
                    dbgObj.executeCmd(cmds, force=True)
                    SysMgr.printPipe("\n")

                # check skip condition #
                if skipAttr:
                    continue

                # print menu #
                _printPipe(
                    (
                        "{0:^19}|{1:^16}|{2:>5}|{3:>5}|{4:>5}|{5:>5}|"
                        "{6:>5}|{7:>5}|{8:>5}| {9} ({10})\n{11}"
                    ).format(
                        "VADDR",
                        "PFN",
                        "PRES",
                        "SWAP",
                        "FILE",
                        "SHR",
                        "SDRT",
                        "EXMAP",
                        "IDLE",
                        "ATTRIBUTES",
                        "BITS",
                        oneLine,
                    )
                )

                # init total variables #
                totalPresent = 0
                totalSwapped = 0
                totalSoftdirty = 0
                totalExmapped = 0
                totalFile = 0
                totalRef = 0
                totalIdle = 0
                totalFlags = 0

                # read pagemap #
                addreNew = addre + offset
                length = addreNew - addrs
                pagemap = PageAnalyzer.getPagemap(pid, addrs, length)

                # create a new pagetable for this area #
                if retList or (saveBitmapFlag and checkIdle):
                    pageSubTable = [1] * long(length / PAGESIZE)

                idx = 0
                for addr in xrange(addrs, addreNew, PAGESIZE):
                    # get page frame info #
                    try:
                        entry = struct.unpack("Q", pagemap[idx : idx + 8])[0]
                    except SystemExit:
                        sys.exit(0)
                    except:
                        entry = 0
                    finally:
                        idx += 8

                    # get page frame number #
                    pfn = PageAnalyzer.getPfn(entry)

                    # no page #
                    isIdle = "-"
                    # mark idle pages #
                    if markIdle:
                        if SysMgr.handleIdlePage(pfn, write=True):
                            isIdle = True
                            totalIdle += 1
                            procIdle += 1
                    # check idle pages #
                    elif checkIdle:
                        isIdle = SysMgr.handleIdlePage(
                            pfn, write=False, bitmap=bitmap
                        )
                        if isIdle is None:
                            isIdle = "-"
                        elif isIdle:
                            totalIdle += 1
                            procIdle += 1

                            # change a bit in a pagetable #
                            if retList or saveBitmapFlag:
                                try:
                                    pageSubTable[long(idx / 8) - 1] = 0
                                except:
                                    SysMgr.printWarn(
                                        "failed to save an idle page status",
                                        always=True,
                                        reason=True,
                                    )

                    # check skip condition #
                    if not verb:
                        continue

                    isPresent = PageAnalyzer.isPresent(entry)
                    if isPresent:
                        totalPresent += 1
                        procPresent += 1

                    isSwapped = PageAnalyzer.isSwapped(entry)
                    if isSwapped:
                        totalSwapped += 1
                        procSwapped += 1

                    isSoftdirty = PageAnalyzer.isSoftdirty(entry)
                    if isSoftdirty:
                        totalSoftdirty += 1
                        procSoftdirty += 1

                    isExmapped = PageAnalyzer.isExmapped(entry)
                    if isExmapped:
                        totalExmapped += 1
                        procExmapped += 1

                    isFile = PageAnalyzer.isFilePage(entry)
                    if isFile:
                        totalFile += 1
                        procFile += 1

                    refCnt = PageAnalyzer.getPageCount(pfn)
                    if refCnt > 1:
                        refCnt = 1
                        totalRef += 1
                        procRef += 1
                    else:
                        refCnt = 0

                    if showAttr:
                        kflags = PageAnalyzer.getPageFlags(pfn)
                        totalFlags |= kflags
                        if kflags:
                            bflags = hex(kflags).rstrip("L")
                            sflags = PageAnalyzer.getFlagTypes(bflags)
                            bflags = "(%s)" % bflags
                        else:
                            sflags = bflags = ""
                    else:
                        sflags = bflags = ""

                    if SysMgr.showAll:
                        SysMgr.addPrint(
                            (
                                "{0:>18} |{1:>15} |{2:>5}|{3:>5}|{4:>5}|"
                                "{5:>5}|{6:>5}|{7:>5}|{8:>5}| {9} {10}\n"
                            ).format(
                                hex(addr).rstrip("L"),
                                hex(pfn).rstrip("L") if pfn else "",
                                isPresent,
                                isSwapped,
                                isFile,
                                convSize(refCnt, isInt=True),
                                isSoftdirty,
                                isExmapped,
                                isIdle,
                                sflags,
                                bflags,
                            ),
                            force=True,
                            isList=True,
                        )

                procTotalFlags |= totalFlags
                totalFlagsStr = hex(totalFlags).rstrip("L")

                # print total stats #
                _printPipe(
                    (
                        "{0:>18} |{1:>15} |{2:>5}|{3:>5}|{4:>5}|"
                        "{5:>5}|{6:>5}|{7:>5}|{8:>5}| {9} ({10})\n{11:1}"
                    ).format(
                        "[TOTAL]",
                        "",
                        convSize(totalPresent, isInt=True),
                        convSize(totalSwapped, isInt=True),
                        convSize(totalFile, isInt=True),
                        convSize(totalRef, isInt=True),
                        convSize(totalSoftdirty, isInt=True),
                        convSize(totalExmapped, isInt=True),
                        convSize(totalIdle, isInt=True),
                        PageAnalyzer.getFlagTypes(totalFlagsStr),
                        totalFlagsStr,
                        oneLine,
                    )
                )

                # print all items #
                if verb:
                    SysMgr.doPrint(newline=False, clear=True, isList=True)
                    if SysMgr.showAll:
                        _printPipe("%s\n" % oneLine)
                else:
                    SysMgr.clearPrint()

                # add a new area to list #
                if retList:
                    idlePageList[pid].append([addrs, addreNew, pageSubTable])

                # merge pagetables #
                if saveBitmapFlag and checkIdle:
                    pageTable += pageSubTable

            if progress:
                UtilMgr.deleteProgress()

            if cmdset:
                # destroy debugger object #
                dbgObj.detach()
                del dbgObj

            # check skip condition #
            if skipAttr:
                continue

            # print process memory info #
            _printPipe(
                "\n[Mem Info] [Proc: %s(%s)] [Area: %s)\n%s"
                % (SysMgr.getComm(pid), pid, convNum(len(targetList)), twoLine)
            )

            _printPipe(
                (
                    "{0:^10}|{1:^8}|{2:^8}|{3:^8}|{4:^8}|"
                    "{5:^8}|{6:^8}|{7:^8}| {8} ({9})\n{10}"
                ).format(
                    "[TOTAL]",
                    "PRESENT",
                    "SWAP",
                    "FILE",
                    "SHR",
                    "SDRT",
                    "EXMAP",
                    "IDLE",
                    "ATTRIBUTES",
                    "BITS",
                    oneLine,
                )
            )

            procTotalFlagsStr = hex(procTotalFlags).rstrip("L")

            # print total page stats #
            _printPipe(
                (
                    "{0:^9} |{1:>7} |{2:>7} |{3:>7} |"
                    "{4:>7} |{5:>7} |{6:>7} |{7:>7} | {8}"
                ).format(
                    "PAGE",
                    convNum(procPresent),
                    convNum(procSwapped),
                    convNum(procFile),
                    convSize(procRef, isInt=True),
                    convNum(procSoftdirty),
                    convNum(procExmapped),
                    convNum(procIdle) if checkIdle or markIdle else "-",
                    PageAnalyzer.getFlagTypes(procTotalFlagsStr),
                )
            )

            # print total size stats #
            _printPipe(
                (
                    "{0:^9} |{1:>7} |{2:>7} |{3:>7} |"
                    "{4:>7} |{5:>7} |{6:>7} |{7:>7} | {8}\n{9}"
                ).format(
                    "SIZE",
                    convSize(procPresent << 12),
                    convSize(procSwapped << 12),
                    convSize(procFile << 12),
                    convSize(procRef << 12, isInt=True),
                    convSize(procSoftdirty << 12),
                    convSize(procExmapped << 12),
                    convSize(procIdle << 12) if checkIdle or markIdle else "-",
                    procTotalFlagsStr,
                    oneLine,
                )
            )

            # save idle bitmap to the file #
            if saveBitmapFlag and checkIdle:
                if SysMgr.printFd:
                    filename = SysMgr.printFd.name
                    name, ext = os.path.splitext(filename)
                    filename = "%s_idlemap%s%s" % (name, pid, ext)
                else:
                    filename = "idlemap.out"

                # backup #
                SysMgr.backupFile(filename)

                # save bitmap to the file #
                try:
                    with open(filename, "wb") as fd:
                        btable = bytes(pageTable)
                        fd.write(btable)
                    os.chmod(filename, 0o777)
                except:
                    SysMgr.printOpenErr(filename)
                    continue

                # get output size #
                fsize = UtilMgr.getFileSizeStr(filename)

                SysMgr.printStat(
                    "saved the bitmap data to '%s'%s successfully"
                    % (filename, fsize)
                )

            # print split line between proceses #
            if pidx < len(pids) - 1:
                _printPipe("\n\n" + splitLine + "\n\n")

        return idlePageList

    @staticmethod
    def printMemoryArea(
        pid,
        start=-1,
        end=-1,
        comm=None,
        lastLine=False,
        showall=True,
        cache=False,
    ):
        buf = []
        count = 0
        switch = False
        fpath = "%s/%s/maps" % (SysMgr.procPath, pid)

        if start == end == -1 and not showall:
            summaryFlag = True
        else:
            summaryFlag = False

        # read all map info #
        try:
            # summary #
            if summaryFlag:
                buf = FileAnalyzer.getProcMapInfo(pid, saveAll=True)
            else:
                # check cache #
                if cache:
                    buf = PageAnalyzer.mapFileCache.get(fpath, [])

                if not buf:
                    with open(fpath, "r") as fd:
                        buf = fd.readlines()

                # save cache #
                if cache:
                    PageAnalyzer.mapFileCache[fpath] = buf
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printOpenErr(fpath)
            sys.exit(-1)

        convSize = UtilMgr.convSize2Unit

        if start == end == -1:
            if not comm:
                comm = SysMgr.getComm(pid)

            # get mem stat #
            mlist = SysMgr.getMemStat(pid)
            mstat = SysMgr.convMemStat(mlist)
            vss = convSize(mstat["vss"])
            rss = convSize(mstat["rss"])
            SysMgr.printPipe(
                "\n[Mem Info] [Proc: %s(%s)] [VSS: %s] [RSS: %s]"
                % (comm, pid, vss, rss)
            )

        startHex = hex(start)
        endHex = hex(end)
        allHex = hex(-1)

        # print menu #
        menuStr = ""
        menuBuf = menuList = [
            "AREA",
            "PERM",
            "%8s" % "OFFSET",
            "%6s" % "DEV",
            "%12s" % "INODE",
            "%5s" % "SIZE",
        ]

        for idx, value in enumerate(menuBuf):
            if idx < 6:
                if idx == 0:
                    text = "{0:^38}".format(menuList[idx])
                else:
                    text = menuList[idx]
            else:
                break

            value = " " * (len(value) - len(text) + 1)
            menuStr = "%s%s%s" % (menuStr, text, value)

        menuStr += "TARGET"
        SysMgr.printPipe("%s\n%s\n%s" % (twoLine, menuStr, oneLine))

        # print summarized map info #
        if summaryFlag:
            for fname, info in sorted(
                buf.items(), key=lambda e: e[1]["vstart"]
            ):
                # skip non-contiguous segments #
                if SysMgr.magicStr in fname:
                    continue

                # convert path from overlayfs mount point #
                overlayInfo = SysMgr.getOverlayfsInfo(pid)
                if overlayInfo:
                    fname = UtilMgr.convOverlayPath(fname, overlayInfo)

                try:
                    soffset = hex(info["vstart"]).rstrip("L")
                    eoffset = hex(info["vend"]).rstrip("L")
                    size = convSize(
                        long(eoffset, 16) - long(soffset, 16), True
                    )

                    if not fname.startswith("/"):
                        fname = "[%s]" % fname

                    SysMgr.printPipe(
                        "%18s %18s %4s %8s %6s %12s %5s %s"
                        % (
                            soffset,
                            eoffset,
                            info["perm"],
                            info["offset"],
                            info["devid"],
                            info["inode"],
                            size,
                            fname,
                        )
                    )

                    count += 1

                except SystemExit:
                    sys.exit(0)
                except:
                    pass

            if count == 0:
                SysMgr.printPipe("no involved memory area")

            if lastLine:
                SysMgr.printPipe(oneLine)

            return

        # get anon size #
        amap = TaskAnalyzer.getProcAnonMap(pid, cache=cache)

        # print all map info #
        for line in buf:
            tmplist = line.split()
            vma = tmplist[0]
            soffset, eoffset = vma.split("-")

            soffset = long(soffset, base=16)
            eoffset = long(eoffset, base=16)

            if startHex == endHex == allHex:
                switch = False
            elif "-" in line:
                if soffset <= start < eoffset:
                    switch = True
                elif not switch:
                    continue
                elif end < eoffset:
                    break

            try:
                target = line[:-1].split(None, 5)

                target[4] = "%12s" % target[4]

                soffsetHex = hex(soffset)
                eoffsetHex = hex(eoffset)

                if not soffsetHex.startswith("0x"):
                    soffsetHex = "0x%s" % soffsetHex

                if not eoffsetHex.startswith("0x"):
                    eoffsetHex = "0x%s" % eoffsetHex

                size = convSize(eoffset - soffset, True)

                # convert path from overlayfs mount point #
                fname = target[5] if len(target) > 5 else " "
                overlayInfo = SysMgr.getOverlayfsInfo(pid)
                if overlayInfo:
                    fname = UtilMgr.convOverlayPath(fname, overlayInfo)

                # add anon size #
                if fname == " " and vma in amap and amap[vma]["size"]:
                    fname = "anon(%s)" % UtilMgr.convNum(
                        long(amap[vma]["size"]) << 10
                    )

                SysMgr.printPipe(
                    "%18s %18s %4s %8s %6s %12s %5s %s"
                    % (
                        soffsetHex,
                        eoffsetHex,
                        target[1],
                        target[2],
                        target[3],
                        target[4],
                        size,
                        fname,
                    )
                )
            except SystemExit:
                sys.exit(0)
            except:
                pass

            count += 1

            if switch and end <= eoffset:
                break

        if count == 0:
            SysMgr.printPipe("no involved memory area")

        if lastLine:
            SysMgr.printPipe(oneLine)

    @staticmethod
    def getFlagTypes(flags):
        nrFlags = long(flags, 16)
        maxBit = nrFlags.bit_length()
        sflags = [
            PageAnalyzer.flagList[idx][4:]
            for idx in xrange(maxBit)
            if nrFlags & (1 << idx)
        ]
        return "|".join(sflags)

    @staticmethod
    def readEntry(path, offset, size=8):
        try:
            f = SysMgr.getCachedFd(path)
            f.seek(offset, 0)

            data = f.read(size)
            if not data:
                return 0

            return struct.unpack("Q" if size == 8 else "I", data)[0]
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printErr(
                "failed to read %s byte from %s of %s" % (size, offset, path),
                reason=True,
            )
            sys.exit(-1)

    @staticmethod
    def getPagemapPath(pid):
        path = "{0}/{1}/pagemap".format(SysMgr.procPath, pid)
        if not os.path.isfile(path):
            SysMgr.printErr("failed to find the process with PID %s" % pid)
            sys.exit(-1)
        return path

    @staticmethod
    def getPagemap(pid, addr, size):
        path = PageAnalyzer.getPagemapPath(pid)
        offset = long(addr / SysMgr.PAGESIZE) * 8
        size = long(size / SysMgr.PAGESIZE) * 8

        try:
            f = SysMgr.getCachedFd(path)
            f.seek(offset, 0)
            return f.read(size)
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printErr(
                "failed to read %s byte from %s of %s" % (size, offset, path),
                reason=True,
            )
            sys.exit(-1)

    @staticmethod
    def getPagemapEntry(pid, addr):
        path = PageAnalyzer.getPagemapPath(pid)
        offset = long(addr / SysMgr.PAGESIZE) * 8
        return PageAnalyzer.readEntry(path, offset)

    @staticmethod
    def getPfn(entry):
        return entry & 0x7FFFFFFFFFFFFF

    @staticmethod
    def isSoftdirty(entry):
        return (entry & (1 << 55)) != 0

    @staticmethod
    def isExmapped(entry):
        return (entry & (1 << 56)) != 0

    @staticmethod
    def isWriteProtected(entry):
        return (entry & (1 << 57)) != 0

    @staticmethod
    def isSwapped(entry):
        return (entry & (1 << 62)) != 0

    @staticmethod
    def isFilePage(entry):
        return (entry & (1 << 61)) != 0

    @staticmethod
    def isPresent(entry):
        return (entry & (1 << 63)) != 0

    @staticmethod
    def getPageCount(pfn):
        path = SysMgr.procPath + "/kpagecount"
        return PageAnalyzer.readEntry(path, pfn * 8)

    @staticmethod
    def getPageFlags(pfn):
        path = SysMgr.procPath + "/kpageflags"
        return PageAnalyzer.readEntry(path, pfn * 8)


class FunctionAnalyzer(object):
    """Analyzer for function profiling"""

    symStackIdxTable = [
        "CPU_TICK",
        "STACK",
        "PAGE_ALLOC",
        "PAGE_FREE",
        "BLK_READ",
        "ARGUMENT",
        "HEAP_EXPAND",
        "HEAP_REDUCE",
        "IGNORE",
        "BLK_WRITE",
        "LOCK_TRY",
        "UNLOCK",
        "SYSCALL",
        "CUSTOM",
    ]

    def __init__(self, logFile):
        self.cpuEnabled = False
        self.memEnabled = False
        self.heapEnabled = False
        self.breadEnabled = False
        self.bwriteEnabled = False
        self.sigEnabled = False
        self.lockEnabled = False
        self.sysEnabled = False

        self.sort = "sym"
        self.connObj = None

        self.finishTime = "0"
        self.lastTime = "0"
        self.totalTime = 0
        self.totalTick = 0
        self.prevTime = "0"
        self.prevTid = "0"

        self.lastCore = None
        self.coreCtx = {}
        self.nowCtx = None
        self.nowEvent = None
        self.savedEvent = None
        self.nestedEvent = None
        self.nowCnt = 0
        self.savedCnt = 0
        self.nestedCnt = 0
        self.nowArg = 0
        self.savedArg = 0
        self.nestedArg = 0

        self.duplicatedPos = 0
        self.periodicEventCnt = 0
        self.periodicContEventCnt = 0
        self.periodicEventInterval = 0
        self.heapExpEventCnt = 0
        self.heapExpSize = 0
        self.heapRedEventCnt = 0
        self.heapRedSize = 0
        self.pageAllocEventCnt = 0
        self.pageAllocCnt = 0
        self.pageFreeEventCnt = 0
        self.pageFreeCnt = 0
        self.pageUnknownFreeCnt = 0
        self.pageUsageCnt = 0
        self.blockRdEventCnt = 0
        self.blockRdUsageCnt = 0
        self.blockWrEventCnt = 0
        self.blockWrUsageCnt = 0
        self.lockTryEventCnt = 0
        self.unlockEventCnt = 0
        self.customCnt = 0
        self.customTotal = 0
        self.syscallCnt = 0

        self.customEventTable = {}
        self.ignoreTable = {}
        self.mapData = []
        self.pageTable = {}
        self.oldPageTable = {}
        self.heapTable = {}
        self.oldHeapTable = {}
        self.posData = {}
        self.userSymData = {}
        self.userFileData = {}
        self.kerSymData = {}
        self.threadData = {}
        self.syscallTable = {}
        self.customCallData = []
        self.lockCallData = []
        self.sysCallData = []
        self.userCallData = []
        self.kernelCallData = []
        """
        userCallData = kernelCallData = \
            [pos, stack, event, eventCnt, eventArg]
        """

        self.init_threadData = {
            "comm": "?",
            "tgid": "-" * 5,
            "target": False,
            "cpuTick": 0,
            "die": False,
            "new": False,
            "nrPages": 0,
            "userPages": 0,
            "cachePages": 0,
            "kernelPages": 0,
            "heapSize": 0,
            "eventCnt": 0,
            "nrWrBlocks": 0,
            "customCnt": 0,
            "nrUnknownFreePages": 0,
            "nrKnownFreePages": 0,
            "nrRdBlocks": 0,
            "nrLockTry": 0,
            "nrUnlock": 0,
            "customTotal": 0,
            "nrSyscall": 0,
            "syscallTable": None,
            "lastNrSyscall": -1,
        }

        self.init_posData = {
            "symbol": "",
            "binary": "",
            "origBin": "",
            "offset": hex(0),
            "posCnt": 0,
            "userPageCnt": 0,
            "cachePageCnt": 0,
            "kernelPageCnt": 0,
            "totalCnt": 0,
            "blockRdCnt": 0,
            "blockWrCnt": 0,
            "pageCnt": 0,
            "heapSize": 0,
            "unknownPageFreeCnt": 0,
            "src": "",
            "customCnt": 0,
            "customTotal": 0,
            "lockTryCnt": 0,
            "unlockCnt": 0,
            "syscallCnt": 0,
        }

        self.init_symData = {
            "pos": "",
            "origBin": "",
            "tickCnt": 0,
            "blockRdCnt": 0,
            "pageCnt": 0,
            "unknownPageFreeCnt": 0,
            "stack": None,
            "symStack": None,
            "userPageCnt": 0,
            "cachePageCnt": 0,
            "kernelPageCnt": 0,
            "heapSize": 0,
            "blockWrCnt": 0,
            "customCnt": 0,
            "customTotal": 0,
            "pagePair": None,
            "pagePairCnt": 0,
            "pagePairTotal": 0.0,
            "pagePairMin": 0.0,
            "pagePairMax": 0.0,
            "pagePairAvr": 0.0,
            "pageRemainMin": 0.0,
            "pageRemainMax": 0.0,
            "pageRemainAvr": 0.0,
            "pageRemainTotal": 0.0,
            "lockTryCnt": 0,
            "unlockCnt": 0,
            "syscallCnt": 0,
            "totalTickCnt": 0,
        }

        self.init_ctxData = {
            "nestedEvent": None,
            "savedEvent": None,
            "nowEvent": None,
            "nested": 0,
            "recStat": False,
            "nestedCnt": 0,
            "savedCnt": 0,
            "nowCnt": 0,
            "nestedArg": None,
            "savedArg": None,
            "prevMode": None,
            "curMode": None,
            "userLastPos": "",
            "userStack": None,
            "kerLastPos": "",
            "kerStack": None,
            "prevKerLastPos": "",
            "prevKerStack": None,
            "nowArg": None,
            "prevTid": None,
            "prevTime": None,
        }

        self.init_pageLinkData = {
            "sym": "0",
            "subStackAddr": 0,
            "ksym": "0",
            "ksubStackAddr": 0,
            "type": "0",
            "time": "0",
        }

        self.init_heapSegData = {
            "tid": "0",
            "size": 0,
            "sym": "0",
            "subStackAddr": 0,
            "ksym": "0",
            "ksubStackAddr": 0,
            "time": 0.0,
            "core": "0",
        }

        self.init_pageData = {
            "tid": "0",
            "page": "0",
            "flags": "0",
            "type": "0",
            "time": "0",
        }

        self.init_glueData = {
            "count": 0,
            "size": 0,
            "timeList": None,
            "valueList": None,
        }

        self.init_subStackPageInfo = [0, 0, 0]
        # subStackPageInfo = [userPageCnt, cachePageCnt, kernelPageCnt]

        # read trace data #
        lines = TaskAnalyzer.readTraceData(logFile)

        # save trace data and stop analysis #
        if SysMgr.outputFile:
            SysMgr.saveTraceData(lines)
            sys.exit(0)

        # Check target thread setting #
        if not SysMgr.filterGroup:
            SysMgr.filterGroup.insert(0, "")
            self.target = []
        else:
            self.target = SysMgr.filterGroup

        # Check root path #
        if SysMgr.userEnable:
            if SysMgr.rootPath == "":
                rootPath = "/"
            else:
                rootPath = SysMgr.rootPath
            SysMgr.printInfo("use '%s' as the sysroot path" % rootPath)

        # Register None pos #
        self.posData["0"] = dict(self.init_posData)

        # get process tree #
        SysMgr.getProcTreeInfo()

        # start parsing logs #
        SysMgr.totalLine = len(lines)
        SysMgr.printStat("start analyzing data... [ STOP(Ctrl+c) ]")

        # parse logs #
        self.parseLines(lines, SysMgr.filterGroup)

        # Check whether data of target thread is collected or nothing #
        if (
            not self.userCallData
            and not self.kernelCallData
            and len(self.target) > 0
        ):
            if not self.target:
                SysMgr.printErr("no collected stack data")
            else:
                targetStr = ", ".join(self.target)
                SysMgr.printErr(
                    "no collected stack related to '%s'" % targetStr
                )
            sys.exit(-1)
        elif (
            SysMgr.userEnable
            and len(self.userCallData) == 1
            and self.userCallData[0][0] == "0"
        ):
            SysMgr.userEnable = False
            if self.target == []:
                SysMgr.printWarn("no collected user stack data", True)
            else:
                targetStr = ", ".join(self.target)
                SysMgr.printWarn(
                    "no collected user stack related to '%s'" % targetStr, True
                )

        # Get symbols from call address #
        SysMgr.printStat("start resolving symbols... [ STOP(Ctrl+c) ]")
        self.getSymbols()

        # Merge callstacks by symbol and address #
        SysMgr.printStat("start summarizing functions... [ STOP(Ctrl+c) ]")
        self.mergeStacks()

    def __del__(self):
        pass

    def handleHeapExpand(self, sym, ksym, stackAddr, kstackAddr, size, arg):
        addr = arg[0]
        time = arg[1]
        core = arg[2]
        tid = arg[3]

        self.userSymData[sym]["heapSize"] += size
        self.kerSymData[ksym]["heapSize"] += size

        self.heapTable.setdefault(addr, dict(self.init_heapSegData))
        self.heapTable[addr]["size"] = size
        self.heapTable[addr]["sym"] = sym
        self.heapTable[addr]["ksym"] = ksym
        self.heapTable[addr]["subStackAddr"] = stackAddr
        self.heapTable[addr]["ksubStackAddr"] = kstackAddr
        self.heapTable[addr]["time"] = time
        self.heapTable[addr]["core"] = core
        self.heapTable[addr]["tid"] = tid

    def handleHeapReduce(self, size, arg):
        addr = arg[0]

        subStackIndex = FunctionAnalyzer.symStackIdxTable.index("STACK")
        heapExpIndex = FunctionAnalyzer.symStackIdxTable.index("HEAP_EXPAND")

        try:
            sym = self.heapTable[addr]["sym"]
            ksym = self.heapTable[addr]["ksym"]
            stackAddr = self.heapTable[addr]["subStackAddr"]
            kstackAddr = self.heapTable[addr]["ksubStackAddr"]

            self.userSymData[sym]["heapSize"] -= size
            self.kerSymData[ksym]["heapSize"] -= size
        except:
            SysMgr.printWarn("failed to find heap segment to be freed")
            return

        # Set user stack list #
        if self.sort == "sym":
            targetStack = self.userSymData[sym]["symStack"]
        elif self.sort == "pos":
            targetStack = self.userSymData[sym]["stack"]

        # Find user stack of symbol allocated this segment #
        for val in targetStack:
            if id(val[subStackIndex]) == stackAddr:
                # Increase heap count of subStack #
                val[heapExpIndex] -= size
                break

        # Set kernel stack list #
        kernelTargetStack = self.kerSymData[ksym]["stack"]

        # Find kernel stack of symbol allocated this segment #
        for val in kernelTargetStack:
            if id(val[subStackIndex]) == kstackAddr:
                # Increase heap count of subStack #
                val[heapExpIndex] -= size
                break

        self.heapTable.pop(addr)

    def handlePageFree(
        self,
        sym,
        ksym,
        stackAddr,
        kstackAddr,
        pageFreeCnt,
        pageType,
        pfn,
        atime,
    ):
        subStackIndex = FunctionAnalyzer.symStackIdxTable.index("STACK")
        pageAllocIndex = FunctionAnalyzer.symStackIdxTable.index("PAGE_ALLOC")
        pageFreeIndex = FunctionAnalyzer.symStackIdxTable.index("PAGE_FREE")
        argIndex = FunctionAnalyzer.symStackIdxTable.index("ARGUMENT")

        for cnt in xrange(pageFreeCnt):
            pfnv = pfn + cnt
            subStackPageInfoIdx = 0

            try:
                # Decrease page count of symbol allocated page #
                # TODO: fix bug about wrong count of pos #
                allocSym = self.pageTable[pfnv]["sym"]
                allocStackAddr = self.pageTable[pfnv]["subStackAddr"]
                allocKernelSym = self.pageTable[pfnv]["ksym"]
                allocKernelStackAddr = self.pageTable[pfnv]["ksubStackAddr"]
                allocTime = self.pageTable[pfnv]["time"]

                self.pageUsageCnt -= 1
                self.userSymData[allocSym]["pageCnt"] -= 1
                self.kerSymData[allocKernelSym]["pageCnt"] -= 1

                if pageType == "USER":
                    self.userSymData[allocSym]["userPageCnt"] -= 1
                    self.kerSymData[allocKernelSym]["userPageCnt"] -= 1
                    subStackPageInfoIdx = 0
                elif pageType == "CACHE":
                    self.userSymData[allocSym]["cachePageCnt"] -= 1
                    self.kerSymData[allocKernelSym]["cachePageCnt"] -= 1
                    subStackPageInfoIdx = 1
                elif pageType == "KERNEL":
                    self.userSymData[allocSym]["kernelPageCnt"] -= 1
                    self.kerSymData[allocKernelSym]["kernelPageCnt"] -= 1
                    subStackPageInfoIdx = 2

                # get page lifetime #
                lifeTime = float(atime) - float(allocTime)

                # Set user page lifetime #
                if lifeTime > self.userSymData[allocSym]["pagePairMax"]:
                    self.userSymData[allocSym]["pagePairMax"] = lifeTime
                if (
                    self.userSymData[allocSym]["pagePairMin"] == 0
                    or lifeTime < self.userSymData[allocSym]["pagePairMin"]
                ):
                    self.userSymData[allocSym]["pagePairMin"] = lifeTime
                self.userSymData[allocSym]["pagePairTotal"] += lifeTime

                # Set kernel page lifetime #
                if lifeTime > self.kerSymData[allocKernelSym]["pagePairMax"]:
                    self.kerSymData[allocKernelSym]["pagePairMax"] = lifeTime
                if (
                    self.kerSymData[allocKernelSym]["pagePairMin"] == 0
                    or lifeTime
                    < self.kerSymData[allocKernelSym]["pagePairMin"]
                ):
                    self.kerSymData[allocKernelSym]["pagePairMin"] = lifeTime
                self.kerSymData[allocKernelSym]["pagePairTotal"] += lifeTime

                # Set user stack list #
                if self.sort == "sym":
                    targetStack = self.userSymData[allocSym]["symStack"]
                elif self.sort == "pos":
                    targetStack = self.userSymData[allocSym]["stack"]

                # Find user stack allocated this page #
                for val in targetStack:
                    if id(val[subStackIndex]) != allocStackAddr:
                        continue

                    val[pageAllocIndex] -= 1
                    val[argIndex][subStackPageInfoIdx] -= 1

                    # Set user stack list to free this page #
                    if self.sort == "sym":
                        subTargetStack = self.userSymData[sym]["symStack"]
                    elif self.sort == "pos":
                        subTargetStack = self.userSymData[sym]["stack"]

                    # Find user stack to free this page #
                    for sval in subTargetStack:
                        if id(sval[subStackIndex]) != stackAddr:
                            continue

                        if not self.userSymData[allocSym]["pagePair"]:
                            self.userSymData[allocSym]["pagePair"] = {}

                        allocCall = ""
                        freeCall = ""

                        try:
                            tsym = val[subStackIndex][0]
                            allocCall = "%s [%s]" % (
                                val[subStackIndex][0],
                                self.userSymData[tsym]["origBin"],
                            )
                            for usym in val[subStackIndex][1:]:
                                allocCall = "%s <- %s [%s]" % (
                                    allocCall,
                                    usym,
                                    self.userSymData[sym]["origBin"],
                                )
                        except:
                            if allocCall == "":
                                allocCall = "None"

                        try:
                            freeCall = "%s [%s]" % (
                                sym,
                                self.userSymData[sym]["origBin"],
                            )
                            for usym in sval[subStackIndex][1:]:
                                freeCall = "%s <- %s[%s]" % (
                                    freeCall,
                                    usym,
                                    self.userSymData[sym]["origBin"],
                                )
                        except:
                            if freeCall == "":
                                freeCall = "None"

                        pairId = "%s#%s" % (allocCall, freeCall)

                        try:
                            self.userSymData[allocSym]["pagePair"][pairId]
                        except:
                            self.userSymData[allocSym]["pagePair"][
                                pairId
                            ] = dict(self.init_glueData)

                        self.userSymData[allocSym]["pagePairCnt"] += 1
                        allocator = self.userSymData[allocSym]["pagePair"][
                            pairId
                        ]
                        allocator["size"] += 1

                        if not allocator["valueList"]:
                            allocator["valueList"] = {}
                        try:
                            allocator["valueList"][pageType] += 1
                        except:
                            allocator["valueList"][pageType] = 1

                        break
                    break

                # Set kernel stack list #
                kernelTargetStack = self.kerSymData[allocKernelSym]["stack"]

                # Find kernel stack allocated this page #
                for val in kernelTargetStack:
                    if id(val[subStackIndex]) != allocKernelStackAddr:
                        continue

                    val[pageAllocIndex] -= 1
                    val[argIndex][subStackPageInfoIdx] -= 1

                    # Set kernel stack list to free this page #
                    subTargetStack = self.kerSymData[ksym]["stack"]

                    # Find kernel stack to free this page #
                    for sval in subTargetStack:
                        if id(sval[subStackIndex]) != kstackAddr:
                            continue

                        kernelData = self.kerSymData[allocKernelSym]

                        if not kernelData["pagePair"]:
                            kernelData["pagePair"] = {}

                        allocCall = ""
                        freeCall = ""

                        try:
                            allocCall = (
                                "%s"
                                % self.posData[val[subStackIndex][0]]["symbol"]
                            )
                            for addr in val[subStackIndex][1:]:
                                allocCall = "%s <- %s" % (
                                    allocCall,
                                    self.posData[addr]["symbol"],
                                )
                        except:
                            if allocCall == "":
                                allocCall = "None"

                        try:
                            freeCall = "%s" % ksym
                            for addr in sval[subStackIndex]:
                                freeCall = "%s <- %s" % (
                                    freeCall,
                                    self.posData[addr]["symbol"],
                                )
                        except:
                            if freeCall == "":
                                freeCall = "None"

                        pairId = "%s#%s" % (allocCall, freeCall)

                        try:
                            kernelData["pagePair"][pairId]
                        except:
                            kernelData["pagePair"][pairId] = dict(
                                self.init_glueData
                            )

                        self.kerSymData[allocKernelSym]["pagePairCnt"] += 1
                        allocator = kernelData["pagePair"][pairId]
                        allocator["size"] += 1

                        if not allocator["valueList"]:
                            allocator["valueList"] = {}
                        try:
                            allocator["valueList"][pageType] += 1
                        except:
                            allocator["valueList"][pageType] = 1

                        break

                    break

                self.pageTable.pop(pfnv, None)
            except SystemExit:
                sys.exit(0)
            except:
                # this page is allocated before starting profile #

                self.pageUnknownFreeCnt += 1
                self.userSymData[sym]["unknownPageFreeCnt"] += 1
                self.kerSymData[ksym]["unknownPageFreeCnt"] += 1

                # Set user stack list #
                if self.sort == "sym":
                    targetStack = self.userSymData[sym]["symStack"]
                elif self.sort == "pos":
                    targetStack = self.userSymData[sym]["stack"]

                # Find subStack allocated this page #
                for val in targetStack:
                    if id(val[subStackIndex]) == stackAddr:
                        val[pageFreeIndex] += 1
                        break

                # Set kernel stack list #
                kernelTargetStack = self.kerSymData[ksym]["stack"]

                # Find subStack allocated this page #
                for val in kernelTargetStack:
                    if id(val[subStackIndex]) == kstackAddr:
                        val[pageFreeIndex] += 1
                        break

                continue

    def handlePageAlloc(
        self,
        sym,
        ksym,
        stackAddr,
        kstackAddr,
        pageAllocCnt,
        pageType,
        pfn,
        atime,
    ):
        subStackPageInfoIdx = 0

        subStackIndex = FunctionAnalyzer.symStackIdxTable.index("STACK")
        pageAllocIndex = FunctionAnalyzer.symStackIdxTable.index("PAGE_ALLOC")
        argIndex = FunctionAnalyzer.symStackIdxTable.index("ARGUMENT")

        # Increase counts of page to be allocated #
        self.userSymData[sym]["pageCnt"] += pageAllocCnt
        self.kerSymData[ksym]["pageCnt"] += pageAllocCnt

        if pageType == "USER":
            self.userSymData[sym]["userPageCnt"] += pageAllocCnt
            self.kerSymData[ksym]["userPageCnt"] += pageAllocCnt
            subStackPageInfoIdx = 0
        elif pageType == "CACHE":
            self.userSymData[sym]["cachePageCnt"] += pageAllocCnt
            self.kerSymData[ksym]["cachePageCnt"] += pageAllocCnt
            subStackPageInfoIdx = 1
        elif pageType == "KERNEL":
            self.userSymData[sym]["kernelPageCnt"] += pageAllocCnt
            self.kerSymData[ksym]["kernelPageCnt"] += pageAllocCnt
            subStackPageInfoIdx = 2

        # Set user stack list #
        if self.sort == "sym":
            targetStack = self.userSymData[sym]["symStack"]
        elif self.sort == "pos":
            targetStack = self.userSymData[sym]["stack"]

        # Find user stack of symbol allocated this page #
        for val in targetStack:
            if id(val[subStackIndex]) == stackAddr:
                # Increase page count of subStack #
                val[argIndex][subStackPageInfoIdx] += pageAllocCnt
                break

        # Set kernel stack list #
        kernelTargetStack = self.kerSymData[ksym]["stack"]

        # Find kernel stack of symbol allocated this page #
        for val in kernelTargetStack:
            if id(val[subStackIndex]) == kstackAddr:
                # Increase page count of subStack #
                val[argIndex][subStackPageInfoIdx] += pageAllocCnt
                break

        # Make PTE in page table #
        for cnt in xrange(pageAllocCnt):
            pfnv = pfn + cnt
            subStackPageInfoIdx = 0

            try:
                # Check whether this page is already allocated #
                allocSym = self.pageTable[pfnv]["sym"]

                allocStackAddr = self.pageTable[pfnv]["subStackAddr"]
                allocKernelSym = self.pageTable[pfnv]["ksym"]
                allocKernelStackAddr = self.pageTable[pfnv]["ksubStackAddr"]

                # Decrease counts of page already allocated but no free log #
                self.pageUsageCnt -= 1
                self.userSymData[allocSym]["pageCnt"] -= 1
                self.kerSymData[allocKernelSym]["pageCnt"] -= 1

                origPageType = self.pageTable[pfnv]["type"]
                if origPageType == "USER":
                    self.userSymData[allocSym]["userPageCnt"] -= 1
                    self.kerSymData[allocKernelSym]["userPageCnt"] -= 1
                    subStackPageInfoIdx = 0
                elif origPageType == "CACHE":
                    self.userSymData[allocSym]["cachePageCnt"] -= 1
                    self.kerSymData[allocKernelSym]["cachePageCnt"] -= 1
                    subStackPageInfoIdx = 1
                elif origPageType == "KERNEL":
                    self.userSymData[allocSym]["kernelPageCnt"] -= 1
                    self.kerSymData[allocKernelSym]["kernelPageCnt"] -= 1
                    subStackPageInfoIdx = 2

                # Set user stack list #
                if self.sort == "sym":
                    targetStack = self.userSymData[allocSym]["symStack"]
                elif self.sort == "pos":
                    targetStack = self.userSymData[allocSym]["stack"]

                # Find user stack of symbol allocated this page #
                for val in targetStack:
                    if id(val[subStackIndex]) == allocStackAddr:
                        # Decrease allocated page count of substack #
                        val[pageAllocIndex] -= 1
                        val[argIndex][subStackPageInfoIdx] -= 1
                        break

                # Set kernel stack list #
                kernelTargetStack = self.kerSymData[allocKernelSym]["stack"]

                # Find kernel stack of symbol allocated this page #
                for val in kernelTargetStack:
                    if id(val[subStackIndex]) == allocKernelStackAddr:
                        # Decrease allocated page count of substack #
                        val[pageAllocIndex] -= 1
                        val[argIndex][subStackPageInfoIdx] -= 1
                        break
            except SystemExit:
                sys.exit(0)
            except:
                self.pageTable[pfnv] = dict(self.init_pageLinkData)

            self.pageTable[pfnv]["sym"] = sym
            self.pageTable[pfnv]["ksym"] = ksym
            self.pageTable[pfnv]["type"] = pageType
            self.pageTable[pfnv]["subStackAddr"] = stackAddr
            self.pageTable[pfnv]["ksubStackAddr"] = kstackAddr
            self.pageTable[pfnv]["time"] = atime

    def mergeStacks(self):
        sym = ""
        ksym = ""
        stackAddr = 0
        kstackAddr = 0
        lineCnt = -1
        lastIdx = len(self.userCallData)

        # Backup page table used previously and Initialize it #
        self.oldPageTable = self.pageTable
        self.pageTable = {}

        # Backup heap table used previously and Initialize it #
        self.oldHeapTable = self.heapTable
        self.heapTable = {}

        subStackIndex = FunctionAnalyzer.symStackIdxTable.index("STACK")
        argIndex = FunctionAnalyzer.symStackIdxTable.index("ARGUMENT")

        # Merge call data by symbol or address #
        for val in self.userCallData:
            lineCnt += 1
            UtilMgr.printProgress(lineCnt, lastIdx - 1)

            pos = val[0]
            stack = val[1]
            event = val[2]
            eventCnt = val[3]
            arg = val[4]

            """
            Do not merge PAGE_FREE count
            because it will be merged with unknownPageFreeCnt
            """
            if event == "PAGE_FREE":
                savedEventCnt = eventCnt
                eventCnt = 0

            try:
                eventIndex = FunctionAnalyzer.symStackIdxTable.index(event)
            except:
                eventIndex = FunctionAnalyzer.symStackIdxTable.index("IGNORE")

            kernelPos = self.kernelCallData[lineCnt][0]
            kernelStack = self.kernelCallData[lineCnt][1]
            subStackPageInfo = list(self.init_subStackPageInfo)

            targetStack = []
            kernelTargetStack = []

            # Resolve user symbol #
            try:
                # No symbol related to last pos #
                if self.posData[pos]["symbol"] == "":
                    self.posData[pos]["symbol"] = pos
                    sym = pos
                else:
                    sym = self.posData[pos]["symbol"]
            except:
                continue

            # Resolve kernel symbol #
            try:
                # No symbol related to last pos #
                if self.posData[kernelPos]["symbol"] == "":
                    self.posData[kernelPos]["symbol"] = kernelPos
                    ksym = kernelPos
                else:
                    ksym = self.posData[kernelPos]["symbol"]
            except:
                continue

            # Make user file table of last pos in stack #
            try:
                path = self.posData[pos]["binary"]
                self.userFileData[path]
            except:
                self.userFileData[path] = dict(self.init_symData)

            # Make user symbol table of last pos in stack #
            try:
                self.userSymData[sym]
            except:
                self.userSymData[sym] = dict(self.init_symData)
                self.userSymData[sym]["stack"] = []
                self.userSymData[sym]["symStack"] = []
                self.userSymData[sym]["pos"] = pos
                self.userSymData[sym]["origBin"] = self.posData[pos]["origBin"]

            # Make kernel symbol table of last pos in stack #
            try:
                self.kerSymData[ksym]
            except:
                self.kerSymData[ksym] = dict(self.init_symData)
                self.kerSymData[ksym]["stack"] = []
                self.kerSymData[ksym]["pos"] = kernelPos

            # Set target user stack #
            if self.sort == "sym":
                tempSymStack = []
                # Make temporary symbol stack to merge stacks by symbol #
                for addr in stack:
                    tempSym = self.posData[addr]["symbol"]

                    # Ignore this function if there is no symbol #
                    if (
                        not SysMgr.showAll
                        and self.posData[addr]["origBin"] == "??"
                        and (
                            tempSym == addr
                            or tempSym == self.posData[addr]["offset"]
                            or addr == "00c0ffee"
                        )
                    ):
                        continue

                    # No symbol data #
                    if tempSym == "":
                        if self.posData[addr]["origBin"] == "??":
                            tempSym = "%x" % long(
                                self.posData[addr]["pos"], 16
                            )
                        else:
                            tempSym = "%x" % long(
                                self.posData[addr]["offset"], 16
                            )

                    try:
                        self.userSymData[tempSym]
                    except:
                        self.userSymData[tempSym] = dict(self.init_symData)
                        self.userSymData[tempSym]["stack"] = []
                        self.userSymData[tempSym]["symStack"] = []
                        self.userSymData[tempSym]["pos"] = addr
                        self.userSymData[tempSym]["origBin"] = self.posData[
                            addr
                        ]["origBin"]

                    tempSymStack.append(tempSym)

                # Switch input stack to symbol stack #
                stack = tempSymStack
                targetStack = self.userSymData[sym]["symStack"]
            elif self.sort == "pos":
                targetStack = self.userSymData[sym]["stack"]

            # First user stack related to this symbol #
            if not targetStack:
                tempList = [0] * len(FunctionAnalyzer.symStackIdxTable)
                tempList[eventIndex] = eventCnt
                tempList[subStackIndex] = stack
                tempList[argIndex] = list(subStackPageInfo)
                targetStack.append(tempList)

                stackAddr = id(stack)
            else:
                found = False

                # Find same stack by pos in stack list #
                for stackInfo in targetStack:
                    stackSet = set(stack)
                    subStackSet = set(stackInfo[subStackIndex])

                    # Found same stack #
                    if stackSet == subStackSet:
                        found = True
                        stackInfo[eventIndex] += eventCnt
                        stackAddr = id(stackInfo[subStackIndex])
                        break

                # New stack related to this symbol #
                if not found:
                    tempList = [0] * len(FunctionAnalyzer.symStackIdxTable)
                    tempList[eventIndex] = eventCnt
                    tempList[subStackIndex] = stack
                    tempList[argIndex] = list(subStackPageInfo)
                    targetStack.append(tempList)

                    stackAddr = id(stack)

            # Set target kernel stack #
            kernelTargetStack = self.kerSymData[ksym]["stack"]

            # First stack related to this symbol #
            if not kernelTargetStack:
                tempList = [0] * len(FunctionAnalyzer.symStackIdxTable)
                tempList[eventIndex] = eventCnt
                tempList[subStackIndex] = kernelStack
                tempList[argIndex] = list(subStackPageInfo)
                kernelTargetStack.append(tempList)

                kstackAddr = id(kernelStack)
            else:
                found = False
                for stackInfo in kernelTargetStack:
                    kerStackSet = set(kernelStack)
                    kerSubStackSet = set(stackInfo[subStackIndex])

                    # Found same stack in stack list #
                    if kerStackSet == kerSubStackSet:
                        found = True
                        stackInfo[eventIndex] += eventCnt
                        kstackAddr = id(stackInfo[subStackIndex])
                        break

                # New stack related to this symbol #
                if not found:
                    tempList = [0] * len(FunctionAnalyzer.symStackIdxTable)
                    tempList[eventIndex] = eventCnt
                    tempList[subStackIndex] = kernelStack
                    tempList[argIndex] = list(subStackPageInfo)
                    kernelTargetStack.append(tempList)

                    kstackAddr = id(kernelStack)

            # Recover PAGE_FREE count to merge with unknownPageFreeCnt #
            if event == "PAGE_FREE":
                eventCnt = savedEventCnt

            # memory allocation event #
            if event == "PAGE_ALLOC":
                pageType = arg[0]
                pfn = arg[1]
                atime = arg[2]

                self.handlePageAlloc(
                    sym,
                    ksym,
                    stackAddr,
                    kstackAddr,
                    eventCnt,
                    pageType,
                    pfn,
                    atime,
                )

            # memory free event #
            elif event == "PAGE_FREE":
                pageType = arg[0]
                pfn = arg[1]
                atime = arg[2]

                self.handlePageFree(
                    sym,
                    ksym,
                    stackAddr,
                    kstackAddr,
                    eventCnt,
                    pageType,
                    pfn,
                    atime,
                )

            # heap expand event #
            elif event == "HEAP_EXPAND":
                self.handleHeapExpand(
                    sym, ksym, stackAddr, kstackAddr, eventCnt, arg
                )

            # heap expand event #
            elif event == "HEAP_REDUCE":
                self.handleHeapReduce(eventCnt, arg)

            # block read event #
            elif event == "BLK_READ":
                self.userSymData[sym]["blockRdCnt"] += eventCnt
                self.kerSymData[ksym]["blockRdCnt"] += eventCnt

            # block write event #
            elif event == "BLK_WRITE":
                self.userSymData[sym]["blockWrCnt"] += eventCnt
                self.kerSymData[ksym]["blockWrCnt"] += eventCnt

            # lock try event #
            elif event == "LOCK_TRY":
                self.userSymData[sym]["lockTryCnt"] += eventCnt
                self.kerSymData[ksym]["lockTryCnt"] += eventCnt
                self.userFileData[path]["lockTryCnt"] += eventCnt

            # unlock event #
            elif event == "UNLOCK":
                self.userSymData[sym]["unlockCnt"] += eventCnt
                self.kerSymData[ksym]["unlockCnt"] += eventCnt
                self.userFileData[path]["unlockCnt"] += eventCnt

            # periodic event such as CPU tick #
            elif event == "CPU_TICK":
                self.userSymData[sym]["tickCnt"] += 1
                self.kerSymData[ksym]["tickCnt"] += 1
                self.userFileData[path]["tickCnt"] += 1

            # syscall event #
            elif event == "SYSCALL":
                self.userSymData[sym]["syscallCnt"] += 1
                self.kerSymData[ksym]["syscallCnt"] += 1
                self.userFileData[path]["syscallCnt"] += 1

            # periodic event such as CPU tick #
            elif event == "CUSTOM":
                if eventCnt > 0:
                    self.userSymData[sym]["customTotal"] += 1
                    self.kerSymData[ksym]["customTotal"] += 1
                    self.userFileData[path]["customTotal"] += 1

                self.userSymData[sym]["customCnt"] += eventCnt
                self.kerSymData[ksym]["customCnt"] += eventCnt
                self.userFileData[path]["customCnt"] += eventCnt

            # etc event #
            elif event == "IGNORE":
                try:
                    self.ignoreTable[arg]["ignCnt"] += 1
                except:
                    self.ignoreTable[arg] = {"ignCnt": long(1)}

            else:
                SysMgr.printWarn("failed to recognize event %s" % event)

        UtilMgr.deleteProgress()

        # Print summary about ignored events #
        self.printIgnoreEvents()

    def printIgnoreEvents(self):
        for idx, value in self.ignoreTable.items():
            SysMgr.printWarn(
                "ignore %s event %d times" % (idx, value["ignCnt"])
            )

    def getBinFromServer(self, remoteObj, src, des):
        if not remoteObj or remoteObj == "NONE":
            SysMgr.printErr(
                "wrong remote address, input in the format {IP:PORT}"
            )
            sys.exit(-1)

        # set download command #
        req = "DOWNLOAD:%s@%s" % (src, des)

        # get connection with server #
        self.connObj = NetworkMgr.getServerConn()
        if not self.connObj:
            return None

        # request download command #
        NetworkMgr.requestCmd(self.connObj, req)

    def getSymbols(self):
        binPath = ""
        offsetList = []
        curIdx = 0
        nrNoFile = 0
        lastIdx = len(self.posData)

        # Set alarm handler to handle hanged addr2line #
        signal.signal(signal.SIGALRM, SysMgr.timerHandler)

        # Get symbols and source pos #
        for idx, value in sorted(
            self.posData.items(), key=lambda e: e[1]["binary"], reverse=True
        ):
            curIdx += 1

            UtilMgr.printProgress(curIdx, lastIdx)

            # Handle thumbcode #
            if idx == "00c0ffee":
                value["binary"] = "??"
                value["origBin"] = "??"
                value["symbol"] = "ThumbCode"
                continue

            # Handle address #
            if value["binary"] == "":
                # user pos without offset #
                if value["symbol"] in ("", "??"):
                    # TODO: find binary path and symbol of pos #
                    value["binary"] = "??"
                    value["origBin"] = "??"
                    value["symbol"] = idx
                continue

            # Get symbols from address list of previous binary #
            if binPath != value["binary"]:
                if binPath != "":
                    # Get symbols #
                    if self.getFileSymbolInfo(binPath, offsetList) == -1:
                        nrNoFile += 1
                    offsetList = []

                if value["offset"] == hex(0):
                    offsetList.append(idx)
                else:
                    offsetList.append(value["offset"])

                # Set new binPath to find symbol from address #
                binPath = value["binary"]

                # Get binary from server #
                if not os.path.isfile(binPath) and SysMgr.remoteServObj:
                    self.getBinFromServer(
                        SysMgr.remoteServObj, value["origBin"], binPath
                    )
            # add address to offsetList #
            else:
                # not relocatable binary #
                if value["offset"] == hex(0):
                    offsetList.append(idx)
                # relocatable binary #
                else:
                    offsetList.append(value["offset"])

        if self.posData:
            UtilMgr.deleteProgress()

        # Get symbols and source path from last binary #
        if binPath != "":
            if self.getFileSymbolInfo(binPath, offsetList) == -1:
                nrNoFile += 1

        if nrNoFile > 0:
            SysMgr.printWarn(
                "failed to find total %s binaries to analyze functions"
                % nrNoFile,
                True,
            )

    def getFileSymbolInfo(self, binPath, offsetList, onlyFunc=True):
        def _updateSymbol(addr, symbol, src, relocated):
            if not addr:
                return -1
            elif symbol == "??":
                symbol = addr

            # Check whether the file is relocatable or not #
            if not relocated:
                try:
                    savedSymbol = self.posData[addr]["symbol"]
                except:
                    return -1

                """
                Check whether saved symbol found by
                previous addr2line is right #
                """
                if (
                    not savedSymbol
                    or savedSymbol in ("", addr)
                    or savedSymbol[0] == "$"
                ):
                    self.posData[addr]["symbol"] = symbol

                    if SysMgr.showAll:
                        self.posData[addr]["src"] = src
                    else:
                        fileIdx = src.rfind("/")
                        if fileIdx >= 0:
                            self.posData[addr]["src"] = src[fileIdx + 1 :]

                return

            inBinArea = False
            for idx, value in sorted(
                self.posData.items(),
                key=lambda e: e[1]["binary"],
                reverse=True,
            ):
                if value["binary"] == binPath:
                    inBinArea = True

                    if value["offset"] == addr:
                        savedSymbol = self.posData[idx]["symbol"]

                        if (
                            not savedSymbol
                            or savedSymbol in ("", addr)
                            or savedSymbol[0] == "$"
                        ):
                            self.posData[idx]["symbol"] = symbol

                            if SysMgr.showAll:
                                self.posData[idx]["src"] = src
                            else:
                                fileIdx = src.rfind("/")
                                if fileIdx >= 0:
                                    self.posData[idx]["src"] = src[
                                        fileIdx + 1 :
                                    ]

                            break
                elif inBinArea:
                    break

        # Recognize binary type #
        relocated = ElfAnalyzer.isRelocFile(binPath)

        # No file exist #
        if not os.path.isfile(binPath):
            for addr in offsetList:
                try:
                    if not relocated:
                        self.posData[addr]["symbol"] = "NoFile"
                        self.posData[addr]["src"] = "NoFile"
                    else:
                        for idx, value in sorted(
                            self.posData.items(),
                            key=lambda e: e[1]["binary"],
                            reverse=True,
                        ):
                            if value["binary"] == binPath and value[
                                "offset"
                            ] == hex(long(addr, 16)):
                                self.posData[idx]["symbol"] = "NoFile"
                                self.posData[idx]["src"] = "NoFile"
                                break
                except SystemExit:
                    sys.exit(0)
                except:
                    SysMgr.printWarn("failed to find address %s" % addr)
            return -1

        # check user-mode enabled #
        if not SysMgr.userEnable:
            return None

        # check addr2line path #
        if not "ADDR2LINE" in SysMgr.environList:
            try:
                symbolList = []
                binObj = ElfAnalyzer.getObject(binPath)
                if not binObj:
                    return None

                for offset in offsetList:
                    symbol, size = binObj.getSymbolByOffset(
                        offset, onlyFunc=onlyFunc
                    )

                    symbolList.append("??")

                    _updateSymbol(offset, symbol, "??", relocated)
            except SystemExit:
                sys.exit(0)
            except:
                SysMgr.printErr("failed to get symbol from %s" % binPath, True)

            return None

            # get addr2line path #
            addr2linePath = UtilMgr.which("addr2line")
            if not addr2linePath:
                SysMgr.printErr(
                    (
                        "failed to find addr2line to analyze functions, "
                        "use -q option with ADDR2LINE to set binary path"
                    )
                )
                sys.exit(-1)

            SysMgr.printInfo(
                "use '%s' as the addr2line path" % ", ".join(addr2linePath)
            )
        else:
            for path in SysMgr.environList["ADDR2LINE"]:
                if not os.path.isfile(path):
                    SysMgr.printErr(
                        "failed to find %s to use addr2line" % path
                    )
                    sys.exit(-1)

        # get subprocess object #
        subprocess = SysMgr.getPkg("subprocess")

        for path in SysMgr.environList["ADDR2LINE"]:
            # Set addr2line command #
            args = [path, "-C", "-f", "-a", "-e", binPath]

            # Limit the number of arguments to be passed because of ARG_MAX #
            # ARG_MAX = $(getconf PAGE_SIZE)*32 = 131072 #
            listLen = len(offsetList)
            maxArgLine = 256
            offset = 0
            timeout = 10

            # Get symbol by address of every maxArgLine elements in list #
            while offset < listLen:
                # Launch addr2line #
                try:
                    workload = offsetList[offset : offset + maxArgLine - 1]
                    proc = subprocess.Popen(
                        args + workload,
                        stdout=subprocess.PIPE,
                        stderr=subprocess.PIPE,
                    )
                except SystemExit:
                    sys.exit(0)
                except:
                    SysMgr.printErr(
                        "failed to execute %s to pick symbols from binary"
                        % path
                    )
                    sys.exit(-1)

                # Increase offset count in address list #
                offset += maxArgLine

                try:
                    # Set alarm to handle hanged addr2line #
                    signal.alarm(timeout)

                    # Wait for addr2line to finish its job #
                    proc.wait()

                    # Cancel alarm after addr2line respond #
                    signal.alarm(0)
                except SystemExit:
                    sys.exit(0)
                except:
                    SysMgr.printWarn(
                        "no response of addr2line for %s" % binPath
                    )
                    continue

                while 1:
                    # Get return of addr2line #
                    addr = (
                        proc.stdout.readline().decode().replace("\n", "")[2:]
                    )
                    try:
                        addr = hex(long(addr, 16)).rstrip("L")
                    except SystemExit:
                        sys.exit(0)
                    except:
                        pass

                    symbol = proc.stdout.readline().decode().replace("\n", "")
                    src = proc.stdout.readline().decode().replace("\n", "")

                    err = proc.stderr.readline().decode().replace("\n", "")
                    if err:
                        SysMgr.printWarn(err[err.find(":") + 2 :])

                    if _updateSymbol(addr, symbol, src, relocated):
                        break

    def initStacks(self):
        self.nowCtx["userLastPos"] = "0"
        self.nowCtx["userStack"] = []
        self.nowCtx["kerLastPos"] = "0"
        self.nowCtx["kerStack"] = []

    def swapEvents(self):
        tempEvent = self.nowCtx["nowEvent"]
        self.nowCtx["nowEvent"] = self.nowCtx["savedEvent"]
        self.nowCtx["savedEvent"] = tempEvent

        tempCnt = self.nowCtx["nowCnt"]
        self.nowCtx["nowCnt"] = self.nowCtx["savedCnt"]
        self.nowCtx["savedCnt"] = tempCnt

        tempArg = self.nowCtx["nowArg"]
        self.nowCtx["nowArg"] = self.nowCtx["savedArg"]
        self.nowCtx["savedArg"] = tempArg

    def saveFullStack(
        self,
        kernelPos,
        kernelStack,
        userPos,
        userStack,
        targetEvent,
        targetCnt,
        targetArg,
    ):
        # Save userstack #
        self.userCallData.append(
            [userPos, userStack, targetEvent, targetCnt, targetArg]
        )

        # Save kernelstack #
        self.kernelCallData.append(
            [kernelPos, kernelStack, targetEvent, targetCnt, targetArg]
        )

        # Save custom event stacks #
        if SysMgr.showAll and targetEvent == "CUSTOM":
            self.customCallData.append(
                [
                    targetArg[0],
                    targetArg[1],
                    self.userCallData[-1],
                    self.kernelCallData[-1],
                ]
            )

        # Save lock event stacks #
        if SysMgr.showAll and targetEvent in ("LOCK_TRY", "UNLOCK"):
            self.lockCallData.append(
                [
                    targetArg[0],
                    targetArg[1:],
                    self.userCallData[-1],
                    self.kernelCallData[-1],
                ]
            )

        # Save syscall event stacks #
        if SysMgr.showAll and targetEvent == "SYSCALL":
            self.sysCallData.append(
                [
                    targetArg[0],
                    targetArg[1:],
                    self.userCallData[-1],
                    self.kernelCallData[-1],
                ]
            )

    def saveEventStack(self, targetEvent, targetCnt, targetArg):
        kpos = self.nowCtx["kerLastPos"]
        upos = self.nowCtx["userLastPos"]

        # save count data #
        if targetEvent == "CPU_TICK":
            self.periodicEventCnt += 1

        elif targetEvent == "PAGE_ALLOC":
            self.pageAllocEventCnt += 1
            self.pageAllocCnt += targetCnt
            self.pageUsageCnt += targetCnt
            self.posData[kpos]["pageCnt"] += targetCnt
            self.posData[upos]["pageCnt"] += targetCnt

            pageType = targetArg[0]
            pfn = targetArg[1]
            time = targetArg[2]
            targetArg = [pageType, pfn, time]

        elif targetEvent == "PAGE_FREE":
            self.pageFreeEventCnt += 1
            self.pageFreeCnt += targetCnt

            pageType = targetArg[0]
            pfn = targetArg[1]
            time = targetArg[2]
            targetArg = [pageType, pfn, time]

        elif targetEvent == "BLK_READ":
            self.blockRdEventCnt += 1
            self.blockRdUsageCnt += targetCnt
            self.posData[kpos]["blockRdCnt"] += targetCnt
            self.posData[upos]["blockRdCnt"] += targetCnt

        elif targetEvent == "BLK_WRITE":
            self.blockWrEventCnt += 1
            self.blockWrUsageCnt += targetCnt
            self.posData[kpos]["blockWrCnt"] += targetCnt
            self.posData[upos]["blockWrCnt"] += targetCnt

        elif targetEvent == "LOCK_TRY":
            self.lockTryEventCnt += 1
            self.posData[kpos]["lockTryCnt"] += targetCnt
            self.posData[upos]["lockTryCnt"] += targetCnt

        elif targetEvent == "UNLOCK":
            self.unlockEventCnt += 1
            self.posData[kpos]["unlockCnt"] += targetCnt
            self.posData[upos]["unlockCnt"] += targetCnt

        elif targetEvent == "HEAP_EXPAND":
            self.heapExpEventCnt += 1
            self.heapExpSize += targetCnt
            self.posData[kpos]["heapSize"] += targetCnt
            self.posData[upos]["heapSize"] += targetCnt

        elif targetEvent == "HEAP_REDUCE":
            self.posData[kpos]["heapSize"] += targetCnt
            self.posData[upos]["heapSize"] += targetCnt

        elif targetEvent == "SYSCALL":
            nrSyscall = targetArg[0]
            self.syscallCnt += 1

            try:
                self.syscallTable[nrSyscall] += 0
            except:
                self.syscallTable[nrSyscall] = 1

            self.posData[kpos]["syscallCnt"] += targetCnt
            self.posData[upos]["syscallCnt"] += targetCnt

        elif targetEvent == "CUSTOM":
            if targetCnt > 0:
                self.customTotal += 1
                self.customCnt += targetCnt

                self.posData[kpos]["customTotal"] += 1
                self.posData[upos]["customTotal"] += 1

                self.posData[kpos]["customCnt"] += targetCnt
                self.posData[upos]["customCnt"] += targetCnt

        else:
            pass

        # cut stacks by depth #
        if SysMgr.funcDepth > 0:
            ksize = len(self.nowCtx["kerStack"])
            if ksize >= SysMgr.funcDepth:
                self.nowCtx["kerLastPos"] = self.nowCtx["kerStack"][
                    -SysMgr.funcDepth
                ]
                self.nowCtx["kerStack"] = self.nowCtx["kerStack"][
                    -SysMgr.funcDepth + 1 :
                ]

            usize = len(self.nowCtx["userStack"])
            if usize >= SysMgr.funcDepth:
                self.nowCtx["userLastPos"] = self.nowCtx["userStack"][
                    -SysMgr.funcDepth
                ]
                self.nowCtx["userStack"] = self.nowCtx["userStack"][
                    -SysMgr.funcDepth + 1 :
                ]

            if SysMgr.funcDepth == 1:
                self.nowCtx["kerStack"] = []
                self.nowCtx["userStack"] = []

        try:
            # save both stacks #
            self.saveFullStack(
                self.nowCtx["kerLastPos"],
                self.nowCtx["kerStack"],
                self.nowCtx["userLastPos"],
                self.nowCtx["userStack"],
                targetEvent,
                targetCnt,
                targetArg,
            )
        except:
            SysMgr.printErr("failed to save full stacks", True)
            sys.exit(-1)

    def saveCallStack(self):
        nowCtx = self.nowCtx

        # stack of kernel thread #
        if (
            not SysMgr.userRecordEnable
            or nowCtx["prevMode"] != nowCtx["curMode"] == "kernel"
        ):
            if not nowCtx["userStack"] and len(nowCtx["kerStack"]) > 0:
                # Set userLastPos to None #
                self.nowCtx["userLastPos"] = "0"
                self.nowCtx["userStack"].append("0")
            if not nowCtx["kerStack"] and len(nowCtx["userStack"]) > 0:
                # Set kerLastPos to None #
                self.nowCtx["kerLastPos"] = "0"
                self.nowCtx["kerStack"].append("0")

        # complicated situation ;( #
        elif nowCtx["prevMode"] == nowCtx["curMode"]:
            # previous user stack loss or nested interval #
            if nowCtx["curMode"] == "kernel":
                # nested interval #
                if nowCtx["nowEvent"] == "CPU_TICK":
                    # Backup kernel stack #
                    self.nowCtx["prevKerLastPos"] = nowCtx["kerLastPos"]
                    self.nowCtx["prevKerStack"] = nowCtx["kerStack"]

                    # Initialize both stacks #
                    self.initStacks()
                # previous user stack loss #
                else:
                    # Set userLastPos to None #
                    self.nowCtx["userLastPos"] = "0"
                    self.nowCtx["userStack"].append("0")
            # nested interval #
            elif nowCtx["curMode"] == "user":
                """
                CORE/0 EVENT0
                CORE/0 <kernel>
                CORE/0 <user>

                CORE/0 EVENT1
                CORE/0 <kernel>
                    CORE/0 EVENT2
                    CORE/0 <kernel>
                    CORE/0 <user>
                CORE/0 <user>
                """
                # Swap nowEvent and savedEvent #
                self.swapEvents()

        """
        Save both stacks of previous event before
        starting to record new kernel stack #
        """
        if (len(nowCtx["userStack"]) > 0 and nowCtx["userLastPos"] != "") and (
            len(nowCtx["kerStack"]) > 0 and nowCtx["kerLastPos"] != ""
        ):
            # Remove pc in each stacks #
            del self.nowCtx["kerStack"][0], self.nowCtx["userStack"][0]

            # Check whether there is nested event or not #
            if nowCtx["nested"] > 0:
                """
                CORE/0 EVENT0
                CORE/0 <kernel>
                CORE/0 <user>

                CORE/0 EVENT1
                    CORE/0 EVENT2
                    CORE/0 <kernel>
                    CORE/0 <user>
                CORE/0 <kernel>
                CORE/0 <user>
                """
                targetEvent = nowCtx["nestedEvent"]
                targetCnt = nowCtx["nestedCnt"]
                targetArg = nowCtx["nestedArg"]

                # Swap nowEvent and savedEvent #
                self.swapEvents()
            else:
                targetEvent = nowCtx["savedEvent"]
                targetCnt = nowCtx["savedCnt"]
                targetArg = nowCtx["savedArg"]

            # Save full stack of previous event #
            self.saveEventStack(targetEvent, targetCnt, targetArg)

            # Recover previous kernel stack after handling nested event #
            if (
                nowCtx["prevMode"] == nowCtx["curMode"] == "user"
                and nowCtx["prevKerLastPos"] != "0"
            ):
                self.nowCtx["kerLastPos"] = nowCtx["prevKerLastPos"]
                self.nowCtx["kerStack"] = nowCtx["prevKerStack"]
                self.nowCtx["prevKerLastPos"] = "0"
                self.nowCtx["prevKerStack"] = []
            else:
                self.nowCtx["kerLastPos"] = ""
                self.nowCtx["kerStack"] = []

            # Initialize user stack #
            self.nowCtx["userLastPos"] = ""
            self.nowCtx["userStack"] = []
            self.nowCtx["nestedEvent"] = ""
            self.nowCtx["nestedCnt"] = 0

        # On stack recording switch #
        self.nowCtx["recStat"] = True

    def savePosData(self, pos, path, offset):
        if self.nowCtx["nested"] > 0:
            targetEvent = self.nowCtx["savedEvent"]
        else:
            targetEvent = self.nowCtx["nowEvent"]

        # Register pos #
        try:
            self.posData[pos]
            if (
                path
                and path[0] == "/"
                and path != self.posData[pos]["origBin"]
            ):
                self.duplicatedPos += 1
                """
                SysMgr.printWarn(
                    "duplicated address %s in both '%s' and '%s'" % \
                    (pos, path, self.posData[pos]['origBin']))
                """
        except:
            self.posData[pos] = dict(self.init_posData)

        # user mode #
        if self.nowCtx["curMode"] == "user":
            # Set path #
            if path:
                self.posData[pos]["origBin"] = path
                self.posData[pos]["binary"] = SysMgr.rootPath + path
                self.posData[pos]["binary"] = os.path.normpath(
                    self.posData[pos]["binary"]
                )

                # Set offset #
                if offset:
                    if ElfAnalyzer.isRelocFile(path):
                        self.posData[pos]["offset"] = offset

            # Save pos #
            if not self.nowCtx["userStack"]:
                self.nowCtx["userLastPos"] = pos

                if targetEvent == "CPU_TICK":
                    self.posData[pos]["posCnt"] += 1
                elif targetEvent == "LOCK_TRY":
                    self.posData[pos]["lockTryCnt"] += 1
                elif targetEvent == "UNLOCK":
                    self.posData[pos]["unlockCnt"] += 1

            self.nowCtx["userStack"].append(pos)
        # kernel mode #
        elif self.nowCtx["curMode"] == "kernel":
            # Save pos #
            if not self.nowCtx["kerStack"]:
                self.nowCtx["kerLastPos"] = pos

                if targetEvent == "CPU_TICK":
                    self.posData[pos]["posCnt"] += 1
                elif targetEvent == "LOCK_TRY":
                    self.posData[pos]["lockTryCnt"] += 1
                elif targetEvent == "UNLOCK":
                    self.posData[pos]["unlockCnt"] += 1

            # Skip pos because it is usercall or no symbol #
            elif not SysMgr.showAll and not path:
                return

            self.posData[pos]["symbol"] = path

            self.nowCtx["kerStack"].append(pos)

        # wrong mode #
        else:
            SysMgr.printWarn("wrong current mode %s" % self.nowCtx["curMode"])

        # Increase total call count #
        if self.nowEvent == "CPU_TICK":
            self.posData[pos]["totalCnt"] += 1

    def allocHeapSeg(self, tid, size):
        try:
            self.heapTable[tid + "-ready"]["size"] = size
            self.heapTable[tid + "-ready"]["tid"] = tid
            self.threadData[tid]["heapSize"] -= size
            SysMgr.printWarn(
                "overwrite heap segment of %s(%s) at %s"
                % (self.threadData[tid]["comm"], tid, SysMgr.dbgEventLine)
            )
        except:
            self.heapTable[tid + "-ready"] = dict(self.init_heapSegData)
            self.heapTable[tid + "-ready"]["size"] = size
            self.heapTable[tid + "-ready"]["tid"] = tid

    def freeHeapSeg(self, addr):
        try:
            self.heapRedEventCnt += 1
            self.heapRedSize += self.heapTable[addr]["size"]

            # get tid #
            try:
                tid = self.heapTable[addr]["tid"]
            except:
                return

            self.threadData[tid]["heapSize"] -= self.heapTable[addr]["size"]

            self.heapTable.pop(addr, None)
        except:
            SysMgr.printWarn(
                "failed to free heap segment %s of %s(%s) at %s"
                % (
                    addr,
                    self.threadData[tid]["comm"],
                    tid,
                    SysMgr.dbgEventLine,
                )
            )

    def setHeapSegAddr(self, tid, addr):
        try:
            self.heapTable[addr] = dict(self.heapTable["%s-ready" % tid])
            del self.heapTable["%s-ready" % tid]
        except:
            SysMgr.printWarn(
                "failed to set address of heap segment %s of %s(%s) at %s"
                % (
                    addr,
                    self.threadData[tid]["comm"],
                    tid,
                    SysMgr.dbgEventLine,
                )
            )

    def parseLines(self, lines, desc):
        curIdx = 0
        lastIdx = len(lines)

        # make custom event table #
        if SysMgr.customCmd:
            for cmd in SysMgr.customCmd:
                cmd = cmd.split(":")

                if len(cmd) > 1:
                    self.customEventTable[cmd[0]] = cmd[1]
                else:
                    self.customEventTable[cmd[0]] = None

        # make kernel event table #
        if SysMgr.kernelCmd:
            for cmd in SysMgr.kernelCmd:
                cmd = cmd.split(":")
                self.customEventTable[cmd[0] + "_enter"] = None
                self.customEventTable[cmd[0] + "_exit"] = None

        # make user event table #
        if SysMgr.userCmd:
            for cmd in SysMgr.userCmd:
                cmd = cmd.split(":")
                self.customEventTable[cmd[0] + "_enter"] = None
                self.customEventTable[cmd[0] + "_exit"] = None

        # get pid filter by comm in advance #
        plist = {}
        if SysMgr.groupProcEnable:
            for key, value in self.getTargetList(lines).items():
                for item in desc:
                    if item in value["comm"]:
                        plist[value["tgid"]] = 0

        # start parsing logs #
        for liter in lines:
            curIdx += 1
            SysMgr.logSize += len(liter)
            SysMgr.curLine += 1
            SysMgr.dbgEventLine += 1

            ret = self.parse(liter, desc, plist)

            UtilMgr.printProgress(curIdx, lastIdx)

            # Skip lines before first meaningful event #
            if not self.lastCore:
                continue

            # Set context of current core #
            self.nowCtx = self.coreCtx[self.lastCore]

            # Save full stack to callData table #
            if ret is True:
                self.saveCallStack()
            elif ret is False:
                """
                Ignore this log because its not event or
                stack info related to target thread #
                """
                self.nowCtx["recStat"] = False
                continue
            # Save pos into target stack #
            elif self.nowCtx["recStat"]:
                # decode return value #
                (pos, path, offset) = ret

                self.savePosData(pos, path, offset)

        if lines:
            UtilMgr.deleteProgress()

        # update finish time #
        if self.finishTime == "0":
            self.finishTime = self.lastTime

        # Save stack of last events per core #
        for idx in list(self.coreCtx):
            self.lastCore = idx
            self.nowCtx = self.coreCtx[idx]

            # Recover previous mode #
            if SysMgr.userEnable:
                self.nowCtx["prevMode"] = "user"
            self.nowCtx["curMode"] = "kernel"

            self.saveEventParam("IGNORE", 0, 0)
            self.nowCtx["nested"] -= 1
            self.saveCallStack()

        if self.duplicatedPos > 0:
            SysMgr.printWarn(
                "found %d addresses duplicated" % self.duplicatedPos
            )

    def getCustomEventValue(self, args, cond):
        if not cond:
            return 1

        # set condition #
        if ">" in cond:
            condVal = cond[cond.find(">") + 1 :]
            condOp = ">"
            condStr = cond[: cond.find(">")]
        elif "<" in cond:
            condVal = cond[cond.find("<") + 1 :]
            condOp = "<"
            condStr = cond[: cond.find("<")]
        elif "==" in cond:
            condVal = cond[cond.find("==") + 2 :]
            condOp = "=="
            condStr = cond[: cond.find("==")]
        else:
            condStr = cond
            condOp = None
            condVal = None

        m = re.match(r"^.+%s=(?P<value>\S+)" % condStr, args)
        if not m:
            return 0

        d = m.groupdict()

        value = d["value"]

        if not condOp and value:
            try:
                return long(value)
            except:
                return 0
        elif condOp == ">":
            try:
                if long(value) > long(condVal):
                    return long(value)
            except:
                pass

            return 0
        elif condOp == "<":
            try:
                if long(value) < long(condVal):
                    return long(value)
            except:
                pass

            return 0
        elif condOp == "==":
            if value == condVal:
                return 1
            else:
                return 0
        else:
            return 0

    def saveEventParam(self, event, count, arg):
        # save context #
        self.nowCtx["nestedEvent"] = self.nowCtx["savedEvent"]
        self.nowCtx["savedEvent"] = self.nowCtx["nowEvent"]
        self.nowCtx["nowEvent"] = event

        self.nowCtx["nestedCnt"] = self.nowCtx["savedCnt"]
        self.nowCtx["savedCnt"] = self.nowCtx["nowCnt"]
        self.nowCtx["nowCnt"] = count

        self.nowCtx["nestedArg"] = self.nowCtx["savedArg"]
        self.nowCtx["savedArg"] = self.nowCtx["nowArg"]
        self.nowCtx["nowArg"] = arg

        self.nowCtx["nested"] += 1

        if self.nowCtx["nested"] > 2:
            # self.printDbgInfo()
            SysMgr.printWarn(
                (
                    "failed to analyze stack data "
                    "because of corruption (overflow) at %s\n"
                    "\tso report the results may differ from actual"
                )
                % SysMgr.dbgEventLine,
                True,
            )

    def printDbgInfo(self):
        data = self.nowCtx

        print(
            "[%s]" % self.lastCore,
            "(now) %s/%s/%s"
            % (data["nowEvent"], data["nowCnt"], data["nowArg"]),
            "(saved) %s/%s/%s"
            % (data["savedEvent"], data["savedCnt"], data["savedArg"]),
            "(nested) %s/%s/%s"
            % (data["nestedEvent"], data["nestedCnt"], data["nestedArg"]),
            "(user) %s/%s" % (data["userLastPos"], len(data["userStack"])),
            "(kernel) %s/%s" % (data["kerLastPos"], len(data["kerStack"])),
            "(backup) %s/%s"
            % (data["prevKerLastPos"], len(data["prevKerStack"])),
            "at %s" % SysMgr.dbgEventLine,
        )

    def parseEventInfo(self, tid, func, args, time, core):
        # check core filter #
        if (
            SysMgr.perCoreList
            and long(core) not in SysMgr.perCoreList
            and func[0] != "<"
        ):
            self.saveEventParam("IGNORE", 0, func[:-1])
            return False

        # check fixed event list #
        if self.customEventTable and (
            func[:-1] in self.customEventTable
            or len(
                [
                    event
                    for event in self.customEventTable
                    if event.endswith(func[:-1])
                ]
            )
            > 0
        ):
            isFixedEvent = False
        else:
            isFixedEvent = True

        # CPU tick event #
        # TODO: find shorter periodic event for sampling #
        if isFixedEvent and func == "hrtimer_start:":
            if "tick_sched_timer" in args:
                self.cpuEnabled = True

                self.saveEventParam("CPU_TICK", 1, 0)
            else:
                self.saveEventParam("IGNORE", 0, func[:-1])

            return False

        # memory allocation event #
        elif isFixedEvent and func == "mm_page_alloc:":
            m = re.match(
                (
                    r"^\s*page=\s*(?P<page>\S+)\s+pfn=(?P<pfn>[0-9]+)\s+"
                    r"order=(?P<order>[0-9]+)\s+migratetype=(?P<mt>[0-9]+)\s+"
                    r"gfp_flags=(?P<flags>\S+)"
                ),
                args,
            )
            if m:
                d = m.groupdict()

                # check whether it is huge page #
                if d["page"] == "(null)":
                    page = "huge"
                else:
                    page = d["page"]

                pfn = long(d["pfn"])
                flags = d["flags"]
                pageCnt = pow(2, long(d["order"]))

                # Increase page count of thread #
                self.threadData[tid]["nrPages"] += pageCnt

                # Increase page counts of thread #
                pageType = None
                if (
                    "NOFS" in flags
                    or "GFP_WRITE" in flags
                    or "0x1000000" in flags
                ):
                    pageType = "CACHE"
                    self.threadData[tid]["cachePages"] += pageCnt
                elif "USER" in flags:
                    pageType = "USER"
                    self.threadData[tid]["userPages"] += pageCnt
                else:
                    pageType = "KERNEL"
                    self.threadData[tid]["kernelPages"] += pageCnt

                # Make PTE in page table #
                for cnt in xrange(pageCnt):
                    pfnv = pfn + cnt

                    try:
                        """
                        Decrease page count of it's owner \
                        because this page was already allocated but no free log
                        """

                        ownerTid = self.pageTable[pfnv]["tid"]
                        self.threadData[ownerTid]["nrPages"] -= 1

                        origPageType = self.pageTable[pfnv]["type"]
                        if origPageType == "USER":
                            self.threadData[ownerTid]["userPages"] -= 1
                        elif origPageType == "CACHE":
                            self.threadData[ownerTid]["cachePages"] -= 1
                        elif origPageType == "KERNEL":
                            self.threadData[ownerTid]["kernelPages"] -= 1
                    except:
                        self.pageTable[pfnv] = dict(self.init_pageData)

                    self.pageTable[pfnv]["tid"] = tid
                    self.pageTable[pfnv]["page"] = page
                    self.pageTable[pfnv]["flags"] = flags
                    self.pageTable[pfnv]["type"] = pageType
                    self.pageTable[pfnv]["time"] = time

                self.memEnabled = True

                self.saveEventParam(
                    "PAGE_ALLOC", pageCnt, [pageType, pfn, time]
                )
            else:
                self.saveEventParam("IGNORE", 0, func[:-1])

                SysMgr.printWarn(
                    "failed to recognize event %s at %d"
                    % (func[:-1], SysMgr.dbgEventLine)
                )

            return False

        # memory free event #
        elif isFixedEvent and func in (
            "mm_page_free:",
            "mm_page_free_direct:",
        ):
            m = re.match(
                (
                    r"^\s*page=(?P<page>\S+)\s+pfn=(?P<pfn>[0-9]+)\s+"
                    r"order=(?P<order>[0-9]+)"
                ),
                args,
            )
            if m:
                d = m.groupdict()

                page = d["page"]
                pfn = long(d["pfn"])
                pageCnt = pow(2, long(d["order"]))

                # Update page table #
                origPageType = None
                for cnt in xrange(pageCnt):
                    pfnv = pfn + cnt

                    try:
                        owner = self.pageTable[pfnv]["tid"]
                        origPageType = self.pageTable[pfnv]["type"]

                        self.threadData[owner]["nrPages"] -= 1

                        if origPageType == "CACHE":
                            self.threadData[owner]["cachePages"] -= 1
                        elif origPageType == "USER":
                            self.threadData[owner]["userPages"] -= 1
                        elif origPageType == "KERNEL":
                            self.threadData[owner]["kernelPages"] -= 1

                        self.threadData[tid]["nrKnownFreePages"] += 1

                        self.pageTable.pop(pfnv)
                    except:
                        # this page was allocated before starting profile #

                        self.threadData[tid]["nrUnknownFreePages"] += 1
                        continue

                self.memEnabled = True

                self.saveEventParam(
                    "PAGE_FREE", pageCnt, [origPageType, pfn, time]
                )

                return False

            SysMgr.printWarn(
                "failed to recognize event %s at %d"
                % (func[:-1], SysMgr.dbgEventLine)
            )

            self.saveEventParam("IGNORE", 0, func[:-1])

            return False

        # syscall / heap / lock events #
        elif isFixedEvent and func == "sys_enter:":
            m = re.match(r"^\s*NR (?P<nr>[0-9]+) (?P<args>.+)", args)
            if m:
                b = m.groupdict()

                num = long(b["nr"])

                self.threadData[tid]["lastNrSyscall"] = num

                # syscall event #
                if SysMgr.sysEnable:
                    self.sysEnabled = True

                    syscallList = SysMgr.syscallList

                    if not syscallList or num in syscallList:
                        args = b["args"][1:-1]

                        self.threadData[tid]["nrSyscall"] += 1

                        # set syscall table #
                        if not self.threadData[tid]["syscallTable"]:
                            self.threadData[tid]["syscallTable"] = {}

                        try:
                            self.threadData[tid]["syscallTable"][num] += 1
                        except:
                            self.threadData[tid]["syscallTable"][num] = 1

                        self.saveEventParam(
                            "SYSCALL", 1, [num, args, time, core, tid]
                        )

                        return False

                # heap increasement event #
                elif num == ConfigMgr.getMmapId():
                    self.heapEnabled = True

                    try:
                        size = long(b["args"].split(",")[1], 16)

                        # just brk call to check data segment address #
                        if size == 0:
                            pass

                        self.threadData[tid]["heapSize"] += size
                    except:
                        self.saveEventParam("IGNORE", 0, func[:-1])

                        return False

                    # make heap segment tid-ready #
                    self.allocHeapSeg(tid, size)

                    self.saveEventParam("IGNORE", 0, func[:-1])

                    return False

                # heap decreasement event #
                elif num == SysMgr.getNrSyscall("sys_munmap"):
                    self.heapEnabled = True

                    try:
                        addr = long(b["args"][1:].split(",")[0], 16)
                        size = self.heapTable[addr]["size"]

                        # remove heap segment #
                        self.freeHeapSeg(addr)

                        self.saveEventParam(
                            "HEAP_REDUCE", size, [addr, time, core, tid]
                        )

                        return False
                    except:
                        pass

                # lock event #
                elif num == SysMgr.getNrSyscall("sys_futex"):
                    n = re.match(
                        (
                            r"^\s*(?P<uaddr>\S+), (?P<op>\S+), "
                            r"(?P<val>\S+), (?P<timer>\S+),"
                        ),
                        b["args"],
                    )
                    if n:
                        l = n.groupdict()

                        FUTEX_CMD_MASK = ~(128 | 256)
                        # FUTEX_PRIVATE_FLAG: 128, FUTEX_CLOCK_REALTIME: 256 #
                        maskedOp = long(l["op"], base=16) & FUTEX_CMD_MASK

                        addr = l["uaddr"][1:]

                        try:
                            event = ConfigMgr.FUTEX_TYPE[maskedOp]
                        except:
                            event = "LOCK"

                        # lock_pi #
                        if event in ("FUTEX_LOCK_PI", "FUTEX_TRYLOCK_PI"):
                            self.lockEnabled = True
                            self.threadData[tid]["nrLockTry"] += 1
                            self.saveEventParam(
                                "LOCK_TRY", 1, [event, addr, time, core, tid]
                            )
                            return False
                        # wait #
                        elif event in (
                            "FUTEX_WAIT",
                            "FUTEX_WAIT_REQUEUE_PI",
                            "FUTEX_WAIT_BITSET",
                        ):
                            self.lockEnabled = True
                            self.threadData[tid]["nrLockTry"] += 1
                            self.saveEventParam(
                                "LOCK_TRY", 1, [event, addr, time, core, tid]
                            )
                            return False
                        # unlock #
                        elif event == "FUTEX_UNLOCK_PI":
                            self.lockEnabled = True
                            self.threadData[tid]["nrUnlock"] += 1
                            self.saveEventParam(
                                "UNLOCK", 1, [event, addr, time, core, tid]
                            )
                            return False

            else:
                SysMgr.printWarn(
                    "failed to recognize event %s at %d"
                    % (func[:-1], SysMgr.dbgEventLine)
                )

            self.saveEventParam("IGNORE", 0, func[:-1])

            return False

        # syscall / heap events #
        elif isFixedEvent and func == "sys_exit:":
            m = re.match(r"^\s*NR (?P<nr>\S+) = (?P<ret>.+)", args)
            if m:
                b = m.groupdict()

                nr = long(b["nr"])

                # handle wrong syscall number #
                if nr < 0:
                    if self.threadData[tid]["lastNrSyscall"] >= 0:
                        nr = self.threadData[tid]["lastNrSyscall"]

                # heap increasement event #
                if nr == ConfigMgr.getMmapId():
                    self.heapEnabled = True

                    addr = long(b["ret"])

                    # rename heap segment from tid-ready to addr #
                    self.setHeapSegAddr(tid, addr)

                    try:
                        size = self.heapTable[addr]["size"]

                        self.saveEventParam(
                            "HEAP_EXPAND", size, [addr, time, core, tid]
                        )

                        return False
                    except:
                        pass

                # heap decreasement event #
                elif nr == SysMgr.getNrSyscall("sys_brk"):
                    self.heapEnabled = True

                    addr = long(b["ret"])

                    try:
                        pid = self.threadData[tid]["tgid"]
                        if pid.startswith("-"):
                            pid = SysMgr.savedProcTree[tid]
                        self.threadData[pid]
                    except:
                        pid = tid

                    try:
                        self.threadData[pid]["lastBrk"]

                        if addr > self.threadData[pid]["lastBrk"]:
                            size = addr - self.threadData[pid]["lastBrk"]

                            self.threadData[pid]["heapSize"] += size

                            self.saveEventParam(
                                "HEAP_EXPAND", size, [addr, time, core, tid]
                            )

                            return False
                    except:
                        self.threadData[pid]["lastBrk"] = addr
            else:
                SysMgr.printWarn(
                    "failed to recognize event %s at %d"
                    % (func[:-1], SysMgr.dbgEventLine)
                )

            self.saveEventParam("IGNORE", 0, func[:-1])

            return False

        # block request event #
        elif isFixedEvent and func == "block_bio_queue:":
            m = re.match(
                (
                    r"^\s*(?P<major>[0-9]+),(?P<minor>[0-9]+)\s*"
                    r"(?P<operation>\S+)\s*(?P<address>\S+)\s+\+\s+"
                    r"(?P<size>[0-9]+)"
                ),
                args,
            )
            if m:
                b = m.groupdict()

                """
                The operation to specific virtual devices such like
                device-mapper is remapped to another device by block_bio_queue
                event. It will probably make the workload look multiple times.
                """

                opt = b["operation"]

                if opt[0] == "R":
                    self.breadEnabled = True

                    blockRdCnt = long(b["size"])
                    self.threadData[tid]["nrRdBlocks"] += blockRdCnt

                    self.saveEventParam("BLK_READ", blockRdCnt, 0)

                    return False
                elif opt == "WS":
                    self.bwriteEnabled = True

                    blockWrCnt = long(b["size"])
                    self.threadData[tid]["nrWrBlocks"] += blockWrCnt

                    self.saveEventParam("BLK_WRITE", blockWrCnt, 0)

                    return False

            SysMgr.printWarn(
                "failed to recognize event %s at %d"
                % (func[:-1], SysMgr.dbgEventLine)
            )

            self.saveEventParam("IGNORE", 0, func[:-1])

            return False

        # block write request event #
        elif isFixedEvent and func == "writeback_dirty_page:":
            m = re.match(
                (
                    r"^\s*bdi\s+(?P<major>[0-9]+):(?P<minor>[0-9]+):\s*"
                    r"ino=(?P<ino>\S+)\s+index=(?P<index>\S+)"
                ),
                args,
            )
            if m:
                b = m.groupdict()
                self.bwriteEnabled = True

                self.threadData[tid]["nrWrBlocks"] += 8

                self.saveEventParam("BLK_WRITE", 8, 0)

                return False

            self.saveEventParam("IGNORE", 0, func[:-1])

            return False

        # block write request event #
        elif isFixedEvent and func == "wbc_writepage:":
            m = re.match(
                (
                    r"^\s*bdi\s+(?P<major>[0-9]+):(?P<minor>[0-9]+):\s*"
                    r"towrt=(?P<towrt>\S+)\s+skip=(?P<skip>\S+)"
                ),
                args,
            )
            if m:
                d = m.groupdict()

                if d["skip"] == "0":
                    self.bwriteEnabled = True

                    self.threadData[tid]["nrWrBlocks"] += 8

                    self.saveEventParam("BLK_WRITE", 8, 0)

                    return False

            self.saveEventParam("IGNORE", 0, func[:-1])

            return False

        # segmentation fault generation event #
        elif isFixedEvent and func == "signal_generate:":
            m = re.match(
                (
                    r"^\s*sig=(?P<sig>[0-9]+) errno=(?P<err>[0-9]+) "
                    r"code=(?P<code>.*) comm=(?P<comm>.*) "
                    r"pid=(?P<pid>[0-9]+)"
                ),
                args,
            )
            if m:
                b = m.groupdict()

                if b["sig"] == str(signal.SIGSEGV):
                    self.sigEnabled = True

                    self.saveEventParam("SIGSEGV_GEN", 0, 0)

                    return False

            SysMgr.printWarn(
                "failed to recognize event %s at %d"
                % (func[:-1], SysMgr.dbgEventLine)
            )

            self.saveEventParam("IGNORE", 0, func[:-1])

            return False

        elif isFixedEvent and func == "signal_deliver:":
            m = re.match(
                (
                    r"^\s*sig=(?P<sig>[0-9]+) errno=(?P<err>[0-9]+) "
                    r"code=(?P<code>.*) sa_handler=(?P<handler>.*) "
                    r"sa_flags=(?P<flags>.*)"
                ),
                args,
            )
            if m:
                b = m.groupdict()

                if b["sig"] == str(signal.SIGSEGV):
                    self.sigEnabled = True

                    self.saveEventParam("SIGSEGV_DLV", 0, 0)
                else:
                    self.saveEventParam("IGNORE", 0, func[:-1])
            else:
                self.saveEventParam("IGNORE", 0, func[:-1])

            return False

        elif isFixedEvent and func == "locks_get_lock_context:":
            m = re.match(
                (
                    r"^\s*dev=(?P<dev>.+)\s+ino=(?P<ino>.+)"
                    r"\s+type=(?P<type>.+)\s+ctx=(?P<ctx>.+)"
                ),
                args,
            )
            if m:
                d = m.groupdict()
                if d["type"] == "F_UNLCK":
                    self.saveEventParam("IGNORE", 0, func[:-1])
                else:
                    self.lockEnabled = True

                    self.threadData[tid]["nrLockTry"] += 1

                    self.saveEventParam(
                        "LOCK_TRY", 1, ["FLOCK", d["ino"], time, core, tid]
                    )
            else:
                self.saveEventParam("IGNORE", 0, func[:-1])

            return False

        # Start to record user stack #
        elif func == "<user":
            self.nowCtx["prevMode"] = self.nowCtx["curMode"]
            self.nowCtx["curMode"] = "user"

            return True

        # Start to record kernel stack #
        elif func == "<stack":
            self.nowCtx["prevMode"] = self.nowCtx["curMode"]
            self.nowCtx["curMode"] = "kernel"
            self.nowCtx["nested"] -= 1

            if self.nowCtx["nested"] < 0:
                # self.printDbgInfo()
                SysMgr.printWarn(
                    (
                        "failed to analyze stack data "
                        "because of corruption (underflow) at %s\n"
                        "\tso report the results may differ from actual"
                    )
                    % SysMgr.dbgEventLine,
                    True,
                )

            return True

        # custom event #
        elif not isFixedEvent:
            try:
                if not [
                    event
                    for event in self.customEventTable
                    if event.endswith(func[:-1])
                ]:
                    cond = self.customEventTable[func[:-1]] = None
                else:
                    cond = self.customEventTable[func[:-1]]

                # set event filter #
                customCnt = self.getCustomEventValue(args, cond)

                if customCnt > 0:
                    self.threadData[tid]["customTotal"] += customCnt

                self.saveEventParam(
                    "CUSTOM", customCnt, [func[:-1], [args, time, core, tid]]
                )
            except:
                self.saveEventParam("IGNORE", 0, func[:-1])

            return False

        # Ignore event #
        else:
            self.saveEventParam("IGNORE", 0, func[:-1])

            return False

    def getTargetList(self, tlist):
        threadData = {}

        for liter in tlist:
            m = SysMgr.getTraceItem(liter)
            if m:
                d = m.groupdict()

                # Make thread entity #
                thread = d["thread"]
                try:
                    threadData[thread]["comm"] = d["comm"]
                except:
                    threadData[thread] = {}
                    threadData[thread]["comm"] = d["comm"]

                # set tgid #
                try:
                    tgid = d["tgid"]
                except:
                    try:
                        tgid = SysMgr.savedProcTree[thread]
                    except:
                        continue

                threadData[thread]["tgid"] = tgid

        return threadData

    def parse(self, string, desc, plist=[]):
        m = SysMgr.getTraceItem(string)
        if m:
            d = m.groupdict()

            self.lastTime = d["time"]

            if (
                SysMgr.countEnable
                and SysMgr.repeatCnt * SysMgr.intervalEnable
                <= float(d["time"]) - float(SysMgr.startTime)
            ):
                self.lastCore = None
                return False

            # Make thread entity #
            thread = d["thread"]
            try:
                self.threadData[thread]["comm"] = d["comm"]
            except:
                self.threadData[thread] = dict(self.init_threadData)
                self.threadData[thread]["comm"] = d["comm"]

            # set tgid #
            tgid = None
            try:
                if d["tgid"].startswith("-"):
                    raise Exception("no tgid")
                tgid = d["tgid"]
            except:
                try:
                    tgid = SysMgr.savedProcTree[thread]
                except:
                    pass
            if tgid:
                self.threadData[thread]["tgid"] = tgid

            # increase event count #
            self.threadData[thread]["eventCnt"] += 1

            # set current core #
            self.lastCore = d["core"]

            # Make core entity #
            try:
                self.coreCtx[self.lastCore]
            except:
                self.coreCtx[self.lastCore] = dict(self.init_ctxData)
                self.coreCtx[self.lastCore]["userStack"] = []
                self.coreCtx[self.lastCore]["kerStack"] = []
                self.coreCtx[self.lastCore]["prevKerStack"] = []

            # set context of current core #
            self.nowCtx = self.coreCtx[self.lastCore]

            # Check core filter #
            if (
                SysMgr.perCoreList
                and long(d["core"]) not in SysMgr.perCoreList
                and not d["func"].startswith("tracing_mark_write")
                and d["func"] != "0:"
            ):
                pass

            # Calculate a total of CPU usage #
            elif (
                d["func"] == "hrtimer_start:"
                and "tick_sched_timer" in d["etc"]
            ):
                self.totalTick += 1
                self.threadData[thread]["cpuTick"] += 1

                # Set global interval #
                if self.nowCtx["prevTid"]:
                    diff = float(d["time"]) - float(self.nowCtx["prevTime"])
                    self.periodicEventInterval += diff
                    self.periodicContEventCnt += 1

                self.nowCtx["prevTid"] = thread
                self.nowCtx["prevTime"] = d["time"]

                # Set max core to calculate CPU usage of thread #
                if SysMgr.maxCore < long(d["core"]):
                    SysMgr.maxCore = long(d["core"])

            # Mark die flag of thread that is not able to be profiled #
            elif d["func"] == "sched_process_exit:":
                m = re.match(
                    r"^\s*comm=(?P<comm>.*)\s+pid=(?P<pid>[0-9]+)", d["etc"]
                )
                if m:
                    p = m.groupdict()

                    pid = p["pid"]

                    try:
                        self.threadData[pid]
                    except:
                        self.threadData[pid] = dict(self.init_threadData)
                        self.threadData[pid]["comm"] = p["comm"]

                    self.threadData[pid]["die"] = True

            # Make thread name #
            elif d["func"] == "sched_process_fork:":
                m = re.match(
                    (
                        r"^\s*comm=(?P<comm>.*)\s+pid=(?P<pid>[0-9]+)\s+"
                        r"child_comm=(?P<child_comm>.*)\s+"
                        r"child_pid=(?P<child_pid>[0-9]+)"
                    ),
                    d["etc"],
                )
                if m:
                    p = m.groupdict()

                    cpid = p["child_pid"]
                    ccomm = p["child_comm"]

                    try:
                        self.threadData[cpid]
                    except:
                        self.threadData[cpid] = dict(self.init_threadData)
                        self.threadData[cpid]["comm"] = ccomm
                        self.threadData[cpid]["new"] = True
                        self.threadData[cpid]["ptid"] = thread

            # Make thread name #
            elif d["func"] == "task_newtask:":
                m = re.match(
                    r"^\s*pid=(?P<pid>[0-9]+)\s+comm=(?P<comm>\S+)", d["etc"]
                )
                if m:
                    p = m.groupdict()

                    pid = p["pid"]

                    try:
                        self.threadData[pid]
                    except:
                        self.threadData[pid] = dict(self.init_threadData)
                        self.threadData[pid]["comm"] = p["comm"]
                        self.threadData[pid]["new"] = True
                        self.threadData[pid]["ptid"] = thread

            # Save user event #
            elif (
                d["func"].startswith("tracing_mark_write") or d["func"] == "0:"
            ):
                m = re.match(r"^.+EVENT_(?P<event>\S+)", d["etc"])
                if m:
                    gd = m.groupdict()

                    EventAnalyzer.addEvent(d["time"], gd["event"])

                    if gd["event"] == "STOP":
                        self.finishTime = float(d["time"])

                # Return False because there is no stack data with this event #
                return False

            # apply filter #
            if SysMgr.isExceptTarget(thread, self.threadData, plist=plist):
                return False
            else:
                self.threadData[thread]["target"] = True

            return self.parseEventInfo(
                thread, d["func"], d["etc"], d["time"], d["core"]
            )

        # Parse call stack #
        else:
            # exist path, offset, pos #
            m = re.match(
                (
                    r" => (?P<path>.+)\[\+0x(?P<offset>.\S*)\] "
                    r"\<(?P<pos>.\S+)\>"
                ),
                string,
            )
            if m:
                d = m.groupdict()
                return (d["pos"], d["path"], hex(long(d["offset"], 16)))

            # exist only pos #
            pos = string.find("=>  <")
            if pos > -1:
                return (string[pos + 5 : len(string) - 2], None, None)

            # no user stack tracing supported #
            if "??" in string:
                if SysMgr.userEnable and SysMgr.userEnableWarn:
                    SysMgr.printWarn(
                        (
                            "enable CONFIG_USER_STACKTRACE_SUPPORT kernel"
                            " option if it is not enabled"
                        ),
                        True,
                    )
                    SysMgr.userEnableWarn = False
                return ("0", None, None)

            # exist symbol, pos #
            m = re.match(r" => (?P<symbol>.+) \<(?P<pos>.\S+)\>", string)
            if m:
                d = m.groupdict()
                return (d["pos"], d["symbol"], None)

            # garbage log #
            return False

    def getBinInfo(self, addr):
        for data in self.mapData:
            if not (
                long(data["startAddr"], 16) <= long(addr, 16)
                and long(data["endAddr"], 16) >= long(addr, 16)
            ):
                continue

            if ElfAnalyzer.isRelocFile(data["binName"]):
                # Return full path and offset in mapping table #
                return SysMgr.rootPath + data["binName"], hex(
                    long(addr, 16) - long(data["startAddr"], 16)
                )
            else:
                return SysMgr.rootPath + data["binName"], hex(long(addr, 16))

        SysMgr.printWarn(
            "failed to get the binary info of %s in mapping table" % addr
        )

    def printSyscallSummary(self):
        # no effective syscall event #
        if self.syscallCnt == 0:
            return

        convNum = UtilMgr.convNum

        SysMgr.clearPrint()
        SysMgr.printPipe(
            "[Function Syscall Info] [Cnt: %s]\n%s"
            % (convNum(self.syscallCnt), twoLine)
        )
        SysMgr.printPipe(
            "{0:>16}({1:>7}/{2:>7}) {3:>30}({4:>3}) {5:>12}\n{6:1}".format(
                "Name", "TID", "PID", "Syscall", "SID", "Count", twoLine
            )
        )

        outputCnt = 0
        for key, value in sorted(
            self.threadData.items(),
            key=lambda e: e[1]["nrSyscall"],
            reverse=True,
        ):
            threadInfo = ""
            syscallInfo = ""

            if key.startswith("0["):
                continue

            try:
                if value["syscallTable"]:
                    threadInfo = "%16s(%7s/%7s)" % (
                        value["comm"],
                        key,
                        value["tgid"],
                    )
                else:
                    continue
            except:
                continue

            for sysId, val in sorted(
                value["syscallTable"].items(), key=lambda e: e[1], reverse=True
            ):
                if val == 0:
                    continue

                try:
                    syscall = ConfigMgr.sysList[sysId][4:]
                except:
                    SysMgr.printErr(
                        "failed to get syscall name by number %s" % sysId
                    )
                    syscall = sysId

                syscallInfo = ("{0:1} {1:>30}({2:>3}) {3:>12}\n").format(
                    syscallInfo + (" " * len(threadInfo)),
                    syscall,
                    sysId,
                    convNum(val),
                )

            if syscallInfo != "":
                outputCnt += 1
                SysMgr.printPipe(threadInfo)
                SysMgr.printPipe(syscallInfo + oneLine)

        if outputCnt == 0:
            SysMgr.printPipe("    None\n%s" % oneLine)

        SysMgr.printPipe("\n\n")

    def printUsage(self):
        targetCnt = 0
        self.totalTime = float(self.finishTime) - float(SysMgr.startTime)

        convSize = UtilMgr.convSize2Unit
        convNum = UtilMgr.convNum

        SysMgr.printLogo(big=True)

        # print system information #
        SysMgr.printInfoBuffer()

        # choose syscall / heap menu in table #
        if self.sysEnabled:
            cmenu = "SYSTEM"
            cmenu2 = "CALLS"
        else:
            cmenu = "HEAP"
            cmenu2 = "EVENTS"

        # Print thread list #
        SysMgr.printPipe(
            "[%s] [%s: %0.3f] [%s: %0.3f] [Threads: %d] [LogSize: %s]\n%s"
            % (
                "Function Thread Info",
                "Elapsed",
                round(self.totalTime, 7),
                "Start",
                round(float(SysMgr.startTime), 7),
                len(self.threadData),
                convSize(SysMgr.logSize),
                twoLine,
            )
        )
        SysMgr.printPipe(
            (
                "{0:_^46}|{1:_^7}|{2:_^54}|{3:_^8}|"
                "{4:_^18}|{5:_^6}|{6:_^8}|"
            ).format("Thread", "CPU", "PAGE", cmenu, "BLOCK", "LOCK", "CUSTOM")
        )
        SysMgr.printPipe(
            (
                "{0:^16}|{0:^7}|{0:^7}|{0:^6}|{0:^6}|{0:^7}|"
                "{0:^9}{0:^8}{0:^8}{0:^12}|{0:^8}|{0:^7}|{0:^8}|"
                "{0:^8}|{0:^9}|{0:^6}|{0:^8}|"
            ).format(" ")
        )

        SysMgr.printPipe(
            (
                "{0:_^16}|{1:_^7}|{2:_^7}|{3:_^6}|{4:_^6}|"
                "{5:_^7}|{6:_^9}({7:_^8}/{8:_^8}/{9:_^8})|{10:_^8}|"
                "{11:_^7}|{12:_^8}|{13:_^8}|{14:_^9}|{15:_^6}|{16:_^8}|"
            ).format(
                "Name",
                "TID",
                "PID",
                "PICK",
                "LIFE",
                "PER",
                "ALLOC",
                "USER",
                "BUF",
                "KERN",
                "FREE",
                "UFREE",
                cmenu2,
                "READ",
                "WRITE",
                "TRY",
                "EVENTS",
            )
        )
        SysMgr.printPipe(twoLine)

        # set sort value #
        if SysMgr.sort == "m":
            sortedThreadData = sorted(
                self.threadData.items(),
                key=lambda e: e[1]["nrPages"],
                reverse=True,
            )
        elif SysMgr.sort == "b":
            sortedThreadData = sorted(
                self.threadData.items(),
                key=lambda e: e[1]["nrRdBlocks"],
                reverse=True,
            )
        elif SysMgr.sort == "L":
            sortedThreadData = sorted(
                self.threadData.items(),
                key=lambda e: e[1]["nrLockTry"],
                reverse=True,
            )
        elif SysMgr.sort == "h":
            sortedThreadData = sorted(
                self.threadData.items(),
                key=lambda e: e[1]["heapSize"],
                reverse=True,
            )
        elif SysMgr.sort == "s":
            sortedThreadData = sorted(
                self.threadData.items(),
                key=lambda e: e[1]["nrSyscall"],
                reverse=True,
            )
        else:
            # set CPU usage as default #
            sortedThreadData = sorted(
                self.threadData.items(),
                key=lambda e: e[1]["cpuTick"],
                reverse=True,
            )

        for idx, value in sortedThreadData:
            targetMark = ""

            # skip no event count thread #
            if value["eventCnt"] == 0:
                continue

            # check target thread #
            if value["target"]:
                targetCnt += 1
                if targetCnt == 2:
                    SysMgr.printWarn("multiple target threads are selected")
                targetMark = "*"

            # get CPU usage #
            if self.totalTick > 0:
                cpuPer = "%.1f%%" % (
                    (float(value["cpuTick"]) / float(self.totalTick)) * 100
                )
            else:
                cpuPer = "0.0%%"

            # set break condition #
            if SysMgr.sort == "m":
                breakCond = value["nrPages"]
            elif SysMgr.sort == "b":
                breakCond = value["nrRdBlocks"]
            else:
                breakCond = long(cpuPer[: cpuPer.rfind(".")])

            # check condition for stop #
            if breakCond < 1 and not SysMgr.showAll:
                pass

            # set lifecycle flags #
            if value["new"]:
                life = "N"
            else:
                life = " "
            if value["die"]:
                life = "%sD" % life

            if self.cpuEnabled:
                # remove percentage if no tick #
                if float(value["cpuTick"]) == 0:
                    cpuPer = "-"
                else:
                    cpuPer = cpuPer
            else:
                cpuPer = "-"

            if self.sysEnabled:
                cval = "%s" % convNum(value["nrSyscall"])
            elif self.heapEnabled:
                cval = "%s" % convSize(value["heapSize"])
            else:
                cval = "-"

            if self.memEnabled:
                allocMem = "%s" % convSize(value["nrPages"] << 12)
                userMem = "%s" % convSize(value["userPages"] << 12)
                cacheMem = "%s" % convSize(value["cachePages"] << 12)
                kernelMem = "%s" % convSize(value["kernelPages"] << 12)
                knownFreeMem = "%s" % convSize(value["nrKnownFreePages"] << 12)
                unknownFreeMem = "%s" % convSize(
                    value["nrUnknownFreePages"] << 12
                )
            else:
                allocMem = "-"
                userMem = "-"
                cacheMem = "-"
                kernelMem = "-"
                knownFreeMem = "-"
                unknownFreeMem = "-"

            if self.breadEnabled:
                readBlock = "%s" % convSize(value["nrRdBlocks"] << 9)
            else:
                readBlock = "-"

            if self.bwriteEnabled:
                writeBlock = "%s" % convSize(value["nrWrBlocks"] << 9)
            else:
                writeBlock = "-"

            if self.lockEnabled:
                nrLock = convNum(value["nrLockTry"])
            else:
                nrLock = "-"

            if self.customTotal > 0:
                nrCustom = convNum(value["customTotal"])
            else:
                nrCustom = "-"

            # update comm #
            if value["comm"] == "<...>" and idx in SysMgr.commCache:
                comm = SysMgr.commCache[idx]
            else:
                comm = value["comm"]

            SysMgr.printPipe(
                (
                    "{0:>16}|{1:>7}|{2:>7}|{3:^6}|{4:^6}|"
                    "{5:>7}|{6:>9}({7:>8}/{8:>8}/{9:>8})|{10:>7}|{11:>8}|"
                    "{12:>8}|{13:>8}|{14:>9}|{15:>6}|{16:>8}|"
                ).format(
                    comm[:16],
                    idx,
                    value["tgid"],
                    targetMark,
                    life,
                    cpuPer,
                    allocMem,
                    userMem,
                    cacheMem,
                    kernelMem,
                    knownFreeMem,
                    unknownFreeMem,
                    cval,
                    readBlock,
                    writeBlock,
                    nrLock,
                    nrCustom,
                )
            )

        if targetCnt == 0:
            SysMgr.printPipe("    None")

        SysMgr.printPipe("%s\n\n\n" % oneLine)

        # Exit because there is no target #
        if not self.target:
            SysMgr.printWarn(
                "no specific thread targeted, input value for TID"
            )

        # Print syscall usage of threads #
        self.printSyscallSummary()

        # Print resource usage of functions #
        self.printCpuUsage()
        self.printMemUsage()
        self.printHeapUsage()
        self.printBlockRdUsage()
        self.printBlockWrUsage()
        self.printLockUsage()
        self.printSyscallUsage()
        self.printCustomUsage()

    def makeKernelSymList(self, subStack, ilen):
        symbolStack = ""
        stackIdx = 0
        tlen = ilen

        if not subStack:
            return " None"

        try:
            for pos in subStack:
                if self.posData[pos]["symbol"] == "":
                    symbolSet = " <- %s" % hex(long(pos, 16))
                elif not self.posData[pos]["symbol"] and SysMgr.showAll:
                    symbolSet = " <- %s" % hex(long(pos, 16))
                else:
                    symbolSet = " <- %s" % str(self.posData[pos]["symbol"])

                lpos = tlen + len(symbolStack[stackIdx:]) + len(symbolSet)
                if symbolStack != "" and lpos > SysMgr.lineLength:
                    stackIdx = len(symbolStack)
                    symbolStack = "%s\n%s" % (symbolStack, " " * ilen)
                    tlen = 0

                symbolStack += symbolSet
        except SystemExit:
            sys.exit(0)
        except:
            pass

        return symbolStack

    def makeUserSymList(self, subStack, ilen):
        symbolStack = ""
        stackIdx = 0
        tlen = ilen

        if self.sort == "sym":
            for sym in subStack:
                if not sym or sym == "0":
                    symbolSet = ""
                elif self.userSymData[sym]["origBin"] == "??":
                    symbolSet = " <- %s" % sym
                else:
                    symbolSet = " <- %s [%s]" % (
                        sym,
                        self.userSymData[sym]["origBin"],
                    )

                lpos = tlen + len(symbolStack[stackIdx:]) + len(symbolSet)
                if symbolStack != "" and lpos > SysMgr.lineLength:
                    stackIdx = len(symbolStack)
                    symbolStack = "%s\n%s" % (symbolStack, " " * ilen)
                    tlen = 0

                symbolStack += symbolSet
        elif self.sort == "pos":
            for pos in subStack:
                if not pos:
                    symbolStack += " <- None"
                # No symbol so that just print pos #
                elif self.posData[pos]["symbol"] == "":
                    symbolStack = "%s <- %s [%s]" % (
                        symbolStack,
                        hex(long(pos, 16)),
                        self.posData[pos]["origBin"],
                    )
                # Print symbol #
                else:
                    symbolStack = "%s <- %s [%s]" % (
                        symbolStack,
                        self.posData[pos]["symbol"],
                        self.posData[pos]["origBin"],
                    )

        if not symbolStack:
            return "    None"
        else:
            return symbolStack

    def printSyscallUsage(self):
        # no effective syscall event #
        if self.syscallCnt == 0 or not SysMgr.userEnable:
            return

        subStackIndex = FunctionAnalyzer.symStackIdxTable.index("STACK")
        eventIndex = FunctionAnalyzer.symStackIdxTable.index("SYSCALL")
        convNum = UtilMgr.convNum

        # Print syscall event #
        SysMgr.clearPrint()
        SysMgr.printPipe(
            "[Function Syscall Info] [Cnt: %s] (USER)\n%s"
            % (convNum(self.syscallCnt), twoLine)
        )

        SysMgr.printPipe(
            "{0:_^9}|{1:_^47}|{2:_^49}|{3:_^46}\n{4:1}".format(
                "Usage", "Function", "Binary", "Source", twoLine
            )
        )

        for idx, value in sorted(
            self.userSymData.items(),
            key=lambda e: e[1]["syscallCnt"],
            reverse=True,
        ):
            if value["syscallCnt"] == 0:
                break

            SysMgr.printPipe(
                "{0:>7}  |{1:^47}| {2:48}| {3:37}".format(
                    convNum(value["syscallCnt"]),
                    idx,
                    self.posData[value["pos"]]["origBin"],
                    self.posData[value["pos"]]["src"],
                )
            )

            # Set target stack #
            targetStack = self.getTargetStack(value, eventIndex)

            # Merge and Print symbols in stack #
            for stack in targetStack:
                eventCnt = stack[eventIndex]
                subStack = list(stack[subStackIndex])

                if eventCnt == 0:
                    break

                if not subStack:
                    continue
                else:
                    ilen = len("\t" * 16) + 3
                    symbolStack = self.makeUserSymList(subStack, ilen)

                SysMgr.printPipe(
                    "\t\t +{0:>7} |{1:32}".format(
                        convNum(eventCnt), symbolStack
                    )
                )

            SysMgr.printPipe(oneLine)

        SysMgr.printPipe()

        # Print syscall file #
        SysMgr.printPipe(
            "[Function Syscall File Info] [Cnt: %s] (USER)\n%s"
            % (twoLine, convNum(self.syscallCnt))
        )

        SysMgr.printPipe(
            "{0:_^9}|{1:_^144}\n{2:1}".format("Usage", "Binary", twoLine)
        )

        for idx, value in sorted(
            self.userFileData.items(),
            key=lambda e: e[1]["syscallCnt"],
            reverse=True,
        ):
            if value["syscallCnt"] == 0:
                break

            SysMgr.printPipe(
                "{0:>8} | {1:<142}\n{2:1}".format(
                    convNum(value["syscallCnt"]), idx, oneLine
                )
            )

        if self.periodicEventCnt == 0:
            SysMgr.printPipe("    None\n%s" % oneLine)

        SysMgr.printPipe()

        # Print syscall history #
        if not SysMgr.showAll or not self.sysCallData:
            SysMgr.printPipe("\n\n")
            return

        SysMgr.clearPrint()
        SysMgr.printPipe(
            "[Function Syscall History] [Cnt: %s]\n%s"
            % (convNum(self.syscallCnt), twoLine)
        )

        SysMgr.printPipe(
            "{0:_^32}|{1:_^17}({2:_^7})|{3:_^8}|{4:_^17}|\n{5:1}".format(
                "Event", "COMM", "TID", "CORE", "TIME", twoLine
            )
        )

        # sort by time #
        for call in self.sysCallData:
            event = ConfigMgr.sysList[call[0]][4:]
            args, time, core, tid = call[1][:4]

            title = "{0:^32}| {1:>16}({2:>7})| {3:>6} | {4:>15} |".format(
                event, self.threadData[tid]["comm"], tid, core, time
            )
            SysMgr.printPipe("%s\n%s" % (title, len(title) * "-"))

            # Make argument info #
            argsInfo = " %s" % args

            # Make user call info #
            ilen = 32
            nowLen = ilen
            try:
                last = call[2][0]
                stack = call[2][1]
                userCall = " %s[%s]" % (
                    self.posData[last]["symbol"],
                    self.posData[last]["binary"],
                )
                nowLen += len(userCall)
                for subcall in stack:
                    try:
                        symbol = self.posData[subcall]["symbol"]
                        binary = self.posData[subcall]["binary"]
                        nextCall = " <- %s[%s]" % (symbol, binary)
                        if SysMgr.lineLength > nowLen + len(nextCall):
                            userCall += nextCall
                            nowLen += len(nextCall)
                        else:
                            userCall = "%s\n%s %s" % (
                                userCall,
                                " " * ilen,
                                nextCall,
                            )
                            nowLen = ilen + len(nextCall)
                    except:
                        pass
            except SystemExit:
                sys.exit(0)
            except:
                pass

            SysMgr.printPipe(
                "{0:>32}| {1:<121}".format("[Args] ", argsInfo.strip())
            )
            SysMgr.printPipe(
                "{0:>32}|{1:<121}\n{2:1}".format("[User] ", userCall, oneLine)
            )

        SysMgr.printPipe("\n\n")

    def printCustomUsage(self):
        # no effective custom event #
        if self.customTotal == 0:
            return

        subStackIndex = FunctionAnalyzer.symStackIdxTable.index("STACK")
        eventIndex = FunctionAnalyzer.symStackIdxTable.index("CUSTOM")
        convNum = UtilMgr.convNum

        # Make custom event list #
        customList = ", ".join(list(self.customEventTable))

        # Print custom event in user space #
        if SysMgr.userEnable:
            SysMgr.clearPrint()
            SysMgr.printPipe(
                "[Function %s Info] [Cnt: %s] [Total: %s] (USER)\n%s"
                % (
                    customList,
                    convNum(self.customTotal),
                    convNum(self.customCnt),
                    twoLine,
                )
            )

            SysMgr.printPipe(
                "{0:_^9}|{1:_^47}|{2:_^49}|{3:_^46}\n{4:1}".format(
                    "Usage", "Function", "Binary", "Source", twoLine
                )
            )

            for idx, value in sorted(
                self.userSymData.items(),
                key=lambda e: e[1]["customCnt"],
                reverse=True,
            ):
                if value["customCnt"] == 0:
                    break

                SysMgr.printPipe(
                    "{0:>7}  |{1:^47}| {2:48}| {3:37}".format(
                        convNum(value["customCnt"]),
                        idx,
                        self.posData[value["pos"]]["origBin"],
                        self.posData[value["pos"]]["src"],
                    )
                )

                # Set target stack #
                targetStack = self.getTargetStack(value, eventIndex)

                # Merge and Print symbols in stack #
                for stack in targetStack:
                    eventCnt = stack[eventIndex]
                    subStack = list(stack[subStackIndex])

                    if eventCnt == 0:
                        break

                    if not subStack:
                        continue
                    else:
                        ilen = len("\t" * 16) + 3
                        symbolStack = self.makeUserSymList(subStack, ilen)

                    SysMgr.printPipe(
                        "\t\t +{0:>7} |{1:32}".format(
                            convNum(eventCnt), symbolStack
                        )
                    )

                SysMgr.printPipe(oneLine)

            SysMgr.printPipe("\n")

            # Print custom event file in user space #
            SysMgr.printPipe(
                "[Function %s File Info] [Cnt: %s] [Total: %s] (USER)\n%s"
                % (
                    customList,
                    convNum(self.customTotal),
                    convNum(self.customCnt),
                    twoLine,
                )
            )

            SysMgr.printPipe(
                "{0:_^9}|{1:_^144}\n{2:1}".format("Usage", "Binary", twoLine)
            )

            for idx, value in sorted(
                self.userFileData.items(),
                key=lambda e: e[1]["customCnt"],
                reverse=True,
            ):
                if value["customCnt"] == 0:
                    break

                SysMgr.printPipe(
                    "{0:>8} | {1:<142}\n{2:1}".format(
                        convNum(value["customCnt"]), idx, oneLine
                    )
                )

            if self.periodicEventCnt == 0:
                SysMgr.printPipe("    None\n%s" % oneLine)

            SysMgr.printPipe("\n")

        # Print custom event in kernel space #
        SysMgr.clearPrint()
        SysMgr.printPipe(
            "[Function %s Info] [Cnt: %s] [Total: %s] (KERNEL)\n%s"
            % (
                customList,
                convNum(self.customTotal),
                convNum(self.customCnt),
                twoLine,
            )
        )

        SysMgr.printPipe(
            "{0:_^9}|{1:_^144}\n{2:1}".format("Usage", "Function", twoLine)
        )

        # Print custom usage of stacks #
        for idx, value in sorted(
            self.kerSymData.items(),
            key=lambda e: e[1]["customCnt"],
            reverse=True,
        ):
            if value["customCnt"] == 0:
                break

            SysMgr.printPipe(
                "{0:>7}  |{1:^134}".format(convNum(value["customCnt"]), idx)
            )

            # Sort stacks by usage #
            value["stack"] = sorted(
                value["stack"], key=lambda x: x[eventIndex], reverse=True
            )

            # Print stacks by symbol #
            for stack in value["stack"]:
                eventCnt = stack[eventIndex]
                subStack = list(stack[subStackIndex])

                if eventCnt == 0:
                    break

                if not subStack:
                    continue
                elif (
                    len(subStack) == 1
                    and not SysMgr.showAll
                    and (
                        not self.posData[subStack[0]]["symbol"]
                        or self.posData[subStack[0]]["symbol"] == "NoFile"
                    )
                ):
                    # Pass unmeaningful part #
                    continue
                else:
                    ilen = len("\t" * 16) + 3
                    symbolStack = self.makeKernelSymList(subStack, ilen)

                SysMgr.printPipe(
                    "\t\t +{0:>7} |{1:32}".format(
                        convNum(eventCnt), symbolStack
                    )
                )

            SysMgr.printPipe(oneLine)

            SysMgr.printPipe("\n")

        # Print custom call history #
        if not SysMgr.showAll or not self.customCallData:
            SysMgr.printPipe("\n\n")
            return

        SysMgr.clearPrint()
        SysMgr.printPipe(
            "[Function %s History] [Cnt: %s] [Total: %s]\n%s"
            % (
                customList,
                convNum(self.customTotal),
                convNum(self.customCnt),
                twoLine,
            )
        )

        SysMgr.printPipe(
            "{0:_^32}|{1:_^17}({2:_^7})|{3:_^8}|{4:_^17}|\n{5:1}".format(
                "Event", "COMM", "TID", "CORE", "TIME", twoLine
            )
        )

        # sort by time #
        for call in sorted(self.customCallData, key=lambda e: e[1][1]):
            event = call[0]
            args = call[1][0]
            time = call[1][1]
            core = call[1][2]
            tid = call[1][3]

            title = "{0:^32}| {1:>16}({2:>7})| {3:>6} | {4:>15} |".format(
                event, self.threadData[tid]["comm"], tid, core, time
            )
            SysMgr.printPipe("%s\n%s" % (title, len(title) * "-"))

            # Make argument info #
            argsInfo = " %s" % args

            # Make user call info #
            ilen = 32
            nowLen = ilen
            try:
                last = call[2][0]
                stack = call[2][1]
                userCall = " %s[%s]" % (
                    self.posData[last]["symbol"],
                    self.posData[last]["binary"],
                )
                nowLen += len(userCall)
                for subcall in stack:
                    try:
                        symbol = self.posData[subcall]["symbol"]
                        binary = self.posData[subcall]["binary"]
                        nextCall = " <- %s[%s]" % (symbol, binary)
                        if SysMgr.lineLength > nowLen + len(nextCall):
                            userCall += nextCall
                            nowLen += len(nextCall)
                        else:
                            userCall = "%s\n%s %s" % (
                                userCall,
                                " " * ilen,
                                nextCall,
                            )
                            nowLen = ilen + len(nextCall)
                    except:
                        pass
            except SystemExit:
                sys.exit(0)
            except:
                pass

            # Make kernel call info #
            ilen = 32
            nowLen = ilen
            try:
                last = call[3][0]
                stack = call[3][1]
                kernelCall = " %s" % (self.posData[last]["symbol"])
                nowLen += len(kernelCall)
                for subcall in stack:
                    try:
                        nextCall = " <- %s" % (self.posData[subcall]["symbol"])
                        if SysMgr.lineLength > nowLen + len(nextCall):
                            kernelCall += nextCall
                            nowLen += len(nextCall)
                        else:
                            kernelCall = "%s\n%s %s" % (
                                kernelCall,
                                " " * ilen,
                                nextCall,
                            )
                            nowLen = ilen + len(nextCall)
                    except:
                        pass
            except SystemExit:
                sys.exit(0)
            except:
                pass

            SysMgr.printPipe(
                "{0:>32}| {1:<121}".format("[Args] ", argsInfo.strip())
            )
            SysMgr.printPipe("{0:>32}|{1:<121}".format("[User] ", userCall))
            SysMgr.printPipe(
                "{0:>32}|{1:<121}\n{2:1}".format(
                    "[Kernel] ", kernelCall, oneLine
                )
            )

        SysMgr.printPipe("\n\n")

    def printCpuUsage(self):
        # no CPU event #
        if not self.cpuEnabled or self.periodicEventCnt == 0:
            return

        subStackIndex = FunctionAnalyzer.symStackIdxTable.index("STACK")
        cpuTickIndex = FunctionAnalyzer.symStackIdxTable.index("CPU_TICK")
        tCnt = UtilMgr.convNum(self.periodicEventCnt)

        # average tick interval #
        if self.periodicContEventCnt > 0:
            self.periodicEventInterval /= self.periodicContEventCnt

        # Print CPU usage in user space #
        if SysMgr.userEnable:
            SysMgr.clearPrint()
            title = "Function CPU-Tick Info"
            SysMgr.printPipe(
                "[%s] [Cnt: %s] [Interval: %dms] (USER)"
                % (title, tCnt, self.periodicEventInterval * 1000)
            )

            # Print call stack #
            SysMgr.printPipe(
                "{3:1}\n{0:_^9}|{1:_^47}|{2:_^96}\n{3:1}".format(
                    "Usage", "Function", "Binary", twoLine
                )
            )

            for idx, value in sorted(
                self.userSymData.items(),
                key=lambda e: e[1]["tickCnt"],
                reverse=True,
            ):
                if value["tickCnt"] == 0:
                    break

                cpuPer = (
                    value["tickCnt"] / float(self.periodicEventCnt)
                ) * 100
                if cpuPer < 1 and not SysMgr.showAll:
                    break

                SysMgr.printPipe(
                    "{0:7.1f}% |{1:^47}| {2:48}".format(
                        cpuPer, idx, self.posData[value["pos"]]["origBin"]
                    )
                )

                # Increase total CPU usage per symbol #
                value["totalTickCnt"] += value["tickCnt"]

                # Set target stack #
                targetStack = self.getTargetStack(value)

                # Merge and Print symbols in stack #
                for stack in targetStack:
                    cpuCnt = stack[cpuTickIndex]
                    subStack = list(stack[subStackIndex])

                    if cpuCnt == 0:
                        break

                    if not subStack:
                        continue
                    else:
                        # Increase total tick count of symbols in stack #
                        for sym in subStack:
                            self.userSymData[sym]["totalTickCnt"] += 1

                        cpuPer = (cpuCnt / float(value["tickCnt"])) * 100
                        if cpuPer < 1 and not SysMgr.showAll:
                            break

                        ilen = len("\t" * 16)
                        symbolStack = self.makeUserSymList(subStack, ilen)

                    SysMgr.printPipe(
                        "\t +{0:7.1f}% |{1:32}".format(cpuPer, symbolStack)
                    )

                SysMgr.printPipe(oneLine)

            if self.periodicEventCnt == 0:
                SysMgr.printPipe("    None\n%s" % oneLine)

            SysMgr.printPipe("\n")

            # Print per-symbol #
            title = "Function CPU-Tick Symbol Info"
            SysMgr.printPipe(
                "[%s] [Cnt: %s] [Interval: %dms] (USER)\n%s"
                % (title, tCnt, self.periodicEventInterval * 1000, twoLine)
            )

            SysMgr.printPipe(
                "{0:_^9}|{1:_^47}|{2:_^96}\n{3:1}".format(
                    "Usage", "Function", "Binary", twoLine
                )
            )

            for idx, value in sorted(
                self.userSymData.items(),
                key=lambda e: e[1]["totalTickCnt"],
                reverse=True,
            ):
                if value["totalTickCnt"] == 0:
                    break

                cpuPer = (
                    value["totalTickCnt"] / float(self.periodicEventCnt)
                ) * 100
                if cpuPer < 1 and not SysMgr.showAll:
                    break

                SysMgr.printPipe(
                    "{0:7.1f}% |{1:^47}| {2:48}\n{3:1}".format(
                        cpuPer,
                        idx,
                        self.posData[value["pos"]]["origBin"],
                        oneLine,
                    )
                )

            if self.periodicEventCnt == 0:
                SysMgr.printPipe("    None\n%s" % oneLine)

            SysMgr.printPipe("\n")

            # Print tick per-file #
            title = "Function CPU-Tick File Info"
            SysMgr.printPipe(
                "[%s] [Cnt: %s] [Interval: %dms] (USER)\n%s"
                % (title, tCnt, self.periodicEventInterval * 1000, twoLine)
            )

            SysMgr.printPipe(
                "{0:_^9}|{1:_^144}\n{2:1}".format("Usage", "Binary", twoLine)
            )

            for idx, value in sorted(
                self.userFileData.items(),
                key=lambda e: e[1]["tickCnt"],
                reverse=True,
            ):
                if value["tickCnt"] == 0:
                    break

                cpuPer = (
                    value["tickCnt"] / float(self.periodicEventCnt)
                ) * 100
                if cpuPer < 1 and not SysMgr.showAll:
                    break

                SysMgr.printPipe(
                    "{0:7.1f}% | {1:<142}\n{2:1}".format(cpuPer, idx, oneLine)
                )

            if self.periodicEventCnt == 0:
                SysMgr.printPipe("    None\n%s" % oneLine)

            SysMgr.printPipe("\n")

        # Print CPU usage in kernel space #
        title = "Function CPU-Tick Info"
        SysMgr.clearPrint()
        titleStr = "[%s] [Cnt: %s] [Interval: %dms] (KERNEL)" % (
            title,
            tCnt,
            self.periodicEventInterval * 1000,
        )
        SysMgr.printPipe(
            "{3:1}\n{2:1}\n{0:_^9}|{1:_^144}\n{2:1}".format(
                "Usage", "Function", twoLine, titleStr
            )
        )

        # Make exception list to remove a redundant part of stack #
        exceptList = {}
        for pos, value in self.posData.items():
            if value["symbol"] in (
                "__irq_usr",
                "__irq_svc",
                "el1_irq",
                "gic_handle_irq",
                "apic_timer_interrupt",
                "asm_sysvec_hyperv_stimer0",
            ):
                exceptList.setdefault(pos, {})

        # Print CPU usage of stacks #
        for idx, value in sorted(
            self.kerSymData.items(),
            key=lambda e: e[1]["tickCnt"],
            reverse=True,
        ):
            if value["tickCnt"] == 0:
                break

            """
            disable to print last symbol because it is only one

            tickCnt = float(value['tickCnt'])
            eventCnt = float(self.periodicEventCnt)
            cpuPer = round(tickCnt / eventCnt * 100, 1)

            if cpuPer < 1 and not SysMgr.showAll:
                break

            SysMgr.printPipe("{0:7}% |{1:^134}".format(cpuPer, idx))
            """

            # Sort stacks by usage #
            value["stack"].sort(reverse=True)

            # Define merge list #
            mergedSymbolChain = {}

            # Merge by symbol chain #
            for stack in value["stack"]:
                cpuCnt = stack[cpuTickIndex]
                subStack = list(stack[subStackIndex])

                if cpuCnt == 0:
                    break
                else:
                    # Find index of the backmost exception value #
                    maxIdx = -1
                    for pos in list(exceptList):
                        try:
                            ridx = subStack.index(pos)
                            if ridx >= 0 and ridx > maxIdx:
                                maxIdx = ridx
                        except:
                            pass

                    # Remove a redundant part #
                    if maxIdx >= 0:
                        maxIdx += 1
                        if maxIdx == len(subStack):
                            subStack = []
                        else:
                            subStack = subStack[maxIdx:]

                if not subStack:
                    symbolStack = " <- USER"
                elif (
                    len(subStack) == 1
                    and not SysMgr.showAll
                    and (
                        not self.posData[subStack[0]]["symbol"]
                        or self.posData[subStack[0]]["symbol"] == "NoFile"
                    )
                ):
                    # Pass unmeaningful part #
                    continue
                else:
                    ilen = 10
                    symbolStack = self.makeKernelSymList(subStack, ilen)

                try:
                    mergedSymbolChain[symbolStack] += cpuCnt
                except:
                    mergedSymbolChain[symbolStack] = cpuCnt

            # define sample list for flame graph #
            callList = {}
            drawflame = "DRAWFLAME" in SysMgr.environList

            # Print stacks by symbol #
            for chain, tick in sorted(
                mergedSymbolChain.items(), key=lambda e: e[1], reverse=True
            ):
                cpuPer = (tick / float(value["tickCnt"])) * 100
                if cpuPer < 1 and not SysMgr.showAll:
                    break

                # add a sample for flame graph #
                if drawflame:
                    callList[chain.lstrip("<- ")] = tick

                SysMgr.printPipe("{0:7.1f}% |{1:32}".format(cpuPer, chain))

            SysMgr.printPipe(oneLine)

            if self.periodicEventCnt == 0:
                SysMgr.printPipe("    None\n%s" % oneLine)

            # draw flame graph #
            if drawflame:
                Debugger.drawFlame(
                    callList=callList,
                    title=titleStr,
                    outFile=SysMgr.outPath + ".cpu.kernel",
                )

        SysMgr.printPipe("\n\n")

    def printUnknownMemFreeInfo(self):
        # check memory event #
        if not self.memEnabled:
            return

        SysMgr.printPipe("\n")

        title = "Function Free-Only-Page Info"
        subStackIndex = FunctionAnalyzer.symStackIdxTable.index("STACK")
        pageFreeIndex = FunctionAnalyzer.symStackIdxTable.index("PAGE_FREE")
        convSize = UtilMgr.convSize2Unit
        size = convSize(self.pageUnknownFreeCnt << 12)

        if SysMgr.userEnable:
            # Print memory reduce by page free in user space #
            SysMgr.clearPrint()
            SysMgr.printPipe(
                "[%s] [Size: %s] (USER)\n%s" % (title, size, twoLine)
            )

            SysMgr.printPipe(
                "{0:_^9}|{1:_^47}|{2:_^49}|{3:_^46}\n{4:1}".format(
                    "Free", "Function", "Binary", "Source", twoLine
                )
            )

            for idx, value in sorted(
                self.userSymData.items(),
                key=lambda e: e[1]["unknownPageFreeCnt"],
                reverse=True,
            ):
                if value["unknownPageFreeCnt"] == 0:
                    break

                SysMgr.printPipe(
                    "{0:>8} |{1:^47}| {2:48}| {3:37}".format(
                        convSize(value["unknownPageFreeCnt"] << 12),
                        idx,
                        self.posData[value["pos"]]["origBin"],
                        self.posData[value["pos"]]["src"],
                    )
                )

                # Set target stack #
                targetStack = self.getTargetStack(value, pageFreeIndex)

                # Merge and Print symbols in stack #
                for stack in targetStack:
                    pageFreeCnt = stack[pageFreeIndex]
                    subStack = list(stack[subStackIndex])

                    if pageFreeCnt == 0:
                        break

                    if not subStack:
                        continue
                    else:
                        ilen = len("\t" * 16)
                        symbolStack = self.makeUserSymList(subStack, ilen)

                    SysMgr.printPipe(
                        "\t+ {0:>8} |{1:32}".format(
                            convSize(pageFreeCnt << 12), symbolStack
                        )
                    )

                SysMgr.printPipe(oneLine)

            if self.pageUnknownFreeCnt == 0:
                SysMgr.printPipe("    None\n%s" % oneLine)

            SysMgr.printPipe("\n")

        # Print memory reduce by page free in kernel space #
        SysMgr.clearPrint()
        SysMgr.printPipe(
            "[%s] [Size: %s] (KERNEL)\n%s" % (title, size, twoLine)
        )

        SysMgr.printPipe(
            "{0:_^9}|{1:_^144}\n{2:1}".format("FREE", "Function", twoLine)
        )

        for idx, value in sorted(
            self.kerSymData.items(),
            key=lambda e: e[1]["unknownPageFreeCnt"],
            reverse=True,
        ):
            if value["unknownPageFreeCnt"] == 0:
                break

            SysMgr.printPipe(
                "{0:>8} |{1:^144}".format(
                    convSize(value["unknownPageFreeCnt"] << 12), idx
                )
            )

            # Sort stacks by usage #
            value["stack"] = sorted(
                value["stack"], key=lambda x: x[pageFreeIndex], reverse=True
            )

            # Print stacks by symbol #
            for stack in value["stack"]:
                pageFreeCnt = stack[pageFreeIndex]
                subStack = list(stack[subStackIndex])

                if pageFreeCnt == 0:
                    continue

                if not subStack:
                    symbolStack = "    None"
                else:
                    ilen = len("\t" * 16)
                    symbolStack = self.makeKernelSymList(subStack, ilen)

                SysMgr.printPipe(
                    "\t+ {0:>8} |{1:32}".format(
                        convSize(pageFreeCnt << 12), symbolStack
                    )
                )

            SysMgr.printPipe(oneLine)

        if self.pageUnknownFreeCnt == 0:
            SysMgr.printPipe("    None\n%s" % oneLine)

    def printKnownMemFreeInfo(self):
        title = "Function Alloc-Free-Page Info"
        lineLength = SysMgr.lineLength
        diff = self.pageAllocCnt - self.pageUsageCnt
        convSize = UtilMgr.convSize2Unit
        size = convSize(diff << 12)

        # Print page alloc-free pair in user space #
        if SysMgr.userEnable:
            SysMgr.clearPrint()
            SysMgr.printPipe(
                "[%s] [Total: %s] (USER)\n%s" % (title, size, twoLine)
            )

            SysMgr.printPipe(
                (
                    "{0:^7}({1:^6}/{2:^6}/{3:^6})|"
                    "{4:_^47}|{5:_^40}|{6:_^35}\n{7:1}"
                ).format(
                    "Usage",
                    "Usr",
                    "Buf",
                    "Ker",
                    "Function",
                    "LifeTime",
                    "Binary",
                    twoLine,
                )
            )

            for idx, value in sorted(
                self.userSymData.items(),
                key=lambda e: e[1]["pagePairCnt"],
                reverse=True,
            ):
                if value["pagePairCnt"] == 0:
                    break

                typeList = {"USER": 0, "KERNEL": 0, "CACHE": 0}

                for pairId, item in value["pagePair"].items():
                    for ptype, cnt in item["valueList"].items():
                        try:
                            typeList[ptype] += cnt
                        except:
                            pass

                try:
                    avrTime = float(
                        value["pagePairTotal"] / value["pagePairCnt"]
                    )
                except:
                    avrTime = 0

                lifeTime = " AVR: %.3f / MIN: %.3f / MAX: %.3f" % (
                    avrTime,
                    value["pagePairMin"],
                    value["pagePairMax"],
                )

                SysMgr.printPipe(
                    (
                        "{0:>7}({1:>6}/{2:>6}/{3:>6})|{4:^47}|{5:40}| {6:1}"
                    ).format(
                        convSize(value["pagePairCnt"] << 12),
                        convSize(typeList["USER"] << 12),
                        convSize(typeList["CACHE"] << 12),
                        convSize(typeList["KERNEL"] << 12),
                        idx,
                        lifeTime,
                        self.posData[value["pos"]]["origBin"],
                    )
                )

                for pairId, item in sorted(
                    value["pagePair"].items(),
                    key=lambda e: e[1]["size"],
                    reverse=True,
                ):
                    try:
                        userPages = item["valueList"]["USER"]
                    except:
                        userPages = 0
                    try:
                        cachePages = item["valueList"]["CACHE"]
                    except:
                        cachePages = 0
                    try:
                        kernelPages = item["valueList"]["KERNEL"]
                    except:
                        kernelPages = 0

                    # get user alloc and free call #
                    allocCall, freeCall = pairId.split("#")

                    printBuf = "{0:4}+ {1:>7}({2:>6}/{3:>6}/{4:>6})| ".format(
                        " ",
                        convSize(item["size"] << 12),
                        convSize(userPages << 12),
                        convSize(cachePages << 12),
                        convSize(kernelPages << 12),
                    )

                    ilen = len(printBuf)
                    tlen = ilen

                    for seq, call in enumerate(allocCall.split(" <- ")):
                        if seq > 0 and tlen + len(call) > lineLength:
                            printBuf = "%s\n%s" % (printBuf, " " * ilen)
                            tlen = ilen
                        printBuf = "%s<- %s " % (printBuf, call)
                        tlen += len(call) + 4

                    SysMgr.printPipe(printBuf)

                    printBuf = "{0:5}{1:>30}|".format(" ", "[FREE]")
                    ilen = len(printBuf)
                    tlen = ilen

                    for index, call in enumerate(freeCall.split(" <- ")):
                        clen = len(call) + 4

                        if index == 0:
                            clen -= 4

                        if index > 0 and tlen + clen > lineLength:
                            printBuf = "%s\n%s" % (printBuf, " " * ilen)
                            tlen = ilen

                        if index == 0:
                            printBuf = "%s %s" % (printBuf, call)
                        else:
                            printBuf = "%s <- %s" % (printBuf, call)

                        tlen += clen

                    SysMgr.printPipe(printBuf)

                SysMgr.printPipe(oneLine)

            if self.pageAllocCnt - self.pageUsageCnt <= 0:
                SysMgr.printPipe("    None\n%s" % oneLine)

            SysMgr.printPipe("\n")

        # Print page alloc-free pair in kernel space #
        SysMgr.clearPrint()
        SysMgr.printPipe(
            "[%s] [Total: %s] (KERNEL)\n%s" % (title, size, twoLine)
        )

        SysMgr.printPipe(
            "{0:^7}({1:^6}/{2:^6}/{3:^6})|{4:_^47}|{5:_^76}\n{6:1}".format(
                "Usage", "Usr", "Buf", "Ker", "Function", "LifeTime", twoLine
            )
        )

        # Print mem usage of stacks #
        for idx, value in sorted(
            self.kerSymData.items(),
            key=lambda e: e[1]["pagePairCnt"],
            reverse=True,
        ):
            if value["pagePairCnt"] == 0:
                break

            typeList = {"USER": 0, "KERNEL": 0, "CACHE": 0}

            for pairId, item in value["pagePair"].items():
                for ptype, cnt in item["valueList"].items():
                    try:
                        typeList[ptype] += cnt
                    except:
                        pass

            try:
                avrTime = float(value["pagePairTotal"] / value["pagePairCnt"])
            except:
                avrTime = 0

            lifeTime = " AVR: %.3f / MIN: %.3f / MAX: %.3f" % (
                avrTime,
                value["pagePairMin"],
                value["pagePairMax"],
            )

            SysMgr.printPipe(
                "{0:>7}({1:>6}/{2:>6}/{3:>6})|{4:^47}|{5:^75}".format(
                    convSize(value["pagePairCnt"] << 12),
                    convSize(typeList["USER"] << 12),
                    convSize(typeList["CACHE"] << 12),
                    convSize(typeList["KERNEL"] << 12),
                    idx,
                    lifeTime,
                )
            )

            for pairId, item in sorted(
                value["pagePair"].items(),
                key=lambda e: e[1]["size"],
                reverse=True,
            ):
                try:
                    userPages = item["valueList"]["USER"]
                except:
                    userPages = 0
                try:
                    cachePages = item["valueList"]["CACHE"]
                except:
                    cachePages = 0
                try:
                    kernelPages = item["valueList"]["KERNEL"]
                except:
                    kernelPages = 0

                # get kernel alloc and free call #
                allocCall, freeCall = pairId.split("#")

                printBuf = "{0:4}+ {1:>7}({2:>6}/{3:>6}/{4:>6})| ".format(
                    " ",
                    convSize(item["size"] << 12),
                    convSize(userPages << 12),
                    convSize(cachePages << 12),
                    convSize(kernelPages << 12),
                )

                ilen = len(printBuf)
                tlen = ilen

                for seq, call in enumerate(allocCall.split(" <- ")):
                    if seq > 0 and tlen + len(call) > lineLength:
                        printBuf = "%s\n%s" % (printBuf, " " * ilen)
                        tlen = ilen
                    printBuf = "%s<- %s " % (printBuf, call)
                    tlen += len(call) + 4

                SysMgr.printPipe(printBuf)

                printBuf = "{0:5}{1:>30}|".format(" ", "[FREE]")
                ilen = len(printBuf)
                tlen = ilen

                for index, call in enumerate(freeCall.split(" <- ")):
                    clen = len(call) + 4

                    if index == 0:
                        clen -= 4

                    if index > 0 and tlen + clen > lineLength:
                        printBuf = "%s\n%s" % (printBuf, " " * ilen)
                        tlen = ilen

                    if index == 0:
                        printBuf = "%s %s" % (printBuf, call)
                    else:
                        printBuf = "%s <- %s" % (printBuf, call)

                    tlen += clen

                SysMgr.printPipe(printBuf)

            SysMgr.printPipe(oneLine)

        if self.pageAllocCnt - self.pageUsageCnt <= 0:
            SysMgr.printPipe("    None\n%s" % oneLine)

        SysMgr.printPipe()

    def printMemUsage(self):
        # check memory event #
        if not self.memEnabled:
            return

        title = "Function Alloc-Only-Page Info"
        subStackIndex = FunctionAnalyzer.symStackIdxTable.index("STACK")
        pageAllocIndex = FunctionAnalyzer.symStackIdxTable.index("PAGE_ALLOC")
        argIndex = FunctionAnalyzer.symStackIdxTable.index("ARGUMENT")

        convSize = UtilMgr.convSize2Unit
        userSize = convSize(self.pageUsageCnt << 12)
        allocSize = convSize(self.pageAllocCnt << 12)
        freeSize = convSize(self.pageFreeCnt << 12)
        allocCnt = UtilMgr.convNum(self.pageAllocEventCnt)
        freeCnt = UtilMgr.convNum(self.pageFreeEventCnt)

        # Calculate page lifetime #
        for pfn, item in self.pageTable.items():
            if not item:
                continue

            # calculate time #
            time = float(item["time"])
            if time > 0:
                lifeTime = float(self.finishTime) - time
            else:
                lifeTime = 0

            # Set user page lifetime #
            self.userSymData[item["sym"]]["pageRemainTotal"] += lifeTime
            if (
                self.userSymData[item["sym"]]["pageRemainMin"] == 0
                or self.userSymData[item["sym"]]["pageRemainMin"] > lifeTime
            ):
                self.userSymData[item["sym"]]["pageRemainMin"] = lifeTime
            if self.userSymData[item["sym"]]["pageRemainMax"] < lifeTime:
                self.userSymData[item["sym"]]["pageRemainMax"] = lifeTime

            # Set kernel page lifetime #
            self.kerSymData[item["ksym"]]["pageRemainTotal"] += lifeTime
            if (
                self.kerSymData[item["ksym"]]["pageRemainMin"] == 0
                or self.kerSymData[item["ksym"]]["pageRemainMin"] > lifeTime
            ):
                self.kerSymData[item["ksym"]]["pageRemainMin"] = lifeTime
            if self.kerSymData[item["ksym"]]["pageRemainMax"] < lifeTime:
                self.kerSymData[item["ksym"]]["pageRemainMax"] = lifeTime

        # Print memory usage by page allocation in user space #
        if SysMgr.userEnable:
            SysMgr.clearPrint()
            SysMgr.printPipe(
                "[%s] [Total: %s] [Alloc: %s(%s)] [Free: %s(%s)] (USER)\n%s"
                % (
                    title,
                    userSize,
                    allocSize,
                    allocCnt,
                    freeSize,
                    freeCnt,
                    twoLine,
                )
            )

            SysMgr.printPipe(
                (
                    "{0:^7}({1:^6}/{2:^6}/{3:^6})|"
                    "{4:_^47}|{5:_^40}|{6:_^35}\n{7:1}"
                ).format(
                    "Usage",
                    "Usr",
                    "Buf",
                    "Ker",
                    "Function",
                    "LifeTime",
                    "Binary",
                    twoLine,
                )
            )

            for idx, value in sorted(
                self.userSymData.items(),
                key=lambda e: e[1]["pageCnt"],
                reverse=True,
            ):
                if value["pageCnt"] == 0:
                    break

                try:
                    avrTime = float(
                        value["pageRemainTotal"] / value["pageCnt"]
                    )
                except:
                    avrTime = 0

                lifeTime = " AVR: %.3f / MIN: %.3f / MAX: %.3f" % (
                    avrTime,
                    value["pageRemainMin"],
                    value["pageRemainMax"],
                )

                SysMgr.printPipe(
                    (
                        "{0:>7}({1:>6}/{2:>6}/{3:>6})|{4:^47}|{5:40}| {6:1}"
                    ).format(
                        convSize(value["pageCnt"] << 12),
                        convSize(value["userPageCnt"] << 12),
                        convSize(value["cachePageCnt"] << 12),
                        convSize(value["kernelPageCnt"] << 12),
                        idx,
                        lifeTime,
                        self.posData[value["pos"]]["origBin"],
                    )
                )

                # Set target stack #
                targetStack = self.getTargetStack(value, pageAllocIndex)

                # Merge and Print symbols in stack #
                for stack in targetStack:
                    subStack = list(stack[subStackIndex])
                    pageCnt = stack[pageAllocIndex]
                    userPageCnt = stack[argIndex][0]
                    cachePageCnt = stack[argIndex][1]
                    kernelPageCnt = stack[argIndex][2]

                    if pageCnt == 0:
                        break

                    if not subStack:
                        continue
                    else:
                        ilen = len("\t" * 36)
                        symbolStack = self.makeUserSymList(subStack, ilen)

                    SysMgr.printPipe(
                        "\t+ {0:>7}({1:>6}/{2:>6}/{3:>6})|{4:32}".format(
                            convSize(pageCnt << 12),
                            convSize(userPageCnt << 12),
                            convSize(cachePageCnt << 12),
                            convSize(kernelPageCnt << 12),
                            symbolStack,
                        )
                    )

                SysMgr.printPipe(oneLine)

            if self.pageUsageCnt == 0:
                SysMgr.printPipe("    None\n%s" % oneLine)

            SysMgr.printPipe("\n")

        # Print memory usage by page allocation in kernel space #
        SysMgr.clearPrint()
        SysMgr.printPipe(
            "[%s] [Total: %s] [Alloc: %s(%s)] [Free: %s(%s)] (KERNEL)\n%s"
            % (
                title,
                userSize,
                allocSize,
                allocCnt,
                freeSize,
                freeCnt,
                twoLine,
            )
        )

        SysMgr.printPipe(
            "{0:^7}({1:^6}/{2:^6}/{3:^6})|{4:_^47}|{5:_^76}\n{6:1}".format(
                "Usage", "Usr", "Buf", "Ker", "Function", "LifeTime", twoLine
            )
        )

        # Print mem usage of stacks #
        for idx, value in sorted(
            self.kerSymData.items(),
            key=lambda e: e[1]["pageCnt"],
            reverse=True,
        ):
            if value["pageCnt"] == 0:
                break

            try:
                avrTime = float(value["pageRemainTotal"] / value["pageCnt"])
            except:
                avrTime = 0

            lifeTime = " AVR: %.3f / MIN: %.3f / MAX: %.3f" % (
                avrTime,
                value["pageRemainMin"],
                value["pageRemainMax"],
            )

            SysMgr.printPipe(
                "{0:>7}({1:>6}/{2:>6}/{3:>6})|{4:^47}|{5:^76}".format(
                    convSize(value["pageCnt"] << 12),
                    convSize(value["userPageCnt"] << 12),
                    convSize(value["cachePageCnt"] << 12),
                    convSize(value["kernelPageCnt"] << 12),
                    idx,
                    lifeTime,
                )
            )

            # Sort stacks by usage #
            value["stack"] = sorted(
                value["stack"], key=lambda x: x[pageAllocIndex], reverse=True
            )

            # Print stacks by symbol #
            for stack in value["stack"]:
                subStack = list(stack[subStackIndex])
                pageCnt = stack[pageAllocIndex]
                userPageCnt = stack[argIndex][0]
                cachePageCnt = stack[argIndex][1]
                kernelPageCnt = stack[argIndex][2]

                if pageCnt == 0:
                    continue

                if not subStack:
                    continue
                else:
                    ilen = len("\t" * 36)
                    symbolStack = self.makeKernelSymList(subStack, ilen)

                SysMgr.printPipe(
                    "\t+ {0:>7}({1:>6}/{2:>6}/{3:>6})|{4:32}".format(
                        convSize(pageCnt << 12),
                        convSize(userPageCnt << 12),
                        convSize(cachePageCnt << 12),
                        convSize(kernelPageCnt << 12),
                        symbolStack,
                    )
                )

            SysMgr.printPipe(oneLine)

        if self.pageUsageCnt == 0:
            SysMgr.printPipe("    None\n%s" % oneLine)

        SysMgr.printPipe("\n")

        self.printKnownMemFreeInfo()

        self.printUnknownMemFreeInfo()

        SysMgr.printPipe("\n\n")

    def printHeapUsage(self):
        # check heap memory event #
        if not self.heapEnabled or not SysMgr.userEnable:
            return

        title = "Function Expand-Heap"
        subStackIndex = FunctionAnalyzer.symStackIdxTable.index("STACK")
        heapExpIndex = FunctionAnalyzer.symStackIdxTable.index("HEAP_EXPAND")
        convSize = UtilMgr.convSize2Unit

        # Print heap usage in user space #
        SysMgr.clearPrint()
        SysMgr.printPipe(
            "[%s Info] [Total: %s] [Alloc: %s(%s)] [Free: %s(%s)] (USER)%s\n"
            % (
                title,
                convSize(self.heapExpSize - self.heapRedSize),
                convSize(self.heapExpSize),
                UtilMgr.convNum(self.heapExpEventCnt),
                convSize(self.heapRedSize),
                UtilMgr.convNum(self.heapRedEventCnt),
                twoLine,
            )
        )

        SysMgr.printPipe(
            "{0:_^9}|{1:_^47}|{2:_^49}|{3:_^46}\n{4:1}".format(
                "Usage", "Function", "Binary", "Source", twoLine
            )
        )

        for idx, value in sorted(
            self.userSymData.items(),
            key=lambda e: e[1]["heapSize"],
            reverse=True,
        ):
            if value["heapSize"] == 0:
                break

            binary = self.posData[value["pos"]]["origBin"]
            source = self.posData[value["pos"]]["src"]
            SysMgr.printPipe(
                "{0:>8} |{1:^47}| {2:48}| {3:37}".format(
                    convSize(value["heapSize"]), idx, binary, source
                )
            )

            if idx == value["pos"]:
                SysMgr.printPipe(oneLine)
                continue

            # Set target stack #
            targetStack = self.getTargetStack(value, heapExpIndex)

            # Merge and Print symbols in stack #
            for stack in targetStack:
                heapSize = stack[heapExpIndex]
                subStack = list(stack[subStackIndex])

                if heapSize == 0:
                    break

                if not subStack:
                    continue
                else:
                    ilen = len("\t" * 16)
                    symbolStack = self.makeUserSymList(subStack, ilen)

                SysMgr.printPipe(
                    "\t+ {0:>8} |{1:32}".format(
                        convSize(heapSize), symbolStack
                    )
                )

            SysMgr.printPipe(oneLine)

        if not self.heapTable:
            SysMgr.printPipe("    None\n%s" % oneLine)

        SysMgr.printPipe("\n")

        # Print remaining heap history #
        if not SysMgr.showAll or not self.heapTable:
            SysMgr.printPipe("\n\n")
            return

        SysMgr.clearPrint()
        SysMgr.printPipe(
            "[%s History] [Cnt: %s]\n%s"
            % (title, UtilMgr.convNum(len(self.heapTable)), twoLine)
        )

        SysMgr.printPipe(
            (
                "{0:_^32}|{1:_^12}|{2:_^17}({3:_^7})|"
                "{4:_^8}|{5:_^17}|\n{6:1}"
            ).format("VAddr", "Size", "COMM", "TID", "CORE", "TIME", twoLine)
        )

        # sort by time #
        for segment in sorted(
            self.heapTable.items(), key=lambda e: e[1]["time"]
        ):
            addr = segment[0]

            size = segment[1]["size"]
            time = segment[1]["time"]
            core = segment[1]["core"]
            tid = segment[1]["tid"]

            usersym = segment[1]["sym"]
            kernelsym = segment[1]["ksym"]
            userstack = segment[1]["subStackAddr"]
            kernelstack = segment[1]["ksubStackAddr"]

            title = (
                "{0:^32}| {1:>10} | {2:>16}({3:>7})|{4:>6} | {5:>15} |"
            ).format(
                addr,
                convSize(size),
                self.threadData[tid]["comm"],
                tid,
                long(core),
                time,
            )
            SysMgr.printPipe("%s\n%s" % (title, len(title) * "-"))

            # Make user call info #
            ilen = 32
            nowLen = ilen
            try:
                userCall = " %s[%s]" % (
                    usersym,
                    self.userSymData[usersym]["origBin"],
                )
                nowLen += len(userCall)

                # Set user stack list #
                if self.sort == "sym":
                    targetStack = self.userSymData[usersym]["symStack"]
                elif self.sort == "pos":
                    targetStack = self.userSymData[usersym]["stack"]

                # Find user stack by addr #
                stack = []
                for val in targetStack:
                    if id(val[subStackIndex]) == userstack:
                        stack = val[subStackIndex]
                        break

                for subcall in stack:
                    try:
                        nextCall = " <- %s[%s]" % (
                            subcall,
                            self.userSymData[subcall]["origBin"],
                        )
                        if SysMgr.lineLength > nowLen + len(nextCall):
                            userCall += nextCall
                            nowLen += len(nextCall)
                        else:
                            userCall = "%s\n%s %s" % (
                                userCall,
                                " " * ilen,
                                nextCall,
                            )
                            nowLen = ilen + len(nextCall)
                    except:
                        pass
            except:
                pass

            # Make kernel call info #
            ilen = 32
            nowLen = ilen
            try:
                kernelCall = " %s" % (kernelsym)
                nowLen += len(kernelCall)

                # Set kernel stack list #
                if self.sort == "sym":
                    targetStack = self.kerSymData[kernelsym]["symStack"]
                elif self.sort == "pos":
                    targetStack = self.kerSymData[kernelsym]["stack"]

                # Find kernel stack by addr #
                stack = []
                for val in targetStack:
                    if id(val[subStackIndex]) == kernelstack:
                        stack = val[subStackIndex]
                        break

                for subcall in stack:
                    try:
                        nextCall = " <- %s" % (subcall)
                        if SysMgr.lineLength > nowLen + len(nextCall):
                            kernelCall += nextCall
                            nowLen += len(nextCall)
                        else:
                            kernelCall = "%s\n%s %s" % (
                                kernelCall,
                                " " * ilen,
                                nextCall,
                            )
                            nowLen = ilen + len(nextCall)
                    except:
                        pass
            except:
                pass

            if userCall != " 0":
                SysMgr.printPipe(
                    "{0:>32}|{1:<121}".format("[User] ", userCall)
                )
            if kernelCall != " 0":
                SysMgr.printPipe(
                    "{0:>32}|{1:<121}".format("[Kernel] ", kernelCall)
                )
            SysMgr.printPipe(oneLine)

        SysMgr.printPipe("\n\n")

    def printLockUsage(self):
        # no lock event #
        if not self.lockEnabled or not SysMgr.userEnable:
            return

        subStackIndex = FunctionAnalyzer.symStackIdxTable.index("STACK")
        lockIndex = FunctionAnalyzer.symStackIdxTable.index("LOCK_TRY")
        unlockIndex = FunctionAnalyzer.symStackIdxTable.index("UNLOCK")

        # Print lock try #
        title = "Function Lock-Try Info"
        SysMgr.clearPrint()
        SysMgr.printPipe(
            "[%s] [Cnt: %d] (USER)\n%s"
            % (title, self.lockTryEventCnt, twoLine)
        )

        SysMgr.printPipe(
            "{0:_^9}|{1:_^47}|{2:_^49}|{3:_^46}\n{4:1}".format(
                "Usage", "Function", "Binary", "Source", twoLine
            )
        )

        for idx, value in sorted(
            self.userSymData.items(),
            key=lambda e: e[1]["lockTryCnt"],
            reverse=True,
        ):
            if value["lockTryCnt"] == 0:
                break

            binary = self.posData[value["pos"]]["origBin"]
            source = self.posData[value["pos"]]["src"]
            SysMgr.printPipe(
                "{0:8} |{1:^47}| {2:48}| {3:37}".format(
                    value["lockTryCnt"], idx, binary, source
                )
            )

            # Set target stack #
            targetStack = self.getTargetStack(value, lockIndex)

            # Merge and Print symbols in stack #
            for stack in targetStack:
                lockTryCnt = stack[lockIndex]
                subStack = list(stack[subStackIndex])

                if lockTryCnt == 0:
                    break

                if not subStack:
                    continue
                else:
                    ilen = len("\t" * 16)
                    symbolStack = self.makeUserSymList(subStack, ilen)

                SysMgr.printPipe(
                    "\t+ {0:8} |{1:32}".format(lockTryCnt, symbolStack)
                )

            SysMgr.printPipe(oneLine)

        if self.lockTryEventCnt == 0:
            SysMgr.printPipe("    None\n%s" % oneLine)

        SysMgr.printPipe("\n")

        # Print lock per-file #
        title = "Function Lock-Try File Info"
        SysMgr.printPipe(
            "[%s] [Cnt: %d] (USER)\n%s"
            % (title, self.lockTryEventCnt, twoLine)
        )

        SysMgr.printPipe(
            "{0:_^9}|{1:_^144}\n{2:1}".format("Usage", "Binary", twoLine)
        )

        for idx, value in sorted(
            self.userFileData.items(),
            key=lambda e: e[1]["lockTryCnt"],
            reverse=True,
        ):
            if value["lockTryCnt"] == 0:
                break

            SysMgr.printPipe(
                "{0:8} | {1:<142}\n{2:1}".format(
                    value["lockTryCnt"], idx, oneLine
                )
            )

        if self.periodicEventCnt == 0:
            SysMgr.printPipe("    None\n%s" % oneLine)

        SysMgr.printPipe("\n")

        # Print unlock #
        title = "Function Unlock Info"
        SysMgr.clearPrint()
        SysMgr.printPipe(
            "[%s] [Cnt: %d] (USER)\n%s" % (title, self.unlockEventCnt, twoLine)
        )

        SysMgr.printPipe(
            "{0:_^9}|{1:_^47}|{2:_^49}|{3:_^46}\n{4:1}".format(
                "Usage", "Function", "Binary", "Source", twoLine
            )
        )

        for idx, value in sorted(
            self.userSymData.items(),
            key=lambda e: e[1]["unlockCnt"],
            reverse=True,
        ):
            if value["unlockCnt"] == 0:
                break

            binary = self.posData[value["pos"]]["origBin"]
            source = self.posData[value["pos"]]["src"]
            SysMgr.printPipe(
                "{0:8} |{1:^47}| {2:48}| {3:37}".format(
                    value["unlockCnt"], idx, binary, source
                )
            )

            # Set target stack #
            targetStack = self.getTargetStack(value, unlockIndex)

            # Merge and Print symbols in stack #
            for stack in targetStack:
                unlockCnt = stack[unlockIndex]
                subStack = list(stack[subStackIndex])

                if unlockCnt == 0:
                    break

                if not subStack:
                    continue
                else:
                    ilen = len("\t" * 16)
                    symbolStack = self.makeUserSymList(subStack, ilen)

                SysMgr.printPipe(
                    "\t+ {0:8} |{1:32}".format(unlockCnt, symbolStack)
                )

            SysMgr.printPipe(oneLine)

        if self.unlockEventCnt == 0:
            SysMgr.printPipe("    None\n%s" % oneLine)

        SysMgr.printPipe("\n")

        # Print unlock per-file #
        title = "Function Unlock File Info"
        SysMgr.printPipe(
            "[%s] [Cnt: %d] (USER)\n%s"
            % (title, self.lockTryEventCnt, twoLine)
        )

        SysMgr.printPipe(
            "{0:_^9}|{1:_^144}\n{2:1}".format("Usage", "Binary", twoLine)
        )

        for idx, value in sorted(
            self.userFileData.items(),
            key=lambda e: e[1]["unlockCnt"],
            reverse=True,
        ):
            if value["unlockCnt"] == 0:
                break

            SysMgr.printPipe(
                "{0:8} | {1:<142}".format(value["unlockCnt"], idx)
            )

            SysMgr.printPipe(oneLine)

        if self.periodicEventCnt == 0:
            SysMgr.printPipe("    None\n%s" % oneLine)

        SysMgr.printPipe("\n")

        # Print lock history #
        if not SysMgr.showAll or not self.lockCallData:
            SysMgr.printPipe("\n\n")
            return

        title = "Function Lock History"
        SysMgr.clearPrint()
        SysMgr.printPipe(
            "[%s] [Lock: %d] [Unlock: %d]\n%s"
            % (title, self.lockTryEventCnt, self.unlockEventCnt, twoLine)
        )

        SysMgr.printPipe(
            (
                "{0:_^32}|{1:_^16}|{2:_^17}({3:_^7})|"
                "{4:_^8}|{5:_^17}|\n{6:1}"
            ).format("Event", "Target", "COMM", "TID", "Core", "Time", twoLine)
        )

        # sort by time #
        for call in self.lockCallData:
            event = call[0]
            target = call[1][0]
            time = call[1][1]
            core = call[1][2]
            tid = call[1][3]
            userstack = call[2]

            comm = self.threadData[tid]["comm"]
            title = (
                "{0:^32}|{1:^16}|{2:>16}({3:>7})| {4:>6} | {5:>15} |".format(
                    event, target, comm, tid, core, time
                )
            )
            SysMgr.printPipe("%s\n%s" % (title, len(title) * "-"))

            # Make user call info #
            ilen = 32
            nowLen = ilen
            try:
                last = userstack[0]
                stack = userstack[1]
                symbol = self.posData[last]["symbol"]
                binary = self.posData[last]["binary"]
                userCall = " %s[%s]" % (symbol, binary)
                nowLen += len(userCall)
                for subcall in stack:
                    try:
                        symbol = self.posData[subcall]["symbol"]
                        binary = self.posData[subcall]["binary"]
                        nextCall = " <- %s[%s]" % (symbol, binary)
                        if SysMgr.lineLength > nowLen + len(nextCall):
                            userCall += nextCall
                            nowLen += len(nextCall)
                        else:
                            userCall = "%s\n%s %s" % (
                                userCall,
                                " " * ilen,
                                nextCall,
                            )
                            nowLen = ilen + len(nextCall)
                    except:
                        pass
            except SystemExit:
                sys.exit(0)
            except:
                pass

            SysMgr.printPipe(
                "{0:>32}|{1:<121}\n{2:1}".format("[User] ", userCall, oneLine)
            )

        SysMgr.printPipe("\n\n")

    def printBlockWrUsage(self):
        # no block write event #
        if not self.bwriteEnabled:
            return

        title = "Function Write-Block Info"
        subStackIndex = FunctionAnalyzer.symStackIdxTable.index("STACK")
        blkWrIndex = FunctionAnalyzer.symStackIdxTable.index("BLK_WRITE")
        convSize = UtilMgr.convSize2Unit
        convNum = UtilMgr.convNum
        size = convSize(self.blockWrUsageCnt << 9)

        # Print block write in user space #
        if SysMgr.userEnable:
            SysMgr.clearPrint()
            SysMgr.printPipe(
                "[%s] [Size: %s] [Cnt: %s] (USER)\n%s"
                % (title, size, convNum(self.blockWrEventCnt), twoLine)
            )

            SysMgr.printPipe(
                "{0:_^9}|{1:_^47}|{2:_^49}|{3:_^46}\n{4:1}".format(
                    "Usage", "Function", "Binary", "Source", twoLine
                )
            )

            for idx, value in sorted(
                self.userSymData.items(),
                key=lambda e: e[1]["blockWrCnt"],
                reverse=True,
            ):
                if value["blockWrCnt"] == 0:
                    break

                binary = self.posData[value["pos"]]["origBin"]
                source = self.posData[value["pos"]]["src"]
                SysMgr.printPipe(
                    "{0:>8} |{1:^47}| {2:48}| {3:37}".format(
                        convSize(value["blockWrCnt"] << 9), idx, binary, source
                    )
                )

                # Set target stack #
                targetStack = self.getTargetStack(value, blkWrIndex)

                # Merge and Print symbols in stack #
                for stack in targetStack:
                    blockWrCnt = stack[blkWrIndex]
                    subStack = list(stack[subStackIndex])

                    if blockWrCnt == 0:
                        break

                    if not subStack:
                        continue
                    else:
                        ilen = len("\t" * 16)
                        symbolStack = self.makeUserSymList(subStack, ilen)

                    SysMgr.printPipe(
                        "\t+ {0:>8} |{1:32}".format(
                            convSize(blockWrCnt << 9), symbolStack
                        )
                    )

                SysMgr.printPipe(oneLine)

            if self.blockWrUsageCnt == 0:
                SysMgr.printPipe("    None\n%s" % oneLine)

            SysMgr.printPipe("\n")

        # Print block write in kernel space #
        SysMgr.clearPrint()
        SysMgr.printPipe(
            "[%s] [Size: %s] [Cnt: %s] (KERNEL)\n%s"
            % (title, size, convNum(self.blockWrEventCnt), twoLine)
        )

        SysMgr.printPipe(
            "{0:_^9}|{1:_^144}\n{2:1}".format("Usage", "Function", twoLine)
        )

        # Print block write usage of stacks #
        for idx, value in sorted(
            self.kerSymData.items(),
            key=lambda e: e[1]["blockWrCnt"],
            reverse=True,
        ):
            if value["blockWrCnt"] == 0:
                break

            SysMgr.printPipe(
                "{0:>8} |{1:^134}".format(
                    convSize(value["blockWrCnt"] << 9), idx
                )
            )

            # Sort stacks by usage #
            value["stack"] = sorted(
                value["stack"], key=lambda x: x[blkWrIndex], reverse=True
            )

            # Print stacks by symbol #
            for stack in value["stack"]:
                blockWrCnt = stack[blkWrIndex]
                subStack = list(stack[subStackIndex])

                if blockWrCnt == 0:
                    continue

                if not subStack:
                    symbolStack = "    None"
                else:
                    ilen = len("\t" * 16)
                    symbolStack = self.makeKernelSymList(subStack, ilen)

                SysMgr.printPipe(
                    "\t+ {0:>8} |{1:32}".format(
                        convSize(blockWrCnt << 9), symbolStack
                    )
                )

            SysMgr.printPipe(oneLine)

        if self.blockWrUsageCnt == 0:
            SysMgr.printPipe("    None\n%s" % oneLine)

        SysMgr.printPipe("\n\n")

    def getExceptionList(self):
        exceptList = {}

        # do not use this method now #
        return exceptList

        for pos, value in self.posData.items():
            if value["symbol"] == "None":
                try:
                    exceptList[pos]
                except:
                    exceptList[pos] = {}

        return exceptList

    def getTargetStack(self, value, index=None):
        targetStack = []
        if self.sort == "sym":
            targetStack = value["symStack"]
        elif self.sort == "pos":
            targetStack = value["stack"]

        # Sort by usage #
        if index:
            targetStack = sorted(
                targetStack, key=lambda x: x[index], reverse=True
            )
        else:
            targetStack.sort(reverse=True)

        return targetStack

    def printBlockRdUsage(self):
        # no block read event #
        if not self.breadEnabled:
            return

        title = "Function Read-Block Info"
        subStackIndex = FunctionAnalyzer.symStackIdxTable.index("STACK")
        blkRdIndex = FunctionAnalyzer.symStackIdxTable.index("BLK_READ")
        convSize = UtilMgr.convSize2Unit
        convNum = UtilMgr.convNum
        size = convSize(self.blockRdUsageCnt << 9)

        # Print block read in user space #
        if SysMgr.userEnable:
            SysMgr.clearPrint()
            SysMgr.printPipe(
                "[%s] [Size: %s] [Cnt: %s] (USER)\n%s"
                % (title, size, convNum(self.blockRdEventCnt), twoLine)
            )

            SysMgr.printPipe(
                "{0:_^9}|{1:_^47}|{2:_^49}|{3:_^46}\n{4:1}".format(
                    "Usage", "Function", "Binary", "Source", twoLine
                )
            )

            for idx, value in sorted(
                self.userSymData.items(),
                key=lambda e: e[1]["blockRdCnt"],
                reverse=True,
            ):
                if value["blockRdCnt"] == 0:
                    break

                binary = self.posData[value["pos"]]["origBin"]
                source = self.posData[value["pos"]]["src"]
                SysMgr.printPipe(
                    "{0:>8} |{1:^47}| {2:48}| {3:37}".format(
                        convSize(value["blockRdCnt"] << 9), idx, binary, source
                    )
                )

                # Set target stack #
                targetStack = self.getTargetStack(value, blkRdIndex)

                # Merge and Print symbols in stack #
                for stack in targetStack:
                    blockRdCnt = stack[blkRdIndex]
                    subStack = list(stack[subStackIndex])

                    if blockRdCnt == 0:
                        break

                    if not subStack:
                        continue
                    else:
                        ilen = len("\t" * 16)
                        symbolStack = self.makeUserSymList(subStack, ilen)

                    SysMgr.printPipe(
                        "\t+ {0:8} |{1:32}".format(
                            convSize(blockRdCnt << 9), symbolStack
                        )
                    )

                SysMgr.printPipe(oneLine)

            SysMgr.printPipe("\n")

        # Print block read in kernel space #
        SysMgr.clearPrint()
        SysMgr.printPipe(
            "[%s] [Size: %s] [Cnt: %s] (KERNEL)\n%s"
            % (title, size, convNum(self.blockRdEventCnt), twoLine)
        )

        SysMgr.printPipe(
            "{0:_^9}|{1:_^144}\n{2:1}".format("Usage", "Function", twoLine)
        )

        # Print block read usage of stacks #
        for idx, value in sorted(
            self.kerSymData.items(),
            key=lambda e: e[1]["blockRdCnt"],
            reverse=True,
        ):
            if value["blockRdCnt"] == 0:
                break

            SysMgr.printPipe(
                "{0:>8} |{1:^144}".format(
                    convSize(value["blockRdCnt"] << 9), idx
                )
            )

            # Sort stacks by usage #
            value["stack"] = sorted(
                value["stack"], key=lambda x: x[blkRdIndex], reverse=True
            )

            # Print stacks by symbol #
            for stack in value["stack"]:
                blockRdCnt = stack[blkRdIndex]
                subStack = list(stack[subStackIndex])

                if blockRdCnt == 0:
                    continue

                if not subStack:
                    symbolStack = "    None"
                else:
                    ilen = len("\t" * 16)
                    symbolStack = self.makeKernelSymList(subStack, ilen)

                SysMgr.printPipe(
                    "\t+ {0:>8} |{1:32}".format(
                        convSize(blockRdCnt << 9), symbolStack
                    )
                )

            SysMgr.printPipe(oneLine)

        SysMgr.printPipe("\n\n")


class LeakAnalyzer(object):
    """Analyzer for leaktracing"""

    startSig = 35  # SIGRT1
    stopSig = 12  # SIGUSR2
    markedIdlePages = False
    idlePageList = []
    repeatCnt = 0

    def __init__(self, f=None, pid=None, kill=False, tobj=None):
        self.pid = pid
        self.comm = None
        self.posData = {}
        self.symData = {}
        self.fileData = {}
        self.callData = []
        self.totalLeakSize = 0
        self.memUsage = "N/A"

        self.init_posData = {
            "offset": 0,
            "path": None,
            "lastPosCnt": 0,
            "callList": None,
            "count": 0,
            "size": 0,
            "lastPosSize": 0,
            "sym": "??",
        }

        self.init_symData = {
            "offset": 0,
            "path": None,
            "lastPosCnt": 0,
            "substack": None,
            "count": 0,
            "size": 0,
            "lastPosSize": 0,
        }

        self.init_fileData = {
            "lastPosCnt": 0,
            "count": 0,
            "size": 0,
            "lastPosSize": 0,
        }

        # get file size #
        fsize = UtilMgr.getFileSizeStr(f)

        # get target info #
        try:
            # get target object #
            proc = Debugger(pid=int(pid), attach=False)

            # save memory usage #
            if tobj:
                memUsage = TaskAnalyzer.getMemStr(tobj, pid)
                if memUsage:
                    self.memUsage = memUsage

            # save comm #
            self.comm = proc.comm

            # save maps #
            try:
                maps = SysMgr.getProcMaps(pid)
                mapFd = MemoryFile(name="buf", data=maps.encode())
                proc.mapFd = mapFd
            except SystemExit:
                sys.exit(0)
            except:
                SysMgr.printErr(
                    "failed to save maps for %s(%s)" % (self.comm, self.pid),
                    True,
                )

            if kill:
                try:
                    os.kill(proc.pid, signal.SIGKILL)
                except:
                    pass
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printErr(
                "failed to analyze leakage for the task with PID %s" % pid,
                True,
            )
            sys.exit(-1)

        # get trace data #
        try:
            SysMgr.printInfo("start loading '%s'%s" % (f, fsize))

            with open(f, "r") as fd:
                self.callData = self.parseLines(fd)
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printOpenErr(f)
            sys.exit(-1)

        # resolve symbols #
        SysMgr.printInfo("start resolving symbols...")
        self.resolveSymbols(proc)

        # merge symbols #
        SysMgr.printInfo("start merging symbols...")
        self.mergeSymbols()

    def printLeakage(self, runtime, profileTime, startTime):
        # define shortcut #
        convSize = UtilMgr.convSize2Unit

        # task info #
        proc = "%s(%s)" % (self.comm, self.pid)

        # set name #
        name = "Leakage"
        if LeakAnalyzer.markedIdlePages:
            if "REPORTIDLE" in SysMgr.environList:
                name = "Idle"
            elif "REPORTACTIVE" in SysMgr.environList:
                name = "Active"

        # set title #
        title = "Function %s Info" % name
        titleStr = (
            "\n\n[%s] [Process: %s] [Start: %s] [Run: %s] [Profile: %s] "
            "[Mem: %s] [%s: %s] [NrCall: %s]"
        ) % (
            title,
            proc,
            startTime,
            runtime,
            profileTime,
            self.memUsage,
            name,
            convSize(self.totalLeakSize),
            convSize(len(self.callData)),
        )
        SysMgr.printPipe(titleStr)

        SysMgr.printPipe(
            "{4:1}\n{0:>7} | {1:>7} | {2:>7} | {3:<122} |".format(
                "Size", "Count", "Avg", "Function", twoLine
            )
        )
        SysMgr.printPipe(
            "{0:>7} | {1:>7} | {2:<132} |\n{3:1}".format(
                " ", "Size", " Backtrace", twoLine
            )
        )

        # init flamegraph variable #
        stackList = {}

        count = 0
        for sym, val in sorted(
            self.symData.items(),
            key=lambda e: long(e[1]["lastPosSize"]),
            reverse=True,
        ):
            if val["lastPosSize"] == 0:
                break

            size = val["lastPosSize"]
            count = val["lastPosCnt"]
            avg = long(size / count)

            SysMgr.printPipe(
                "{0:>7} | {1:>7} | {2:>7} | {3:<122} ".format(
                    convSize(size),
                    convSize(count),
                    convSize(avg),
                    "%s[%s]" % (sym, val["path"]),
                )
            )

            for substack, size in sorted(
                val["substack"].items(), key=lambda e: e[1], reverse=True
            ):
                SysMgr.printPipe(
                    "{0:>7} | {1:>7} | {2:<132} ".format(
                        "", convSize(size), substack
                    )
                )

                # register fullstack to the list for flamegraph #
                fullStack = " ".join(
                    [
                        item.strip()
                        for item in (
                            "%s[%s]%s" % (sym, val["path"], substack)
                        ).split("\n")
                    ]
                )
                stackList[fullStack] = size

            count += 1

            SysMgr.printPipe(oneLine)

        if count == 0:
            SysMgr.printPipe("    None\n%s" % oneLine)

        # set title #
        title = "File %s Info" % name
        SysMgr.printPipe(
            (
                "\n[%s] [Process: %s] [Start: %s] [Run: %s] [Profile: %s] "
                "[Mem: %s] [%s: %s] [NrCall: %s]\n%s"
            )
            % (
                title,
                proc,
                startTime,
                runtime,
                profileTime,
                self.memUsage,
                name,
                convSize(self.totalLeakSize),
                convSize(len(self.callData)),
                twoLine,
            )
        )

        SysMgr.printPipe(
            "{0:>7} | {1:>7} | {2:>7} | {3:<122} |\n{4:1}".format(
                "Size", "Count", "Avg", "Path", twoLine
            )
        )

        count = 0
        for fpath, val in sorted(
            self.fileData.items(),
            key=lambda e: long(e[1]["lastPosSize"]),
            reverse=True,
        ):
            if val["lastPosSize"] == 0:
                break
            elif val["lastPosCnt"] == 0:
                continue

            size = val["lastPosSize"]
            count = val["lastPosCnt"]
            avg = long(size / count)

            SysMgr.printPipe(
                "{0:>7} | {1:>7} | {2:>7} | {3:<122} |".format(
                    convSize(size),
                    convSize(count),
                    convSize(avg),
                    fpath if fpath else "",
                )
            )

            count += 1

        if count == 0:
            SysMgr.printPipe("    None")

        SysMgr.printPipe(oneLine)

        # check exit condition #
        if not self.callData:
            SysMgr.printWarn("no sample data", True)
            return

        # draw flamegraph #
        if SysMgr.inputFile:
            inputFile = SysMgr.inputFile
        else:
            inputFile = "guider.out"
        inputFile = os.path.realpath(inputFile)
        SysMgr.printStat(r"start drawing a flame graph...")
        Debugger.drawFlame(inputFile, stackList, titleStr, suffix=True)

        # print histo stats for size #
        sizeStats = UtilMgr.convList2Histo(
            [v["size"] for v in self.callData.values()]
        )
        UtilMgr.printHist(sizeStats, "Allocation", "byte")

        # check exit condition #
        if not SysMgr.showAll:
            return

        # leakage history #
        title = "Leakage History"
        SysMgr.printPipe(
            "\n[%s] [Process: %s] [Start: %s] [Total: %s] [Count: %s]\n%s"
            % (
                title,
                proc,
                startTime,
                convSize(self.totalLeakSize, True),
                convSize(len(self.callData), True),
                twoLine,
            )
        )

        SysMgr.printPipe(
            "{0:^16} | {1:^16} | {2:^6} |{3:^50}| {4:^53}\n{5:1}".format(
                "Time", "Addr", "Size", "Data", "Stack", oneLine
            )
        )

        if not self.callData:
            SysMgr.printPipe("\tNone\n%s" % oneLine)
            return

        for timeval, items in sorted(
            self.callData.items(), key=lambda e: e[0], reverse=False
        ):
            stack = " <- ".join(
                [
                    "%s(%s)[%s]" % (item[1], item[0], item[2])
                    for item in list(items["symstack"])
                ]
            )

            SysMgr.printPipe(
                "{0:>16} | {1:>16} | {2:>6} |{3:<50}| {4:<53}\n{5:1}".format(
                    timeval,
                    items["addr"] if "addr" in items else " ",
                    long(items["size"]),
                    items["data"][:-1],
                    stack,
                    oneLine,
                )
            )

    def mergeSymbols(self):
        cnt = 0
        total = len(self.posData)
        dobj = Debugger()
        for pos, val in self.posData.items():
            UtilMgr.printProgress(cnt, total)
            cnt += 1

            # merge by symbol #
            sym = val["sym"]

            try:
                symData = self.symData[sym]
                symData["count"] += val["count"]
                symData["size"] += val["size"]
                symData["lastPosSize"] += val["lastPosSize"]
                if val["lastPosSize"] > 0:
                    symData["lastPosCnt"] += val["count"]
            except SystemExit:
                sys.exit(0)
            except:
                self.symData[sym] = dict(self.init_symData)
                symData = self.symData[sym]
                symData["offset"] = val["offset"]
                symData["path"] = val["path"]
                symData["count"] = val["count"]
                symData["size"] = val["size"]
                symData["lastPosSize"] = val["lastPosSize"]
                if val["lastPosSize"] > 0:
                    symData["lastPosCnt"] += val["count"]
                symData["substack"] = {}

            # merge by backtrace #
            path = val["path"]
            if val["callList"]:
                for timeval in list(val["callList"]):
                    callinfo = self.callData[timeval]
                    substack = dobj.getBtStr(callinfo["symstack"][1:])
                    dobj.btStr = None
                    size = long(callinfo["size"])

                    try:
                        symData["substack"][substack] += size
                    except:
                        symData["substack"][substack] = size

            # merge by file #
            try:
                fileData = self.fileData[path]
                fileData["count"] += val["count"]
                fileData["size"] += val["size"]
                fileData["lastPosSize"] += val["lastPosSize"]
                if val["lastPosSize"] > 0:
                    fileData["lastPosCnt"] += val["count"]
            except SystemExit:
                sys.exit(0)
            except:
                self.fileData[path] = dict(self.init_fileData)
                fileData = self.fileData[path]
                fileData["count"] = val["count"]
                fileData["size"] = val["size"]
                fileData["lastPosSize"] = val["lastPosSize"]
                if val["lastPosSize"] > 0:
                    fileData["lastPosCnt"] = val["count"]

            self.totalLeakSize += val["lastPosSize"]

        if cnt:
            UtilMgr.deleteProgress()

    def resolveSymbols(self, proc):
        cnt = 0
        total = len(self.posData) + len(self.callData)

        # disable load messages #
        SysMgr.addEnvironVar("NOLOADMSG")

        # resolve all symbols #
        for pos, val in self.posData.items():
            UtilMgr.printProgress(cnt, total)
            cnt += 1

            try:
                ret = proc.getSymbolInfo(long(pos, 16))
                if ret and len(ret) > 3:
                    val["sym"] = ret[0]
                    val["path"] = ret[1]
                    val["offset"] = ret[2]
            except:
                SysMgr.printWarn(
                    "failed to get symbol for %s" % pos, reason=True
                )
                continue

        posCache = {}

        # resolve symbols in stacks #
        for pos, val in self.callData.items():
            UtilMgr.printProgress(cnt, total)
            cnt += 1

            if not "stack" in val:
                continue

            symstack = list(val["stack"])

            for idx, offset in enumerate(val["stack"]):
                data = self.posData[offset]
                # save stack item info in cache list #
                if not offset in posCache:
                    posCache[offset] = [
                        data["offset"],
                        data["sym"],
                        data["path"],
                    ]
                symstack[idx] = posCache[offset]

            val["symstack"] = symstack

            try:
                posid = val["stack"][0]
                self.posData[posid].setdefault("callList", {})
                self.posData[posid]["callList"][pos] = None
            except SystemExit:
                sys.exit(0)
            except:
                SysMgr.printWarn("failed to get stack from %s" % val, True)

        if cnt:
            UtilMgr.deleteProgress()

    def parseLines(self, fd):
        callinfo = {}
        total = os.fstat(fd.fileno()).st_size
        pageSize = SysMgr.PAGESIZE

        # check REPORTIDLE variable to report only idle page info #
        bitVal = -1
        if LeakAnalyzer.markedIdlePages:
            if "REPORTIDLE" in SysMgr.environList:
                bitVal = 0
            elif "REPORTACTIVE" in SysMgr.environList:
                bitVal = 1
            else:
                LeakAnalyzer.markedIdlePages = False

        # define stack cache #
        stackCache = {}

        # get size check function #
        _sizeChecker = UtilMgr.getSizeFilterFunc()

        while 1:
            try:
                line = fd.readline()
            except SystemExit:
                sys.exit(0)
            except:
                continue

            # check EOF #
            if not line:
                break

            # print progress #
            cur = fd.tell()
            if cur > total:
                total = os.fstat(fd.fileno()).st_size
            UtilMgr.printProgress(cur, total)

            # check prefix #
            if not line.startswith("leak,"):
                continue

            items = line.split(", ")
            time = None
            item = {}

            for content in items[1:]:
                try:
                    name, body = content.split("=", 1)
                except SystemExit:
                    sys.exit(0)
                except:
                    continue

                if name == "time":
                    time = body
                elif name == "stack":
                    # split callstack #
                    try:
                        if body in stackCache:
                            item[name] = stackCache[body]
                        else:
                            item[name] = body.split()
                            item[name] = item[name][1:]
                            stackCache[body] = item[name]
                    except SystemExit:
                        sys.exit(0)
                    except:
                        pass
                elif not SysMgr.showAll and name == "data":
                    continue
                else:
                    item[name] = body

            # check size value #
            if not item or not "size" in item or not item["size"].isdigit():
                continue

            # get allocation size #
            size = long(item["size"])
            if not _sizeChecker(size):
                continue

            # filter parts on idle pages #
            if LeakAnalyzer.markedIdlePages and "addr" in item:
                # get start address for a chunk #
                addr = long(item["addr"], 16)
                addrStart = long(addr / pageSize) * pageSize
                addrDiff = addr - addrStart

                # get end address for a chunk #
                sizeAligned = long((size + addrDiff + pageSize - 1) / pageSize)
                addrEnd = addrStart + sizeAligned

                # ignore active part for a chunk #
                try:
                    for mem in LeakAnalyzer.idlePageList[self.pid]:
                        mapStart, mapEnd, bitmap = mem

                        # check scope #
                        if addrStart >= mapEnd or addrEnd <= mapStart:
                            continue

                        # get start address for check #
                        start = addrStart - mapStart
                        if start < 0:
                            sizeAligned += start / pageSize
                            start = 0
                        idx = long((start) / pageSize)

                        # subtract the size of used pages #
                        for offset in range(sizeAligned):
                            targetIdx = idx + offset
                            if len(bitmap) <= targetIdx:
                                break
                            elif bitmap[targetIdx] != bitVal:
                                size -= pageSize

                        # TODO: consider spanning chunks between two pages #
                except SystemExit:
                    sys.exit(0)
                except:
                    SysMgr.printWarn(
                        "failed to check an idle chunk", True, True
                    )

                # skip all used chunks #
                if size < 1:
                    continue

                # update stack size #
                item["size"] = size

            # save pos in common area #
            for pos in item["stack"]:
                try:
                    self.posData[pos]["count"] += 1
                    self.posData[pos]["size"] += size
                except SystemExit:
                    sys.exit(0)
                except:
                    self.posData[pos] = dict(self.init_posData)
                    self.posData[pos]["count"] = 1
                    self.posData[pos]["size"] = size
                    self.posData[pos]["callList"] = {}

            try:
                lastPos = item["stack"][0]
            except:
                continue

            self.posData[lastPos]["lastPosSize"] += size

            callinfo[time] = item

        UtilMgr.deleteProgress()

        return callinfo


class FileAnalyzer(object):
    """Analyzer for file profiling"""

    procMapCache = {}
    procMapStrCache = {}
    procFdMapCache = {}

    init_mapData = {
        "offset": 0,
        "size": 0,
        "pageCnt": 0,
        "fd": None,
        "totalSize": 0,
        "fileMap": None,
        "pids": None,
        "linkCnt": 0,
        "inode": None,
        "accessTime": None,
        "devid": None,
        "isRep": True,
        "perm": None,
        "nrOpen": 0,
        "repFile": None,
        "hardLink": long(1),
        "linkList": None,
        "vstart": 0,
        "vend": 0,
        "elfInfo": None,
        "nrMap": 0,
    }

    def __init__(self):
        self.profSuccessCnt = 0
        self.profFailedCnt = 0
        self.profPageCnt = 0
        self.pgRclmBg = 0
        self.pgRclmFg = 0
        self.procData = {}
        self.fileData = {}
        self.inodeData = {}
        self.target = [""]
        self.readaheadStr = ""

        self.procList = {}
        self.fileList = {}

        self.intervalProcData = []
        self.intervalFileData = []

        self.init_procData = {
            "tids": None,
            "pageCnt": 0,
            "procMap": None,
            "comm": "",
        }
        self.init_threadData = {"comm": ""}
        self.init_inodeData = {}

        # set system maximum fd number #
        SysMgr.setMaxFd()

        # set specific file #
        targetFiles = []

        # convert target path to abspath #
        for item in SysMgr.customCmd:
            if "*" in item:
                continue
            path = os.path.realpath(item)
            targetFiles.append(path)

        # convert target path to realpath #
        if SysMgr.customCmd:
            SysMgr.printInfo(
                "start checking specific files related to [ %s ]"
                % ", ".join(SysMgr.customCmd)
            )

            # get specific file list #
            convList = UtilMgr.getFileList(SysMgr.customCmd)
            for fname in convList:
                try:
                    if os.path.isdir(fname):
                        for subfname in UtilMgr.getFiles(
                            fname, recursive=SysMgr.recursionEnable
                        ):
                            SysMgr.getCachedFd(subfname)
                            targetFiles.append(os.path.realpath(subfname))
                    elif os.path.isfile(fname):
                        SysMgr.getCachedFd(fname)
                        targetFiles.append(os.path.realpath(fname))
                except SystemExit:
                    sys.exit(0)
                except:
                    SysMgr.printErr("failed to open '%s'" % fname, reason=True)
                    sys.exit(-1)

        # remove redundant filters #
        targetFiles = set(targetFiles)

        # add file filters #
        targetFilter = SysMgr.environList.get("TARGETFILE")

        # handle no target case #
        if SysMgr.filterGroup:
            self.target = SysMgr.filterGroup

        if not SysMgr.guiderObj:
            # load libc #
            SysMgr.loadLibcObj(exit=True)

            # define munmap types #
            SysMgr.libcObj.munmap.argtypes = [POINTER(None), c_size_t]
            SysMgr.libcObj.munmap.restype = c_int

            # define mincore types #
            SysMgr.libcObj.mincore.argtypes = [
                POINTER(None),
                c_size_t,
                POINTER(c_ubyte),
            ]
            SysMgr.libcObj.mincore.restype = c_int

        # set filter list #
        targetFiles = targetFiles if targetFiles else [""]
        exceptFiles = (
            SysMgr.environList["EXCEPTFILE"]
            if "EXCEPTFILE" in SysMgr.environList
            else None
        )
        onlyTarget = "ONLYTARGET" in SysMgr.environList

        while 1:
            # print plan #
            realTargets = set(targetFiles) - {""}
            if realTargets:
                SysMgr.printStat(
                    "start collecting a total of %s files..."
                    % UtilMgr.convNum(len(realTargets))
                )

            SysMgr.printStat("start collecting all files on memory...")

            # scan proc directory and save map information of processes #
            self.scanProcs(
                targetFilter=targetFilter,
                targetFile=targetFiles,
                exceptFilter=exceptFiles,
            )

            # merge maps of processes into a integrated file map #
            self.mergeFileMapInfo(
                targetFilter=targetFilter,
                targetFile=targetFiles,
                exceptFilter=exceptFiles,
                onlyTarget=onlyTarget,
            )

            # get file map info on memory #
            self.getFilePageMaps()

            # fill file map of each processes #
            self.fillFileMaps()

            if SysMgr.intervalEnable:
                # save previous file usage and initialize all variables #
                self.intervalProcData.append(self.procData)
                self.intervalFileData.append(self.fileData)
                self.procData = {}
                self.fileData = {}
                self.inodeData = {}
                self.profSuccessCnt = 0
                self.profFailedCnt = 0

                # check exit condition for interval profile #
                if not SysMgr.condExit:
                    SysMgr.waitEvent()
                else:
                    break
            else:
                break

        # save system reclaim info
        try:
            # get vmstat #
            vmData = SysMgr.getVmstat(retDict=True)

            for name in list(ConfigMgr.BGRECLAIMSTAT):
                if name in vmData:
                    self.pgRclmBg += vmData[name]

            for name in list(ConfigMgr.FGRECLAIMSTAT):
                if name in vmData:
                    self.pgRclmFg += vmData[name]
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printErr("failed to get system reclaim stat", True)

    def __del__(self):
        pass

    def printIntervalInfo(self):
        # merge process info into a global list #
        for data in self.intervalProcData:
            for pid, info in data.items():
                try:
                    if self.procList[pid]["pageCnt"] < info["pageCnt"]:
                        self.procList[pid]["pageCnt"] = info["pageCnt"]
                except:
                    self.procList[pid] = dict(self.init_procData)
                    self.procList[pid]["tids"] = {}
                    self.procList[pid]["pageCnt"] = info["pageCnt"]
                    self.procList[pid]["comm"] = info["comm"]

                for tid, val in info["tids"].items():
                    try:
                        self.procList[pid]["tids"][tid]
                    except:
                        self.procList[pid]["tids"][tid] = dict(
                            self.init_threadData
                        )
                        self.procList[pid]["tids"][tid]["comm"] = val["comm"]

        if not self.procList:
            SysMgr.printErr("no process profiled")
            sys.exit(-1)

        # merge file info into a global list #
        fileData = self.intervalFileData
        for data in fileData:
            for name, stat in data.items():
                try:
                    fl = self.fileList[name]
                    if fl["pageCnt"] < stat["pageCnt"]:
                        fl["pageCnt"] = stat["pageCnt"]
                except:
                    self.fileList[name] = dict(FileAnalyzer.init_mapData)
                    self.fileList[name]["pageCnt"] = stat["pageCnt"]
                    self.fileList[name]["totalSize"] = stat["totalSize"]

        # check result #
        if not self.fileList:
            SysMgr.printErr("no file profiled")
            sys.exit(-1)

        self.procData = self.procList

        SysMgr.printLogo(big=True)

        # print system information #
        SysMgr.printInfoBuffer()

        # define alias #
        pageSize = SysMgr.PAGESIZE
        convSize = UtilMgr.convSize2Unit
        convColor = UtilMgr.convColor
        convNum = UtilMgr.convNum
        uptime = UtilMgr.convTime(SysMgr.updateUptime())

        # print process list #
        self.printProcUsage()

        # print file list #
        SysMgr.printPipe(
            (
                "[%s] [File: %s] [LastRAM: %s] [Reclaim: %s/%s] "
                "[Uptime: %s] [Keys: Foward/Back/Save/Quit]\n%s"
            )
            % (
                "File Usage Info",
                convNum(len(self.fileList)),
                convSize(self.profPageCnt * 4 << 10),
                convSize(self.pgRclmBg * 4 << 10),
                convSize(self.pgRclmFg * 4 << 10),
                uptime,
                twoLine,
            )
        )

        printMsg = "{0:_^8}|{1:_^8}|{2:_^3}|".format("InitRAM", "File", "%")

        if len(fileData) > 1:
            for idx in xrange(1, len(fileData)):
                printMsg += "{0:_^15}|".format(str(idx))

        # print title #
        lineLength = SysMgr.lineLength
        printMsg += "{0:_^8}|{1:_^3}|".format("LastRAM", "%")
        printMsg += "_" * (long((lineLength - len(printMsg)) / 2) - 2)
        printMsg += "Library"
        printMsg += "_" * (lineLength - len(printMsg))
        printMsg += "\n%s" % twoLine
        SysMgr.printPipe(printMsg)

        # print interval usage #
        for fileName, val in sorted(
            self.fileList.items(),
            key=lambda e: long(e[1]["pageCnt"]),
            reverse=True,
        ):
            # memory size #
            try:
                memSize = fileData[0][fileName]["pageCnt"] * pageSize
            except:
                memSize = 0

            try:
                idx = val["totalSize"] + pageSize - 1
                fileSize = long(idx / pageSize) * pageSize
            except:
                fileSize = 0

            # set percentage #
            if fileSize != 0:
                per = long((long(memSize) / float(fileSize)) * 100)
                per = UtilMgr.convCpuColor(per, size=3)
            else:
                per = 0

            if memSize > 0:
                memSize = convColor(convSize(memSize), "YELLOW", 7)

            # check whether this file was profiled or not #
            isRep = False
            for fileData in reversed(fileData):
                if fileName in fileData and fileData[fileName]["isRep"]:
                    printMsg = "{0:>7} |{1:>7} |{2:>3}|".format(
                        memSize, convSize(fileSize), per
                    )
                    isRep = True
                    break

            if not isRep:
                continue

            # calculate diff of on-memory file size #
            if len(fileData) > 1:
                for idx in xrange(len(fileData) - 1):
                    diffNew = 0
                    diffDel = 0

                    try:
                        nowFileMap = fileData[idx][fileName]["fileMap"]
                    except:
                        nowFileMap = None

                    try:
                        prevFileMap = fileData[idx - 1][fileName]["fileMap"]
                    except:
                        prevFileMap = None

                    if not nowFileMap:
                        if prevFileMap:
                            diffDel = fileData[idx - 1][fileName]["pageCnt"]
                    else:
                        if not prevFileMap:
                            diffNew = fileData[idx][fileName]["pageCnt"]
                        else:
                            if len(nowFileMap) == len(prevFileMap):
                                for i in xrange(len(nowFileMap)):
                                    if nowFileMap[i] > prevFileMap[i]:
                                        diffNew += 1
                                    elif nowFileMap[i] < prevFileMap[i]:
                                        diffDel += 1

                    if diffNew > 0:
                        diffNew = convColor(
                            convSize(diffNew * pageSize), "RED", 6
                        )
                    else:
                        diffNew = " "

                    if diffDel > 0:
                        diffDel = convColor(
                            convSize(diffDel * pageSize), "RED", 6
                        )
                    else:
                        diffDel = " "

                    printMsg += "+%6s/-%6s|" % (diffNew, diffDel)

            finalData = fileData[-1][fileName]

            totalMemSize = finalData["pageCnt"] * pageSize

            if fileSize != 0:
                per = long((long(totalMemSize) / float(fileSize)) * 100)
                per = UtilMgr.convCpuColor(per, size=3)
            else:
                per = 0

            if totalMemSize > 0:
                totalMemSize = convColor(convSize(totalMemSize), "YELLOW", 7)

            printMsg += "{0:>7} |{1:>3}| {2:1}".format(
                totalMemSize, per, fileName
            )

            SysMgr.printPipe(printMsg)

        SysMgr.printPipe("%s\n\n\n" % oneLine)

    @staticmethod
    def getMappedList(pid, f):
        rpath = "%s/%s/map_files" % (SysMgr.procPath, pid)
        try:
            dirList = os.listdir(rpath)
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printWarn(
                "failed to get the mapped file list for %s(%s)"
                % (SysMgr.getComm(pid), pid),
                reason=True,
            )

        # convert filter type to list #
        if not isinstance(f, list):
            f = [f]

        retList = []
        for item in dirList:
            try:
                fpath = os.path.join(rpath, item)
                fpathName = os.readlink(fpath)
                if UtilMgr.isValidStr(fpathName, f):
                    retList.append(fpath)
            except SystemExit:
                sys.exit(0)
            except:
                SysMgr.printWarn(
                    "failed to get mapped file from %s for %s(%s)"
                    % (fpath, SysMgr.getComm(pid), pid),
                    reason=True,
                )

        return retList

    @staticmethod
    def isDeletedFile(fileName):
        return fileName.endswith(" (deleted)")

    @staticmethod
    def isValidFile(fileName, special=False, incDel=False):
        # skip non-files #
        if not fileName.startswith("/"):
            return False
        # skip device nodes #
        elif fileName.startswith("/dev/"):
            return False
        # skip proc nodes #
        elif fileName.startswith("/proc/"):
            return False
        # skip sys nodes #
        elif fileName.startswith("/sys/"):
            return False
        # skip non-contiguous segments #
        elif SysMgr.magicStr in fileName:
            return False
        # skip deleted files #
        elif not incDel and FileAnalyzer.isDeletedFile(fileName):
            return False

        # check special #
        if special:
            if fileName == "vdso":
                return True

        return True

    @staticmethod
    def getMapAddr(pid, fname, fd=None, retList=False):
        if not fname:
            SysMgr.printWarn("no file name to be searched for address")
            return None
        elif not fd:
            fd = FileAnalyzer.getMapFd(pid)
            if not fd:
                return None

        addrList = []
        allAnon = allFile = allPages = noPerm = fileAnon = something = False

        # check anon #
        if fname == "anon":
            allAnon = True
        # check file #
        elif fname == "file":
            allFile = True
        elif fname == "fileanon":
            fileAnon = True
        # check all #
        elif fname == "all":
            allPages = True
        # check noperm #
        elif fname == "noperm":
            noPerm = True
        else:
            something = True

        # get anon area #
        if not allPages and not noPerm:
            anonMap = TaskAnalyzer.getProcAnonMap(pid)
        else:
            anonMap = None

        # check condition #
        if fileAnon and not anonMap:
            SysMgr.printErr("no smaps info")
            sys.exit(0)

        # read maps #
        fd.seek(0, 0)
        for item in fd.readlines():
            mdict = FileAnalyzer.parseMapLine(
                item, needName=allFile or something
            )
            if not mdict:
                continue

            area = item.split(" ", 1)[0]

            # all #
            if allPages:
                pass
            # no perm #
            elif noPerm:
                if not mdict["perm"].startswith("--"):
                    continue
            # anons #
            elif allAnon:
                # include file-mapped anon pages #
                if anonMap:
                    if area in anonMap and anonMap[area]["size"] == 0:
                        continue
                elif mdict["inode"] != "0" and not mdict["perm"].startswith(
                    "rw"
                ):
                    continue
            # file-mapped anons #
            elif fileAnon:
                # check file-mapped anon pages #
                if mdict["inode"] == "0":
                    continue
                elif anonMap:
                    if area in anonMap and anonMap[area]["size"] == 0:
                        continue
            # files #
            else:
                if not "binName" in mdict or not mdict["binName"]:
                    continue
                # all files #
                elif allFile and area in anonMap and anonMap[area]["size"]:
                    continue
                elif (
                    allFile
                    and mdict["binName"].startswith("/")
                    and mdict["inode"] != "0"
                ):
                    pass
                # a specific file #
                elif not mdict["binName"].endswith(
                    fname
                ) and not UtilMgr.isValidStr(mdict["binName"], [fname]):
                    continue

            start = str(mdict["startAddr"])
            end = str(mdict["endAddr"])

            if retList:
                addrList.append([start, end])
            else:
                return start, end

        if retList:
            return addrList
        else:
            return None

    @staticmethod
    def getMapFilePath(pid, fname, fd=None):
        if not fname:
            SysMgr.printWarn("no file name to be searched for path")
            return None
        elif not fd:
            fd = FileAnalyzer.getMapFd(pid)
            if not fd:
                return None

        # read maps #
        fd.seek(0, 0)
        for item in fd.readlines():
            mdict = FileAnalyzer.parseMapLine(item)
            if mdict and mdict["binName"]:
                if os.path.basename(mdict["binName"]).startswith(fname):
                    path = str(mdict["binName"])
                    overlayInfo = SysMgr.getOverlayfsInfo(pid)
                    if overlayInfo:
                        path = UtilMgr.convOverlayPath(path, overlayInfo)
                    return path

        return None

    @staticmethod
    def getEmptyMapAddr(pid, fd=None, size=0, onlyExec=False):
        if not fd:
            fd = FileAnalyzer.getMapFd(pid)
            if not fd:
                return None

        # search empty space #
        for item in fd.readlines():
            mdict = FileAnalyzer.parseMapLine(item)
            if not mdict:
                mapLine = item.split()
                addrs = list(map(lambda x: long(x, 16), mapLine[0].split("-")))
                perm = mapLine[1]
                if onlyExec and not "x" in perm:
                    continue

                return addrs[0]

    @staticmethod
    def getProcFiles(pid, incMap=True, incFd=True):
        fileList = []

        SysMgr.checkRootPerm()

        if incMap:
            files = FileAnalyzer.getProcMapInfo(pid, saveAll=True)
            for f in files:
                if FileAnalyzer.isValidFile(f):
                    fileList.append(f)

        if incFd:
            try:
                fdlistPath = "%s/%s/fd" % (SysMgr.procPath, pid)
                for fd in os.listdir(fdlistPath):
                    fdPath = "%s/%s" % (fdlistPath, fd)
                    fname = os.readlink(fdPath)
                    if FileAnalyzer.isValidFile(fname):
                        fileList.append(fname)
            except SystemExit:
                sys.exit(0)
            except:
                SysMgr.printErr(
                    "failed to get open file list for %s(%s)"
                    % (SysMgr.getComm(pid), pid),
                    True,
                )

        return fileList

    @staticmethod
    def getProcMapFd(pid, verb=False):
        fd = FileAnalyzer.getMapFd(pid, verb)
        if fd:
            return fd

        # get comm #
        comm = SysMgr.getComm(pid)

        # check alive #
        if not SysMgr.isAlive(pid):
            reason = " because it is terminated"
        # check root #
        elif not SysMgr.isRoot():
            reason = " because there is no root permission"
        else:
            reason = ""

        SysMgr.printErr(
            "failed to get memory map for %s(%s)%s" % (comm, pid, reason)
        )

        # quit #
        if SysMgr.forceEnable:
            return
        else:
            sys.exit(-1)

    @staticmethod
    def getMapFd(pid, verb=False):
        # open maps #
        try:
            path = "%s/%s/maps" % (SysMgr.procPath, pid)
            return open(path, "r")
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printOpenWarn(path, verb)
            return None

    @staticmethod
    def getAnonMapInfo(pid, fd=None, onlyExec=True):
        # set file descriptor #
        if fd:
            fd.seek(0, 0)
        else:
            fd = FileAnalyzer.getProcMapFd(pid)

        # read maps #
        mapBuf = fd.readlines()

        # define map dictionary #
        anonMap = []

        # parse lines #
        for string in mapBuf:
            m = re.match(
                (
                    r"^(?P<startAddr>.\S+)-(?P<endAddr>.\S+) (?P<perm>.\S+) "
                    r"(?P<offset>.\S+) (?P<devid>.\S+) 0"
                ),
                string,
            )
            if not m:
                continue

            d = m.groupdict()

            # get execution permission #
            if onlyExec and "perm" in d and d["perm"][-2] == "-":
                continue

            # get size info #
            startAddr = long(d["startAddr"], 16)
            endAddr = long(d["endAddr"], 16)

            anonMap.append([startAddr, endAddr])

        return anonMap

    @staticmethod
    def getFileMapSize(pid, target, reverse=False):
        try:
            # get file descriptor #
            fd = FileAnalyzer.getProcMapFd(pid)

            # read maps #
            mapBuf = fd.readlines()
            if reverse:
                mapBuf = reversed(mapBuf)
        except SystemExit:
            sys.exit(0)
        except:
            return 0

        # define map dictionary #
        fileMap = {}

        # parse and merge lines in maps #
        for val in mapBuf:
            FileAnalyzer.mergeMapLine(val, fileMap)
            finfo = fileMap.get(target, {})
            if finfo:
                return finfo["size"]

        return 0

    @staticmethod
    def getProcMapInfo(pid, fd=None, onlyExec=False, saveAll=False):
        try:
            # set file descriptor #
            if fd:
                fd.seek(0, 0)
            else:
                fd = FileAnalyzer.getProcMapFd(pid)

            # read maps #
            mapBuf = fd.readlines()
        except SystemExit:
            sys.exit(0)
        except:
            return {}

        # check map cache #
        try:
            if FileAnalyzer.procMapStrCache[pid] == mapBuf:
                return FileAnalyzer.procMapCache[pid]
        except SystemExit:
            sys.exit(0)
        except:
            pass
        finally:
            FileAnalyzer.procMapStrCache[pid] = mapBuf

        # define map dictionary #
        fileMap = {}

        # parse and merge lines in maps #
        for val in mapBuf:
            FileAnalyzer.mergeMapLine(val, fileMap, saveAll=saveAll)

        # remove non-executable files #
        if onlyExec:
            # check non-executable files having discontiguous segments #
            exeFileMap = {}
            for fname in fileMap:
                if not fname.startswith("/") or not fileMap[fname]["exec"]:
                    continue
                exeFileMap[fname.split(SysMgr.magicStr, 1)[0]] = True

            # exclude non-executable files #
            for fname in list(fileMap):
                if (
                    fname != "stack"
                    and not fileMap[fname]["exec"]
                    and not fname.split(SysMgr.magicStr, 1)[0] in exeFileMap
                ):
                    fileMap.pop(fname, None)

        # save map cache #
        FileAnalyzer.procMapCache[pid] = fileMap

        return fileMap

    @staticmethod
    def addMapLine(dataObj, fileName, newOffset, newSize):
        newEnd = newOffset + newSize

        try:
            savedOffset = dataObj[fileName]["offset"]
            savedSize = dataObj[fileName]["size"]
            savedEnd = savedOffset + savedSize

            # start address bigger than saved one #
            if savedOffset <= newOffset:
                # merge bigger end address than saved one #
                if savedEnd < newEnd:
                    dataObj[fileName]["size"] += (
                        newEnd - savedOffset - savedSize
                    )
                # ignore lesser end address than saved one #
                else:
                    pass
            # start address lesser than saved one #
            else:
                if savedEnd >= newEnd:
                    dataObj[fileName]["size"] += savedOffset - newOffset
                else:
                    dataObj[fileName]["size"] = newSize

                dataObj[fileName]["offset"] = newOffset
        except SystemExit:
            sys.exit(0)
        except:
            dataObj[fileName] = dict(FileAnalyzer.init_mapData)
            dataObj[fileName]["offset"] = newOffset
            dataObj[fileName]["size"] = newSize
            dataObj[fileName]["nrMap"] = 1

    @staticmethod
    def parseMapLine(string, needName=True):
        if needName:
            matchStr = (
                r"^(?P<startAddr>.\S+)-(?P<endAddr>.\S+) (?P<perm>.\S+) "
                r"(?P<offset>.\S+) (?P<devid>.\S+) (?P<inode>[0-9]+)"
                r".\s*(?P<binName>.+)"
            )
        else:
            matchStr = (
                r"^(?P<startAddr>.\S+)-(?P<endAddr>.\S+) (?P<perm>.\S+) "
                r"(?P<offset>.\S+) (?P<devid>.\S+) (?P<inode>[0-9]+)"
            )

        m = re.match(matchStr, string)
        if not m:
            return None

        return m.groupdict()

    @staticmethod
    def getReadaheadListStr(raList, raSummary):
        if not raList:
            SysMgr.printWarn("no readahead item", True)
            return ""

        # define variables #
        convNum = UtilMgr.convNum
        printStr = ""

        # print readahead stat #
        printStr += (
            "\n[Thread Readahead Info] (NrFiles: %s) (NrCalls: %s)\n%s\n"
            % (convNum(len(raSummary)), convNum(len(raList)), twoLine)
        )
        printStr += "{0:>12} {1:>12} {2:>1}\n{3:1}\n".format(
            "Size", "Count", "Path", twoLine
        )

        for fname, stat in sorted(
            raSummary.items(), key=lambda e: e[1]["size"], reverse=True
        ):
            printStr += "{0:>12} {1:>12} {2:>1}\n".format(
                UtilMgr.convSize2Unit(stat["size"]),
                convNum(stat["count"]),
                fname,
            )

        if not raSummary:
            printStr += "    None\n"

        printStr += "%s\n" % oneLine

        return printStr

    @staticmethod
    def makeReadaheadFile(
        raPath, readaheadList, pathConvList, raMin, raAddList
    ):
        # check readahead file path #
        if not raPath:
            return readaheadList, {}

        # apply add list #
        for item in raAddList:
            # parse file info #
            finfo = item.split(":")
            try:
                item = finfo[0]
                offset = size = 0

                if len(finfo) == 1:
                    pass
                elif len(finfo) == 2:
                    if finfo[1]:
                        size = UtilMgr.convUnit2Size(finfo[1])
                else:
                    offset = UtilMgr.convUnit2Size(finfo[1])
                    if finfo[2]:
                        size = UtilMgr.convUnit2Size(finfo[2])
            except SystemExit:
                sys.exit(0)
            except:
                fname = SysMgr.environList["RAADDLIST"][0]
                SysMgr.printErr(
                    "failed to apply readahead add list from '%s'" % fname,
                    reason=True,
                )
                sys.exit(-1)

            # get full path #
            fpath = os.path.realpath(item)
            if not os.path.exists(fpath):
                SysMgr.printWarn(
                    "skipped adding '%s' to readahead list because not exists"
                    % fpath,
                    True,
                )
                continue

            # get file size #
            fsize = UtilMgr.getFileSize(fpath, False)
            fsizeStr = UtilMgr.convSize2Unit(fsize)
            finfo = "%s[%s]" % (fpath, fsizeStr)

            # set full size #
            if size == 0:
                size = fsize

            # get path index #
            if finfo in pathConvList:
                idx = pathConvList[finfo][1]
            else:
                idx = len(pathConvList)
                pathConvList.setdefault(finfo, [fpath, idx])

            # add readahead list #
            readaheadList.append([idx, offset, size])

        # backup exist readahead list #
        SysMgr.backupFile(raPath)

        # create a new file for readahead list #
        try:
            raFd = open(raPath, "wb")
            os.chmod(raPath, 0o777)
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printOpenErr(raPath)
            sys.exit(-1)

        # merge readahead chunks #
        if "RAMERGE" in SysMgr.environList:
            SysMgr.printInfo("merge continuous readahead chunks")

            prevChunk = None
            origRaList = readaheadList
            readaheadList = []
            for chunk in origRaList:
                fid, offset, size = chunk
                if not prevChunk:
                    prevChunk = [fid, offset, size]
                    readaheadList.append(prevChunk)
                    continue

                # merge chunks #
                if (
                    fid == prevChunk[0]
                    and offset == prevChunk[1] + prevChunk[2]
                ):
                    prevChunk[2] += size
                    readaheadList[-1] = prevChunk
                    continue

                # add original chunk to list #
                prevChunk = [fid, offset, size]
                readaheadList.append(prevChunk)

        # write readahead list to file #
        try:
            raSummary = {}

            # encode file list #
            fileList = [
                value[0]
                for path, value in sorted(
                    pathConvList.items(), key=lambda e: e[1][1]
                )
            ]

            raDict = {
                "fileList": fileList,
                "readaheadList": [],
            }

            fileStr = "#".join(fileList)
            fileStr = fileStr.encode()

            # write file list size #
            raFd.write(struct.pack("I", len(fileStr)))

            # write file list #
            raFd.write(fileStr)

            # write readahead chunks #
            for chunk in readaheadList:
                # skip chunks lesser than minimum size #
                if raMin > chunk[2]:
                    continue

                fid, offset, size = chunk

                # write chunks #
                if SysMgr.jsonEnable:
                    raDict["readaheadList"].append(chunk)
                else:
                    raFd.write(struct.pack("HQI", fid, offset, size))

                # save readahead stat #
                fname = fileList[fid]
                raSummary.setdefault(fname, dict({"count": 0, "size": 0}))
                raSummary[fname]["count"] += 1
                raSummary[fname]["size"] += size

            if SysMgr.jsonEnable:
                raFd.write(UtilMgr.convDict2Str(raDict).encode())

            # close output file #
            raFd.close()

            # print file size #
            fsize = UtilMgr.getFileSizeStr(raFd.name)

            SysMgr.printInfo(
                "saved the readahead list to '%s'%s successfully"
                % (raFd.name, fsize)
            )
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printErr(
                "failed to save the readahead list to '%s'" % raFd.name, True
            )

        return readaheadList, raSummary

    @staticmethod
    def getReadaheadItems():
        # get readahead list path #
        if "RALIST" in SysMgr.environList:
            # set list file #
            raPath = SysMgr.environList["RALIST"][0]
            if raPath == "SET":
                raPath = "readahead.list"

            # convert to absolute path #
            raPath = os.path.realpath(raPath)

            # set inode scan flag #
            if not "CONVINODE" in SysMgr.environList:
                SysMgr.addEnvironVar("CONVINODE")
        else:
            raPath = None

        # set minimum size #
        raMin = 0
        if "RAMIN" in SysMgr.environList:
            try:
                raMin = SysMgr.environList["RAMIN"][0]
                raMin = long(raMin)
            except SystemExit:
                sys.exit(0)
            except:
                SysMgr.printErr(
                    "failed to set the minimum size to '%s' for readahead chunk"
                    % raMin,
                    True,
                )

        # get readahead allow list path #
        raAllowList = []
        if "RAALLOWLIST" in SysMgr.environList:
            # get list file #
            try:
                fname = os.path.realpath(SysMgr.environList["RAALLOWLIST"][0])

                SysMgr.printInfo(
                    "apply readahead target list from '%s'" % fname
                )

                with open(fname, "r") as fd:
                    raAllowList = fd.readlines()
                    raAllowList = list(map(lambda x: x.rstrip(), raAllowList))
            except SystemExit:
                sys.exit(0)
            except:
                SysMgr.printOpenErr(fname)
                sys.exit(-1)

        # get readahead deny list path #
        raDenyList = []
        if "RADENYLIST" in SysMgr.environList:
            # get list file #
            try:
                fname = os.path.realpath(SysMgr.environList["RADENYLIST"][0])

                SysMgr.printInfo(
                    "apply readahead exception list from '%s'" % fname
                )

                with open(fname, "r") as fd:
                    raDenyList = fd.readlines()
                    raDenyList = list(map(lambda x: x.rstrip(), raDenyList))
            except SystemExit:
                sys.exit(0)
            except:
                SysMgr.printOpenErr(fname)
                sys.exit(-1)

        # get readahead add list path #
        raAddList = []
        if "RAADDLIST" in SysMgr.environList:
            # get list file #
            try:
                fname = os.path.realpath(SysMgr.environList["RAADDLIST"][0])

                SysMgr.printInfo("apply readahead add list from '%s'" % fname)

                with open(fname, "r") as fd:
                    raAddList = fd.readlines()
                    raAddList = list(map(lambda x: x.rstrip(), raAddList))
            except SystemExit:
                sys.exit(0)
            except:
                SysMgr.printOpenErr(fname)
                sys.exit(-1)

        return raPath, raMin, raAllowList, raDenyList, raAddList

    @staticmethod
    def mergeMapLine(string, procMap, onlyExec=False, saveAll=False):
        d = FileAnalyzer.parseMapLine(string)
        if not d:
            return

        # get execution permission #
        if d["perm"][-2] == "-":
            isExec = False
        else:
            isExec = True

        # check execution permission #
        if onlyExec and not isExec:
            return

        # convert file name #
        fileName = d["binName"]
        if fileName.startswith("["):
            fileName = fileName[1:-1]
        if FileAnalyzer.isDeletedFile(fileName):
            fileName = fileName[:-10]

        # set address #
        startAddr = long(d["startAddr"], 16)
        endAddr = long(d["endAddr"], 16)

        newOffset = long(d["offset"], 16)
        newSize = endAddr - startAddr

        # handle discontiguous segments #
        if fileName in procMap and procMap[fileName]["vend"] != startAddr:
            cnt = 0
            while 1:
                newFileName = "%s%s%s" % (fileName, SysMgr.magicStr, cnt)

                # check next segment is contiguous with this line #
                if (
                    newFileName in procMap
                    and procMap[newFileName]["vend"] != startAddr
                ):
                    cnt += 1
                    continue
                else:
                    break

            fileName = newFileName

        # merge map line #
        FileAnalyzer.addMapLine(procMap, fileName, newOffset, newSize)

        # apply attributes #
        if saveAll:
            procMap[fileName]["perm"] = d["perm"]
            procMap[fileName]["devid"] = d["devid"]
            procMap[fileName]["inode"] = d["inode"]
            procMap[fileName]["offset"] = d["offset"]
            try:
                procMap[fileName]["offset"] = long(d["offset"], 16)
            except SystemExit:
                sys.exit(0)
            except:
                procMap[fileName]["offset"] = d["offset"]

        # set mapped addr #
        if procMap[fileName]["vstart"] == 0:
            procMap[fileName]["vstart"] = startAddr

        # set executable flag #
        if "exec" not in procMap[fileName] or not procMap[fileName]["exec"]:
            procMap[fileName]["exec"] = isExec

        procMap[fileName]["vend"] = endAddr

    def printProcUsage(self):
        # define alias #
        convert = UtilMgr.convSize2Unit
        convColor = UtilMgr.convColor
        pageSize = SysMgr.PAGESIZE
        uptime = UtilMgr.convTime(SysMgr.updateUptime())

        # Print process list #
        SysMgr.printPipe(
            (
                "[%s] [Process : %s] [RAM: %s] [Reclaim: %s/%s] "
                "[Uptime: %s] [Keys: Foward/Back/Save/Quit] "
                "[Capture: Ctrl+\\]\n%s"
            )
            % (
                "File Process Info",
                UtilMgr.convNum(len(self.procData)),
                convert(self.profPageCnt * 4 << 10),
                convert(self.pgRclmBg * 4 << 10),
                convert(self.pgRclmFg * 4 << 10),
                uptime,
                twoLine,
            )
        )

        procTitle = "%s(%s)" % ("Thread", "TID")
        SysMgr.printPipe(
            "{0:_^16}({1:_^7})|{2:_^7}|{3:_^120}\n{4:1}".format(
                "Process", "PID", "RAM", procTitle, twoLine
            )
        )

        procInfo = "{0:^16}({0:^7})|{0:7} |".format("")
        threadInfo = " {0:^16}({0:^7})".format("")
        procLength = len(procInfo)
        threadLength = len(threadInfo)
        lineLength = SysMgr.lineLength
        commLen = SysMgr.commLen

        for pid, val in sorted(
            self.procData.items(),
            key=lambda e: long(e[1]["pageCnt"]),
            reverse=True,
        ):
            try:
                rsize = val["pageCnt"] * pageSize
            except:
                SysMgr.printWarn(
                    "failed to get total mapped size for %s" % val["comm"],
                    reason=True,
                )
                continue

            if rsize > 0:
                rsize = convColor(convert(rsize), "YELLOW", 7)

            printMsg = "{0:>16}({1:>7})|{2:>7}|".format(
                val["comm"][:commLen], pid, rsize
            )
            linePos = len(printMsg)

            for tid, threadVal in sorted(val["tids"].items()):
                threadInfo = "{0:>1}({1:>1})".format(
                    threadVal["comm"][:commLen], tid
                )
                threadInfo = "{0:>25}".format(threadInfo)

                linePos += threadLength

                if linePos > lineLength:
                    linePos = procLength + threadLength
                    printMsg += "\n" + (" " * (procLength - 2)) + "|"

                printMsg += threadInfo

            SysMgr.printPipe(printMsg)

        SysMgr.printPipe("%s\n\n" % oneLine)

        # remove invalid files #
        for fileName in list(self.fileData):
            if not FileAnalyzer.isValidFile(fileName):
                if fileName in FileAnalyzer.procFdMapCache:
                    continue
                self.fileData.pop(fileName, None)
            elif self.fileData[fileName]["totalSize"] == 0:
                self.fileData.pop(fileName, None)

    def printUsage(self):
        if not self.procData:
            SysMgr.printErr("no process profiled")
            sys.exit(-1)
        if not self.fileData:
            SysMgr.printErr("no file profiled")
            sys.exit(-1)

        SysMgr.printLogo(big=True)

        # print system information #
        SysMgr.printInfoBuffer()

        # define alias #
        convert = UtilMgr.convSize2Unit
        convColor = UtilMgr.convColor
        convNum = UtilMgr.convNum
        pageSize = SysMgr.PAGESIZE
        uptime = UtilMgr.convTime(SysMgr.updateUptime())

        # print process list #
        self.printProcUsage()

        # get res check function #
        _resChecker = UtilMgr.getSizeFilterFunc("RESFILTER")

        # get pss check function #
        _pssChecker = UtilMgr.getSizeFilterFunc("PSSFILTER")

        # Print file list #
        SysMgr.printPipe(
            (
                "[%s] [File: %s] [RAM: %s] [Reclaim: %s/%s] [Uptime: %s]"
                " [Keys: Foward/Back/Save/Quit]\n%s"
            )
            % (
                "File Usage Info",
                UtilMgr.convNum(len(self.fileData)),
                convert(self.profPageCnt * 4 << 10),
                convert(self.pgRclmBg * 4 << 10),
                convert(self.pgRclmFg * 4 << 10),
                uptime,
                twoLine,
            )
        )
        SysMgr.printPipe(
            "{0:_^8}|{1:_^8}|{2:_^5}|{3:_^8}|{4:_^121}\n{5:1}".format(
                "RAM", "File", "%", "PSS", "Library & Process", twoLine
            )
        )

        # set sort value #
        if SysMgr.sort == "s":
            sortOpt = "totalSize"
        elif SysMgr.sort == "p":
            for path in self.fileData:
                self.fileData[path]["path"] = path
            sortOpt = "path"
        else:
            sortOpt = "pageCnt"

        for fileName, val in sorted(
            self.fileData.items(),
            key=lambda e: e[1][sortOpt]
            if isinstance(e[1][sortOpt], str)
            else long(e[1][sortOpt]),
            reverse=True,
        ):
            # get stat #
            memSize = val["pageCnt"] * pageSize
            nrFilePage = long((val["totalSize"] + pageSize - 1) / pageSize)
            fileSize = nrFilePage * pageSize
            if fileSize != 0:
                per = long((long(memSize) / float(fileSize)) * 100)
                per = UtilMgr.convCpuColor(per, size=4)
            else:
                per = 0

            try:
                pss = long(memSize / len(val["pids"]))
                if not _pssChecker(pss):
                    continue
                elif pss > 0:
                    pss = convColor(convert(pss), "GREEN", 7)
            except:
                pss = 0
                if not _pssChecker(pss):
                    continue

            # check resident size #
            if not _resChecker(memSize):
                continue

            if memSize > 0:
                memSize = convColor(convert(memSize), "YELLOW", 7)

            if not val["isRep"]:
                continue
            else:
                if val["nrMap"] - 1 > 0:
                    cntStr = " [Map: %s]" % convNum(val["nrMap"] - 1)
                else:
                    cntStr = ""

                if val["nrOpen"] > 0:
                    cntStr += " [Open: %s]" % convNum(val["nrOpen"])
                else:
                    cntStr += ""

                SysMgr.printPipe(
                    (
                        "{0:>7} |{1:>7} |{2:>4} |{3:>7} | {4:1} "
                        "[Proc: {5:1}] [Link: {6:1}]{7:1}"
                    ).format(
                        memSize,
                        convert(fileSize),
                        per,
                        pss,
                        fileName,
                        len(val["pids"]),
                        convNum(val["hardLink"]),
                        cntStr,
                    )
                )

            # prepare for printing process list #
            pidInfo = ""
            lineLength = SysMgr.lineLength
            pidLength = len(" %16s (%6s) |" % ("", ""))
            ilength = len("{0:7} |{0:7} |{0:4} |{0:7} ".format(""))
            linePos = ilength + pidLength

            # print hard-linked list #
            if val["hardLink"] > 1:
                for fileLink, tmpVal in val["linkList"].items():
                    if fileName != fileLink:
                        SysMgr.printPipe((" " * ilength) + "| -> " + fileLink)

            # print process list #
            if not "NOTASKINFO" in SysMgr.environList:
                for pid, comm in val["pids"].items():
                    if linePos > lineLength:
                        linePos = ilength + pidLength
                        pidInfo += "\n" + (" " * ilength) + "|"

                    pidInfo += " %16s (%7s) |" % (comm[: SysMgr.commLen], pid)

                    linePos += pidLength
                SysMgr.printPipe((" " * ilength) + "|" + pidInfo)

            SysMgr.printPipe(oneLine)

        if self.readaheadStr:
            SysMgr.printPipe(self.readaheadStr)

        SysMgr.printPipe("\n\n\n")

    def scanProcs(self, targetFilter=None, targetFile=None, exceptFilter=None):
        # get process list in procfs #
        pids = SysMgr.getPidList()

        # initialize proc object #
        procObj = TaskAnalyzer(onlyInstance=True)

        # get size check function #
        _sizeChecker = UtilMgr.getSizeFilterFunc()

        # scan tasks #
        for idx, pid in enumerate(pids):
            try:
                nrPid = long(pid)

                # skip myself #
                if SysMgr.pid == nrPid and targetFile == [""]:
                    continue
            except SystemExit:
                sys.exit(0)
            except:
                continue

            mappedList = []

            # print progress #
            UtilMgr.printProgress(idx, len(pids))

            # make process path #
            procPath = "%s/%s" % (SysMgr.procPath, pid)

            # save stat of process #
            try:
                procObj.saveProcData(procPath, pid)
                pidComm = procObj.procData[pid]["comm"]
            except SystemExit:
                sys.exit(0)
            except:
                SysMgr.printOpenWarn(procPath)
                continue

            # skip kernel tasks #
            if procObj.isKernelThread(pid):
                continue

            # make path of tid #
            taskPath = "%s/%s" % (procPath, "task")

            try:
                tids = os.listdir(taskPath)
            except SystemExit:
                sys.exit(0)
            except:
                SysMgr.printOpenWarn(taskPath)
                continue

            # scan threads #
            for tid in tids:
                try:
                    long(tid)
                except:
                    continue

                # make thread path #
                threadPath = "%s/%s" % (taskPath, tid)

                # save stat of thread #
                try:
                    procObj.saveProcData(threadPath, tid)
                    comm = procObj.procData[tid]["comm"]
                except SystemExit:
                    sys.exit(0)
                except:
                    SysMgr.printOpenWarn(threadPath)
                    continue

                # check condition #
                if (
                    self.target != [""]
                    and not tid in self.target
                    and not UtilMgr.isValidStr(comm, self.target)
                ):
                    continue

                # update procData #
                if not pid in self.procData:
                    self.procData[pid] = dict(self.init_procData)
                    self.procData[pid]["tids"] = {}
                    self.procData[pid]["procMap"] = {}
                    self.procData[pid]["comm"] = pidComm

                    # update mapInfo per process #
                    self.procData[pid][
                        "procMap"
                    ] = FileAnalyzer.getProcMapInfo(pid)

                    # save file info per process #
                    try:
                        fdlist = []
                        fdlistPath = "%s/fd" % procPath
                        fdlist = os.listdir(fdlistPath)
                    except SystemExit:
                        sys.exit(0)
                    except:
                        SysMgr.printOpenWarn(fdlistPath)

                    if SysMgr.pid == nrPid:
                        SysMgr.printStat(
                            "start collecting custom files on memory..."
                        )
                        isMe = True
                    else:
                        isMe = False

                    # scan file descriptors #
                    for gidx, fd in enumerate(fdlist):
                        try:
                            size = 0

                            # print progress #
                            if isMe:
                                UtilMgr.printProgress(gidx, len(fdlist))

                            # get real path #
                            fdPath = "%s/%s" % (fdlistPath, long(fd))
                            fname = os.readlink(fdPath)

                            # check files #
                            if not FileAnalyzer.isValidFile(
                                fname, incDel=True
                            ):
                                continue
                            elif (
                                tid != pid
                                and fname in self.procData[pid]["procMap"]
                            ):
                                continue
                            elif targetFile != [""] and not UtilMgr.isValidStr(
                                fname, targetFile
                            ):
                                continue
                            elif targetFilter and not UtilMgr.isValidStr(
                                fname, targetFilter
                            ):
                                continue
                            elif exceptFilter and UtilMgr.isValidStr(
                                fname, exceptFilter
                            ):
                                continue

                            # init file info #
                            if not size:
                                size = UtilMgr.getFileSize(fname, False)
                            if size == 0 or not _sizeChecker(size):
                                if FileAnalyzer.isDeletedFile(fname):
                                    if not mappedList:
                                        mappedList = (
                                            FileAnalyzer.getMappedList(
                                                pid, fname
                                            )
                                        )
                                    if mappedList:
                                        size = UtilMgr.getFileSize(
                                            mappedList[0], False
                                        )
                                if size == 0:
                                    continue

                            procMap = self.procData[pid]["procMap"]
                            procMap[fname] = dict(FileAnalyzer.init_mapData)
                            procMap[fname]["size"] = size
                            procMap[fname]["nrOpen"] = 1
                            if mappedList:
                                FileAnalyzer.procFdMapCache[
                                    fname
                                ] = mappedList[0]
                        except SystemExit:
                            sys.exit(0)
                        except:
                            pass

                # print progress #
                if isMe:
                    UtilMgr.deleteProgress()

                # check thread list #
                if tid in self.procData[pid]["tids"]:
                    continue

                # update threadData #
                threadList = SysMgr.getThreadList(pid)
                for ttid in threadList:
                    if ttid in self.procData[pid]["tids"]:
                        continue

                    self.procData[pid]["tids"][ttid] = dict(
                        self.init_threadData
                    )

                    tcomm = SysMgr.getComm(ttid)
                    self.procData[pid]["tids"][ttid]["comm"] = tcomm

        # remove progress #
        UtilMgr.deleteProgress()

    def fillFileMaps(self):
        self.profPageCnt = 0

        for fileName, val in self.fileData.items():
            if val["fileMap"] and val["isRep"]:
                val["pageCnt"] = val["fileMap"].count(1)
                self.profPageCnt += val["pageCnt"]

        pageSize = SysMgr.PAGESIZE
        for pid, val in self.procData.items():
            for fileName, mapInfo in val["procMap"].items():
                if not fileName in self.fileData:
                    continue
                elif not self.fileData[fileName]["fileMap"] or not mapInfo:
                    continue

                # convert address and size to index in mapping table #
                offset = mapInfo["offset"] - self.fileData[fileName]["offset"]
                offset = long((offset + pageSize - 1) / pageSize)
                size = long((mapInfo["size"] + pageSize - 1) / pageSize)

                mapInfo["fileMap"] = list(
                    self.fileData[fileName]["fileMap"][offset:size]
                )
                mapInfo["pageCnt"] = mapInfo["fileMap"].count(1)
                val["pageCnt"] += mapInfo["pageCnt"]

    def mergeFileMapInfo(
        self,
        targetFilter=None,
        targetFile=None,
        exceptFilter=None,
        onlyTarget=False,
    ):
        SysMgr.printStat("start merging file info...")

        # get size check function #
        _sizeChecker = UtilMgr.getSizeFilterFunc()

        pos = 0
        myPid = str(SysMgr.pid)
        for pid, val in self.procData.items():
            for fileName, scope in val["procMap"].items():
                UtilMgr.printProgress(pos, len(self.procData))

                # check file filter #
                if (
                    targetFile != [""]
                    and onlyTarget
                    and not UtilMgr.isValidStr(fileName, targetFile)
                ):
                    continue
                elif targetFilter and not UtilMgr.isValidStr(
                    fileName, targetFilter
                ):
                    continue
                elif exceptFilter and UtilMgr.isValidStr(
                    fileName, exceptFilter
                ):
                    continue
                elif not _sizeChecker(scope["size"]):
                    continue

                newOffset = scope["offset"]
                newSize = scope["size"]

                # merge map line #
                FileAnalyzer.addMapLine(
                    self.fileData, fileName, newOffset, newSize
                )

                # add map count #
                if scope["nrMap"]:
                    self.fileData[fileName]["nrMap"] += scope["nrMap"]

                # add open count #
                if scope["nrOpen"]:
                    self.fileData[fileName]["nrOpen"] += scope["nrOpen"]

                # add pid into file info #
                if not self.fileData[fileName]["pids"]:
                    self.fileData[fileName]["pids"] = {}
                if not pid in self.fileData[fileName]["pids"] and pid != myPid:
                    self.fileData[fileName]["pids"][pid] = val["comm"]

        UtilMgr.deleteProgress()

    def getFilePageMaps(self):
        # pylint: disable=no-member

        pageSize = SysMgr.PAGESIZE
        self.profSuccessCnt = 0
        self.profFailedCnt = 0

        # get ctypes object #
        if not SysMgr.guiderObj:
            SysMgr.importPkgItems("ctypes")

        # define readahead list #
        pathConvList = {}
        readaheadList = []
        skipFiles = {}

        # get readahead items #
        (
            raPath,
            raMin,
            raAllowList,
            raDenyList,
            raAddList,
        ) = FileAnalyzer.getReadaheadItems()

        for fileName, val in self.fileData.items():
            # check exceptional file #
            if not FileAnalyzer.isValidFile(fileName, incDel=True):
                continue

            if self.intervalFileData:
                # use file descriptor already saved as possible #
                try:
                    fileData = self.intervalFileData
                    fileInfo = fileData[len(self.intervalFileData) - 1][
                        fileName
                    ]

                    val["fd"] = fileInfo["fd"]

                    val["totalSize"] = fileInfo["totalSize"]

                    val["isRep"] = fileInfo["isRep"]
                except SystemExit:
                    sys.exit(0)
                except:
                    pass

                if not val["isRep"]:
                    continue

            if not val["fd"]:
                """
                no fd related to this file
                case 1) no opened
                case 2) closed by mincore error
                case 3) closed because of rlimit
                """

                try:
                    if fileName in FileAnalyzer.procFdMapCache:
                        fileNameReal = FileAnalyzer.procFdMapCache[fileName]
                    else:
                        fileNameReal = fileName

                    # check whether pages are on memory or not #
                    stat = os.stat(fileNameReal)

                    devid = stat.st_dev
                    inode = stat.st_ino

                    # check whether this file was profiled or not #
                    if inode in self.inodeData:
                        found = False
                        repFile = ""
                        fileList = {}
                        procList = dict(val["pids"].items())

                        for fileIdx, devid in self.inodeData[inode].items():
                            # this hard-lined file was already profiled #
                            if devid != devid:
                                continue

                            found = True

                            # add file into same file list #
                            fileList[fileName] = True
                            fileList[fileIdx] = True

                            # merge process list related to this file #
                            procList.update(self.fileData[fileIdx]["pids"])

                            if self.fileData[fileIdx]["isRep"]:
                                repFile = fileIdx

                        self.inodeData[inode][fileName] = devid

                        if found:
                            self.fileData[fileName]["isRep"] = False
                            hardLinkCnt = len(fileList)

                            # set representative file #
                            for fileIdx, value in fileList.items():
                                self.fileData[fileIdx]["repFile"] = repFile
                                self.fileData[fileIdx][
                                    "hardLink"
                                ] = hardLinkCnt

                            # assign merged process list to representative file #
                            self.fileData[repFile]["pids"] = procList
                            self.fileData[repFile]["hardLink"] = hardLinkCnt

                            if self.fileData[repFile]["linkList"]:
                                self.fileData[repFile]["linkList"].update(
                                    fileList
                                )
                            else:
                                self.fileData[repFile]["linkList"] = fileList

                            continue
                    else:
                        self.inodeData[inode] = dict(self.init_inodeData)
                        self.inodeData[inode][fileName] = devid

                    # get meta data #
                    size = stat.st_size
                    linkCnt = stat.st_nlink
                    time = stat.st_atime

                    # update meta data #
                    val["inode"] = inode
                    val["totalSize"] = size
                    val["linkCnt"] = linkCnt
                    val["accessTime"] = time

                    # open file #
                    val["fd"] = open(fileNameReal, "r")
                except SystemExit:
                    sys.exit(0)
                except:
                    self.profFailedCnt += 1
                    if SysMgr.warnEnable:
                        SysMgr.printOpenWarn(fileName)
                    continue

            # check file size whether it is readable or not #
            if val["totalSize"] <= 0:
                self.profFailedCnt += 1
                if SysMgr.warnEnable:
                    SysMgr.printWarn("failed to mmap %s" % fileName)
                continue

            # prepare variables for mincore syscall #
            fd = val["fd"].fileno()
            offset = val["offset"]
            size = val["totalSize"]

            # get page-aligned size #
            tsize = long((size + pageSize - 1) / pageSize)
            size = tsize * pageSize

            if SysMgr.guiderObj:
                # map a file to ram with PROT_NONE(0), MAP_SHARED(0x10) flags #
                mm = SysMgr.guiderObj.mmap(0, size, 0, 2, fd, offset)

                # call mincore syscall by standard libc library #
                pagemap = SysMgr.guiderObj.mincore(mm, size)

                # unmap #
                SysMgr.guiderObj.munmap(mm, size)
            else:
                # map a file to ram with PROT_NONE(0), MAP_SHARED(0x10) flags #
                mm = SysMgr.mmap(POINTER(c_char)(), size, 0, 2, fd, offset)

                # make a pagemap table #
                pagemap = (tsize * c_ubyte)()

                # call mincore syscall by standard libc library #
                ret = SysMgr.libcObj.mincore(
                    mm, size, cast(pagemap, POINTER(c_ubyte))
                )
                if ret < 0:
                    pagemap = None

                # unmap #
                SysMgr.libcObj.munmap(mm, size)

            # save the on-memory file page table #
            if pagemap:
                try:
                    if SysMgr.guiderObj:
                        val["fileMap"] = [
                            ord(pagemap[i]) for i in xrange(tsize)
                        ]
                    else:
                        val["fileMap"] = [pagemap[i] for i in xrange(tsize)]

                    self.profSuccessCnt += 1

                    # fd resource is about to run out #
                    if SysMgr.maxKeepFd < fd:
                        val["fd"].close()
                        val["fd"] = None
                except SystemExit:
                    sys.exit(0)
                except:
                    SysMgr.printWarn("failed to access '%s'" % fileName)
                    val["fileMap"] = None
                    self.profFailedCnt += 1

                # add filemap info to readahead list #
                if raPath:
                    # check allow list #
                    if raAllowList and not UtilMgr.isValidStr(
                        fileName, raAllowList
                    ):
                        skip = True
                    # check deny list #
                    elif raDenyList and UtilMgr.isValidStr(
                        fileName, raDenyList
                    ):
                        skip = True
                    else:
                        skip = False

                    # check skip condition #
                    if skip:
                        skipFiles.setdefault(fileName, None)
                        continue

                    fileIdx = len(pathConvList)
                    pathConvList.setdefault(fileName, [fileName, fileIdx])

                    start = -1
                    for idx, bit in enumerate(pagemap):
                        if start == -1:
                            if bit:
                                start = idx
                            else:
                                continue

                        # check contiguous chunk #
                        if bit and idx != len(pagemap) - 1:
                            continue

                        # add contiguous chunks to readahead list #
                        offset = start * pageSize
                        size = (idx - start + 1) * pageSize
                        readaheadList.append([fileIdx, offset, size])

                        start = -1
            else:
                val["fd"].close()
                val["fd"] = None

        if not self.fileData:
            SysMgr.printErr("failed to profile files")
            sys.exit(-1)

        SysMgr.printGood(
            "profiled a total of %s files"
            % UtilMgr.convNum(self.profSuccessCnt)
        )

        if self.profFailedCnt > 0:
            SysMgr.printWarn(
                "failed to open a total of %s files"
                % UtilMgr.convNum(self.profFailedCnt)
            )

        # print skip files #
        for path in sorted(list(skipFiles)):
            SysMgr.printWarn("skipped adding '%s' to readahead list" % path)

        # make readahead list file #
        if raPath:
            readaheadList, raSummary = FileAnalyzer.makeReadaheadFile(
                raPath, readaheadList, pathConvList, raMin, raAddList
            )

            # print readahead list info #
            self.readaheadStr = FileAnalyzer.getReadaheadListStr(
                readaheadList, raSummary
            )


class LogMgr(object):
    """Manager for error log"""

    # define log level #
    LOG_EMERG = 0
    LOG_ALERT = 1
    LOG_CRIT = 2
    LOG_ERR = 3
    LOG_WARNING = 4
    LOG_NOTICE = 5
    LOG_INFO = 6
    LOG_DEBUG = 7

    # define syslog type #
    SYSLOG_ACTION_CLOSE = 0
    SYSLOG_ACTION_OPEN = 1
    SYSLOG_ACTION_READ = 2
    SYSLOG_ACTION_READ_ALL = 3
    SYSLOG_ACTION_READ_CLEAR = 4
    SYSLOG_ACTION_CLEAR = 5
    SYSLOG_ACTION_CONSOLE_OFF = 6
    SYSLOG_ACTION_CONSOLE_ON = 7
    SYSLOG_ACTION_CONSOLE_LEVEL = 8
    SYSLOG_ACTION_SIZE_UNREAD = 9
    SYSLOG_ACTION_SIZE_BUFFER = 10

    # define journal type #
    SD_JOURNAL_LOCAL_ONLY = 1 << 0
    SD_JOURNAL_RUNTIME_ONLY = 1 << 1
    SD_JOURNAL_SYSTEM = 1 << 2
    SD_JOURNAL_CURRENT_USER = 1 << 3
    SD_JOURNAL_OS_ROOT = 1 << 4
    SD_JOURNAL_SYSTEM_ONLY = SD_JOURNAL_SYSTEM

    # define android type #
    ANDROID_LOG_UNKNOWN = 0
    ANDROID_LOG_DEFAULT = 1
    ANDROID_LOG_VERBOSE = 2
    ANDROID_LOG_DEBUG = 3
    ANDROID_LOG_INFO = 4
    ANDROID_LOG_WARN = 5
    ANDROID_LOG_ERROR = 6
    ANDROID_LOG_FATAL = 7
    ANDROID_LOG_SILENT = 8

    lockStat = {}

    def __init__(self, target="error"):
        self.errFd = None

        if target == "error":
            self.terminal = sys.stderr
            self.notified = False
            self.error = False
        else:
            # open #
            try:
                self.terminal = open(target, "a")
                self.error = True
                os.chmod(target, 0o777)
            except SystemExit:
                sys.exit(0)
            except:
                SysMgr.printErr("failed to open '%s'" % target, True)

    def write(self, message):
        # write message #
        try:
            self.terminal.write(message)
            self.terminal.flush()
        except SystemExit:
            sys.exit(0)
        except:
            return

        # check error status #
        try:
            if self.error:
                return
        except SystemExit:
            sys.exit(0)
        except:
            pass

        # TODO: check log file limitation #

        # check cache dir #
        if not os.path.exists(SysMgr.cacheDirPath):
            if not SysMgr.mkdirs(SysMgr.cacheDirPath):
                SysMgr.cacheDirPath = "/tmp"
                SysMgr.printWarn("use /tmp as a cache directory", True)

        # set file path for error log #
        if not self.errFd:
            errorFile = "%s/guider.err" % SysMgr.cacheDirPath
            if not SysMgr.isWritable(errorFile):
                SysMgr.printWarn(
                    (
                        "failed to get write permission for %s "
                        "so that use /tmp/guider.err"
                    )
                    % errorFile,
                    True,
                )
                SysMgr.cacheDirPath = "/tmp"
                errorFile = "%s/guider.err" % SysMgr.cacheDirPath

            # open log file #
            try:
                self.errFd = open(errorFile, "a")
            except SystemExit:
                sys.exit(0)
            except:
                SysMgr.printOpenErr(errorFile)
                self.error = True

            # change permission #
            try:
                os.chmod(errorFile, 0o777)
            except SystemExit:
                sys.exit(0)
            except:
                pass
        try:
            if hasattr(self, "notified") and not self.notified:
                SysMgr.printErr(
                    (
                        "please report '%s' to "
                        "https://github.com/iipeace/guider/issues"
                    )
                    % self.errFd.name
                    if self.errFd
                    else "error"
                )
                self.notified = True

            # write log to the file #
            if self.errFd:
                SysMgr.writeErr(self.errFd, message)
        except SystemExit:
            sys.exit(0)
        except:
            pass

    def flush(self):
        pass

    def __getattr__(self, attr):
        return getattr(self.terminal, attr)

    @staticmethod
    def _lock(fd, lock=True):
        # pylint: disable=undefined-variable
        if not SysMgr.isLinux:
            return

        if lock:
            lname = "lock"
        else:
            lname = "unlock"

        if fd in LogMgr.lockStat:
            print(
                UtilMgr.convColor(
                    ("\n[ERROR] tried to %s %s again" % (lname, fd)), "RED"
                )
            )
            return

        # save lock status #
        LogMgr.lockStat[fd] = True

        try:
            if SysMgr.importPkgItems("fcntl", isExit=False):
                lockf(fd, LOCK_EX if lock else LOCK_UN, 1, 0, 0)
        except SystemExit:
            sys.exit(0)
        except:
            name = fd.name if fd else "logger"
            reason = SysMgr.getErrMsg()
            print(
                "\n[ERROR] failed to %s for %s because %s"
                % (lname, name, reason)
            )

        # remove lock status #
        LogMgr.lockStat.pop(fd, None)

    @staticmethod
    def lock(fd):
        return LogMgr._lock(fd, lock=True)

    @staticmethod
    def unlock(fd):
        return LogMgr._lock(fd, lock=False)

    @staticmethod
    def printTrace(console=False):
        fileSize = 0
        compressed = False

        # get target trace points #
        if SysMgr.hasMainArg():
            targets = SysMgr.getMainArgs()
        else:
            targets = []

        # print trace point list #
        if SysMgr.findOption("l"):
            if not SysMgr.findOption("Q"):
                SysMgr.streamEnable = False
            SysMgr.printDirs(
                os.path.join(
                    SysMgr.getTraceEventPath(), targets[0] if targets else ""
                ),
                SysMgr.funcDepth if SysMgr.funcDepth > 0 else 1,
            )
            sys.exit(0)

        # flush the ring buffer #
        if "FLUSH" in SysMgr.environList:
            SysMgr.clearTraceBuffer()

        # enable events #
        SysMgr.enableEvents(targets)

        # disable events #
        SysMgr.disableEvents()

        # clear filters #
        SysMgr.clearFilters()

        # set trace option #
        SysMgr.setTraceOpt()

        if SysMgr.inputParam:
            # open the target file #
            try:
                path = SysMgr.inputParam

                # open compressed file #
                traceFd = SysMgr.getCompFd(path)
                if traceFd:
                    compressed = True
                else:
                    traceFd = open(path, "r")

                    # get file size #
                    fileSize = UtilMgr.getFileSize(path, string=False)
            except SystemExit:
                sys.exit(0)
            except:
                SysMgr.printOpenErr(path)
                return -1
        else:
            # check root permission #
            SysMgr.checkRootPerm()

            # open trace file #
            try:
                eventPath = SysMgr.getTraceEventPath()
                tracePath = "%s/../trace" % eventPath
                origStatus = None

                # change trace file to pipe #
                if "CONSUME" in SysMgr.environList:
                    tracePath += "_pipe"
                # save tracing status #
                else:
                    origStatus = SysMgr.readTraceFile("../tracing_on")

                # set trace fd #
                traceFd = open(tracePath, "r")

                # check tracing status change #
                if origStatus == "1":
                    newStatus = SysMgr.readTraceFile("../tracing_on")
                    if newStatus == "0":
                        SysMgr.printWarn(
                            "tracing is stopped because '%s' is open"
                            % tracePath,
                            True,
                        )
            except SystemExit:
                sys.exit(0)
            except:
                SysMgr.printOpenErr(tracePath)
                return -1

        # get item for log filter #
        if "WATCHLOG" in SysMgr.environList:
            watchcond = SysMgr.environList["WATCHLOG"][0].split("+")
            refilter = SysMgr.environList.get("REFILTER")
        else:
            refilter = watchcond = None

        # get times for tail and until option #
        current = SysMgr.getUptime()
        since, until = SysMgr.getTimeValues(["TAIL", "UNTIL"], current)
        if since or until:
            checkTime = True
            if since and since < 0:
                since += current
            if until and until < 0:
                until += until
        else:
            checkTime = False

        # get INCPROCNAME flag #
        incProcName = "INCPROCNAME" in SysMgr.environList

        SysMgr.printInfo("start printing trace logs... [ STOP(Ctrl+c) ]")

        initialized = False

        while 1:
            log = traceFd.readline()

            # decode and remove newlines #
            if compressed:
                log = log.decode()
            log = log.rstrip()

            if not initialized:
                initialized = True if TaskAnalyzer.setTraceAttr(log) else False

            # check log #
            if not log:
                if SysMgr.inputParam:
                    if traceFd.tell() == fileSize:
                        break
                else:
                    if console:
                        break
                    else:
                        time.sleep(0.1)
                continue
            elif log[0] == "#" or not UtilMgr.isValidStr(log, inc=True):
                continue

            # parse log #
            m = SysMgr.getTraceItem(log)
            if not m:
                continue

            if checkTime:
                d = m.groupdict()
                try:
                    ltime = float(d["time"])

                    if since:
                        if since > ltime:
                            continue
                        else:
                            since = 0

                    if until:
                        if until < ltime:
                            continue
                        else:
                            break

                    if since == until == 0:
                        checkTime = False
                except SystemExit:
                    sys.exit(0)
                except:
                    pass

            if SysMgr.jsonEnable:
                # pick each key and value set #
                d = m.groupdict()
                addAttrs = re.findall("(\w+)=(.+?(?=\s\w+=|\Z))", d["etc"])
                for name, val in addAttrs:
                    d[name] = val.rstrip()

                # set process name #
                if incProcName:
                    tgid = d.get("tgid")
                    if tgid:
                        if tgid == d["thread"]:
                            tgcomm = d["comm"]
                        else:
                            tgcomm = SysMgr.getComm(d["tgid"])
                        d["tgcomm"] = tgcomm

                log = UtilMgr.convDict2Str(d, pretty=not console).rstrip()

            if SysMgr.outPath and console:
                print(log)

            SysMgr.printPipe(log)

            # check log command #
            SysMgr.checkLogCond(log, watchcond, refilter)

    @staticmethod
    def printAndroid(console=False):
        # check logcat #
        logcatPath = UtilMgr.which("logcat")
        if not logcatPath:
            SysMgr.printErr("failed to find logcat")
            sys.exit(-1)

        # create pipe #
        rd, wr = os.pipe()

        # get item for log filter #
        if "WATCHLOG" in SysMgr.environList:
            watchcond = SysMgr.environList["WATCHLOG"][0].split("+")
            refilter = SysMgr.environList.get("REFILTER")
        else:
            refilter = watchcond = None

        SysMgr.printInfo("start printing android logs... [ STOP(Ctrl+c) ]")

        # create a new process #
        pid = SysMgr.createProcess()

        # parent #
        if pid > 0:
            # set pipe #
            os.close(wr)
            rdPipe = os.fdopen(rd, "r")
            SysMgr.setPipeSize(rdPipe)

            # get length of pid #
            pidlen = len(str(SysMgr.maxPid))

            while 1:
                log = rdPipe.readline()

                if not log:
                    if console:
                        break
                    else:
                        time.sleep(0.1)
                elif not UtilMgr.isValidStr(log, inc=True):
                    continue

                try:
                    meta, msg = log.split(": ", 1)
                except SystemExit:
                    sys.exit(0)
                except:
                    continue

                # split items #
                items = meta.split()

                # local time #
                ltime = items[0] + "T" + items[1]

                # others #
                pid, tid, lev = items[2:5]

                if SysMgr.jsonEnable:
                    msgDict = {
                        "pid": pid,
                        "tid": tid,
                        "level": lev,
                    }
                else:
                    msgDict = {}

                # proc #
                pcomm = " "
                if pid != "0":
                    pcomm = SysMgr.getComm(pid, cache=True, save=True)
                    if not pcomm:
                        pcomm = " "

                if SysMgr.jsonEnable:
                    msgDict["pcomm"] = pcomm

                proc = "{0:>16}({1:>{pidlen}})".format(
                    pcomm, pid, pidlen=pidlen
                )

                # thread #
                thread = ""
                if not SysMgr.processEnable:
                    if tid == "0":
                        tcomm = " "
                    else:
                        tcomm = SysMgr.getComm(tid, cache=True, save=True)
                        if not tcomm:
                            tcomm = " "

                    if SysMgr.jsonEnable:
                        msgDict["tcomm"] = tcomm
                    else:
                        thread = "{0:>16}({1:>{pidlen}})".format(
                            tcomm, tid, pidlen=pidlen
                        )
                        proc += "->" + thread

                # remove useless spaces #
                msg = msg.strip()

                if len(items) > 5:
                    msg = items[5] + ": " + msg

                if SysMgr.jsonEnable:
                    msgDict["message"] = msg
                    log = UtilMgr.convDict2Str(msgDict, pretty=not console)
                else:
                    # build log line #
                    log = "%s %s %s %s" % (ltime, proc, lev, msg)

                if SysMgr.outPath and console:
                    print(log)

                SysMgr.printPipe(log)

                # check log command #
                SysMgr.checkLogCond(log, watchcond, refilter)

        # child #
        elif pid == 0:
            # redirect stdout to pipe #
            os.close(rd)
            wrPipe = os.fdopen(wr, "w")
            SysMgr.setPipeSize(wrPipe)
            os.dup2(wr, 1)

            # set SIGPIPE handler for termination of parent #
            SysMgr.setPipeHandler()

            SysMgr.executeProcess(
                cmd=["logcat", "-v", "threadtime"], closeFd=False
            )

            sys.exit(0)

    @staticmethod
    def printSyslog(console=False):
        # check root permission #
        SysMgr.checkRootPerm()

        # open syslog file #
        # TODO: implement reading syslog using syslog syscall #
        try:
            if not SysMgr.syslogFd:
                SysMgr.syslogFd = open(SysMgr.syslogPath, "r")
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printOpenErr(SysMgr.syslogPath)
            sys.exit(-1)

        # set nonblock attribute #
        try:
            if console:
                SysMgr.setBlock(SysMgr.syslogFd, False)
                stream = True
            else:
                SysMgr.setBlock(SysMgr.syslogFd)
                stream = False
        except SystemExit:
            sys.exit(0)
        except:
            pass

        SysMgr.printInfo("start printing syslog... [ STOP(Ctrl+c) ]")

        # set file posiion #
        SysMgr.syslogFd.seek(0)

        # get item for log filter #
        if "WATCHLOG" in SysMgr.environList:
            watchcond = SysMgr.environList["WATCHLOG"][0].split("+")
            refilter = SysMgr.environList.get("REFILTER")
        else:
            refilter = watchcond = None

        while 1:
            log = SysMgr.syslogFd.readline()
            if not log:
                if stream:
                    break
                else:
                    time.sleep(0.1)
            elif not UtilMgr.isValidStr(log, inc=True):
                continue

            if SysMgr.outPath and console:
                print(log)

            SysMgr.printPipe(log, newline=False)

            # check log command #
            SysMgr.checkLogCond(log, watchcond, refilter)

    @staticmethod
    def printJournal(console=False):
        # get ctypes object #
        SysMgr.importPkgItems("ctypes")

        """
        struct sd_journal {
                int toplevel_fd;

                char *path;
                char *prefix;
                char *namespace;

                OrderedHashmap *files;
                IteratedCache *files_cache;
                MMapCache *mmap;

                Location current_location;

                JournalFile *current_file;
                uint64_t current_field;

                Match *level0, *level1, *level2;

                pid_t original_pid;

                int inotify_fd;
                unsigned current_invalidate_counter, last_invalidate_counter;
                usec_t last_process_usec;
                unsigned generation;

                /* Iterating through unique fields and their data values */
                char *unique_field;
                JournalFile *unique_file;
                uint64_t unique_offset;

                /* Iterating through known fields */
                JournalFile *fields_file;
                uint64_t fields_offset;
                uint64_t fields_hash_table_index;
                char *fields_buffer;
                size_t fields_buffer_allocated;

                int flags;

                bool on_network:1;
                bool no_new_files:1;
                bool no_inotify:1;
                bool unique_file_lost:1; /* File we were iterating over got
                                            removed, and there were no more
                                            files, so sd_j_enumerate_unique
                                            will return a value equal to 0. */
                bool fields_file_lost:1;
                bool has_runtime_files:1;
                bool has_persistent_files:1;

                size_t data_threshold;

                Hashmap *directories_by_path;
                Hashmap *directories_by_wd;

                Hashmap *errors;
        };
        """

        errstr = "failed to print journal"

        # load libsystemd library #
        try:
            if not SysMgr.systemdObj:
                SysMgr.systemdObj = SysMgr.loadLib(SysMgr.libsystemdPath)
                if not SysMgr.systemdObj:
                    raise Exception("no %s" % SysMgr.libsystemdPath)

            func = "sd_journal_open"
            if not hasattr(SysMgr.systemdObj, func):
                raise Exception("no %s in %s" % (func, SysMgr.libsystemdPath))
        except:
            SysMgr.printErr(errstr, True)
            sys.exit(-1)

        # define shortcut for object #
        systemdObj = SysMgr.systemdObj

        # open journal #
        jrl = c_void_p(0)
        flag = LogMgr.SD_JOURNAL_LOCAL_ONLY
        res = systemdObj.sd_journal_open(byref(jrl), c_int(flag))
        if res < 0:
            SysMgr.printErr(errstr + " because there is no journal")
            return

        SysMgr.printInfo("start printing journal... [ STOP(Ctrl+c) ]")

        # set head #
        res = systemdObj.sd_journal_seek_head(jrl)
        if res < 0:
            SysMgr.printErr(errstr + " because there is no journal head")
            return

        # get item for log filter #
        if "WATCHLOG" in SysMgr.environList:
            watchcond = SysMgr.environList["WATCHLOG"][0].split("+")
            refilter = SysMgr.environList.get("REFILTER")
        else:
            refilter = watchcond = None

        # get times for tail and until option #
        current = time.time()
        since, until = SysMgr.getTimeValues(["TAIL", "UNTIL"], current)

        # apply relative time #
        if since < 0:
            since += current
        if until < 0:
            until += current

        # initialize variables #
        data = c_void_p(0)
        size = c_size_t(0)
        usec = c_uint64(0)
        timeout = c_uint64(10000)

        # set fields #
        if SysMgr.inputParam:
            fieldList = SysMgr.inputParam.split(",")
        else:
            fieldList = [
                b"_TIME",
                b"_HOSTNAME",
                b"_TRANSPORT",
                b"_COMM",
                b"_PID",
                b"MESSAGE",
            ]

        # move to the end of journal #
        if not SysMgr.showAll:
            systemdObj.sd_journal_seek_tail(jrl)

        # define summary table #
        table = {}

        SysMgr.printPipe("\n")

        # start reading journal in loop #
        try:
            while 1:
                res = systemdObj.sd_journal_next(jrl)
                if res == 0:
                    ret = systemdObj.sd_journal_wait(jrl, timeout)
                    # SD_JOURNAL_NOP / SD_JOURNAL_APPEND / SD_JOURNAL_INVALID #
                    if ret == 0:
                        if console:
                            break
                        else:
                            continue
                    elif ret in (1, 2):
                        continue
                    elif ret < 0:
                        break
                elif res < 1:
                    break

                # check time condition #
                if since or until:
                    ret = systemdObj.sd_journal_get_realtime_usec(
                        jrl, byref(usec)
                    )
                    if ret == 0:
                        realtime = usec.value / 1000000
                        if since and realtime < since:
                            continue
                        elif until and realtime >= until:
                            SysMgr.printWarn(
                                "all previous journal logs are printed", True
                            )
                            sys.exit(0)

                # traverse specific fields #
                if SysMgr.inputParam:
                    res = systemdObj.sd_journal_restart_data(jrl)
                    while 1:
                        res = systemdObj.sd_journal_enumerate_data(
                            jrl, byref(data), byref(size)
                        )
                        if res < 1:
                            break

                        SysMgr.printPipe(
                            cast(data, c_char_p).value.decode("latin-1"),
                            flush=True,
                        )
                    SysMgr.printPipe(flush=True)
                    continue

                # traverse all fields #
                jrlStr = b""
                jrlDict = {}
                for field in fieldList:
                    if field == b"_TIME":
                        # get time #
                        ret = systemdObj.sd_journal_get_realtime_usec(
                            jrl, byref(usec)
                        )
                        if ret < 0:
                            realtime = 0
                        else:
                            realtime = usec.value

                        wtime = time.strftime(
                            "%m %d %H:%M:%S ",
                            time.localtime(realtime / float(1000000)),
                        )
                        """
                        ret = systemdObj.sd_journal_get_monotonic_usec(
                            jrl, byref(usec), boottime)
                        """

                        # set time #
                        if SysMgr.jsonEnable:
                            jrlDict[field.decode().lstrip("_")] = wtime.strip()

                        jrlStr += wtime.encode()

                        continue

                    res = systemdObj.sd_journal_get_data(
                        jrl, field, byref(data), byref(size)
                    )
                    if res < 0:
                        continue

                    val = cast(data, c_char_p).value[len(field) + 1 :]
                    if SysMgr.jsonEnable:
                        pass
                    elif field == b"_COMM":
                        if SysMgr.outPath:
                            val = val.decode("latin-1").rstrip("\x01")
                            table.setdefault(val, 0)
                            table[val] += 1
                    elif field == b"_PID":
                        val = b"[%s]: " % val
                    elif field == b"_TRANSPORT" and val == b"kernel":
                        val += b": "
                    else:
                        val += b" "

                    if SysMgr.jsonEnable:
                        jrlDict[
                            field.decode().lstrip("_")
                        ] = val.decode().strip()

                    if isinstance(val, str):
                        val = val.encode()
                    jrlStr += val

                # print journal #
                if jrlStr:
                    try:
                        decstr = jrlStr.decode("latin-1")

                        # check filter #
                        if not UtilMgr.isValidStr(decstr):
                            continue

                        if SysMgr.jsonEnable:
                            output = UtilMgr.convDict2Str(jrlDict)
                        else:
                            output = decstr

                        if SysMgr.outPath and console:
                            print(output)

                        SysMgr.printPipe(output, flush=True)

                        # check log command #
                        SysMgr.checkLogCond(decstr, watchcond, refilter)
                    except SystemExit:
                        sys.exit(0)
                    except:
                        if SysMgr.jsonEnable:
                            output = jrlDict
                        else:
                            output = jrlStr

                        SysMgr.printPipe(output, flush=True)

            # close journal #
            systemdObj.sd_journal_close(jrl)
        except SystemExit:
            if not table:
                return

            SysMgr.printPipe("\n[Journal Summary]\n%s\n" % twoLine)

            SysMgr.printPipe(
                "{0:>32} {1:>16}\n{2:1}".format("COMM", "COUNT", oneLine)
            )

            total = 0
            for comm, count in sorted(
                table.items(), key=lambda x: x[1], reverse=True
            ):
                SysMgr.printPipe(
                    "{0:>32} {1:>16}".format(comm, UtilMgr.convNum(count))
                )
                total += count

            SysMgr.printPipe(
                "\n{0:>32} {1:>16}\n{2:1}".format(
                    "TOTAL", UtilMgr.convNum(total), oneLine
                )
            )

    @staticmethod
    def getKmsg(line=0):
        try:
            fd = os.open(SysMgr.kmsgPath, os.O_RDONLY | os.O_NONBLOCK)
        except:
            return None

        logs = []
        while 1:
            try:
                data = os.read(fd, SysMgr.PAGESIZE).decode()
                logs.append(data)
            except SystemExit:
                sys.exit(0)
            except:
                break

        # convert logs #
        retList = []
        for log in logs[-line:]:
            # parse log #
            pos = log.find(";")

            meta = log[:pos].split(",")
            if len(meta) > 2:
                nrLevel = long(meta[0])
                try:
                    level = ConfigMgr.LOG_LEVEL[nrLevel]
                except:
                    level = nrLevel

                # time #
                ltime = str(meta[2])
                if len(ltime) < 7:
                    ltime = "0.%s" % ltime
                else:
                    ltime = "%s.%s" % (ltime[:-6], ltime[-6:])

                # name & log #
                log = log[pos + 1 :]
                npos = log.find(":")
                name = log[:npos]
                if log[-1] == "\n":
                    log = log[npos + 1 : -1]
                else:
                    log = log[npos + 1 :]

                retList.append("[%s] (%s) %s: %s" % (ltime, level, name, log))

        return retList[-line:]

    @staticmethod
    def printKmsg(console=False):
        # open kmsg device node #
        try:
            SysMgr.kmsgFd = open(SysMgr.kmsgPath, "r")
            if SysMgr.findOption("Q"):
                SysMgr.setBlock(SysMgr.kmsgFd, False)
        except SystemExit:
            sys.exit(0)
        except:
            # get ctypes object #
            SysMgr.importPkgItems("ctypes")

        SysMgr.printInfo("start printing kernel log... [ STOP(Ctrl+c) ]")

        # get item for log filter #
        if "WATCHLOG" in SysMgr.environList:
            watchcond = SysMgr.environList["WATCHLOG"][0].split("+")
            refilter = SysMgr.environList.get("REFILTER")
        else:
            refilter = watchcond = None

        # get times for tail and until option #
        current = SysMgr.getUptime()
        tail, until = SysMgr.getTimeValues(["TAIL", "UNTIL"], current)

        # apply relative time #
        if tail < 0:
            tail += current
        if until < 0:
            until += current

        # syslog #
        if not SysMgr.kmsgFd:
            # get kernel ring-buffer size #
            size = SysMgr.syscall(
                "syslog", LogMgr.SYSLOG_ACTION_SIZE_BUFFER, 0, 0
            )

            # allocate buffer #
            buf = (c_char * size)()

            ret = SysMgr.syscall(
                "syslog", LogMgr.SYSLOG_ACTION_READ_ALL, buf, size
            )
            if ret > 0:
                logBuf = memoryview(buf).tobytes().decode()
                for line in logBuf.split("\n"):
                    if not UtilMgr.isValidStr(line):
                        continue

                    # convert log level #
                    if line.startswith("<"):
                        try:
                            level, line = line[1:].split(">", 1)

                            # get level number #
                            nrLevel = long(level)
                            try:
                                level = ConfigMgr.LOG_LEVEL[nrLevel]
                            except:
                                level = nrLevel

                            ts, line = line.split("] ", 1)

                            line = "%s] <%s> %s" % (ts, level, line)
                        except SystemExit:
                            sys.exit(0)
                        except:
                            pass

                    # check time condition #
                    if tail or until:
                        try:
                            if not line.startswith("["):
                                raise Exception("no time")

                            ltime = float(line[1:].split("]", 1)[0])

                            if tail and ltime < tail:
                                continue
                            elif until and ltime >= until:
                                SysMgr.printWarn(
                                    "all previous kernel logs are printed",
                                    True,
                                )
                                sys.exit(0)

                        except SystemExit:
                            sys.exit(0)
                        except:
                            pass

                    # print to console #
                    if SysMgr.outPath and console:
                        print(line)

                    # print #
                    SysMgr.printPipe(line)

                    # check log command #
                    SysMgr.checkLogCond(line, watchcond, refilter)

            while 1:
                memset(buf, 0, size)
                ret = SysMgr.syscall(
                    "syslog", LogMgr.SYSLOG_ACTION_READ, buf, size
                )
                if ret < 1:
                    continue

                logBuf = memoryview(buf).tobytes().decode()
                if not UtilMgr.isValidStr(line):
                    continue

                if SysMgr.outPath and console:
                    print(logBuf)

                SysMgr.printPipe(logBuf)

            return

        # kmsg node #
        while 1:
            jsonResult = {}
            log = SysMgr.kmsgFd.readline()
            if not log:
                break

            # apply filter #
            if not UtilMgr.isValidStr(log):
                continue

            # parse log #
            # PRIORITY,SEQUENCE_NUM,TIMESTAMP,-;MESSAGE #
            pos = log.find(";")
            meta = log[:pos].split(",")
            if len(meta) > 2:
                nrLevel = long(meta[0])
                try:
                    level = ConfigMgr.LOG_LEVEL[nrLevel]
                except:
                    level = nrLevel

                # time #
                ltime = str(meta[2])
                if len(ltime) < 7:
                    ltime = "0.%s" % ltime
                else:
                    ltime = "%s.%s" % (ltime[:-6], ltime[-6:])

                # check time condition #
                if tail or until:
                    try:
                        ltime = float(ltime)
                        if tail and ltime < tail:
                            continue
                        elif until and ltime >= until:
                            SysMgr.printWarn(
                                "all previous kernel logs are printed", True
                            )
                            sys.exit(0)

                    except SystemExit:
                        sys.exit(0)
                    except:
                        pass

                # update pos #
                log = log[pos + 1 :]

                # name #
                npos = log.find(":")
                if npos < 0:
                    name = ""
                else:
                    name = log[:npos]

                # log #
                if log[-1] == "\n":
                    log = log[npos + 1 : -1]
                else:
                    log = log[npos + 1 :]

                # update message #
                if name:
                    log = "%s: %s" % (name, log)

                # set message #
                if SysMgr.jsonEnable:
                    jsonResult = dict(
                        uptime=ltime, level=level, name=name, message=log
                    )
                else:
                    if not SysMgr.outPath:
                        level = UtilMgr.convColor(level, "BOLD")
                        name = UtilMgr.convColor(name, "SPECIAL")
                        ltime = UtilMgr.convColor(ltime, "GREEN")
                    log = "[%s] (%s) %s" % (ltime, level, log)
            # skip special info #
            elif tail or until:
                continue

            # print message #
            if SysMgr.jsonEnable:
                if jsonResult:
                    jsonResult = UtilMgr.convDict2Str(
                        jsonResult, pretty=not SysMgr.streamEnable
                    )
                    SysMgr.printPipe(jsonResult)
            else:
                if SysMgr.outPath and console:
                    print(log.rstrip())

                SysMgr.printPipe(log.rstrip())

            # check log command #
            SysMgr.checkLogCond(log, watchcond, refilter)

    @staticmethod
    def doLogTrace(msg=None, level=None):
        # print trace point list #
        if SysMgr.findOption("l"):
            if not SysMgr.findOption("Q"):
                SysMgr.streamEnable = False
            SysMgr.printDirs(
                SysMgr.getTraceEventPath(),
                SysMgr.funcDepth if SysMgr.funcDepth > 0 else 1,
            )
            sys.exit(0)

        # enable events #
        SysMgr.enableEvents()

        # disable events #
        SysMgr.disableEvents()

        # clear filters #
        SysMgr.clearFilters()

        # open trace_marker #
        try:
            if not SysMgr.traceFd:
                tracePath = "%s/../trace_marker" % SysMgr.getTraceEventPath()
                SysMgr.traceFd = open(tracePath, "w")
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printOpenErr(tracePath)
            return -1

        # get trace status #
        traceStat = SysMgr.readTraceFile("../tracing_on")

        # tracing on #
        if SysMgr.forceEnable and traceStat == "0":
            recoverTraceStat = True
            SysMgr.writeTraceCmd("../tracing_on", "1")
        else:
            recoverTraceStat = False

        # print messages #
        try:
            msgList = msg.split("\n")
            for line in msgList:
                SysMgr.traceFd.write(line)
            SysMgr.traceFd.flush()
        except SystemExit:
            sys.exit(0)
        except:
            try:
                if traceStat == "0":
                    SysMgr.printErr(
                        "failed to log trace message because tracing is off"
                    )
            except SystemExit:
                sys.exit(0)
            except:
                SysMgr.printWarn("failed to get trace status", reason=True)
            return -1
        finally:
            if recoverTraceStat:
                SysMgr.writeTraceCmd("../tracing_on", traceStat)

        return 0

    @staticmethod
    def doLogKmsg(msg=None, level=None):
        # open kmsg device node #
        try:
            if not SysMgr.kmsgFd:
                SysMgr.kmsgFd = open(SysMgr.kmsgPath, "w")
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printOpenErr(SysMgr.kmsgPath)
            return -1

        # print messages #
        try:
            msgList = msg.split("\n")
            for line in msgList:
                SysMgr.kmsgFd.write("guider: %s\n" % line)
            SysMgr.kmsgFd.flush()
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printWarn("failed to log kernel message", reason=True)
            return -1

        return 0

    @staticmethod
    def doLogSyslog(msg=None, level=None):
        if not msg:
            return -1

        # load libc #
        SysMgr.loadLibcObj(exit=True)

        # set default level #
        if level is None:
            level = LogMgr.LOG_NOTICE

        # write message #
        SysMgr.libcObj.syslog(level, msg.encode())

        return 0

    @staticmethod
    def doLogAndroid(msg=None, level=None, tag="guider"):
        # get ctypes object #
        SysMgr.importPkgItems("ctypes")

        if not msg:
            return -1

        # set default priority #
        if level is None:
            level = LogMgr.ANDROID_LOG_INFO

        # load liblog library #
        try:
            if not SysMgr.andlogObj:
                SysMgr.andlogObj = SysMgr.loadLib(SysMgr.liblogPath)
                if not SysMgr.andlogObj:
                    raise Exception("no %s" % SysMgr.liblogPath)

                func = "__android_log_write"
                if not hasattr(SysMgr.andlogObj, func):
                    SysMgr.andlogObj = None
                    raise Exception("no %s in %s" % (func, SysMgr.liblogPath))

                # prototype #
                SysMgr.andlogFunc = getattr(SysMgr.andlogObj, func)
                SysMgr.andlogFunc.argtypes = [c_int, c_char_p, c_char_p]
                SysMgr.andlogFunc.restype = c_int
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printErr("failed to log android message", True)
            return -1

        # print message #
        # pylint: disable=not-callable
        return SysMgr.andlogFunc(level, tag.encode(), msg.encode())

    @staticmethod
    def doLogJournal(msg=None, level=None):
        # get ctypes object #
        SysMgr.importPkgItems("ctypes")

        if not msg:
            return -1

        # set default level #
        if level is None:
            level = LogMgr.LOG_NOTICE

        # load libsystemd library #
        try:
            if not SysMgr.systemdObj:
                SysMgr.systemdObj = SysMgr.loadLib(SysMgr.libsystemdPath)
                if not SysMgr.systemdObj:
                    raise Exception("no %s" % SysMgr.libsystemdPath)

                func = "sd_journal_print"
                if not hasattr(SysMgr.systemdObj, func):
                    SysMgr.systemdObj = None
                    raise Exception(
                        "no %s in %s" % (func, SysMgr.libsystemdPath)
                    )
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printErr("failed to log journal message", True)
            return -1

        # print message #
        return SysMgr.systemdObj.sd_journal_print(level, msg.encode())


class SysMgr(object):
    """Manager for system"""

    arch = None
    origArgs = []
    kernelVersion = None
    platform = ""
    isLinux = True
    isDarwin = False
    isWindows = False
    isAndroid = False
    drawMode = False
    archOption = None

    # page size #
    try:
        PAGESIZE = os.sysconf("SC_PAGE_SIZE")
    except:
        PAGESIZE = 4096

    # define many-core number #
    NRMANYCORE = 8

    # set tick #
    try:
        HZ = 250  # 4ms tick #
        if isLinux:
            TICK = os.sysconf(os.sysconf_names["SC_CLK_TCK"])
        else:
            TICK = long((1 / float(HZ)) * 1000)
    except:
        TICK = long((1 / float(HZ)) * 1000)

    # python call function #
    if sys.version_info < (3, 6):
        pyCallFunc = "PyEval_EvalFrameEx"
    elif sys.version_info < (3, 11):
        pyCallFunc = "_PyEval_EvalFrameDefault"
    else:
        # TODO: change return function #
        pyCallFunc = "_PyThreadState_PopFrame"

    startInitTime = 0  # init time for Guider #
    startTime = 0  # start time for Guider #
    startRecTime = 0  # start time for Recording #
    startRunTime = 0  # start time for Process #
    startOverheadTime = 0  # start overhead time for Process #

    blockSize = 512
    bufferSize = -1
    termGetId = None
    termSetId = None
    ttyRows = 43
    ttyRowsMargin = 2
    ttyCols = 156
    ttyData = None
    encoding = None
    remoteRun = False
    magicStr = "@@@@@"
    launchBuffer = ""
    lineLength = 154
    pid = 0
    comm = __module__
    commLen = 16
    masterPid = 0
    parentPid = 0
    prio = None
    ioprio = None
    funcDepth = 0
    maxPid = 32768
    maxRdCnt = 1024
    maxSize = 2147483647
    pidDigit = 5
    stdlog = None
    stderr = sys.stderr
    packetSize = 32767
    defaultCliPort = 5554
    defaultServPort = 5555
    bgProcList = None
    waitDelay = 0.5
    repeatCnt = 0
    progressCnt = 0
    skipCnt = 0
    termCnt = 0
    repeatInterval = 0
    maxIntervalTick = 0
    ipAddr = None
    swappiness = 0
    vmpressure = 0
    overcommit = 0
    limitRepDirSize = 0
    nrRun = 0
    nrReport = 0
    nrProcMemReport = 6

    # watermark constants #
    cpuPerHighThr = 80
    cpuPerLowThr = 10
    memAvailPerThr = 90
    memHighThr = 1024
    memLowThr = 100
    swapPerThr = 90
    diskPerHighThr = 90
    fdHighThr = 8192

    # print condition #
    printCond = dict(
        CPUCOND=-1,
        MEMFREECOND=sys.maxsize,
        MEMAVLCOND=sys.maxsize,
        BLKRDCOND=-1,
        BLKWRCOND=-1,
    )

    # lifecycle condition #
    startUptime = 0
    deadlineUptime = 0
    startCondCpuMore = -1
    startCondCpuLess = -1
    startCondMemMore = -1
    startCondMemLess = -1
    exitCondCpuMore = -1
    exitCondCpuLess = -1
    exitCondMemMore = -1
    exitCondMemLess = -1
    cpuUsage = -1
    memAvail = -1
    memTotal = -1

    # path #
    procPath = "/proc"
    imagePath = None
    mountCmd = None
    cgroupPathList = {}
    drawFormat = "svg"
    traceEventPath = None
    tracefsPath = None
    defaultTracefsPath = "/sys/kernel/tracing"
    debugfsPath = None
    defaultDebugfsPath = "/sys/kernel/debug"
    debugDirPath = "/usr/lib/debug"
    cacheDirPath = "/var/log/guider"
    outFilePath = "guider.out"
    confFileName = "guider.conf"
    cmdFileName = "guider.cmd"
    tmpPath = "/tmp"
    kmsgPath = "/dev/kmsg"
    nullPath = "/dev/null"
    rootdevPath = None
    syslogPath = "/var/log/syslog"
    lmkPath = "/sys/module/lowmemorykiller/parameters/minfree"
    pythonPath = sys.executable
    pyLibPath = None
    objdumpPath = []
    rootPath = ""
    fontPath = None
    libdltPath = "libdlt"
    libcPath = "libc"
    libgobjPath = "libgobject-2.0"
    libgioPath = "libgio-2.0"
    libdbusPath = "libdbus-1"
    libcppPath = "libstdc++"
    libsystemdPath = "libsystemd"
    liblogPath = "liblog"
    libglesPath = "libGLESv2"
    ldCachePath = "/etc/ld.so.cache"
    libcorkscrewPath = "libcorkscrew"
    libbacktracePath = "libbacktrace"
    libdemanglePath = libcppPath
    libLLVMPath = "libLLVM.so"
    environList = {}
    externList = {}
    environ = {}
    eventLogPath = None
    inputFile = None
    outputFile = None
    inputParam = None
    outPath = None
    origOutPath = None
    outPathList = {}
    freezerPath = None

    # list #
    signalCmd = "trap 'kill $$' INT\n"
    saveCmd = None
    boundaryLine = None
    nrTop = 0
    pipeForPager = None
    printFd = None
    fileSuffix = None
    parsedAnalOption = False
    optionList = []
    customCmd = []
    pyFuncFilter = []
    userCmd = []
    kernelCmd = []
    udpListCache = {}
    tcpListCache = {}
    udsListCache = {}
    ipAddrCache = {}
    customEventList = []
    userEventList = []
    kernelEventList = []
    perfEventChannel = {}
    perfEventData = {}
    perfEventList = {}
    perfTargetEvent = []
    psiData = {}
    prevPsiData = {}
    ignoreItemList = []
    idList = []
    commCache = {}
    commFdCache = {}
    fdCache = {}
    libCache = {}
    rawFdCache = {}
    syscallCache = {}
    netAddrCache = {}
    pciList = []
    limitDirList = {}
    limitDirCntList = {}
    fixedTaskList = []
    overlayfsCache = {}
    vmflagList = []

    # threshold #
    thrData = {}
    prevThrData = {}
    thrTarget = {}
    thrRefreshList = []
    thrTimerList = []
    thrEvtList = {}
    thrEvtHist = {}
    thrEvtCondList = {}
    thrEvtCntList = {}
    thrItemList = {}
    eventLockList = {}
    eventCommandList = {}

    # object #
    impPkg = {}
    impGlbPkg = {}
    skipImpPkg = {}
    exitFuncList = []
    exitPrivateFuncList = {}
    dltObj = None
    dltCtx = None
    shmObj = None
    systemdObj = None
    libcObj = None
    libgioObj = None
    libdbusObj = None
    libcorkscrewObj = None
    libbacktraceObj = None
    libgObj = None
    libglesObj = None
    statvfsObj = None
    guiderObj = None
    libcppObj = None
    libdemangleObj = None
    libLLVMObj = None
    demangleFunc = None
    geterrnoFunc = None
    maxErrStrLen = 256
    raFunc = None
    raMaxSize = 1048576  # 1MB #
    matplotlibVersion = 0
    matplotlibDpi = 500
    matplotlibSize = None
    sigsetObj = None
    sigsetOldObj = None
    ansiObj = None
    localServObj = None
    remoteServObj = None
    netlinkObj = None
    andlogObj = None
    andlogFunc = None

    geAttr = [0] * 9
    addrListForPrint = {}
    addrListForReport = {}

    # number #
    maxCore = 0
    nrCore = 0
    utilProc = 0
    logSize = 0
    kmsgLine = 100
    curLine = 0
    totalLine = 0
    dbgEventLine = 0
    uptime = 0
    prevUptime = 0
    dltTime = 0
    uptimeDiff = 0
    diskStats = []
    prevDiskStats = []
    netstat = ""
    prevNetstat = ""
    loadavg = ""
    battery = ""
    netInIndex = -1
    nrUDPSock = 0
    nrTCPSock = 0
    nrUDSSock = 0
    nrTCPConn = 0
    cpuStealUsage = 0
    nrOOMKill = 0
    nrTotalOOMKill = -1
    nrIrq = 0

    # peak #
    maxBlkRd = 0
    maxBlkWr = 0
    maxCpuUsage = 0
    maxCpuSteal = 0
    maxGpuMemUsage = 0
    maxGpuUsage = 0
    maxInterval = 0
    maxLoad1m = 0
    maxLoad5m = 0
    maxLoad15m = 0
    maxNetIn = 0
    maxNetOut = 0
    maxNrContextSwitch = 0
    maxNrCore = 0
    maxNrFd = 0
    maxNrIrq = 0
    maxNrProcess = 0
    maxNrTaskAbnormal = 0
    maxNrTaskDie = 0
    maxNrTaskNew = 0
    maxNrThread = 0
    maxPsiCpu = 0
    maxPsiIo = 0
    maxPsiMem = 0
    maxSwapPer = 0
    minDiskFree = maxSize
    minMemAvail = maxSize

    # report #
    reportObject = None
    terminalOver = False
    procBuffer = []
    fixedProcList = {}
    topInstance = None
    procInstance = {}
    fileInstance = {}
    sysInstance = None
    procBufferSize = 0
    bufferOverflowed = False
    bufferString = ""
    bufferList = []
    bufferRows = 0
    sysinfoBuffer = ""
    kerSymTable = {}
    jsonData = {}
    nrTopRank = 10
    layout = None
    reportReason = None
    lastHost = None

    showAll = False
    enableAll = False
    disableAll = False
    intervalNow = 0
    recordStatus = False
    bgStatus = False
    condExit = False
    sort = None
    sortCond = None
    sortCondOp = None
    newlined = True
    nsType = "pid"

    # descriptor #
    maxFd = 512
    maxKeepFd = maxFd - 16
    batteryFd = None
    cmdFd = None
    diskStatsFd = None
    cgroupFd = None
    eventLogFd = None
    irqFd = None
    kmsgFd = None
    traceFd = None
    lmkFd = None
    loadavgFd = None
    memFd = None
    mountFd = None
    msgqFd = None
    netdevFd = None
    netstatFd = None
    nullFd = None
    semFd = None
    shmFd = None
    slabFd = None
    softirqFd = None
    statFd = None
    swapFd = None
    syslogFd = None
    uptimeFd = None
    vmstatFd = None
    vmallocFd = None
    zoneFd = None
    swappinessFd = None
    vmpressureFd = None
    overcommitFd = None

    # flag #
    affinityEnable = False
    andlogEnable = False
    attrEnable = False
    barGraphEnable = None
    binderEnable = False
    blockEnable = False
    bufferLossEnable = False
    cgroupEnable = False
    cgroupRootEnable = False
    cloneEnable = True
    cmdEnable = False
    cmdlineEnable = False
    colorEnable = True
    compressEnable = False
    countEnable = False
    cpuAvgEnable = True
    cpuEnable = True
    dbusTopEnable = False
    dbusUnitEnable = False
    delayEnable = False
    demangleEnable = True
    depEnable = False
    diskEnable = False
    dltEnable = False
    dltTimeEnable = False
    dltTopEnable = False
    dmabufEnable = False
    dwarfEnable = False
    elasticEnable = False
    encodeEnable = True
    eventHandleEnable = True
    exceptCommFilter = False
    exeEnable = False
    execEnable = None
    exitFlag = False
    fileEnable = False
    fileTopEnable = False
    fixTargetEnable = False
    floatEnable = False
    forceColorEnable = False
    forceEnable = False
    freeMemEnable = False
    fsEnable = False
    functionEnable = False
    generalEnable = True
    gpuEnable = True
    gpuMemEnable = True
    graphEnable = False
    groupProcEnable = False
    heapEnable = False
    i2cEnable = False
    idlePageEnable = False
    inExit = False
    inWaitStatus = False
    initWssEnable = False
    inotifyEnable = False
    intervalEnable = 0
    ioschedEnable = False
    iouringEnable = False
    irqEnable = False
    journalEnable = False
    jsonEnable = False
    keventEnable = False
    kmsgEnable = False
    kvmEnable = False
    latEnable = cpuEnable
    leakEnable = False
    lockEnable = False
    logEnable = True
    loggingEnable = False
    loggingOpsEnable = False
    logoEnable = True
    memEnable = False
    minStatEnable = False
    networkEnable = False
    nsAllEnable = False
    nsEnable = False
    oomEnable = False
    peakEnable = False
    perfEnable = False
    perfGroupEnable = False
    pipeEnable = False
    powerEnable = False
    printEnable = True
    processEnable = True
    psiEnable = True
    pssEnable = False
    rankProcEnable = True
    recursionEnable = False
    reportEnable = False
    reportFileEnable = False
    resetEnable = False
    rssEnable = False
    schedEnable = False
    schedstatEnable = True
    selectEnable = True
    sigHandlerEnable = False
    stackEnable = False
    streamEnable = False
    sttyEnable = None
    sysEnable = False
    syslogEnable = False
    systemEnable = False
    taskEnable = True
    taskThrEnable = False
    termFlag = True
    tgidEnable = True
    threadEnable = False
    thresholdEnable = False
    totalEnable = False
    truncEnable = True
    ttyEnable = False
    ueventEnable = False
    userEnable = False
    userEnableWarn = False
    userRecordEnable = False
    ussEnable = False
    vssEnable = False
    waitEnable = False
    warnEnable = False
    wchanEnable = False
    wfcEnable = False
    wqEnable = False
    wssEnable = False

    cmdList = {}
    rcmdList = {}
    savedProcTree = {}
    savedProcComm = {}
    savedMountTree = {}
    preemptGroup = []
    filterGroup = []
    schedFilter = []
    affinityFilter = []
    killFilter = []
    syscallList = []
    syscallExceptList = []
    perCoreList = []
    perCoreDrawList = []
    childList = {}
    gpuMemGetters = None
    pidFilter = ""
    fileWatchList = {}
    fileWatchTotalList = {}
    procWatchList = {}
    hostList = {}

    traceTgidFormat1 = (
        r"^\s*(?P<comm>\S+)-(?P<thread>[0-9]+)\s+"
        r"\(\s*(?P<tgid>\S+)\)\s+\[(?P<core>[0-9]+)\]\s+"
        r"(?P<irq>(\S{5}\s)*)(?P<time>\S+):\s+(?P<func>\S+):(?P<etc>.+)"
    )

    traceTgidFormat2 = (
        r"^\s*(?P<comm>\S+)-(?P<thread>[0-9]+)\s+"
        r"\[(?P<core>[0-9]+)\]\s+\(\s*(?P<tgid>.+)\)\s+"
        r"(?P<irq>(\S{5}\s)*)(?P<time>\S+):\s+(?P<func>\S+)(?P<etc>.+)"
    )

    traceNoTgidFormat = (
        r"^\s*(?P<comm>\S+)-(?P<thread>[0-9]+)\s+"
        r"\[(?P<core>[0-9]+)\]\s+(?P<irq>(\S{5}\s)*)(?P<time>\S+):\s+"
        r"(?P<func>\S+):(?P<etc>.+)"
    )

    cliCmdStr = """
Commands:
    - RUN:Command
    - BROADCAST:Command
    - UPSTREAM:Command
    - NOTIFY:Event
    - DOWNLOAD:RemotePath@LocalPath
    - UPLOAD:LocalPath@RemotePath
    - DIR{:RemotePath}
    - LIST
    - JOBS
    - PING
    - CLEAR
    - RESTART
    - HISTORY
    - QUIT\n"""

    def __init__(self, onlyInstance=False):
        self.cpuInfo = {}
        self.cpuCacheInfo = {}
        self.cpuCacheData = {}
        self.memInfo = {}
        self.devInfo = {}
        self.diskInfo = {}
        self.mountInfo = {}
        self.networkInfo = {}
        self.systemInfo = {}
        self.partitionInfo = {}
        self.devArchInfo = {}
        self.devNodeInfo = {}

        self.cpuData = None
        self.gpuData = {}
        self.memData = {}
        self.diskData = {}
        self.blockData = {}
        self.storageData = {}
        self.prevStorageData = {}
        self.storageMapData = {}
        self.ipcData = {}
        self.prevIpcData = {}
        self.userData = {}
        self.mountData = None
        self.mountUpdated = False
        self.loadData = None
        self.nrMaxThread = 0
        self.cmdlineData = None
        self.osData = None
        self.devData = None
        self.procData = None
        self.macAddr = None
        self.uname = []
        self.openFileData = {}
        self.limitData = []

        # resource update time #
        self.netUpdate = None
        self.storageUpdate = None
        self.ipcUpdate = None

        SysMgr.sysInstance = self
        if onlyInstance:
            return
        if not SysMgr.isLinux:
            return

        # save system info first #
        self.saveSysStat(False)

    def __del__(self):
        pass

    @staticmethod
    def writeErr(fd, log):
        LogMgr.lock(fd)

        try:
            fd.write(log)
        except SystemExit:
            sys.exit(0)
        except:
            return

        LogMgr.unlock(fd)

    @staticmethod
    def setRemoteEnv():
        for item in ("REMOTERUN", "NO_ENCODE", "MAXTTY"):
            os.environ[item] = "1"

    @staticmethod
    def setErrorLogger():
        if SysMgr.isLinux:
            sys.stderr = LogMgr()

    @staticmethod
    def getTotalUsage(pid):
        try:
            dobj = Debugger(pid, attach=False)
            dobj.initValues()
            dobj.getCpuUsage()

            cpu = dobj.prevCpuStat[0]
            rss = dobj.getMemUsage()
            proc = "%s(%s)" % (dobj.comm, dobj.pid)

            return [cpu, rss, proc]
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printWarn(
                "failed to get usage info of the task having PID %s" % pid,
                True,
                True,
            )
            return [0, 0, "?"]

    @staticmethod
    def getMaxPid():
        path = "%s/sys/kernel/pid_max" % SysMgr.procPath
        try:
            with open(path, "r") as fd:
                maxPid = fd.readline()[:-1]

                # update pid length #
                if len(maxPid) > SysMgr.pidDigit:
                    SysMgr.pidDigit = len(maxPid)

                SysMgr.maxPid = long(maxPid)
        except SystemExit:
            sys.exit(0)
        except:
            pass

    @staticmethod
    def waitForResource(cpuRes=-1, cpuCond="MORE", memRes=-1, memCond="MORE"):
        # check condition #
        if cpuRes >= 0:
            SysMgr.printInfo(
                "wait for system CPU usage %s than %s%%"
                % (cpuCond, UtilMgr.convNum(cpuRes))
            )
        elif memRes >= 0:
            SysMgr.printInfo(
                ("wait for system available memory %s than %s MB")
                % (memCond, UtilMgr.convNum(memRes))
            )
        else:
            return False

        # convert condition #
        if cpuCond:
            cpuCond = cpuCond.upper()
        if memCond:
            memCond = memCond.upper()

        # init variables #
        dobj = Debugger(SysMgr.pid, attach=False)
        dobj.initValues()

        while 1:
            if cpuRes >= 0:
                cpuStat = dobj.getCpuUsage(system=True)
                if cpuStat and cpuStat[3]:
                    cpuUsage = 100 - (cpuStat[3] / SysMgr.uptimeDiff)
                    SysMgr.cpuUsage = cpuUsage
                    if cpuCond == "MORE" and cpuUsage >= cpuRes:
                        return True
                    elif cpuCond == "LESS" and cpuUsage <= cpuRes:
                        return True

            if memRes >= 0:
                memAvail = SysMgr.getAvailMemInfo(retstr=False)
                if memAvail:
                    SysMgr.memAvail = memAvail = memAvail >> 20
                    if memCond == "MORE" and memAvail >= memRes:
                        return True
                    elif memCond == "LESS" and memAvail <= memRes:
                        return True

            time.sleep(1)
            SysMgr.updateUptime()

    @staticmethod
    def loadLibcObj(exit=False):
        if not SysMgr.isLinux:
            return False
        elif SysMgr.libcObj:
            return True
        elif SysMgr.libcObj is False and not exit:
            return False

        try:
            ret = SysMgr.loadLib(SysMgr.libcPath)
            if ret:
                SysMgr.libcObj = ret
                return True
            else:
                SysMgr.libcObj = False
                if exit:
                    SysMgr.printErr("failed to load libc")
                    sys.exit(-1)
                else:
                    SysMgr.printWarn("failed to load libc", True)
                    return False
        except SystemExit:
            sys.exit(0)
        except:
            return False

    @staticmethod
    def getCpuUsageStr(obj, diff, color=True, proc=False):
        # get CPU usage for me #
        if diff >= 1:
            cpuUsage = obj.getCpuUsage(system=True, proc=proc)
        else:
            cpuUsage = [0, 0, 0, 100]

        ttime = cpuUsage[0] / diff
        utime = cpuUsage[1] / diff
        stime = cpuUsage[2] / diff

        # get CPU usage for myself #
        mcpu = "%d%%" % ttime
        if color:
            mcpu = UtilMgr.convCpuColor(ttime, mcpu)
        mcpuStr = "%s(U%d%%+S%d%%)" % (mcpu, utime, stime)

        # get CPU usage for system #
        ctime = 100 - (cpuUsage[3] / diff)
        ctime = ctime if ctime > 0 else 0
        sysCpuStr = "%d%%" % ctime
        if color:
            sysCpuStr = UtilMgr.convCpuColor(ctime, sysCpuStr)

        return mcpuStr, sysCpuStr

    @staticmethod
    def shrinkHeap():
        if not SysMgr.isLinux:
            return

        # load libc #
        if not SysMgr.loadLibcObj():
            return

        # check malloc_trim #
        if not hasattr(SysMgr.libcObj, "malloc_trim"):
            SysMgr.printWarn("no malloc_trim in %s" % SysMgr.libcPath)
            return

        # check env #
        if "NOSHRINK" in SysMgr.environList:
            return

        # int malloc_trim (size_t pad) #
        SysMgr.libcObj.malloc_trim(0)

    @staticmethod
    def getTimeValues(times, default):
        retList = []
        for item in times:
            if not item in SysMgr.environList:
                retList.append(0)
                continue

            if (
                SysMgr.environList[item][0] == "SET"
                or not SysMgr.environList[item][0]
            ):
                retList.append(default)
                continue

            try:
                val = SysMgr.environList[item][0]
                ret = UtilMgr.convStr2Time(val, isSec=True, verb=False)
                if ret:
                    retList.append(ret)
                else:
                    retList.append(float(val))
            except:
                SysMgr.printErr(
                    "failed to convert %s to float for %s" % (val, item), True
                )
                sys.exit(-1)

        # return value #
        if len(retList) == 1:
            return retList[0]
        else:
            return retList

    @staticmethod
    def doLogMode(mode):
        umode = mode.upper()
        if umode == "KMSG":
            func = LogMgr.doLogKmsg
            mtype = "kernel"
        elif umode == "TRACE":
            func = LogMgr.doLogTrace
            mtype = "trace"
        elif umode == "DLT":
            func = DltAnalyzer.doLogDlt
            mtype = "DLT"
        elif umode == "ANDROID":
            func = LogMgr.doLogAndroid
            mtype = "android"
        elif umode == "JOURNAL":
            func = LogMgr.doLogJournal
            mtype = "journal"
        elif umode == "SYSLOG":
            func = LogMgr.doLogSyslog
            mtype = "syslog"

        SysMgr.printLogo(big=True, onlyFile=True)

        # get message #
        if SysMgr.hasMainArg():
            msg = SysMgr.getMainArg()
        elif SysMgr.inputParam:
            msg = SysMgr.inputParam
        else:
            SysMgr.printErr("no input message for %s" % mtype)
            sys.exit(-1)

        # get log level #
        if not SysMgr.customCmd:
            level = None
        else:
            level = SysMgr.customCmd[0]

        # get repeat count #
        repeatCnt = UtilMgr.getEnvironNum("REPEAT", False, 1, False, True)

        # get interval #
        if SysMgr.intervalEnable:
            interval = SysMgr.intervalEnable / 1000.0
        else:
            interval = 0

        for idx in xrange(repeatCnt):
            ret = func(msg=msg, level=level)
            if ret == 0 or (func == LogMgr.doLogAndroid and ret > 0):
                SysMgr.printInfo(
                    "logged %s message '%s' successfully" % (mtype, msg)
                )
                time.sleep(interval)
            else:
                SysMgr.printErr("failed to log %s message" % mtype)
                break

    @staticmethod
    def getPidList():
        try:
            if SysMgr.isLinux:
                pids = os.listdir(SysMgr.procPath)
            else:
                pids = list(map(str, SysMgr.getPkg("psutil").pids()))
            return pids
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printOpenErr(SysMgr.procPath)
            return []

    @staticmethod
    def checkTimeout(func, timeout=1):
        errStr = "failed to call the function for timeout"

        # create a new process #
        pid = SysMgr.createProcess(keepPrintFd=True)
        if pid > 0:
            try:
                # change SIGALRM handler #
                handle = signal.getsignal(signal.SIGALRM)
                signal.signal(signal.SIGALRM, SysMgr.exceptHandler)
                signal.alarm(timeout)

                # wait for child #
                try:
                    os.waitpid(pid, 0)
                    signal.alarm(0)
                    return True
                except SystemExit:
                    sys.exit(0)
                except:
                    SysMgr.killChildren(children=[pid])
            except SystemExit:
                sys.exit(0)
            except:
                SysMgr.printErr(errStr, True)

            # clear alarm #
            signal.alarm(0)

            # recover SIGALRM handler #
            if handle:
                signal.signal(signal.SIGALRM, handle)
        elif pid == 0:
            try:
                # run target function #
                func()
            except SystemExit:
                pass
            except:
                SysMgr.printErr(errStr, True)
            finally:
                os._exit(0)
        else:
            return False

    @staticmethod
    def killProcGroup(pid, sig=None):
        try:
            if not sig:
                sig = signal.SIGINT

            pgrp = os.getpgid(pid)
            os.killpg(pgrp, signal.SIGINT)
        except SystemExit:
            sys.exit(0)
        except:
            pass

    @staticmethod
    def getNrSysFdHandle():
        try:
            with open("%s/sys/fs/file-nr" % SysMgr.procPath) as fd:
                stats = list(map(long, fd.readline()[:-1].split()))
                """
                0: the total allocated file handles
                1: the number of currently used file handles (kernel 2.4)
                2: the maximum file handles that can be allocated
                """
                return stats
        except SystemExit:
            sys.exit(0)
        except:
            return None

    @staticmethod
    def setMaxFd():
        if not SysMgr.isLinux:
            return

        # define resource and value #
        rtype = "RLIMIT_NOFILE"
        slim = 1048576
        hlim = 1048576

        # try to change maximum open file #
        try:
            ret = SysMgr.chRlimit(os.getpid(), rtype, slim, hlim)
            if not ret:
                raise Exception("N/A")

            # update the number of maximum open file #
            SysMgr.maxFd = ret[0]
            SysMgr.maxKeepFd = SysMgr.maxFd - 16
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printWarn(
                "failed to get the number of maximum file descriptor",
                reason=True,
            )

    @staticmethod
    def setReportAttr():
        # get argument #
        if SysMgr.hasMainArg():
            SysMgr.inputFile = SysMgr.getMainArg()
        elif SysMgr.inputParam:
            SysMgr.inputFile = SysMgr.inputParam
        else:
            SysMgr.inputFile = "guider.dat"

        # set default output path #
        if not SysMgr.outPath:
            SysMgr.outPath = "%s.out" % os.path.splitext(SysMgr.inputFile)[0]

    @staticmethod
    def execSystemView():
        # parse all options and make output file path #
        SysMgr.parseAnalOption()

        SysMgr.printStat(r"start recording... [ STOP(Ctrl+c), MARK(Ctrl+\) ]")

        # wait for interval #
        if SysMgr.repeatInterval > 0:
            try:
                time.sleep(SysMgr.repeatInterval)
            except:
                pass
        # wait for user input #
        else:
            SysMgr.waitEvent()

        # save system info #
        SysMgr.saveSysStats()

        # get and remove process tree from data file #
        SysMgr.getProcTreeInfo()

        SysMgr.printLogo(big=True)

        # print system information #
        SysMgr.printInfoBuffer()

    @staticmethod
    def getVmstat(retDict=False):
        # read vmstat buf #
        vmBuf = SysMgr.readProcStat(
            SysMgr.vmstatFd, "vmstat", SysMgr, "vmstatFd"
        )
        if not vmBuf:
            return vmBuf

        # convert list to dictionary #
        if retDict:
            newDict = {}

            for line in vmBuf:
                try:
                    item = line.strip().split()
                    newDict.setdefault(item[0], long(item[1]))
                except SystemExit:
                    sys.exit(0)
                except:
                    pass

            return newDict

        return vmBuf

    @staticmethod
    def setRecordAttr():
        # iorec #
        if SysMgr.checkMode("iorec"):
            SysMgr.blockEnable = True
            SysMgr.cpuEnable = False

        # function #
        elif SysMgr.checkMode("funcrec"):
            SysMgr.functionEnable = True

        # syscall #
        elif SysMgr.checkMode("sysrec"):
            SysMgr.sysEnable = True
            SysMgr.cpuEnable = False

        # file #
        elif SysMgr.checkMode("filerec"):
            SysMgr.fileEnable = True

        # general #
        elif SysMgr.checkMode("genrec"):
            SysMgr.systemEnable = True

        # update record status #
        SysMgr.inputFile = "%s/../trace" % SysMgr.getTraceEventPath()

        SysMgr.parseRecordOption()

        # change the CPU scheduling priority for process #
        if SysMgr.prio is None:
            SysMgr.setPriority(SysMgr.pid, "C", -20)

        SysMgr.printProfileOption()
        SysMgr.printProfileCmd()

    @staticmethod
    def removeFd(fname):
        SysMgr.fdCache.pop(fname, None)

    @staticmethod
    def getProcMaps(pid):
        mpath = "%s/%s/maps" % (SysMgr.procPath, pid)
        return SysMgr.readFile(mpath)

    @staticmethod
    def saveProcMapsFile(pid, path=None):
        # convert PID and check root permission #
        if pid == "self":
            pid = SysMgr.pid
        else:
            SysMgr.checkRootPerm()

        procInfo = "%s(%s)" % (SysMgr.getComm(pid), pid)

        # set temporary path name #
        if not path:
            path = "%s/%s.maps" % (SysMgr.tmpPath, procInfo)

        try:
            maps = SysMgr.getProcMaps(pid)
            if not maps:
                return False

            ret = SysMgr.writeFile(path, maps + "\n", False)
            if not ret:
                return False

            SysMgr.printInfo(
                "saved maps of %s to '%s'%s successfully"
                % (procInfo, path, UtilMgr.getFileSizeStr(path))
            )
            return True
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printErr("failed to save maps of %s" % procInfo, True)
            return False

    @staticmethod
    def setPipeSize(fd, size=0):
        def _printErr(msg):
            SysMgr.printWarn("failed to change pipe size because " + msg)

        fcntl = SysMgr.getPkg("fcntl", False)
        if not fcntl:
            _printErr("no fcntl package")
            return

        # set size to max #
        if not size:
            size = SysMgr.getMaxPipeSize()

        # define constant value #
        F_SETPIPE_SZ = 1024 + 7
        F_GETPIPE_SZ = 1024 + 8

        # change pipe size #
        try:
            ret = fcntl.fcntl(fd, F_SETPIPE_SZ, size)
        except SystemExit:
            sys.exit(0)
        except:
            _printErr(SysMgr.getErrMsg())

    @staticmethod
    def getMaxPipeSize():
        if not SysMgr.isLinux:
            return

        # return max pipe size #
        try:
            return long(SysMgr.readFile("/proc/sys/fs/pipe-max-size"))
        except:
            return None

    @staticmethod
    def removeFiles(flist):
        for f in flist:
            try:
                if not os.path.exists(f):
                    continue

                os.remove(f)
                SysMgr.printInfo("removed '%s'" % f)
            except SystemExit:
                sys.exit(0)
            except:
                SysMgr.printErr("failed to remove '%s'" % f, True)

    @staticmethod
    def getCompFd(fname, perm="r", fd=None, verb=True):
        try:
            # TODO: support zip, tar #
            # gz + tar.gz #
            if UtilMgr.isGzipFile(fname, fd):
                gzip = SysMgr.getPkg("gzip", False)
                fd = (
                    gzip.GzipFile(fileobj=fd) if fd else gzip.open(fname, perm)
                )
            # xz (lzma) #
            elif UtilMgr.isXzFile(fname, fd):
                param = fd if fd else fname
                fd = SysMgr.getPkg("lzma", False).open(param, perm)
            # raw data #
            else:
                fd = None
        except SystemExit:
            sys.exit(0)
        except:
            if verb:
                SysMgr.printWarn("failed to open '%s'" % fname, True, True)
            fd = None
        return fd

    @staticmethod
    def getCachedFd(fname, perm="rb", verb=True):
        if fname in SysMgr.fdCache and SysMgr.fdCache[fname]["perm"] == perm:
            return SysMgr.fdCache[fname]["fd"]

        try:
            SysMgr.fdCache[fname] = {"fd": open(fname, perm), "perm": perm}
            return SysMgr.fdCache[fname]["fd"]
        except SystemExit:
            sys.exit(0)
        except:
            if verb:
                SysMgr.printOpenErr(fname)
            return None

    @staticmethod
    def setVisualAttr():
        # set default input path #
        if len(sys.argv) <= 2:
            sys.argv.insert(2, SysMgr.outFilePath)

        SysMgr.graphEnable = True

        # ignore user warning #
        SysMgr.ignoreWarn()

        # apply regex for first path #
        flist = UtilMgr.convPath(sys.argv[2])
        flist = UtilMgr.getFileList(flist, exceptDir=True)
        if isinstance(flist, list) and len(flist) > 0:
            sys.argv = sys.argv[:2] + flist + sys.argv[3:]

        # thread draw mode #
        if TaskAnalyzer.getInitTime(sys.argv[2]) > 0:
            # apply launch option #
            SysMgr.applyLaunchOption()

            # check data type #
            if SysMgr.isFuncMode():
                SysMgr.printErr(
                    "failed to draw because this data is for function"
                )
                sys.exit(-1)
            elif (
                SysMgr.isThreadMode()
                or not SysMgr.sysinfoBuffer
                or SysMgr.forceEnable
            ):
                pass
            else:
                SysMgr.printErr(
                    "failed to draw because this data is not supported"
                )
                sys.exit(-1)

            # set input path #
            SysMgr.inputFile = sys.argv[1] = sys.argv[2]
            SysMgr.intervalEnable = 1
            if not SysMgr.outPath:
                SysMgr.outPath = (
                    "%s.out" % os.path.splitext(SysMgr.inputFile)[0]
                )
            del sys.argv[2]
        # top draw mode #
        else:
            # CPU #
            if SysMgr.checkMode("drawcpu") or SysMgr.checkMode("drawcpuavg"):
                SysMgr.layout = "CPU"
            # delay #
            elif SysMgr.checkMode("drawdelay"):
                SysMgr.layout = "DELAY"
            # memory #
            elif SysMgr.checkMode("drawmem") or SysMgr.checkMode("drawmemavg"):
                SysMgr.layout = "MEM"
            # VSS #
            elif SysMgr.checkMode("drawvss") or SysMgr.checkMode("drawvssavg"):
                SysMgr.layout = "MEM"
                SysMgr.vssEnable = True
            # RSS #
            elif (
                SysMgr.checkMode("drawrss")
                or SysMgr.checkMode("drawrssavg")
                or "DRAWRSS" in SysMgr.environList
            ):
                SysMgr.layout = "MEM"
                SysMgr.rssEnable = True
            # leak #
            elif SysMgr.checkMode("drawleak"):
                SysMgr.layout = "MEM"
                SysMgr.leakEnable = True
            # I/O #
            elif SysMgr.checkMode("drawio"):
                SysMgr.layout = "IO"
            # prio #
            elif SysMgr.checkMode("drawpri"):
                SysMgr.layout = "PRIO"

            # modify args for drawing multiple input files #
            sys.argv[1] = "top"
            args = sys.argv[2:]
            SysMgr.inputParam = UtilMgr.getFileList(args)

    @staticmethod
    def execFileAnalysis():
        SysMgr.checkRootPerm()

        # parse analysis option #
        SysMgr.parseAnalOption()

        SysMgr.printStat(r"start analyzing... [ STOP(Ctrl+c), MARK(Ctrl+\) ]")

        # start analyzing files #
        try:
            pi = FileAnalyzer()
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printErr(
                "failed to analyze memory-mapped files", reason=True
            )
            sys.exit(-1)

        # save system info #
        SysMgr.saveSysStats()

        # get and remove process tree from data file #
        SysMgr.getProcTreeInfo()

        if SysMgr.intervalEnable == 0:
            # print total file usage per process #
            pi.printUsage()
        else:
            # print file usage per process on timeline #
            pi.printIntervalInfo()

    @staticmethod
    def execRecordLoop():
        skipWait = False

        while SysMgr.repeatInterval > 0:
            # set alarm #
            signal.alarm(SysMgr.repeatInterval)

            # get init time in buffer for verification #
            if SysMgr.graphEnable:
                skipWait = True
                initTime = 0
            else:
                initTime = TaskAnalyzer.getInitTime(SysMgr.inputFile)

            # wait for timer #
            try:
                for cnt in xrange(SysMgr.repeatInterval):
                    UtilMgr.printProgress(cnt, SysMgr.repeatInterval)
                    time.sleep(1)
            except SystemExit:
                sys.exit(0)
            except:
                pass

            # real-time copy from pipe to file #
            if SysMgr.pipeEnable:
                if SysMgr.outputFile:
                    # backup #
                    SysMgr.backupFile(SysMgr.outputFile)

                    # copy #
                    SysMgr.copyPipeToFile(
                        "%s%s" % (SysMgr.inputFile, "_pipe"), SysMgr.outputFile
                    )
                else:
                    SysMgr.printErr(
                        "wrong option used, use also -s option to save data"
                    )
                sys.exit(-1)

            # check counter #
            if SysMgr.repeatCnt <= SysMgr.progressCnt and SysMgr.termFlag:
                # check exit condition #
                if SysMgr.graphEnable:
                    break

                SysMgr.printWarn("terminated by timer\n", True)
                sys.exit(-1)

            # compare init time with now time for buffer verification #
            if 0 < initTime < TaskAnalyzer.getInitTime(SysMgr.inputFile):
                SysMgr.printErr(
                    "buffer size is not enough (%sKB)" % SysMgr.getBufferSize()
                )
                sys.exit(-1)
            else:
                SysMgr.clearTraceBuffer()

        # start writing logs to file through pipe #
        if SysMgr.pipeEnable:
            if SysMgr.outputFile:
                pipePath = "%s%s" % (SysMgr.inputFile, "_pipe")
                SysMgr.copyPipeToFile(pipePath, SysMgr.outputFile)
            else:
                SysMgr.printErr(
                    "wrong option used, use also -s option to save data"
                )
            sys.exit(-1)

        if not SysMgr.graphEnable:
            # get init time from buffer for verification #
            initTime = TaskAnalyzer.getInitTime(SysMgr.inputFile)

        # wait for user input #
        while 1:
            if skipWait:
                break
            elif SysMgr.recordStatus:
                SysMgr.condExit = True

                SysMgr.waitEvent()
                if SysMgr.condExit:
                    break
            else:
                break

        # FUNCTION GRAPH MODE #
        if SysMgr.graphEnable:
            if not SysMgr.outputFile:
                SysMgr.stopRecording()

                # read trace data #
                data = SysMgr.readFile(SysMgr.inputFile)

                # parse options #
                SysMgr.parseAnalOption()

                # apply core filter #
                if SysMgr.perCoreList:
                    data = data.split("\n")
                    data = SysMgr.applyCoreFilter(data)
                    data = "\n".join(data)

                # print trace data #
                SysMgr.printPipe(data)

                sys.exit(0)
            elif SysMgr.repeatInterval:
                sys.exit(0)
        # THREAD & FUNCTION MODE #
        else:
            # compare init time with now time for buffer verification #
            if initTime < TaskAnalyzer.getInitTime(SysMgr.inputFile):
                SysMgr.printErr(
                    "buffer size %sKB is not enough" % SysMgr.getBufferSize()
                )
                sys.exit(-1)

            # save system info #
            SysMgr.saveSysStats()

    @staticmethod
    def dumpFileData():
        if (
            not "DUMPFILE" in SysMgr.environList
            and not "DUMPFILEB64" in SysMgr.environList
        ):
            return False

        # get size #
        commonSize = UtilMgr.getEnvironNum(
            "DUMPFILESIZE", False, 0, False, True
        )

        def _dumpFile(path, size=0, base64=True):
            path = os.path.realpath(path)

            SysMgr.printInfo("dump '%s'" % path)

            if os.path.isdir(path):
                fdata = ",".join(os.listdir(path))
                fsize = ""
            else:
                size = commonSize

                # update size #
                if size < 0:
                    tail = True
                    size = abs(size)
                else:
                    tail = False
                    if size == 0:
                        size = -1

                fdata = SysMgr.readFile(path, size, tail=tail, byte=True)
                fsize = UtilMgr.getFileSizeStr(path)

            if not fdata:
                return

            # print file info #
            SysMgr.printPipe(
                "[ File Dump ] (%s%s)\n%s\n" % (path, fsize, twoLine)
            )

            # write file data base64-encoded #
            if base64:
                fdata = UtilMgr.encodeBase64(fdata)
            SysMgr.printFd.write(fdata)

            SysMgr.printPipe("\n%s\n" % oneLine)

        # save file data for this interval #
        printCnt = 0
        for name in ["DUMPFILE", "DUMPFILEB64"]:
            if not name in SysMgr.environList:
                continue

            base64 = "DUMPFILEB64" in SysMgr.environList

            for path in SysMgr.environList[name]:
                # parse path snd size #
                items = path.split("#", 1)
                path = items[0]
                try:
                    size = long(items[1]) if len(items) > 1 else commonSize
                except SystemExit:
                    sys.exit(0)
                except:
                    SysMgr.printErr(
                        "failed to parse size info of '%s' for dump" % path
                    )
                    size = commonSize

                # dump file #
                _dumpFile(path, size, base64)

                printCnt += 1

        if printCnt > 0:
            return True
        else:
            return False

    @staticmethod
    def getTrimVals(timeline):
        imin = 0
        imax = len(timeline)

        if "TRIM" in SysMgr.environList:
            trim = SysMgr.environList["TRIM"][0].split(":")
        elif "TRIMIDX" in SysMgr.environList:
            trim = SysMgr.environList["TRIMIDX"][0].split(":")
        else:
            return timeline

        try:
            if len(trim) == 1:
                condMin = long(trim[0])
                condMax = SysMgr.maxSize
            elif len(trim) >= 2:
                # first tick #
                if trim[0].strip():
                    condMin = long(trim[0])
                else:
                    condMin = 0

                # last tick #
                if trim[1].strip():
                    condMax = long(trim[1])
                else:
                    condMax = long(SysMgr.maxSize)
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printErr(
                "failed to recognize '%s' as the START:END time"
                % ":".join(trim)
            )
            sys.exit(-1)

        if "TRIM" in SysMgr.environList:
            # define default values #
            imin = timeline[0]
            imax = timeline[-1]

            # get min index #
            for itime in timeline:
                if itime >= condMin:
                    imin = itime
                    break

            # get max index #
            for itime in timeline:
                if itime >= condMax:
                    imax = itime
                    break

            # convert index range #
            imin = timeline.index(imin)
            imax = timeline.index(imax)
        elif "TRIMIDX" in SysMgr.environList:
            imin = condMin
            imax = condMax

        return timeline[imin:imax]

    @staticmethod
    def applyCoreFilter(lines):
        newList = []
        filterList = set(map(str, SysMgr.perCoreList))

        for line in lines:
            m = re.match(r"^\s*(?P<time>.+) \|\s*(?P<core>[0-9]+)\)", line)
            if not m:
                newList.append(line)
                continue

            # check core number #
            d = m.groupdict()
            if d["core"] in filterList:
                newList.append(line)

        return newList

    @staticmethod
    def getGpuMem(incAll=False):
        if SysMgr.gpuMemGetters is None:
            SysMgr.gpuMemGetters = [
                SysMgr.getNvGpuMem,  # nvidia #
                SysMgr.getMaliGpuMem,  # mali #
            ]

        totalStat = {}
        memStat = {}
        removeList = []
        getters = SysMgr.gpuMemGetters

        # pylint: disable-next=not-an-iterable
        for func in getters:
            gpuStat, procStat = func(incAll)

            # merge total size #
            if gpuStat:
                for item in gpuStat:
                    if item in totalStat:
                        totalStat[item]["size"] += gpuStat[item]["size"]
                    else:
                        totalStat[item] = gpuStat[item]

                # make total stat #
                if not "0" in totalStat:
                    totalMem = 0
                    for stats in totalStat.values():
                        totalMem += stats["size"]
                    totalStat["0"] = {"comm": "TOTAL", "size": totalMem}
            else:
                removeList.append(func)

            # merge per-type size #
            for pid, data in procStat.items():
                memStat.setdefault(pid, {})
                for mtype, size in data.items():
                    memStat[pid].setdefault(mtype, 0)
                    memStat[pid][mtype] += size

        # set total size of each process #
        for pid, data in memStat.items():
            total = 0
            for size in data.values():
                total += size
            memStat[pid]["TOTAL"] = total

        # remove invalid getters #
        if removeList:
            SysMgr.gpuMemGetters = list(set(getters) - set(removeList))

        return totalStat, memStat

    @staticmethod
    def getMaliGpuMem(incAll=False):
        gpuStat = {}
        procStat = {}

        # read Mali GPU memory info #
        try:
            path = "/sys/kernel/debug/mali/gpu_memory"
            if not os.path.exists(path):
                return gpuStat, procStat

            with open(path, "rb") as fd:
                gpuInfo = fd.readlines()[2:]
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printOpenWarn(path)
            return gpuStat, procStat

        # parse per-process memory info #
        for item in gpuInfo:
            try:
                line = item.decode().split()

                if line[0] == "Mali":
                    if line[2] == "usage":
                        pid = "0"
                        comm = "TOTAL"
                    else:
                        continue
                else:
                    comm, pid, mali, malimax, extern, ump, dma = line
            except SystemExit:
                sys.exit(0)
            except:
                continue

            # save stat #
            gpuStat.setdefault(pid, {"comm": comm, "size": long(mali)})

            # save stat to per-process list #
            if incAll:
                procStat.setdefault(pid, {})
                procStat[pid]["mali"] = long(mali)
                procStat[pid]["extern"] = long(extern)
                procStat[pid]["ump"] = long(ump)
                procStat[pid]["dma"] = long(dma)

        return gpuStat, procStat

    @staticmethod
    def getOatHeader(path):
        header = {}

        try:
            if not UtilMgr.isOatFile(path):
                raise ValueError("not oat image")

            # OAT header starts from oatdata symbol adderss (0x1000) #
            data = SysMgr.readFile(
                path, size=80, verb=False, byte=True, offset=0x1000
            )

            # convert values #
            v = struct.unpack("20I", data)
            header = {
                "magic": struct.pack("I", v[0]),
                "version": struct.pack("I", v[1]),
                "checksum": v[2],
                "inst": v[3],
                "instFeat": v[4],
                "dexCnt": v[5],
                "dexOffset": v[6],
                "execOffset": v[7],
                "intp2intpOffset": v[8],
                "compiledOffset": v[9],
                "jniDlsymOffset": v[10],
                "jniTrampOffset": v[11],
                "imtTrampOffset": v[12],
                "resTrampOffset": v[13],
                "compiled2intpOffset": v[14],
                "imgPatchDeltaOffset": v[15],
                "oatFileLocChecksum": v[16],
                "oatDataOffset": v[17],
                "keyValSize": v[18],
                "keyValOffset": v[19],
            }
        except ValueError:
            pass
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printErr("failed to get OAT header from '%s'" % path, True)
        finally:
            return header

    @staticmethod
    def symbolizeOat(path, debug=False, printer=None):
        try:
            if not UtilMgr.isOatFile(path):
                raise ValueError("not oat image")

            # make symbolized file path #
            symPath = os.path.join(
                SysMgr.cacheDirPath, path.replace("/", "_") + ".symbolized"
            )

            # print header #
            if debug:
                header = SysMgr.getOatHeader(path)
                printer("[OAT Header]\n%s" % twoLine)
                for n, v in header.items():
                    if isinstance(v, long):
                        v = "%s (%s)" % (str(v), hex(v))
                    printer("{0:1}: {1:<1}".format(n, str(v)))
                if not header:
                    printer("\tNone")
                printer("%s\n" % oneLine)

            # check skip condition #
            if not "NOELFCACHE" in SysMgr.environList and os.path.exists(
                symPath
            ):
                # update path #
                path = symPath

                raise ValueError("already symbolized")

            SysMgr.printInfo(
                "symbolize OAT image for %s%s"
                % (path, UtilMgr.getFileSizeStr(path))
            )

            # check oatdump binary #
            if "OATDUMP" in SysMgr.environList:
                oatdump = SysMgr.environList["OATDUMP"][0]
            elif not UtilMgr.which("oatdump"):
                SysMgr.printErr("no found oatdump")
                raise ValueError("no oatdump")
            else:
                oatdump = "oatdump"

            # execute oatdump #
            cmd = [
                oatdump,
                "--symbolize=" + path,
                "--output=" + symPath,
                "--only-keep-debug",
            ]
            ret = SysMgr.executeCmdSync(cmd)

            # check output #
            if not os.path.exists(symPath):
                raise Exception("N/A")

            # update path #
            path = symPath
        except ValueError:
            pass
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printErr("failed to symbolize '%s'" % path, True)
        finally:
            return path

    @staticmethod
    def mkdirs(path):
        try:
            if not os.path.isdir(path):
                os.makedirs(path)
            return True
        except SystemExit:
            sys.exit(0)
        except:
            err = SysMgr.getErrMsg()
            SysMgr.printWarn(
                "failed to make %s directory because %s" % (path, err),
                True,
            )
            return False

    @staticmethod
    def getNvGpuMem(incAll=False):
        gpuStat = {}
        procStat = {}
        commonPath = "/sys/kernel/debug/nvmap/"

        # check dir #
        if not os.path.exists(commonPath):
            return gpuStat, procStat

        # find all clients files #
        if incAll or "GPUALLMEM" in SysMgr.environList:
            targetList = UtilMgr.getFiles(commonPath, ["clients"])
        else:
            targetList = ["/sys/kernel/debug/nvmap/iovmm/clients"]

        for path in targetList:
            # read NVIDIA GPU memory info #
            try:
                with open(path, "rb") as fd:
                    gpuInfo = fd.readlines()[1:]
            except SystemExit:
                sys.exit(0)
            except:
                SysMgr.printOpenWarn(path)
                continue

            # get memory type #
            mtype = UtilMgr.lstrip(path, commonPath).rsplit("/", 1)[0]

            # parse per-process memory info #
            for item in gpuInfo:
                try:
                    line = item.decode().split()
                    if line[0] == "total":
                        pid = "0"
                        comm = "TOTAL"
                        size = line[1]
                    else:
                        comm, pid, size = line[1:]
                except SystemExit:
                    sys.exit(0)
                except:
                    continue

                # convert size to number #
                size = UtilMgr.convUnit2Size(size)

                # save stat #
                gpuStat.setdefault(pid, {"comm": comm, "size": 0})
                gpuStat[pid]["size"] += size

                # save stat to per-process list #
                if incAll:
                    procStat.setdefault(pid, {})
                    procStat[pid][mtype] = size

        return gpuStat, procStat

    @staticmethod
    def execSshCmd(cmd, addr, port=22, user="root", passwd=None):
        # get subprocess object #
        subprocess = SysMgr.getPkg("subprocess")

        # get environment variables #
        env = SysMgr.getEnvList()

        cmdList = []

        if passwd:
            # get sshpass path #
            sshpassPath = UtilMgr.which("sshpass")
            if not sshpassPath:
                SysMgr.printErr("failed to get sshpass path")
                sys.exit(-1)
            sshpassPath = sshpassPath[0]

            # add sshpass command #
            cmdList += [sshpassPath, "-p", passwd]

        # get ssh path #
        sshPath = UtilMgr.which("ssh")
        if not sshPath:
            SysMgr.printErr("failed to get ssh path")
            sys.exit(-1)
        sshPath = sshPath[0]

        # set option #
        opt = "-oStrictHostKeyChecking=no"

        # add ssh command #
        cmdList += [sshPath, opt, "-p", str(port), "%s@%s" % (user, addr), cmd]

        # save start time #
        startTime = time.time()

        # create a new worker process #
        procObj = subprocess.Popen(
            cmdList,
            stdout=subprocess.PIPE,
            stdin=subprocess.PIPE,
            stderr=subprocess.PIPE,
            universal_newlines=True,
            bufsize=0,
            env=env,
        )

        # handle output #
        try:
            select = SysMgr.getPkg("select")
            while 1:
                res = select.select(
                    [procObj.stdout, procObj.stderr], [], [], 1
                )
                if not res[0]:
                    continue
                for item in res[0]:
                    val = item.read()
                    if val:
                        print(val)
                    else:
                        raise Exception()
        except SystemExit:
            sys.exit(0)
        except:
            pass
        finally:
            duration = time.time() - startTime

        # kill subprocess group #
        SysMgr.killProcGroup(procObj.pid)

        SysMgr.printInfo("elapsed %.6f for '%s'" % (duration, cmd))

        sys.exit(0)

    @staticmethod
    def execTopCmd():
        # check background processes #
        if not "FASTINIT" in SysMgr.environList and not SysMgr.isDrawMode():
            SysMgr.checkBgProcs()

        # write user command #
        SysMgr.runProfCmd("BEFORE")

        # set signal handler #
        if SysMgr.isDrawMode():
            SysMgr.setDefaultSignal()
        # print profile option #
        else:
            SysMgr.printProfileOption()
            SysMgr.printProfileCmd()

            # set handler for exit #
            SysMgr.setNormalSignal()

        # thread #
        if SysMgr.checkMode("ttop"):
            SysMgr.processEnable = False

            # check GPU option #
            if not (
                "GPUMEM" in SysMgr.environList
                or "GPUMEMSUM" in SysMgr.environList
            ):
                SysMgr.gpuMemEnable = False
        # file #
        elif SysMgr.checkMode("ftop"):
            SysMgr.fileTopEnable = True

        # stack #
        elif SysMgr.checkMode("stacktop"):
            if SysMgr.checkStackTopCond():
                SysMgr.processEnable = False
                SysMgr.stackEnable = True
            else:
                sys.exit(-1)

        # perf #
        elif SysMgr.checkMode("ptop"):
            if SysMgr.checkPerfTopCond():
                SysMgr.perfEnable = True
                if SysMgr.findOption("g"):
                    SysMgr.processEnable = False
                    SysMgr.perfGroupEnable = True
            else:
                sys.exit(-1)

        # mem #
        elif SysMgr.checkMode("mtop"):
            if SysMgr.checkMemTopCond():
                SysMgr.memEnable = True
                SysMgr.sort = "m"
            else:
                sys.exit(-1)

        # WSS (Working Set Size) #
        elif SysMgr.checkMode("wtop"):
            if SysMgr.checkWssTopCond():
                if not "SYSMEM" in SysMgr.environList:
                    SysMgr.addEnvironVar("NOSYSMEM")
                SysMgr.memEnable = True
                SysMgr.wssEnable = True
                SysMgr.sort = "m"
            else:
                sys.exit(-1)

        # disk #
        elif SysMgr.checkMode("disktop"):
            if SysMgr.checkDiskTopCond():
                SysMgr.diskEnable = True
                SysMgr.blockEnable = True
                SysMgr.sort = "b"
            else:
                sys.exit(-1)

        # all #
        elif SysMgr.checkMode("atop"):
            SysMgr.cpuEnable = True
            SysMgr.diskEnable = True
            SysMgr.irqEnable = True
            SysMgr.memEnable = True
            SysMgr.networkEnable = True

            if SysMgr.isRoot():
                SysMgr.blockEnable = True
                SysMgr.perfEnable = True
            else:
                SysMgr.printWarn(
                    "block stat is disabled "
                    "because there is no root permission"
                )

        # tree #
        elif SysMgr.checkMode("trtop"):
            # get target #
            if SysMgr.hasMainArg():
                targets = SysMgr.getMainArgs()
            else:
                targets = None

            # set default interval #
            if not SysMgr.intervalEnable:
                SysMgr.intervalEnable = 1

            # init dbg object #
            TaskAnalyzer.dbgObj = Debugger(SysMgr.pid, attach=False)
            TaskAnalyzer.dbgObj.initValues()
            TaskAnalyzer.dbgObj.getCpuUsage(system=True)

            # add environment variables #
            SysMgr.addEnvironVar("CURCPU")
            if not "NOMEM" in SysMgr.environList:
                SysMgr.addEnvironVar("ADDMEM")

            SysMgr.doPstree(targets=targets, canRepeat=True)

        # condition #
        elif SysMgr.checkMode("ctop"):
            # check config file #
            if not SysMgr.getOption("C"):
                # search config file #
                SysMgr.updateConfigPath()

            # load config file #
            if ConfigMgr.confData:
                pass
            elif not SysMgr.loadConfig(SysMgr.confFileName):
                sys.exit(-1)

            # ignore output #
            if not SysMgr.outPath:
                if SysMgr.isWindows:
                    SysMgr.outPath = "NUL"
                else:
                    SysMgr.outPath = SysMgr.nullPath

            # get output dir #
            if SysMgr.outPath == SysMgr.nullPath:
                if os.path.exists(SysMgr.tmpPath):
                    targetDir = SysMgr.tmpPath
                else:
                    targetDir = "."
            elif os.path.isdir(SysMgr.outPath):
                targetDir = SysMgr.outPath
            else:
                targetDir = os.path.dirname(SysMgr.outPath)

            # check output dir #
            if not SysMgr.isWritable(targetDir):
                SysMgr.printErr("wrong path '%s' for output" % targetDir)
                sys.exit(-1)

            # remove SIGINT handler to ignore creating the report file #
            signal.signal(signal.SIGINT, SysMgr.exitHandler)

            # update run number #
            if not SysMgr.nrRun:
                try:
                    SysMgr.updateNrRun(targetDir)
                except SystemExit:
                    sys.exit(0)
                except:
                    SysMgr.printErr(
                        "failed to update sequence number of execution", True
                    )

            # set buffer strip #
            SysMgr.bufferLossEnable = True

            # set threshold handle flag #
            SysMgr.thresholdEnable = True

            # disable color and encoding #
            SysMgr.colorEnable = False
            SysMgr.encodeEnable = False

        # DLT #
        elif SysMgr.checkMode("dlttop"):
            SysMgr.dltTopEnable = True
            SysMgr.dltTimeEnable = True

        # D-Bus #
        elif SysMgr.checkMode("dbustop"):
            SysMgr.dbusTopEnable = True
            SysMgr.floatEnable = True

            # set default interval to 3 for performance #
            if not SysMgr.findOption("i") and not SysMgr.findOption("R"):
                SysMgr.intervalEnable = 3

        # usercall #
        elif SysMgr.checkMode("utop"):
            SysMgr.doTrace("usercall")

        # pycall #
        elif SysMgr.checkMode("pytop"):
            SysMgr.doTrace("pycall")

        # breakcall #
        elif SysMgr.checkMode("btop"):
            SysMgr.doTrace("breakcall")

        # syscall #
        elif SysMgr.checkMode("systop"):
            SysMgr.doTrace("syscall")

        # kernelcall #
        elif SysMgr.checkMode("ktop"):
            SysMgr.doTrace("kernel")

        # kernelfunctioncall #
        elif SysMgr.checkMode("kftop"):
            SysMgr.runKernelTop()

        # network #
        elif SysMgr.checkMode("ntop"):
            SysMgr.networkEnable = True

        # background #
        elif SysMgr.checkMode("bgtop"):
            # check condition #
            if not SysMgr.checkBgTopCond():
                sys.exit(-1)

            # check permission #
            if SysMgr.isRoot():
                SysMgr.blockEnable = True
            else:
                SysMgr.printWarn(
                    "block stat is disabled because "
                    "there is no root permission"
                )

            SysMgr.diskEnable = True
            SysMgr.networkEnable = True

            SysMgr.runBackgroundMode()

        # report #
        elif SysMgr.checkMode("rtop"):
            SysMgr.jsonEnable = True

            # check root permission #
            if SysMgr.isRoot():
                SysMgr.diskEnable = True
                SysMgr.networkEnable = True

                # disable resources by option #
                disableList = SysMgr.getOption("d")
                if disableList:
                    if "d" in disableList:
                        SysMgr.diskEnable = False
                    if "n" in disableList:
                        SysMgr.networkEnable = False
            else:
                SysMgr.printWarn(
                    "failed to get stats for disk and network "
                    "because there is no root permission"
                )

            # check command requirement #
            if not SysMgr.checkRepTopCond():
                sys.exit(-1)

        # init network resource data #
        if SysMgr.networkEnable and not SysMgr.sysInstance.networkInfo:
            SysMgr.sysInstance.updateNetworkInfo()

        # init disk resource data #
        if SysMgr.diskEnable and not SysMgr.sysInstance.storageData:
            SysMgr.sysInstance.updateStorageInfo()

        # run process / file monitoring #
        TaskAnalyzer()

    @staticmethod
    def loadLibCache():
        try:
            # no ld cache file #
            if not SysMgr.ldCachePath:
                return

            if not os.path.exists(SysMgr.ldCachePath):
                reason = SysMgr.ldCachePath
                SysMgr.ldCachePath = None
                raise Exception("no %s" % reason)

            libDict = {}

            # load libs from cache #
            libList = UtilMgr.convBin2Str(SysMgr.ldCachePath)
            for idx, item in enumerate(libList):
                try:
                    if libList[idx + 1].startswith("/"):
                        value = libList[idx + 1]
                        libDict.setdefault(item, [])
                        libDict[item].append(value)
                except SystemExit:
                    sys.exit(0)
                except:
                    pass

            # load libs form /usr/local/lib #
            for path in ("/usr/local/lib", "/usr/local/lib64"):
                if not os.path.exists(path):
                    continue

                for item in os.listdir(path):
                    if not ".so" in item:
                        continue

                    value = os.path.join(path, item)

                    libDict.setdefault(item, [])
                    libDict[item].append(value)

            # save lib list #
            SysMgr.libCache = libDict
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printWarn("failed to load library cache", reason=True)
            return False

    @staticmethod
    def readZramStats(dev, target=[]):
        stats = {}
        prefix = "/sys/block/" + dev + "/"
        convNum = UtilMgr.convNum
        convSize = UtilMgr.convSize2Unit
        PAGESIZE = SysMgr.PAGESIZE

        for item in os.listdir(prefix):
            if not target or item in target:
                res = SysMgr.readFile(prefix + item, verb=False)
                if not res:
                    continue

                if item == "mm_stat":
                    """
                    orig_data_size   uncompressed size of data stored in this disk.
                                     This excludes same-element-filled pages (same_pages) since
                                     no memory is allocated for them.
                                     Unit: bytes
                    compr_data_size  compressed size of data stored in this disk
                    mem_used_total   the amount of memory allocated for this disk. This
                                     includes allocator fragmentation and metadata overhead,
                                     allocated for this disk. So, allocator space efficiency
                                     can be calculated using compr_data_size and this statistic.
                                     Unit: bytes
                    mem_limit        the maximum amount of memory ZRAM can use to store
                                     the compressed data
                    mem_used_max     the maximum amount of memory zram have consumed to
                                     store the data
                    same_pages       the number of same element filled pages written to this disk.
                                     No memory is allocated for such pages.
                    pages_compacted  the number of pages freed during compaction
                    huge_pages	  the number of incompressible pages
                    """
                    res = list(map(long, res.split(None, 8)[:8]))

                    items = {
                        "orig_data_size": convSize(res[0]),
                        "compr_data_size": convSize(res[1]),
                        "compr_rate": "%d%%"
                        % ((res[1] / float(res[0])) * 100),
                        "mem_used_total": convSize(res[2]),
                        "mem_limit": convSize(res[3]),
                        "mem_used_max": convSize(res[4]),
                        "same_pages": convSize(res[5] * PAGESIZE),
                        "pages_compacted": convNum(res[6] * PAGESIZE),
                        "huge_pages": convNum(res[7] * PAGESIZE),
                    }

                    # remove zero values #
                    for name in list(items):
                        if items[name] in ("0", 0):
                            items.pop(name, None)

                    # merge stats #
                    stats.update(items)
                else:
                    stats[item] = res

        return stats

    @staticmethod
    def findLib(lib, inc=False):
        if not SysMgr.libCache:
            SysMgr.loadLibCache()

        libs = []

        for key, val in SysMgr.libCache.items():
            # convert for specific caches are only consist of absolute paths #
            if key.startswith("/"):
                val = [key]
                key = os.path.basename(key)

            # check path #
            if key == lib:
                pass
            elif not key.startswith(lib):
                continue
            elif inc and key.startswith(lib):
                pass
            elif len(key) <= len(lib):
                continue
            elif key[len(lib)] in (".", "-"):
                pass
            else:
                continue

            # return found values #
            if len(val) > 1:
                SysMgr.printWarn(
                    "multiple libraries [ %s ] exist for %s"
                    % (", ".join(val), key)
                )

            # add path to list #
            libs += val

        return libs

    @staticmethod
    def convSchedPrio(pri, fromNum=False):
        try:
            if fromNum:
                # RT / DEADLINE #
                if pri > 39:
                    if pri == 140:
                        policy = "D"
                    else:
                        policy = "R"

                    pri -= 40
                # NORMAL #
                else:
                    policy = "N"
                    pri = 19 - pri

                return policy + str(pri)
            else:
                policy = pri[0]
                prio = long(pri[1:])

                # convert priority to number(-101 ~ 40) #
                if policy in ("C", "B"):
                    prio += 20
                elif policy in ("R", "F"):
                    prio = (-1) - prio
                elif policy == "D":
                    prio = -101
                elif policy == "I":
                    prio = 40

                return prio
        except SystemExit:
            sys.exit(0)
        except:
            return None

    @staticmethod
    def convertTaskIdInput(taskList=[]):
        ret = SysMgr.selectTaskId()
        if ret:
            newList = []
            for item in taskList:
                newList.append(item.replace("PID", ret))
            return newList
        else:
            return taskList

    @staticmethod
    def selectTaskId():
        if not "TASKMON" in SysMgr.environList:
            return

        SysMgr.printInfo("start monitoring tasks to get target ID...\n")
        netObj = NetworkMgr("server", None, 0)

        # run monitor #
        cid = SysMgr.runTaskMonitor(
            [""],
            wait=False,
            addOpt=["-cGUIDER send PID -X%s:%s" % (netObj.ip, netObj.port)],
        )

        # wait for target ID #
        try:
            SysMgr.exitFlag = True

            ret = netObj.recv().decode()

            SysMgr.killChildren(children=[cid])

            # remove variable to prevent handling again by child processes #
            SysMgr.environList.pop("TASKMON", None)

            SysMgr.exitFlag = False
            return ret
        except SystemExit:
            os._exit(0)
        except:
            SysMgr.printErr("failed to get target ID to be traced", True)
            os._exit(-1)

    @staticmethod
    def getPrintFlag():
        return not "QUIET" in SysMgr.environList

    @staticmethod
    def loadLib(lib, path=False, inc=False):
        if not SysMgr.importPkgItems("ctypes", False):
            return

        # absolute path #
        if path:
            try:
                return CDLL(lib)
            except SystemExit:
                sys.exit(0)
            except:
                SysMgr.printErr("failed to load '%s'" % lib, True)
                return

        # search ld.so.cache #
        target = SysMgr.findLib(lib, inc)
        if not target:
            target = ["%s.so" % lib]
            ret = FileAnalyzer.getMapFilePath(SysMgr.pid, lib)
            if ret:
                target.append(ret)

        # load libraries #
        for item in target:
            try:
                res = cdll.LoadLibrary(item)
                if res:
                    return res
            except SystemExit:
                sys.exit(0)
            except:
                SysMgr.printWarn("failed to load '%s'" % item, reason=True)

    @staticmethod
    def importNative():
        try:
            # do not use native library to improve initialization time #
            raise Exception

            import guider

            guider.check()  # pylint: disable=no-member
            SysMgr.guiderObj = guider
        except:
            pass

    @staticmethod
    def applyKillVal(value):
        if not value:
            SysMgr.printErr("no value to send signal")
            sys.exit(-1)

        SysMgr.checkRootPerm()

        jobs = value.split(",")
        for job in jobs:
            try:
                value = job.split(":")

                if len(value) > 3:
                    raise Exception("wrong input")

                # set task #
                tid = value[0]

                # set signal #
                if len(value) == 1:
                    sig = signal.SIGKILL
                else:
                    sig = SysMgr.getSigNum(value[1])
                    if not sig:
                        raise Exception("wrong signal %s" % value[1])

                if len(value) > 2 and value[2].upper() == "CONT":
                    flag = "CONT"
                else:
                    flag = "ONCE"

                SysMgr.killFilter.append([tid, sig, flag])
            except SystemExit:
                sys.exit(0)
            except:
                SysMgr.printErr("failed to set signal", True)
                sys.exit(-1)

    @staticmethod
    def parseAffinityOption(jobs, launch=False):
        if not jobs:
            SysMgr.printErr("wrong option value")
            sys.exit(-1)

        doneList = set()
        for origVal in jobs:
            try:
                value = origVal.split(":")

                if not len(value) in (2, 3):
                    raise Exception("wrong input")

                # set task #
                tid = value[0]
                if tid == "":
                    tid = str(SysMgr.pid)

                # set mask #
                mask = value[1]
                if not mask:
                    raise Exception("wrong input")

                # convert mask values #
                if "+" in mask:
                    maskList = mask.split("+")
                    mask = 0
                    for val in maskList:
                        if not val:
                            continue
                        mask |= 1 << long(val)
                    mask = str(hex(mask))

                if launch:
                    sibling = SysMgr.groupProcEnable
                    targetList = SysMgr.getTids(tid, sibling=sibling)
                    targetList = list(map(long, targetList))
                    targetList = list(set(targetList) - doneList)
                    if targetList:
                        SysMgr.setAffinity(mask, targetList)
                        doneList = set(list(doneList) + targetList)
                    elif doneList:
                        pass
                    else:
                        SysMgr.printWarn("no thread related to '%s'" % tid)

                if len(value) == 3 and value[2].upper() == "CONT":
                    flag = "CONT"
                else:
                    flag = "ONCE"

                SysMgr.affinityFilter.append([tid, mask, flag])
            except SystemExit:
                sys.exit(0)
            except:
                SysMgr.printErr("wrong option value '%s'" % origVal, True)
                sys.exit(-1)

    @staticmethod
    def clearStats():
        if SysMgr.sysInstance:
            SysMgr.sysInstance.intervalData = {}
            SysMgr.sysInstance.intervalFileData = {}

        SysMgr.thrEvtCntList = {}

        SysMgr.maxBlkRd = 0
        SysMgr.maxBlkWr = 0
        SysMgr.maxCpuUsage = 0
        SysMgr.maxCpuSteal = 0
        SysMgr.maxGpuMemUsage = 0
        SysMgr.maxGpuUsage = 0
        SysMgr.maxInterval = 0
        SysMgr.maxLoad15m = 0
        SysMgr.maxLoad1m = 0
        SysMgr.maxLoad5m = 0
        SysMgr.maxNetIn = 0
        SysMgr.maxNetOut = 0
        SysMgr.maxNrContextSwitch = 0
        SysMgr.maxNrCore = 0
        SysMgr.maxNrFd = 0
        SysMgr.maxNrIrq = 0
        SysMgr.maxNrProcess = 0
        SysMgr.maxNrTaskAbnormal = 0
        SysMgr.maxNrTaskDie = 0
        SysMgr.maxNrTaskNew = 0
        SysMgr.maxNrThread = 0
        SysMgr.maxPsiCpu = 0
        SysMgr.maxPsiIo = 0
        SysMgr.maxPsiMem = 0
        SysMgr.maxSwapPer = 0
        SysMgr.minDiskFree = SysMgr.maxSize
        SysMgr.minMemAvail = SysMgr.maxSize
        SysMgr.nrOOMKill = 0
        SysMgr.nrTotalOOMKill = -1

    @staticmethod
    def getPeakList(incInter=True):
        peakList = {}
        if incInter and SysMgr.topInstance:
            for n, v in SysMgr.topInstance.intervalPeakData.items():
                if v:
                    peakList[n] = sum(v) / len(v)

        peakList.update(
            {
                "blkRdMB": SysMgr.maxBlkRd,
                "blkWrMB": SysMgr.maxBlkWr,
                "cpuPer": SysMgr.maxCpuUsage,
                "cpuStealPer": SysMgr.maxCpuSteal,
                "diskFreeMB": SysMgr.minDiskFree,
                "gpuPer": SysMgr.maxGpuUsage,
                "gpumemMB": SysMgr.maxGpuMemUsage,
                "intervalMS": SysMgr.maxInterval * 1000,
                "intervalOptSec": SysMgr.intervalEnable,
                "memAvailMB": SysMgr.minMemAvail,
                "netInMB": SysMgr.maxNetIn,
                "netOutMB": SysMgr.maxNetOut,
                "nrContextSwitch": SysMgr.maxNrContextSwitch,
                "nrCore": SysMgr.maxNrCore,
                "nrFd": SysMgr.maxNrFd,
                "nrIrq": SysMgr.maxNrIrq,
                "nrLoad1m": SysMgr.maxLoad1m,
                "nrLoad5m": SysMgr.maxLoad5m,
                "nrLoad15m": SysMgr.maxLoad15m,
                "nrOOMKill": SysMgr.nrTotalOOMKill,
                "nrProcess": SysMgr.maxNrProcess,
                "nrTaskAbnormal": SysMgr.maxNrTaskAbnormal,
                "nrTaskDie": SysMgr.maxNrTaskDie,
                "nrTaskNew": SysMgr.maxNrTaskNew,
                "nrThread": SysMgr.maxNrThread,
                "psiCpuPer": SysMgr.maxPsiCpu,
                "psiIoPer": SysMgr.maxPsiIo,
                "psiMemPer": SysMgr.maxPsiMem,
                "swapPer": SysMgr.maxSwapPer,
                "uptimeSec": SysMgr.uptime,
            }
        )

        # add custom data #
        if SysMgr.topInstance:
            peakList.update(SysMgr.topInstance.customPeakData)

        # remove maximum stats #
        for n in list(peakList):
            if peakList[n] == SysMgr.maxSize:
                peakList.pop(n, None)

        return peakList

    @staticmethod
    def splitFile(
        infile, size=None, outfile=None, count=0, which="START", verb=True
    ):
        pathList = []
        try:
            # check error condition #
            if not size and not count:
                SysMgr.printErr("no size and count info")
                return pathList

            # convert infile to realpath #
            infile = os.path.realpath(infile).rstrip("/")

            # set direction #
            if which == "END":
                direction = 2
                fromStart = False
            elif which == "START":
                direction = 0
                fromStart = True
            else:
                SysMgr.printErr(
                    "failed to split '%s' because wrong direction '%s'"
                    % (infile, which)
                )
                return pathList

            # split file #
            with open(infile, "rb") as fd:
                # get total size #
                fd.seek(0, 2)
                totalSize = fd.tell()

                # convert size #
                if size:
                    nsize = UtilMgr.convUnit2Size(size)
                    if not count:
                        count = long(totalSize / nsize)
                        if totalSize % nsize:
                            count += 1
                elif count:
                    nsize = long(totalSize / count)
                    if totalSize % count:
                        nsize += 1

                # set position #
                fd.seek(0, direction)

                # init file count #
                if fromStart:
                    nrCnt = 0
                else:
                    if count:
                        nrCnt = long(totalSize / nsize) - 1
                        if totalSize % nsize:
                            nrCnt += 1
                        count = nrCnt - count
                    else:
                        nrCnt = count
                        count = 0

                while 1:
                    # get current pos #
                    pos = fd.tell()

                    # get chunk size #
                    if fromStart:
                        chunk = (
                            nsize
                            if pos + nsize <= totalSize
                            else totalSize - pos
                        )
                    else:
                        chunk = nsize if pos - nsize >= 0 else pos
                        fd.seek(0 - chunk, 1)

                    # set pos from current #
                    buf = fd.read(chunk)

                    # rewind pos from end #
                    if not fromStart:
                        fd.seek(0 - chunk, 1)
                        pos = fd.tell()

                    # make a new outfile path #
                    if outfile:
                        if os.path.isdir(outfile):
                            filename = os.path.basename(infile)
                            outpath = os.path.join(outfile, filename)
                            outpath += "_{0:03}".format(nrCnt)
                        else:
                            outpath = outfile + "_{0:03}".format(nrCnt)
                    else:
                        outpath = infile + "_{0:03}".format(nrCnt)
                    outpath = os.path.realpath(outpath)

                    # backup file #
                    if os.path.exists(outpath):
                        SysMgr.backupFile(outpath)

                    # save to a new file #
                    with open(outpath, "wb") as wfd:
                        wfd.write(buf)

                    if verb:
                        SysMgr.printInfo(
                            "%s bytes from offset %s in '%s' are written to '%s'"
                            % (
                                UtilMgr.convNum(chunk),
                                UtilMgr.convNum(pos),
                                infile,
                                outpath,
                            ),
                            prefix=False,
                        )

                    pathList.append(outpath)

                    # get current pos #
                    pos = fd.tell()

                    # increase file count #
                    if fromStart:
                        nrCnt += 1
                    else:
                        nrCnt -= 1

                    # check exit condition #
                    if fromStart:
                        if pos == totalSize:
                            break
                        elif count and nrCnt >= count:
                            break
                    else:
                        if pos == 0:
                            break
                        elif nrCnt <= count:
                            break

            # sort list #
            pathList.sort()

            return pathList
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printErr(
                "failed to split '%s' by %s" % (infile, size), True
            )
            return pathList

    @staticmethod
    def doLess(inputArg=None):
        pipeInput = None
        stdin = sys.stdin
        wbuf = []
        cursor = 0
        lastWord = ""
        colorWord = ""

        # check input #
        if inputArg:
            if not isinstance(inputArg, list):
                inputArg = [inputArg]
        elif SysMgr.hasMainArg():
            inputArg = SysMgr.getMainArgs()
        elif SysMgr.inputParam:
            inputArg = str(SysMgr.inputParam).split(",")
            inputArg = UtilMgr.cleanItem(inputArg, True)
        elif SysMgr.isWindows:
            # no support pipe for windows #
            SysMgr.printErr("no input for PATH")
            return -1
        elif not os.isatty(0) or os.getenv("LESSPIPE"):
            # check stdin pipe #
            buf = sys.stdin.readline()
            if buf:
                wbuf.append(buf)

            # set file descriptors #
            pipeInput = sys.stdin
            freeFd = SysMgr.getFreeFd()
            os.dup2(1, freeFd)
            stdin = os.fdopen(freeFd, "r")
        else:
            SysMgr.printErr("no input for PATH")
            return -1

        def _printPage(wbuf, cursor):
            SysMgr.clearScreen()

            pbuf = wbuf[cursor : cursor + SysMgr.ttyRows - 2]

            # apply color #
            if colorWord:
                coloredWord = UtilMgr.convColor(colorWord, "REVERSE")
                for pidx in range(len(pbuf)):
                    pbuf[pidx] = UtilMgr.removeColor(pbuf[pidx]).replace(
                        colorWord, coloredWord
                    )

            SysMgr.printPipe(pbuf, pager=False)

        def _readPipe(wbuf, pipeInput):
            while 1:
                l = pipeInput.readline()
                if not l:
                    break
                wbuf.append(l)

        if pipeInput:
            # read data from stdin pipe #
            _readPipe(wbuf, pipeInput)

            # clear screen and print a page #
            _printPage(wbuf, cursor)
        else:
            # convert input files #
            inputArg = UtilMgr.getFileList(inputArg, sort=True, exceptDir=True)
            if not inputArg:
                SysMgr.printErr("no target file")
                return -1

            # print files #
            for path in inputArg:
                fsize = UtilMgr.getFileSizeStr(path)
                SysMgr.printStat(r"start reading '%s%s'..." % (path, fsize))

                try:
                    lines = SysMgr.readFile(path, strip=False).split("\n")
                    wbuf += [
                        twoLine,
                        "%s%s <%s lines>"
                        % (path, fsize, UtilMgr.convNum(len(lines))),
                        twoLine,
                    ]
                    wbuf += lines
                except SystemExit:
                    sys.exit(0)
                except:
                    SysMgr.printErr(
                        "failed to less '%s%s'" % (path, fsize), True
                    )
                    continue

                # clear screen and print a page #
                _printPage(wbuf, cursor)

        def _printStatus(wbuf, cursor):
            if SysMgr.ttyRows == 0:
                return

            totalPage = long(len(wbuf) / SysMgr.ttyRows)
            curPage = totalPage - long((len(wbuf) - cursor) / SysMgr.ttyRows)
            convNum = UtilMgr.convNum

            sys.stdout.write(
                "Line: %s-%s | Page: %s-%s | Size: %s > "
                % (
                    convNum(cursor),
                    convNum(len(wbuf)),
                    convNum(curPage),
                    convNum(totalPage),
                    convNum(sum(len(s) for s in wbuf)),
                )
            )
            sys.stdout.flush()

        def _handleInput(key, cursor, wbuf, lastWord, lastKey):
            # set cmd #
            if key:
                k = key[0]
            else:
                k = ""

            # use cached key #
            if k == "":
                if not lastKey:
                    lastKey = "k"
                k = lastKey

            # update last key #
            lastKey = k

            # exit #
            if k == "q":
                sys.exit(0)

            # start #
            elif k == "g":
                cursor = 0

            # end #
            elif k == "G":
                cursor = len(wbuf)

            # next page #
            elif k in ("\x1b[C", "f"):
                cursor += SysMgr.ttyRows

            # prev page #
            elif k in ("\x1b[D", "b"):
                cursor -= SysMgr.ttyRows

            # prev line #
            elif k in ("\x1b[A", "j"):
                cursor -= 1

            # next line #
            elif k in ("\x1b[B", "k"):
                cursor += 1

            # search #
            elif k == "/" or (lastWord and k in ("n", "N")):
                # update word #
                idx = -1
                if k == "/":
                    lastWord = key[1:]

                # set prev index #
                if k == "N":
                    src = cursor - 1
                    des = -1
                    inc = -1
                # set next index #
                else:
                    src = cursor + 1
                    des = len(wbuf)
                    inc = 1

                # find word #
                for i in xrange(src, des, inc):
                    try:
                        if lastWord in wbuf[i] or re.match(lastWord, wbuf[i]):
                            cursor = idx = i
                            break
                    except SystemExit:
                        sys.exit(0)
                    except:
                        pass

                # print error #
                if idx == -1:
                    sys.stdout.write("< not found: %s >" % lastWord)
                    sys.stdout.flush()
                    stdin.readline()

                return cursor, lastWord, lastKey

            # move to line #
            elif key.isdigit():
                cursor = long(key)

            # update cursor #
            if cursor < 0 or len(wbuf) < SysMgr.ttyRows:
                cursor = 0
            elif cursor > len(wbuf) - SysMgr.ttyRows + 1:
                cursor = max(len(wbuf) - SysMgr.ttyRows, 0) + 2

            return cursor, lastWord, lastKey

        # print status #
        _printStatus(wbuf, cursor)

        lastKey = None

        # handle input #
        while 1:
            try:
                # read input from stdin #
                k = stdin.readline()

                # read data from stdin pipe #
                if pipeInput:
                    _readPipe(wbuf, pipeInput)

                # handle key #
                cursor, lastWord, lastKey = _handleInput(
                    k.strip(), cursor, wbuf, lastWord, lastKey
                )
                if lastWord:
                    colorWord = lastWord

                # clear screen and print a page #
                _printPage(wbuf, cursor)

                # print status #
                _printStatus(wbuf, cursor)
            except SystemExit:
                sys.exit(0)
            except:
                SysMgr.printWarn("failed to move page", True, True)

    @staticmethod
    def doDump():
        # get argument #
        ret = SysMgr.selectTaskId()
        if ret:
            inputParam = [ret]
        elif SysMgr.hasMainArg():
            inputParam = SysMgr.getMainArgs()
        elif SysMgr.filterGroup:
            inputParam = SysMgr.filterGroup
        else:
            SysMgr.printErr("no input for COMM or PID")
            sys.exit(-1)

        # check condition #
        if not inputParam:
            SysMgr.printErr("no input for COMM or PID")
            sys.exit(-1)
        elif not SysMgr.outPath:
            SysMgr.printErr("no output path")
            sys.exit(-1)
        elif not SysMgr.inputParam:
            SysMgr.printErr("no input for memory area")
            sys.exit(-1)

        # convert comm to pid #
        targetList = []
        for item in inputParam:
            targetList += SysMgr.getTids(item, isThread=False)
        targetList = list(set(targetList))

        # check target #
        if not targetList:
            SysMgr.printErr("no target process")
            sys.exit(-1)
        elif len(targetList) > 1:
            SysMgr.printErr(
                "found multiple tasks [%s]" % SysMgr.getCommList(targetList)
            )
            sys.exit(-1)

        # set args #
        pid = targetList[0]
        meminfo = SysMgr.inputParam
        output = SysMgr.outPath
        if os.path.isdir(output):
            output = "%s/dump.out" % output

        # dump memory #
        Debugger.dumpTaskMemory(pid, meminfo, output)

    @staticmethod
    def readProcStat(fd, name, obj, attr, err=False, retry=True):
        try:
            buf = None
            while 1:
                fd.seek(0)
                buf = fd.readlines()
                # retry to read for shared descriptor #
                if buf or not retry:
                    break
        except SystemExit:
            sys.exit(0)
        except:
            try:
                bufPath = "%s/%s" % (SysMgr.procPath, name)
                fd = open(bufPath, "r")
                setattr(obj, attr, fd)
                buf = fd.readlines()
            except SystemExit:
                sys.exit(0)
            except:
                if err:
                    SysMgr.printOpenErr(bufPath)
                else:
                    SysMgr.printOpenWarn(bufPath)
        return buf

    @staticmethod
    def getBtStatsKernel(path, verb=False):
        return SysMgr.getBtStatsCrash(path, SysMgr.parseKernel, verb)

    @staticmethod
    def getBtStatsLinux1(path, verb=False):
        return SysMgr.getBtStatsCrash(path, SysMgr.parseCrash, verb)

    @staticmethod
    def getBtStatsAndroid(path, verb=False):
        return SysMgr.getBtStatsCrash(path, SysMgr.parseTombstone, verb)

    @staticmethod
    def getBtStatsCrash(path, parser, verb=False):
        if verb:
            # get output size #
            fsize = UtilMgr.getFileSizeStr(path)
            SysMgr.printStat(r"start loading '%s'%s..." % (path, fsize))

        # read data #
        buf = SysMgr.readFile(path, comp=True)
        if not buf:
            return [], []

        # decode data #
        try:
            buf = buf.decode()
        except SystemExit:
            sys.exit(0)
        except:
            pass

        # get crash info #
        sampleList = parser(buf)
        if isinstance(sampleList, dict):
            sampleList = [sampleList]

        resList = []
        taskList = []
        isValidStr = UtilMgr.isValidStr

        for sample in sampleList:
            # init lists #
            resList.append({})
            res = resList[-1]
            taskList.append({})
            taskInfo = taskList[-1]

            # get process name #
            for name in ("name", "Processname"):
                target = sample.get(name, "")
                if not target:
                    continue

                # android tombstone #
                if name == "name":
                    thread, process = target.split(">>>")
                    process = process.strip(" <")
                    taskInfo["process"] = process
                    thread = thread.strip()
                    taskInfo["thread"] = thread
                else:
                    taskInfo["process"] = target
                break

            # get thread name #
            for name in ("Threadname",):
                target = sample.get(name, "")
                if not target:
                    continue
                else:
                    taskInfo["thread"] = target
                    break

            # get ID #
            for n in ("pid", "tid"):
                idinfo = sample.get(n)
                taskInfo[n] = idinfo

            # get timestamp #
            for t in ("date", "Timestamp"):
                ts = sample.get(t)
                if ts:
                    try:
                        ts = str(
                            SysMgr.getPkg("datetime").datetime.strptime(
                                ts, "%a %b %d %H:%M:%S %Y"
                            )
                        )
                    except SystemExit:
                        sys.exit(0)
                    except:
                        pass
                    taskInfo["time"] = ts
                    break

            # check time filter #
            timeFilter = SysMgr.environList.get("TIMEFILTER")
            timeInfo = taskInfo.get("time")
            if timeFilter and timeInfo:
                if not isValidStr(timeInfo, timeFilter):
                    continue

            # check task filter #
            isValid = True
            for filterName in ("PROCFILTER", "THREADFILTER"):
                taskFilter = SysMgr.environList.get(filterName, [])
                taskExFilter = SysMgr.environList.get(
                    filterName.rstrip("FILTER") + "EXFILTER", []
                )
                if not taskFilter and not taskExFilter:
                    continue

                # get target #
                if filterName == "PROCFILTER":
                    target = taskInfo.get("process")
                else:
                    target = taskInfo.get("thread")
                if not target:
                    continue

                # check filter #
                if taskFilter and not isValidStr(target, taskFilter):
                    isValid = False
                    break
                elif taskExFilter and isValidStr(target, taskExFilter):
                    isValid = False
                    break
            if not isValid:
                continue

            # convert backtrace #
            backtrace = sample.get("backtrace", [])
            stack = ["%s [%s]" % (a[2], a[0].strip("[]")) for a in backtrace]
            if stack:
                stackName = " <- ".join(stack)
                res[stackName] = res.get(stackName, 0) + 1

        return resList, taskList

    @staticmethod
    def parseKernel(buf):
        # check backtrace #
        try:
            if not "Backtrace:" in buf and not "Call trace:":
                return {}
        except:
            return {}

        resList = []

        # get header #
        idx = -1
        lines = buf.split("\n")
        while 1:
            idx += 1
            if idx >= len(lines):
                break
            else:
                l = lines[idx]

            # get context #
            m = re.match(
                (
                    r"^\[\s*(?P<time>\S+)\] CPU:\s*(?P<cpu>[0-9]+)\s*"
                    r"PID:\s*(?P<pid>[0-9]+)\s*Comm:\s*(?P<comm>.*) (Not tainted|Tainted:)"
                ),
                l,
            )
            if not m:
                continue

            # get task info #
            d = m.groupdict()
            pid = d["pid"]
            comm = d["comm"]

            # find backtrace #
            while 1:
                idx += 1
                if idx >= len(lines):
                    break
                else:
                    l = lines[idx]

                m = re.match(
                    r"^\[\s*(?P<time>\S+)\] (Backtrace|Call trace):", l
                )
                if m:
                    break
                else:
                    continue

            if idx >= len(lines):
                break

            resList.append({"pid": pid, "Processname": comm})
            res = resList[-1]
            res["backtrace"] = []

            # parse backtrace #
            while 1:
                idx += 1
                if idx >= len(lines):
                    break
                else:
                    l = lines[idx]

                # check start #
                if re.match(
                    r"^\[\s*(?P<time>\S+)\] (Exception stack\(|---\[ end trace )",
                    l,
                ):
                    break

                # get function #
                m = None
                for s in (
                    r"^\[\s*(?P<time>\S+)\] \[\<(?P<addr>\S+)\>\] \((?P<func>\S+)\)",
                    r"^\[\s*(?P<time>\S+)\] \[\<(?P<addr>\S+)\>\] (?P<func>\S+)\+",
                    r"^\[\s*(?P<time>\S+)\]\s*(?P<func>\S+)\+",
                ):
                    m = re.match(s, l)
                    if m:
                        break
                if not m:
                    continue

                d = m.groupdict()
                res["backtrace"].append(
                    ["KERNEL", long(d.get("addr", "0"), 16), d["func"], 0x0]
                )

        return resList

    @staticmethod
    def parseCrash(buf):
        if buf.startswith("[task info]"):
            return SysMgr.parseCrash1(buf)
        if buf.startswith("ProblemType: Crash"):
            return SysMgr.parseCrash2(buf)
        else:
            return {}

    @staticmethod
    def parseCrash1(buf):
        res = {}
        kfilter = (
            "Processname:*",
            "Threadname:*",
            "signal:*",
        )

        # get header #
        idx = 1
        lines = buf.split("\n")[1:]
        for l in lines:
            if not l or not l[0]:
                idx += 1
                continue
            elif l.startswith("[backtrace stack]"):
                break
            elif UtilMgr.isValidStr(l, kfilter):
                d = l.split(":", 1)
                res[d[0].strip()] = d[1].strip(" '")
            elif l.startswith("pid:"):
                for d in l.split(","):
                    d = d.split(":", 1)
                    res[d[0].strip()] = d[1].strip()
            elif l.startswith("date ="):
                d = l.split(" ", 2)
                res[d[0].strip()] = d[2].strip()

            idx += 1

        # get backtrace #
        res["backtrace"] = []
        for l in lines[idx + 1 :]:
            if not l.lstrip():
                break

            try:
                # split items #
                items = l.strip().split(" ", 5)
                if items[1]:
                    start = 1
                else:
                    start = 2

                # get values #
                items = items[start:]
                if len(items) == 4:
                    pc, boffset, sp, func = items
                else:
                    pc, boffset, sp, func, etc = items
                    func += etc

                # remove useless data #
                path = func.rsplit(" ", 1)[1]
                boffset = long(boffset.split("=", 1)[1], 16)
                func, foffset = (
                    func.rsplit(")", 1)[0].lstrip("(").rsplit("+", 1)
                )
                foffset = long(foffset, 16)

                res["backtrace"].append([path, boffset, func, foffset])
            except SystemExit:
                sys.exit(0)
            except:
                SysMgr.printWarn(
                    "failed to parse '%s'" % l.strip(), True, True
                )

        return res

    @staticmethod
    def parseCrash2(buf):
        res = {}
        kfilter = (
            "Date:*",
            "ProcCmdline:*",
        )

        # get header #
        idx = 0
        lines = buf.split("\n")
        for l in lines:
            if not l or not l[0]:
                idx += 1
                continue
            elif l.startswith("Stacktrace:"):
                break
            elif UtilMgr.isValidStr(l, kfilter):
                d = l.split(":", 1)
                res[d[0].strip()] = d[1].strip(" '")

            idx += 1

        # get backtrace #
        res["backtrace"] = []
        for l in lines[idx + 1 :]:
            if not l or l[0] != " ":
                break
            elif not l.startswith(" #"):
                continue

            try:
                m = re.match(
                    r"^ #[0-9]+\s*(?P<addr>\S+) in (?P<func>.*) \(.*\)", l
                )
                if not m:
                    continue

                d = m.groupdict()
                res["backtrace"].append(
                    ["??", d.get("addr", "0"), d.get("func", "??"), 0x0]
                )
            except SystemExit:
                sys.exit(0)
            except:
                SysMgr.printWarn(
                    "failed to parse '%s'" % l.strip(), True, True
                )

        return res

    @staticmethod
    def parseTombstoneJava(lines, res, idx):
        res["backtrace"] = []
        for l in lines[idx + 1 :]:
            if not l.lstrip():
                break

            try:
                # split items #
                items = l.strip().split(" ", 1)[1]
                func, path = items.split("(", 1)
                path, line = path.split(":", 1)
                line = line.rstrip(")")

                res["backtrace"].append([path, line, func, "0x0"])
            except SystemExit:
                sys.exit(0)
            except:
                SysMgr.printWarn(
                    "failed to parse '%s'" % l.strip(), True, True
                )
        return res

    @staticmethod
    def parseTombstoneLines(lines, res, idx, cnt, start):
        res["backtrace"] = []
        for l in lines[idx + 1 :]:
            if not l.lstrip():
                break

            try:
                # split items #
                items = l.strip().split(" ", cnt)[start:]
                if len(items) == 4:
                    boffset, _, path, func = items
                else:
                    boffset, _, path = items
                    func = "??"

                # unknown #
                if path.startswith("<"):
                    func = path = "??"
                # remove offset info #
                elif func.startswith("(offset "):
                    func = func.split(")", 1)[1]
                    if not func:
                        func = "??"

                # remove BuildId and split function and offset #
                func = (
                    func.rsplit("(BuildId:", 1)[0].strip("( )").rsplit("+", 1)
                )
                if len(func) == 2:
                    func, foffset = func
                else:
                    func = func[0]
                    foffset = 0

                res["backtrace"].append(
                    [path, "0x" + boffset.lstrip("0"), func, foffset]
                )
            except SystemExit:
                sys.exit(0)
            except:
                SysMgr.printWarn(
                    "failed to parse '%s'" % l.strip(), True, True
                )
        return res

    @staticmethod
    def parseOldTombstone(buf):
        res = {}
        kfilter = (
            "Package:*",
            "Flags:*",
            "Build:*",
            "Revision:*",
        )

        # get header #
        idx = 0
        lines = buf.split("\n")
        isNative = True
        for l in lines:
            if not l or not l[0]:
                idx += 1
                continue
            elif l.startswith("backtrace:"):
                isNative = True
                break
            elif l.startswith("java.lang.NullPointerException"):
                isNative = False
                break
            elif UtilMgr.isValidStr(l, kfilter):
                d = l.split(":", 1)
                res[d[0].strip()] = d[1].strip(" '")
            elif l.startswith("pid:"):
                for d in l.split(","):
                    d = d.split(":", 1)
                    res[d[0].strip()] = d[1].strip()
            elif l.startswith("signal "):
                d = l.split(" ", 1)
                res[d[0].strip()] = d[1].strip()
            elif l.startswith("Process:"):
                res["Processname"] = l.split(": ", 1)[1]

            idx += 1

        # get backtrace #
        if isNative:
            res = SysMgr.parseTombstoneLines(lines, res, idx, 6, 3)
        else:
            res = SysMgr.parseTombstoneJava(lines, res, idx)

        return res

    @staticmethod
    def parseNewTombstone(buf):
        res = {}
        kfilter = (
            "ABI:*",
            "Timestamp:*",
            "Process uptime:*",
            "Cmdline:*",
            "uid:*",
        )

        # get header #
        idx = 1
        lines = buf.split("\n")[1:]
        for l in lines:
            if not l or not l[0]:
                idx += 1
                continue
            elif l.startswith("backtrace:"):
                break
            elif UtilMgr.isValidStr(l, kfilter):
                d = l.split(":", 1)
                res[d[0].strip()] = d[1].strip(" '")
            elif l.startswith("pid:"):
                for d in l.split(","):
                    d = d.split(":", 1)
                    res[d[0].strip()] = d[1].strip()
            elif l.startswith("signal "):
                d = l.split(" ", 1)
                res[d[0].strip()] = d[1].strip()

            idx += 1

        # get backtrace #
        res = SysMgr.parseTombstoneLines(lines, res, idx, 5, 2)

        return res

    @staticmethod
    def parseTombstone(buf):
        if buf.startswith("*** "):
            return SysMgr.parseNewTombstone(buf)
        elif buf.startswith("Process:"):
            return SysMgr.parseOldTombstone(buf)
        else:
            return {}

    @staticmethod
    def getFreeFd(pid="self"):
        try:
            return (
                long(
                    sorted(
                        os.listdir("%s/self/fd" % SysMgr.procPath),
                        key=lambda x: long(x),
                    )[-1]
                )
                + 1
            )
        except SystemExit:
            sys.exit(0)
        except:
            return -1

    @staticmethod
    def getNrFd(pid):
        # get the number of fds #
        try:
            path = "%s/%s/fd" % (SysMgr.procPath, pid)
            return len(os.listdir(path))
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printOpenWarn(path)
            return 0

    @staticmethod
    def getIdleMap():
        try:
            with open("/sys/kernel/mm/page_idle/bitmap", "rb") as fd:
                return fd.read(8888888)
        except SystemExit:
            sys.exit(0)
        except:
            return None

    @staticmethod
    def handleIdlePage(pfn, write=True, flush=True, bitmap=None):
        try:
            # check pfn 0 #
            if not pfn:
                return None

            # set permission #
            if write:
                perm = "wb"
            else:
                perm = "rb"

            # get fd #
            fd = SysMgr.getCachedFd("/sys/kernel/mm/page_idle/bitmap", perm)
            if not fd:
                sys.exit(-1)

            # define size #
            word = 8
            bit = word * 8

            # set pos #
            offset = long(pfn / bit) * word
            if not bitmap:
                fd.seek(offset, 0)

            # handle operation #
            if write:
                val = fd.write(struct.pack("Q", 2**bit - 1))
                if flush:
                    fd.flush()
                return val
            else:
                if bitmap:
                    val = bitmap[offset : offset + 8]
                else:
                    val = fd.read(word)

                val = struct.unpack("Q", val)[0]
                if val & 1 << (pfn % 64):
                    return True
                else:
                    return False
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printWarn(
                "failed to mark idle page (PFN: %s)" % hex(pfn), reason=True
            )
            return None

    @staticmethod
    def clearPageRefs(pid, val):
        """
        The /proc/PID/clear_refs is used to reset the PG_Referenced and ACCESSED/YOUNG
        bits on both physical and virtual pages associated with a process, and the
        soft-dirty bit on pte (see Documentation/admin-guide/mm/soft-dirty.rst
        for details).

        To clear the bits for all the pages associated with the process
            > echo 1 > /proc/PID/clear_refs

        To clear the bits for the anonymous pages associated with the process
            > echo 2 > /proc/PID/clear_refs

        To clear the bits for the file mapped pages associated with the process
            > echo 3 > /proc/PID/clear_refs

        To clear the soft-dirty bit
            > echo 4 > /proc/PID/clear_refs

        To reset the peak resident set size ("high water mark") to the process's
        current value:
            > echo 5 > /proc/PID/clear_refs

        Any other value written to /proc/PID/clear_refs will have no effect.
        """

        try:
            path = "%s/%s/clear_refs" % (SysMgr.procPath, pid)
            with open(path, "w") as fd:
                fd.write(val)
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printErr("failed to write '%s' to %s" % (val, path), True)

    @staticmethod
    def doMount(args=None):
        # get argument #
        if args:
            value = args
        elif SysMgr.hasMainArg():
            value = SysMgr.getMainArgs(False, ":")
        elif SysMgr.inputParam:
            value = SysMgr.inputParam
        else:
            SysMgr.printErr(
                (
                    "wrong value to mount filesystem, "
                    "input in the format DEV:DIR:FS:FLAGS:DATA"
                )
            )
            sys.exit(-1)

        # backup input value #
        origVal = deepcopy(value)

        # mount a filesystem #
        try:
            # convert flags #
            flags = 0
            if len(value) >= 4:
                for item in value[3].split(","):
                    try:
                        name = "MS_%s" % item.upper()
                        flags |= ConfigMgr.MOUNT_TYPE[name]
                    except SystemExit:
                        sys.exit(0)
                    except:
                        pass

                # update flags #
                value[3] = flags

            # get flags #
            if flags > 0:
                flagStr = "|".join(
                    UtilMgr.getFlagList(flags, ConfigMgr.MOUNT_TYPE)
                )
                flagStr = " with %s" % flagStr
            else:
                flagStr = ""

            # mount #
            ret = SysMgr.mount(*value)
            if ret == 0:
                SysMgr.printInfo(
                    "mounted %s on '%s' to '%s'%s successfully"
                    % (
                        value[2],
                        os.path.realpath(value[0]),
                        os.path.realpath(value[1]),
                        flagStr,
                    )
                )
            else:
                errReason = SysMgr.getErrReason()
                SysMgr.printErr(
                    "failed to mount '%s' because %s"
                    % (":".join(origVal), errReason)
                )
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printErr("failed to mount '%s'" % ":".join(origVal), True)

    @staticmethod
    def doUnmount(args=None):
        # get argument #
        if args:
            value = args
        elif SysMgr.hasMainArg():
            value = SysMgr.getMainArgs(False, ":")
        elif SysMgr.inputParam:
            value = SysMgr.inputParam
        else:
            SysMgr.printErr(
                (
                    "wrong value to unmount filesystem, "
                    "input in the format TARGET:FLAGS"
                )
            )
            sys.exit(-1)

        # backup input value #
        origVal = deepcopy(value)

        # mount a filesystem #
        try:
            # convert flags #
            flags = 0
            if len(value) >= 2:
                for item in value[1].split(","):
                    try:
                        name = "MNT_%s" % item.upper()
                        flags |= ConfigMgr.UMOUNT_TYPE[name]
                    except SystemExit:
                        sys.exit(0)
                    except:
                        pass

                # update flags #
                value[1] = flags

            # get flags #
            if flags > 0:
                flagStr = "|".join(
                    UtilMgr.getFlagList(flags, ConfigMgr.UMOUNT_TYPE)
                )
                flagStr = " with %s" % flagStr
            else:
                flagStr = ""

            # mount #
            ret = SysMgr.umount(*value)
            if ret == 0:
                SysMgr.printInfo(
                    "unmounted '%s'%s successfully"
                    % (os.path.realpath(value[0]), flagStr)
                )
            else:
                errReason = SysMgr.getErrReason()
                SysMgr.printErr(
                    "failed to unmount '%s' because %s"
                    % (":".join(origVal), errReason)
                )
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printErr("failed to unmount '%s'" % ":".join(origVal), True)

    @staticmethod
    def initLibbacktrace():
        if "NOLIBBT" in SysMgr.environList:
            SysMgr.libbacktraceObj = -1
        elif SysMgr.libbacktraceObj:
            return
        else:
            SysMgr.libbacktraceObj = SysMgr.loadLib(SysMgr.libbacktracePath)
            if SysMgr.libbacktraceObj:
                libcc = SysMgr.libbacktraceObj

                # define BacktraceMap* BacktraceMap::Create(pid_t pid, bool /*uncached*/) #
                libcc._ZN12BacktraceMap6CreateEib.argtypes = [c_int, c_bool]
                libcc._ZN12BacktraceMap6CreateEib.restype = c_void_p

                # define Backtrace* Backtrace::Create(pid_t pid, pid_t tid, BacktraceMap* map) #
                libcc._ZN9Backtrace6CreateEiiP12BacktraceMap.argtypes = [
                    c_int,
                    c_int,
                    c_void_p,
                ]
                libcc._ZN9Backtrace6CreateEiiP12BacktraceMap.restype = c_void_p

                # define bool UnwindStackPtrace::Unwind(size_t num_ignore_frames, void* context) #
                libcc._ZN17UnwindStackPtrace6UnwindEjPv.argtypes = [
                    c_void_p,
                    c_uint,
                    c_void_p,
                ]
                libcc._ZN17UnwindStackPtrace6UnwindEjPv.restype = c_bool

                # define std::string Backtrace::FormatFrameData(size_t frame_num) #
                libcc._ZN9Backtrace15FormatFrameDataEj.argtypes = [
                    c_void_p,
                    c_uint,
                ]
                libcc._ZN9Backtrace15FormatFrameDataEj.restype = c_void_p

                SysMgr.printInfo("enabled libbacktrace")

                # TODO: implement printing backtrace part #
                # assure ptrace_attached status #
                TID = long(SysMgr.filterGroup[0])
                PID = long(SysMgr.getTgid(TID))
                btmap = libcc._ZN12BacktraceMap6CreateEib(PID, 0)
                bt = libcc._ZN9Backtrace6CreateEiiP12BacktraceMap(
                    PID, TID, c_void_p(btmap)
                )
                while 1:
                    bt = libcc._ZN9Backtrace6CreateEiiP12BacktraceMap(
                        PID, TID, 0
                    )
                    print(libcc._ZN17UnwindStackPtrace6UnwindEjPv(bt, 0, 0))
                    print(
                        libcc._ZN9Backtrace15FormatFrameDataEj(bt, c_uint(0))
                    )
                    time.sleep(1)
            else:
                SysMgr.libbacktraceObj = -1

    @staticmethod
    def initLibcorkscrew():
        if "NOLIBCORK" in SysMgr.environList:
            SysMgr.libcorkscrewObj = -1
        elif SysMgr.libcorkscrewObj:
            return
        else:
            SysMgr.libcorkscrewObj = SysMgr.loadLib(SysMgr.libcorkscrewPath)
            if SysMgr.libcorkscrewObj:
                libcc = SysMgr.libcorkscrewObj

                # define load_ptrace_context #
                libcc.load_ptrace_context.argtypes = [c_int]
                libcc.load_ptrace_context.restype = c_void_p

                # define free_ptrace_context #
                libcc.free_ptrace_context.argtypes = [c_void_p]
                libcc.free_ptrace_context.restype = None

                # define unwind_backtrace_ptrace #
                libcc.unwind_backtrace_ptrace.argtypes = [
                    c_int,
                    c_void_p,
                    c_void_p,
                    c_size_t,
                    c_size_t,
                ]
                libcc.unwind_backtrace_ptrace.restype = c_size_t

                # define get_backtrace_symbols_ptrace #
                libcc.get_backtrace_symbols_ptrace.argtypes = [
                    c_void_p,
                    c_void_p,
                    c_size_t,
                    c_void_p,
                ]
                libcc.get_backtrace_symbols_ptrace.restype = None

                SysMgr.printInfo("enabled libcorkscrew")
            else:
                SysMgr.libcorkscrewObj = -1

    @staticmethod
    def doRlimit():
        # get argument #
        if SysMgr.hasMainArg():
            value = SysMgr.getMainArgs()
        elif SysMgr.filterGroup:
            value = SysMgr.filterGroup
        else:
            SysMgr.printErr(
                (
                    "wrong value to change resource limit, "
                    "input in the format TID|COMM:RTYPE:SLIM:HLIM"
                )
            )
            sys.exit(-1)

        for item in value:
            conf = item.split(":")
            if len(conf) != 1 and len(conf) != 4:
                SysMgr.printErr(
                    (
                        "wrong value to change resource limit, "
                        "input in the format TID|COMM:RTYPE:SLIM:HLIM"
                    )
                )
                sys.exit(-1)

            conf = UtilMgr.cleanItem(conf, False)

            # get tasks #
            target = SysMgr.getTids(conf[0])
            if not target:
                SysMgr.printErr("no task related to '%s'" % conf[0])
                sys.exit(-1)

            # get return type #
            if len(conf) == 4:
                rtype = conf[1]
                slim, hlim = list(map(long, conf[2:4]))

            # apply new limits #
            for tid in target:
                # get comm for the task #
                comm = SysMgr.getComm(tid, cache=True)

                if len(conf) == 1:
                    rlist = SysMgr.readProcData(tid, "limits")
                    if rlist:
                        SysMgr.printPipe(
                            "\n[Task Limit Info] [%s(%s)]\n%s\n%s%s"
                            % (comm, tid, twoLine, "".join(rlist), oneLine)
                        )
                    continue

                # convert resource type #
                rtype = rtype.upper()
                if not rtype in ConfigMgr.RLIMIT_TYPE:
                    candidate = "RLIMIT_" + rtype
                    if not candidate in ConfigMgr.RLIMIT_TYPE:
                        SysMgr.printErr(
                            "failed to change rlimit for %s because wrong resource"
                            % rtype
                        )
                        return False
                    rtype = candidate

                # change resource limit #
                ret = SysMgr.chRlimit(long(tid), rtype, slim, hlim)
                if ret:
                    SysMgr.printInfo(
                        "changed %s to %s/%s for %s(%s)"
                        % (
                            rtype,
                            UtilMgr.convNum(ret[0]),
                            UtilMgr.convNum(ret[1]),
                            comm,
                            tid,
                        )
                    )
                else:
                    SysMgr.printErr(
                        "failed to change %s to %s/%s for %s(%s)"
                        % (
                            rtype,
                            UtilMgr.convNum(slim),
                            UtilMgr.convNum(hlim),
                            comm,
                            tid,
                        )
                    )

    @staticmethod
    def getAvailMemInfo(retstr=True, unit=None):
        try:
            # read memory stats #
            memBuf = SysMgr.getMemInfo()

            # parse lines #
            memData = {}
            for line in memBuf:
                if line.startswith("Buffers"):
                    break

                memList = line.split()
                memData[memList[0][:-1]] = long(memList[1])

            # get available memory #
            if memData.get("MemAvailable"):
                sysMemStr = memData["MemAvailable"] << 10
            elif "MemFree" in memData:
                sysMemStr = memData["MemFree"] << 10
            else:
                sysMemStr = 0

            # calculate again #
            try:
                if sysMemStr != 0:
                    raise Exception()

                for line in memBuf:
                    memList = line.split()
                    if memList[0][:-1] in memData:
                        continue
                    memData[memList[0][:-1]] = long(memList[1])

                sysMemStr = (
                    memData["MemFree"]
                    + memData["Inactive(file)"]
                    + memData["SReclaimable"]
                    + memData["SwapCached"]
                )
                sysMemStr = sysMemStr << 10
            except SystemExit:
                sys.exit(0)
            except:
                pass

            if not retstr:
                return sysMemStr

            sysMemStr = UtilMgr.convSize2Unit(
                sysMemStr, isInt=False, unit=unit
            )
        except SystemExit:
            sys.exit(0)
        except:
            sysMemStr = 0
        finally:
            return sysMemStr

    @staticmethod
    def updateBatInfo():
        try:
            batPath = "/sys/class/power_supply"
            if not os.path.exists(batPath):
                return

            SysMgr.battery = ""

            stats = {
                "capacity": 0,
                "status": "",
                "energy_now": 0,
                "power_now": 0,
            }

            for d in os.listdir(batPath):
                subpath = os.path.join(batPath, d)
                for sub in os.listdir(subpath):
                    if not sub in stats:
                        continue

                    data = SysMgr.readFile(os.path.join(subpath, sub))
                    if sub in ("capacity", "energy_now", "power_now"):
                        stats[sub] += long(data)
                    elif sub == "status" and not data.upper().startswith(
                        "DIS"
                    ):
                        stats[sub] = "+"

            # check capacity #
            if stats["capacity"] == 0:
                return

            SysMgr.battery = (
                stats["capacity"],
                long(stats["energy_now"] / float(stats["power_now"]) * 3600)
                if stats["power_now"]
                else 0,
                stats["status"],
            )
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printWarn("failed to get battery info", reason=True)

    @staticmethod
    def updateNrRun(path):
        if not path or not os.path.isdir(path):
            SysMgr.printErr(
                (
                    "failed to update sequence number of execution "
                    "because wrong directory path '%s'"
                )
                % path
            )
            return

        # get the file list sorted by name in the output dir #
        flist = sorted(
            filter(
                lambda x: os.path.isfile(os.path.join(path, x)),
                os.listdir(path),
            ),
            reverse=True,
        )

        # get the number of last execution #
        for fname in flist:
            m = re.match(r"^guider_(?P<run>[0-9]+)_*", fname)
            if not m:
                continue

            # update the execution number #
            d = m.groupdict()
            SysMgr.nrRun = long(d["run"]) + 1
            return

    @staticmethod
    def getMemInfo():
        # return mem buf #
        return SysMgr.readProcStat(
            SysMgr.memFd, "meminfo", SysMgr, "memFd", True
        )

    @staticmethod
    def getVmFlags(flags=None, retList=False, retStr=False):
        vmList = {
            "READ": "rd",  # readable
            "WRITE": "wr",  # writable
            "EXE": "ex",  # executable
            "SHARED": "sh",  # shared
            "MAYRD": "mr",  # may read
            "MAYWR": "mw",  # may write
            "MAYEXE": "me",  # may execute
            "MAYSHARE": "ms",  # may share
            "STACKDOWN": "gd",  # stack segment grows down
            "PFNRANGE": "pf",  # pure PFN range
            "DISABLEWR": "dw",  # disabled write to the mapped file
            "LOCKED": "lo",  # pages are locked in memory
            "IO": "io",  # memory mapped I/O area
            "SEQRD": "sr",  # sequential read advise provided
            "RANDRD": "rr",  # random read advise provided
            "DONTCOPY": "dc",  # do not copy area on fork
            "DONTREMAP": "de",  # do not expand area on remapping
            "ACCOUNT": "ac",  # area is accountable
            "NOSWAP": "nr",  # swap space is not reserved for the area
            "HUGETLB": "ht",  # area uses huge tlb pages
            "SYNCFAULT": "sf",  # perform synchronous page faults (since Linux 4.15)
            "NOLINEAR": "nl",  # non-linear mapping (removed in Linux 4.0)
            "ARCH": "ar",  # architecture specific flag
            "WIPEONFORK": "wf",  # wipe on fork (since Linux 4.14)
            "DONTDUMP": "dd",  # do not include area into core dump
            "SOFTDIRTY": "sd",  # soft-dirty flag (since Linux 3.13)
            "MIXMAP": "mm",  # mixed map area
            "HUGEPAGE": "hg",  # huge page advise flag
            "NOHUGEPAGE": "nh",  # no-huge page advise flag
            "MERGE": "mg",  # mergeable advise flag
            "MISSING": "um",  # userfaultfd missing pages tracking (since Linux 4.3)
            "WPROTECT": "uw",  # userfaultfd wprotect pages tracking (since Linux 4.3)
        }

        # check vmflag list #
        if retList:
            return vmList
        elif retStr:
            return " ".join(vmList)

        # check type #
        if not isinstance(flags, list):
            SysMgr.printErr(
                "failed to convert '%s' to vmflags because it is not list"
                % flags
            )
            sys.exit(0)

        flagList = []
        for item in flags:
            try:
                flagList.append(vmList[item.upper()])
            except SystemExit:
                sys.exit(0)
            except:
                SysMgr.printErr("failed to convert '%s' to vmflag" % item)
                sys.exit(0)

        return flagList

    @staticmethod
    def madvise(addr, length, advise):
        try:
            # load libc #
            ret = SysMgr.loadLibcObj()
            if not ret:
                raise Exception("no library")
        except SystemExit:
            sys.exit(0)
        except:
            sys.exit(-1)

        # prlimit #
        if not hasattr(SysMgr.libcObj, "madvise"):
            SysMgr.printErr("no madvise in libc")
            sys.exit(-1)

        # define prototype #
        SysMgr.libcObj.madvise.argtypes = (POINTER(None), c_size_t, c_int)
        SysMgr.libcObj.madvise.restype = c_int

        # convert page-aligned size #
        start, end, addr, length = UtilMgr.getPageAddrs(addr, length)

        return SysMgr.libcObj.madvise(addr, length, advise)

    @staticmethod
    def chRlimit(pid, rtype, slim, hlim):
        rindex = ConfigMgr.RLIMIT_TYPE.index(rtype)

        # resource package #
        try:
            resource = SysMgr.getPkg("resource", False, True)
            if resource:
                resource.prlimit(pid, rindex, (slim, hlim))
                soft, hard = resource.prlimit(pid, rindex)
                return (soft, hard)
        except SystemExit:
            sys.exit(0)
        except:
            pass

        # get ctypes object #
        if not SysMgr.importPkgItems("ctypes", False):
            return

        class rlimit(Structure):
            _fields_ = (("rlim_cur", c_ulong), ("rlim_max", c_ulong))

        # try to get maxFd by standard library call #
        try:
            # load libc #
            ret = SysMgr.loadLibcObj()
            if not ret:
                raise Exception("no library")

            # create new object #
            rlim = rlimit()
            rlim.rlim_cur = c_ulong(slim)
            rlim.rlim_max = c_ulong(hlim)

            # prlimit #
            if hasattr(SysMgr.libcObj, "prlimit") or hasattr(
                SysMgr.libcObj, "prlimit64"
            ):
                # define real function #
                if hasattr(SysMgr.libcObj, "prlimit"):
                    func = SysMgr.libcObj.prlimit
                else:
                    func = SysMgr.libcObj.prlimit64

                func.argtypes = (
                    c_int,
                    c_int,
                    POINTER(rlimit),
                    POINTER(rlimit),
                )
                func.restype = c_int

                # create origin object #
                oldrlim = rlimit()

                ret = func(
                    c_int(os.getpid()), rindex, byref(rlim), byref(oldrlim)
                )
                if ret != 0:
                    raise Exception("error return %s" % ret)
            # getrlimit / setrlimit #
            elif hasattr(SysMgr.libcObj, "getrlimit"):
                SysMgr.libcObj.getrlimit.argtypes = (c_int, POINTER(rlimit))
                SysMgr.libcObj.getrlimit.restype = c_int

                # set resource value #
                ret = SysMgr.libcObj.setrlimit(rindex, byref(rlim))
                if ret != 0:
                    raise Exception("error return %s" % ret)

                # get new resource value #
                ret = SysMgr.libcObj.getrlimit(rindex, byref(rlim))
            else:
                raise Exception("no function")

            return (rlim.rlim_cur, rlim.rlim_max)
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printWarn(
                "failed to change %s to (%s,%s)" % (rtype, slim, hlim),
                reason=True,
            )

    @staticmethod
    def doStrings():
        # get argument #
        if SysMgr.hasMainArg():
            inputParam = SysMgr.getMainArg()
        elif SysMgr.inputParam:
            inputParam = SysMgr.inputParam
        else:
            SysMgr.printErr("no input for PATH")
            sys.exit(-1)

        # SysMgr.setStream(cut=False)

        # get file size #
        if os.path.exists(inputParam):
            # get output size #
            fsize = UtilMgr.getFileSizeStr(inputParam)
        else:
            SysMgr.printErr("no %s file" % inputParam)
            sys.exit(-1)

        SysMgr.printStat("start reading %s%s..." % (inputParam, fsize))

        # convert binary file to string #
        clist = UtilMgr.convBin2Str(inputParam, pos=SysMgr.showAll)
        if clist:
            SysMgr.printPipe()
        else:
            SysMgr.printErr("no available string")
            return

        convColor = UtilMgr.convColor

        # print position #
        if SysMgr.showAll:
            lastPos = sorted(clist.values())[-1]
            maxDigit = len(hex(lastPos))

            # print strings #
            for string, pos in sorted(clist.items(), key=lambda e: e[1]):
                if not UtilMgr.isValidStr(string, inc=True, ignCap=True):
                    continue
                SysMgr.printPipe(
                    "{0:>{digit}} {1}".format(
                        convColor(hex(pos), "BLUE"),
                        convColor(string, "RED"),
                        digit=maxDigit,
                    )
                )
        else:
            if SysMgr.filterGroup:
                for string in clist:
                    if not UtilMgr.isValidStr(string, inc=True, ignCap=True):
                        continue
                    SysMgr.printPipe(convColor(string, "RED"))
            else:
                SysMgr.printPipe("\n".join(clist))

    @staticmethod
    def restart(cmd=[], env=None):
        try:
            if not cmd:
                cmd = SysMgr.getCmdline(os.getpid()).split()

            if not env:
                env = deepcopy(os.environ)

            os.execvpe(cmd[0], cmd, env)
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printErr("failed to restart %s" % __module__, True)

    @staticmethod
    def removeCgroup(path):
        try:
            # init system context #
            if not SysMgr.sysInstance:
                SysMgr.initSystemContext()

            # get cgroup path #
            cgroupPath = SysMgr.sysInstance.getCgroupPath()
            if not cgroupPath:
                raise Exception("access to cgroup filesystem failed")

            # check and get root path #
            found = False
            for item in cgroupPath:
                if not path.startswith(item):
                    continue
                elif cgroupPath[item].startswith("cgroup2"):
                    continue

                found = True
                item = item[: item.rfind("/", 1)]
                root = UtilMgr.lstrip(path, item).split("/")
                root = UtilMgr.cleanItem(root)
                rootPath = os.path.join(item, root[0], "tasks")
                break

            if not found:
                raise Exception("wrong directory path")

            # find all tasks files #
            targetList = UtilMgr.getFiles(path, ["tasks"])

            # get all tids #
            while 1:
                tids = []
                for f in targetList:
                    try:
                        tids += SysMgr.readFile(f).split("\n")
                    except SystemExit:
                        sys.exit(0)
                    except:
                        pass

                # check items #
                if not UtilMgr.cleanItem(tids):
                    break

                # remove all tasks from sub-groups #
                SysMgr.writeFile(rootPath, tids)

                # wait for a moment #
                time.sleep(0.1)

            # remove target directory #
            for p in sorted(
                targetList, key=lambda x: x.count("/"), reverse=True
            ):
                try:
                    os.rmdir(UtilMgr.rstrip(p, "/tasks"))
                except SystemExit:
                    sys.exit(0)
                except:
                    SysMgr.printWarn("failed to remove '%s'" % p, True, True)

            return True
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printWarn("failed to remove '%s'" % path, True, True)
            return False

    @staticmethod
    def limitBlock(pids, attrs, devices=[]):
        if not pids:
            return

        # check root permission #
        SysMgr.checkRootPerm(msg="limit block usage using cgroup")

        # check subsystem #
        blkPath = SysMgr.getCgroupSubPath("blkio", False)
        if not blkPath:
            return

        # set device list for all #
        if not devices:
            blkPath = os.path.join(blkPath, "blkio.throttle.io_service_bytes")
            if not os.path.exists(blkPath):
                SysMgr.printErr("failed to find '%s'" % blkPath)
                sys.exit(-1)

            blkPathList = SysMgr.readFile(blkPath)
            if blkPathList:
                devices = {}
                for line in blkPathList.split("\n"):
                    if line.startswith("Total"):
                        continue
                    try:
                        devices[line.split(None, 1)[0]] = 0
                    except SystemExit:
                        sys.exit(0)
                    except:
                        pass

                devices = list(devices)
            else:
                # save diskstats #
                SysMgr.updateDiskStats()

                for line in SysMgr.diskStats:
                    try:
                        devices.append(":".join(line.split(None, 3)[:2]))
                    except SystemExit:
                        sys.exit(0)
                    except:
                        pass

        name = str(time.time()).replace(".", "")

        # set commands for block #
        cmds = ["CREATE:blkio:guider_{0:1}:".format(name)]

        # set commands #
        for item in attrs:
            for dev in devices:
                cmds.append(
                    (
                        "WRITE:blkio:guider_{0:1}:{3:1} {2:1}"
                        "@blkio.throttle.{1:1}"
                    ).format(
                        name,
                        item[0],
                        item[1],
                        dev,
                    ),
                )

        # set commands #
        for pid in pids:
            cmds.append("ADD:blkio:guider_{0:1}:{1:1}".format(name, pid))

        # execute commands to limit block usage #
        SysMgr.doCgroup(cmds, make=True, remove=True, verb=False)

    @staticmethod
    def limitPid(pids, attrs):
        if not pids:
            return

        # check root permission #
        SysMgr.checkRootPerm(msg="limit pids using cgroup")

        # check subsystem #
        if not SysMgr.getCgroupSubPath("pids", False):
            return

        name = str(time.time()).replace(".", "")

        # set commands for pids #
        cmds = ["CREATE:pids:guider_{0:1}:".format(name)]

        # set commands #
        for item in attrs:
            cmds.append(
                "WRITE:pids:guider_{0:1}:{2:1}@pids.{1:1}".format(
                    name, item[0], item[1]
                ),
            )

        # set commands #
        for pid in pids:
            cmds.append("ADD:pids:guider_{0:1}:{1:1}".format(name, pid))

        # execute commands to limit pids #
        SysMgr.doCgroup(cmds, make=True, remove=True, verb=False)

    @staticmethod
    def limitCpu(pids, attrs):
        if not pids:
            return

        # check root permission #
        SysMgr.checkRootPerm(msg="limit cpu usage using cgroup")

        # check subsystem #
        if not SysMgr.getCgroupSubPath("cpu", False):
            return

        name = str(time.time()).replace(".", "")

        # set commands for CPU usage #
        cmds = ["CREATE:cpu:guider_{0:1}:".format(name)]

        # get node list #
        nodes = [item[0] for item in attrs]

        # add default values #
        for node in ("cfs_period_us", "rt_period_us"):
            if node in nodes:
                continue

            cmds.append(
                "WRITE:cpu:guider_{0:1}:{2:1}@cpu.{1:1}".format(
                    name, node, 1000000
                )
            )

        # set commands #
        for item in attrs:
            cmds.append(
                "WRITE:cpu:guider_{0:1}:{2:1}@cpu.{1:1}".format(
                    name, item[0], item[1]
                ),
            )

        # set commands #
        for pid in pids:
            cmds.append("ADD:cpu:guider_{0:1}:{1:1}".format(name, pid))

        # execute commands to limit CPU usage #
        SysMgr.doCgroup(cmds, make=True, remove=True, verb=False)

    @staticmethod
    def limitCpuset(pids, attrs):
        if not pids:
            return

        # check root permission #
        SysMgr.checkRootPerm(msg="limit cpuset using cgroup")

        # check subsystem #
        cpusetdir = SysMgr.getCgroupSubPath("cpuset", False)
        if not cpusetdir:
            return

        # get prefix #
        if os.path.exists(os.path.join(cpusetdir, "cpuset.cpus")):
            prefix = "cpuset."
        else:
            prefix = ""

        name = str(time.time()).replace(".", "")

        # set commands for CPU set #
        cmds = ["CREATE:cpuset:guider_{0:1}:".format(name)]

        # get node list #
        nodes = [item[0] for item in attrs]

        # set commands #
        for item in attrs:
            cmds.append(
                "WRITE:cpuset:guider_{0:1}:{2:1}@{3:}{1:1}".format(
                    name, item[0], item[1], prefix
                ),
            )

        # init commands #
        addset = [["mems", 0]]
        if "EXCLUSIVE" in SysMgr.environList:
            addset += [["cpu_exclusive", 1]]
        for vals in addset:
            if vals[0] in nodes:
                continue

            cmds.append(
                "WRITE:cpuset:guider_{0:1}:{2:1}@{3:}{1:1}".format(
                    name, vals[0], vals[1], prefix
                )
            )

        # set commands #
        for pid in pids:
            cmds.append("ADD:cpuset:guider_{0:1}:{1:1}".format(name, pid))

        # execute commands to limit CPU set #
        SysMgr.doCgroup(cmds, make=True, remove=True, verb=False)

    @staticmethod
    def limitMemory(pids, attrs):
        if not pids:
            return

        # check root permission #
        SysMgr.checkRootPerm(msg="limit memory usage using cgroup")

        # check subsystem #
        if not SysMgr.getCgroupSubPath("memory", False):
            return

        name = str(time.time()).replace(".", "")

        # set commands for memory size #
        cmds = ["CREATE:memory:guider_{0:1}:".format(name)]

        # set commands #
        for pid in pids:
            cmds.append("ADD:memory:guider_{0:1}:{1:1}".format(name, pid))

        # set commands #
        for item in attrs:
            cmds.append(
                "WRITE:memory:guider_{0:1}:{2:1}@memory.{1:1}".format(
                    name, item[0], item[1]
                ),
            )

        # execute commands to limit memory #
        SysMgr.doCgroup(cmds, make=True, remove=True, verb=False)

    @staticmethod
    def doCgroup(cmds=[], make=False, remove=False, verb=True):
        def _moveTasks(srcFile, desFile, targetTasks):
            # check files #
            if not os.path.exists(srcFile) or not os.path.exists(desFile):
                return

            # read target tasks #
            tasks = SysMgr.readFile(srcFile)
            if tasks:
                tasks = tasks.split("\n")
            else:
                tasks = []

            # filter target tasks #
            tasks = list(set(tasks) & set(targetTasks))
            if not tasks:
                return []

            # move tasks #
            SysMgr.writeFile(desFile, tasks)

            # read moved tasks #
            movedTasks = SysMgr.readFile(desFile)
            if movedTasks:
                movedTasks = movedTasks.split("\n")
            else:
                movedTasks = []

            # get moved tasks #
            tasks = list(set(tasks) & set(movedTasks))
            if not tasks:
                return []

            return tasks

        def _checkFile(targetFile):
            if os.path.exists(targetFile):
                return True
            else:
                SysMgr.printWarn("failed to find '%s'" % targetFile, True)
                return False

        # get argument #
        if cmds:
            value = cmds
        elif SysMgr.hasMainArg():
            value = SysMgr.getMainArgs(False)
        elif SysMgr.filterGroup:
            value = SysMgr.filterGroup
        else:
            SysMgr.printErr("failed to get commands for cgroup")
            sys.exit(-1)

        # get cgroup list #
        cgroupList = SysMgr.getCgroupList()

        # verify and convert command strings #
        cmds = []
        for item in value:
            try:
                # CMD:SUB:NAME:TARGET #
                cmdset = item.split(":", 3)
                cmd, sub, name, target = cmdset
                if target == "PID":
                    target = str(SysMgr.pid)
                cmds.append([cmd.upper(), sub, name, target.strip()])
            except SystemExit:
                sys.exit(0)
            except:
                SysMgr.printErr(
                    (
                        "failed to convert cgroup command '%s' "
                        "in CMD:SUB:NAME:TARGET format"
                    )
                    % item,
                    True,
                )
                sys.exit(-1)

        # check task type #
        if SysMgr.processEnable:
            isThread = False
            taskType = "process"
            taskNode = "cgroup.procs"
        else:
            isThread = True
            taskType = "thread"
            taskNode = "tasks"

        # define shortcut variable #
        convNum = UtilMgr.convNum

        # handle commands #
        for clist in cmds:
            # get command info #
            cmd, sub, name, target = clist

            # check subsystem #
            rsub = sub.split("/", 1)[0]
            if not rsub in cgroupList:
                SysMgr.printWarn("no support '%s' subsystem" % rsub, True)
            elif cgroupList[rsub]["enable"] == "0":
                SysMgr.printWarn("'%s' subsystem is disabled" % rsub, True)

            # check command #
            if not cmd in (
                "CREATE",
                "ADD",
                "MOVE",
                "REMOVE",
                "DELETE",
                "READ",
                "WRITE",
                "LIST",
            ):
                SysMgr.printErr("no support '%s' command" % cmd)
                sys.exit(-1)

            # get target tasks #
            if cmd in ("CREATE", "ADD", "MOVE", "REMOVE", "LIST"):
                if cmd == "CREATE" and not target:
                    targetTasks = None
                else:
                    targetTasks = SysMgr.getTids(
                        target,
                        isThread=isThread,
                        sibling=SysMgr.groupProcEnable,
                    )
                    if not targetTasks:
                        SysMgr.printErr("no target %s" % taskType, True)
                        continue
            else:
                targetTasks = None

            # handle read / write command #
            if cmd in ("READ", "WRITE"):
                # get target info #
                targetDir = SysMgr.getCgroup(sub, name, make, remove)
                if not targetDir:
                    continue

                try:
                    if cmd == "READ":
                        name = target
                    else:
                        value, name = target.split("@")
                except SystemExit:
                    sys.exit(0)
                except:
                    SysMgr.printErr(
                        "wrong '%s' for %s in VALUE@FILE format"
                        % (target, cmd),
                        True,
                    )
                    continue

                # convert dir list #
                targetDir = (
                    UtilMgr.convPath(targetDir)
                    if "*" in targetDir
                    else [targetDir]
                )

                for item in targetDir:
                    # get target path #
                    targetFiles = os.path.join(item, name)
                    targetFiles = (
                        UtilMgr.convPath(targetFiles)
                        if "*" in targetFiles
                        else [targetFiles]
                    )

                    for targetFile in targetFiles:
                        if not _checkFile(targetFile):
                            continue

                        # read value #
                        if cmd == "READ":
                            ret = [SysMgr.readFile(targetFile)]

                            # print result #
                            SysMgr.printInfo(
                                "read '%s' from '%s'" % (ret, targetFile)
                            )
                        # write value #
                        else:
                            ret = SysMgr.writeFile(targetFile, value)
                            if ret:
                                # print result #
                                SysMgr.printInfo(
                                    "wrote '%s' to '%s'"
                                    % (
                                        convNum(str(value))
                                        if str(value).isdigit()
                                        else value,
                                        targetFile,
                                    )
                                )

                continue
            # handle list command #
            elif cmd == "LIST":
                # get target path #
                targetDir = SysMgr.getCgroup(sub, name, make, remove)
                if not targetDir:
                    continue

                # convert dir list #
                targetDir = (
                    UtilMgr.convPath(targetDir)
                    if "*" in targetDir
                    else [targetDir]
                )

                for item in targetDir:
                    targetFile = os.path.join(item, taskNode)
                    if not _checkFile(targetFile):
                        continue

                    # print target tasks #
                    tasks = SysMgr.readFile(targetFile)
                    if tasks:
                        tasks = tasks.split("\n")
                    else:
                        SysMgr.printInfo(
                            "no %s in '%s'" % (taskType, targetFile)
                        )
                        continue

                    # apply filter #
                    tasks = list(
                        map(long, list(set(tasks) & set(targetTasks)))
                    )

                    # print message #
                    SysMgr.printInfo(
                        "print %s %s from '%s'"
                        % (convNum(len(tasks)), taskType, targetFile)
                    )

                    # print tasks #
                    for idx, tid in enumerate(sorted(tasks)):
                        if not verb:
                            continue

                        SysMgr.printPipe(
                            "[%s] %s(%s)"
                            % (convNum(idx), SysMgr.getComm(tid), tid),
                            pager=False,
                        )

                continue
            # handle move command #
            elif cmd == "MOVE":

                def _getPathInfo(path, name, make):
                    try:
                        targetInfo = path.split("/", 1)
                        if len(targetInfo) == 1:
                            targetPath = targetInfo[0]
                            targetName = "/"
                        else:
                            targetPath, targetName = targetInfo
                            if not targetName:
                                targetName = "/"

                        targetDir = SysMgr.getCgroup(
                            targetPath, targetName, make, remove
                        )
                        if not targetDir:
                            raise Exception("no dir")

                        targetFile = os.path.join(targetDir, taskNode)
                        if not os.path.exists(targetFile):
                            raise Exception("no file")

                        return targetDir, targetFile
                    except SystemExit:
                        sys.exit(0)
                    except:
                        SysMgr.printErr(
                            "wrong %s of cgroup '%s' for move" % (name, path),
                            True,
                        )
                        return None, None

                # source #
                srcDir, srcFile = _getPathInfo(sub, "source", make)
                if not srcDir:
                    continue

                # destination #
                desDir, desFile = _getPathInfo(name, "destination", make)
                if not desDir:
                    continue

                # move tasks #
                tasks = _moveTasks(srcFile, desFile, targetTasks)
                if not tasks:
                    SysMgr.printErr(
                        "no %s to be moved from '%s' to '%s'"
                        % (taskType, srcFile, desFile)
                    )
                    continue

                # print message #
                SysMgr.printInfo(
                    "moved %s %s from '%s' to '%s'"
                    % (convNum(len(tasks)), taskType, srcDir, desDir)
                )

                # print tasks #
                for idx, tid in enumerate(tasks):
                    if not verb:
                        continue

                    SysMgr.printPipe(
                        "[%s] %s(%s)"
                        % (convNum(idx), SysMgr.getComm(tid), tid),
                        pager=False,
                    )

                continue

            # set make flag #
            if cmd == "CREATE":
                make = True
            else:
                make = False

            # get directory #
            targetDir = SysMgr.getCgroup(sub, name, make, remove)
            if not targetDir:
                cgroupPath = SysMgr.sysInstance.getCgroupPath()
                if not os.path.exists(os.path.join(cgroupPath, sub)):
                    SysMgr.printErr("no %s cgroup subsystem" % sub)
                else:
                    SysMgr.printErr(
                        "failed to get target cgroup '%s/%s'" % (sub, name)
                    )
                continue

            # convert dir list #
            targetDir = (
                UtilMgr.convPath(targetDir)
                if "*" in targetDir
                else [targetDir]
            )

            for item in targetDir:
                if cmd == "DELETE":
                    SysMgr.removeCgroup(item)
                    SysMgr.printInfo("deleted '%s' cgroup" % item)
                    continue
                elif cmd == "CREATE" and targetTasks is None:
                    SysMgr.printInfo("created '%s' cgroup" % item)
                    continue

                # register tasks to cgroup node #
                targetFile = os.path.join(item, taskNode)
                if not _checkFile(targetFile):
                    continue

                # ADD #
                if cmd in ("CREATE", "ADD"):
                    # move tasks #
                    SysMgr.writeFile(targetFile, targetTasks)

                    # print message #
                    SysMgr.printInfo(
                        "added %s %s in total to '%s' cgroup"
                        % (convNum(len(targetTasks)), taskType, item)
                    )

                    SysMgr.printWarn(SysMgr.getCommList(targetTasks))

                    tasks = targetTasks
                # REMOVE #
                else:
                    # get target file #
                    desFile = os.path.join(
                        os.path.dirname(targetFile), "..", taskNode
                    )
                    if not _checkFile(desFile):
                        continue

                    # move tasks #
                    tasks = _moveTasks(targetFile, desFile, targetTasks)
                    if not tasks:
                        SysMgr.printErr(
                            "no %s to be removed from '%s'" % (taskType, item)
                        )
                        continue

                    # print message #
                    SysMgr.printInfo(
                        "removed %s %s from '%s' cgroup"
                        % (convNum(len(tasks)), taskType, item)
                    )

                    SysMgr.printWarn(SysMgr.getCommList(tasks))

                # sort targets #
                tasks = sorted(list(map(long, tasks)))

                # print tasks #
                if verb:
                    for idx, tid in enumerate(tasks):
                        SysMgr.printPipe(
                            "[%s] %s(%s)"
                            % (convNum(idx), SysMgr.getComm(tid), tid),
                            pager=False,
                        )

    @staticmethod
    def getCgroupSubPath(sub, exception=True):
        cgroupInfo = SysMgr.printCgroupList(json=True, ret=True)

        if sub in cgroupInfo:
            return cgroupInfo[sub]["mount"].split("(", 1)[0].strip()
        elif exception:
            raise Exception("access to %s cgroup subsystem failed" % sub)
        else:
            SysMgr.printErr("failed to access to %s cgroup subsystem" % sub)
            return None

    @staticmethod
    def clearCgroupPath():
        SysMgr.cgroupPathList = {}

    @staticmethod
    def umountCgroups(subs=[]):
        umountList = {}

        cgroupInfo = SysMgr.printCgroupList(json=True, ret=True)
        for sub, values in cgroupInfo.items():
            if not values["mount"]:
                continue
            elif subs and not sub in subs:
                continue

            # get mount path #
            mountPath = values["mount"].split("(", 1)[0].strip()
            if mountPath in umountList:
                continue

            # umount a cgroup subsystem #
            SysMgr.umountCgroup(sub, mountPath)

            umountList[mountPath] = True

        SysMgr.clearCgroupPath()

    @staticmethod
    def mountCgroups(subs=[]):
        defPath = "/sys/fs/cgroup"
        if not os.path.exists(defPath):
            SysMgr.printErr("no default cgroup path '%s'" % defPath)
            sys.exit(-1)

        # get mount info #
        cgroupInfo = SysMgr.printCgroupList(json=True, ret=True)

        # merge same hierarch items #
        mergedInfo = {}
        for sub, values in cgroupInfo.items():
            if values["mount"]:
                continue
            elif subs and not sub in subs:
                continue

            # append to list #
            mid = "%s-%s" % (values["hierarchy"], values["num"])
            mergedInfo.setdefault(mid, [])
            mergedInfo[mid].append(sub)

        for _, sub in sorted(mergedInfo.items()):
            sub = ",".join(sub)

            # make a new directory #
            desDir = defPath + "/" + sub
            if not SysMgr.mkdirs(desDir):
                continue

            # mount a cgroup subsystem #
            SysMgr.mountCgroup(sub, desDir)

        SysMgr.clearCgroupPath()

    @staticmethod
    def printCgroupList(json=False, ret=False):
        cgroupList = SysMgr.getCgroupList()
        if not cgroupList:
            SysMgr.printErr("no cgroup mount point")
            return

        # check SysMgr instance #
        if not SysMgr.sysInstance:
            SysMgr(onlyInstance=True)

        # get cgroup2 path list #
        cgroupV2Path = SysMgr.sysInstance.getCgroupPath(
            "cgroup2", force=True, save=False
        )
        if cgroupV2Path:
            v2Info = "(v2: %s)" % list(cgroupV2Path.keys())[0]
        else:
            v2Info = ""

        if json:
            cgDict = {}
        else:
            SysMgr.printPipe(
                (
                    "\n[Cgroup Info] {6:1}\n{5:1}\n{0:>16} {1:>10} "
                    "{2:>8} {3:>8}  {4:1}\n{5:1}"
                ).format(
                    "subsystem",
                    "hierarchy",
                    "number",
                    "enabled",
                    "mount",
                    twoLine,
                    v2Info,
                )
            )

        # get cgroup path list #
        cgroupPath = SysMgr.sysInstance.getCgroupPath(force=True, save=True)

        for sub, values in sorted(
            cgroupList.items(), key=lambda x: long(x[1]["hierarchy"])
        ):
            try:
                # get mount info #
                mountinfo = ""
                for path, option in cgroupPath.items():
                    if sub in option.split()[2].split(
                        ","
                    ) or sub in path.rsplit("/", 1)[1].split(","):
                        mountinfo = "%s (%s)" % (path, cgroupPath[path])
                        break

                if json:
                    cgDict[sub] = values
                    cgDict[sub]["mount"] = mountinfo
                else:
                    SysMgr.printPipe(
                        "{0:>16} {1:>10} {2:>8} {3:>8}  {4:1}".format(
                            sub,
                            values["hierarchy"],
                            values["num"],
                            values["enable"],
                            mountinfo,
                        )
                    )
            except SystemExit:
                sys.exit(0)
            except:
                SysMgr.printWarn("failed to print cgroup info", reason=True)

        if json:
            if ret:
                return cgDict

            SysMgr.printPipe(
                UtilMgr.convDict2Str(cgDict, pretty=not SysMgr.streamEnable)
            )
        else:
            SysMgr.printPipe(oneLine)

    @staticmethod
    def getCgroupList():
        # read cgroup list #
        cgroups = SysMgr.readProcStat(
            SysMgr.cgroupFd, "cgroups", SysMgr, "cgroupFd"
        )
        if not cgroups:
            return {}

        cgroupList = {}
        for line in cgroups[1:]:
            try:
                sub, hierarchy, num, enable = line.strip().split("\t")
                cgroupList[sub] = {
                    "hierarchy": hierarchy,
                    "num": num,
                    "enable": enable,
                }
            except SystemExit:
                sys.exit(0)
            except:
                SysMgr.printWarn("failed to get cgroup items", reason=True)

        return cgroupList

    @staticmethod
    def getCgroup(sub, name=None, make=False, remove=False):
        try:
            # get subsystem path #
            cgroupPath = SysMgr.getCgroupSubPath(sub)
            if not cgroupPath:
                raise Exception("no subsystem '%s'" % sub)

            # set dir name #
            if not name:
                name = "guider_%s" % SysMgr.pid
            elif name == "/":
                name = ""

            # make target dir #
            targetDir = os.path.join(cgroupPath, name)
            if make:
                if not SysMgr.mkdirs(targetDir):
                    raise Exception("make fail for '%s'" % targetDir)

            # register handler to remove directory #
            if remove:
                SysMgr.addExitFunc(
                    SysMgr.removeCgroup,
                    [targetDir],
                    private=True,
                    redundant=False,
                )

            return targetDir
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printWarn("failed to get cgroup %s" % sub, True, True)
            return False

    @staticmethod
    def getCgroupFreezer():
        try:
            # check result #
            if SysMgr.freezerPath or SysMgr.freezerPath is False:
                return SysMgr.freezerPath

            # get cgroup #
            tempDir = SysMgr.getCgroup("freezer", make=True, remove=True)
            if not tempDir:
                return

            # freeze tasks #
            SysMgr.writeFile(os.path.join(tempDir, "freezer.state"), "FROZEN")

            # register tasks to cgroup node #
            SysMgr.freezerPath = os.path.join(tempDir, "tasks")
            return SysMgr.freezerPath
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printWarn("failed to create cgroup freezer", True, True)
            SysMgr.freezerPath = False
            return False

    @staticmethod
    def doFreeze(targetList=[]):
        # check root permissino #
        SysMgr.checkRootPerm()

        # get argument #
        ret = SysMgr.selectTaskId()
        if ret:
            value = [ret]
        elif SysMgr.hasMainArg():
            value = SysMgr.getMainArgs()
        elif SysMgr.filterGroup:
            value = SysMgr.filterGroup
        else:
            SysMgr.printErr(
                "failed to freeze tasks because there is no target"
            )
            sys.exit(-1)

        # check task type #
        if SysMgr.processEnable:
            isThread = False
            taskType = "process"
            taskNode = "cgroup.procs"
        else:
            isThread = True
            taskType = "thread"
            taskNode = "tasks"

        # get target tasks #
        targetTasks = []
        if targetList:
            targetTasks = list(map(str, targetList))
        else:
            for item in value:
                targetTasks += SysMgr.getTids(
                    item, isThread=isThread, sibling=SysMgr.groupProcEnable
                )
            if not targetTasks:
                SysMgr.printErr("no target %s" % taskType)
                sys.exit(-1)

        # get cgroup #
        targetDir = SysMgr.getCgroup("freezer", make=True, remove=True)
        if not targetDir:
            SysMgr.mountCgroups("freezer")
            # try again #
            targetDir = SysMgr.getCgroup("freezer", make=True, remove=True)
            if not targetDir:
                return

        # register tasks to cgroup node #
        taskPushFile = os.path.join(targetDir, taskNode)
        SysMgr.writeFile(taskPushFile, targetTasks)

        # freeze tasks #
        SysMgr.writeFile(os.path.join(targetDir, "freezer.state"), "FROZEN")

        if targetList:
            return

        # print target tasks #
        tasks = SysMgr.readFile(taskPushFile).split("\n")
        for tid in tasks:
            if not tid:
                continue
            comm = SysMgr.getComm(tid)
            SysMgr.printInfo("freezed %s(%s) %s" % (comm, tid, taskType))

        # set alarm #
        signal.signal(signal.SIGALRM, SysMgr.onAlarm)
        signal.alarm(SysMgr.intervalEnable)

        # wait for events #
        SysMgr.waitEvent()

    @staticmethod
    def doSetAffinity():
        SysMgr.warnEnable = True

        # get argument #
        if SysMgr.hasMainArg():
            value = SysMgr.getMainArgs()
        elif SysMgr.filterGroup:
            value = SysMgr.filterGroup
        else:
            SysMgr.printErr(
                "failed to set CPU affinity of task because there is no target"
            )
            sys.exit(-1)

        # convert task ID #
        ret = SysMgr.selectTaskId()
        if ret:
            newval = []
            for item in value:
                newval.append(item.replace("PID", ret))
            value = newval

        while 1:
            SysMgr.parseAffinityOption(value, launch=True)
            if SysMgr.intervalEnable:
                time.sleep(SysMgr.intervalEnable)
            else:
                break

        sys.exit(0)

    @staticmethod
    def importPkgItems(pkg, isExit=True):
        if pkg in SysMgr.impGlbPkg:
            return True

        module = SysMgr.getPkg(pkg, isExit)
        if not module:
            return False

        moduleDict = module.__dict__

        try:
            importList = module.__all__
        except SystemExit:
            sys.exit(0)
        except AttributeError:
            importList = [
                name for name in moduleDict if not name.startswith("_")
            ]

        newDict = {}
        for name in importList:
            newDict[name] = moduleDict[name]
        globals().update(newDict)

        SysMgr.impGlbPkg[pkg] = True

        return True

    @staticmethod
    def setBlock(fd, block=True):
        fcntl = SysMgr.getPkg("fcntl", False)

        try:
            if not fcntl:
                raise Exception("no fcntl")

            if not UtilMgr.isNumber(fd):
                fd = fd.fileno()

            flag = fcntl.fcntl(fd, fcntl.F_GETFL)

            if block:
                fcntl.fcntl(fd, fcntl.F_SETFL, flag & ~os.O_NONBLOCK)
            else:
                fcntl.fcntl(fd, fcntl.F_SETFL, flag | os.O_NONBLOCK)

            return True
        except SystemExit:
            sys.exit(0)
        except:
            SysMgr.printWarn(
                "failed to set block attribute to the file descriptor",
                reason=True,
            )
            return False

    @staticmethod
    def getHomePath():
        try:
            return os.environ["HOME"]
        except:
            return None

    @staticmethod
    def getLogEvents(tail=0, until=0, sort="seconds"):
        logEvents = []

        # get print flag #
        printFlag = "PRINTEVENT" in SysMgr.environList

        for logtype, logcmd in (
            ("DLTEVENT", "printdlt"),
            ("JOURNALEVENT", "printjrl"),
            ("KERNELEVENT", "printkmsg"),
            ("ANDROIDEVENT", "printand"),
            ("SYSLOGEVENT", "printsyslog"),
            ("TRACEEVENT", "printtrace"),
        ):
            if not logtype in SysMgr.environList:
                continue

            logname = UtilMgr.rstrip(logtype, "EVENT")

            SysMgr.printStat("start reading %s events..." % logname)

            for item in SysMgr.environList[logtype]:
                # get parameters #
                values = item.split("|", 2)
                if len(values) == 1:
                    path = None
                    name = values[0]
                    keyword = ""
                elif len(values) == 2:
                    path = None
                    name, keyword = values
                else:
                    path, name, keyword = values

                # get last file path #
                if logtype == "DLTEVENT":
                    if values[0] == "LASTFILE":
                        # convert path #
                        traceFileList = DltAnalyzer.getTraceFileList()
                        if traceFileList:
                            path = traceFileList[-1]
                        else:
                            path = None

                    # convert time #
                    if tail or until:
                        diff = SysMgr.getUptime() - UtilMgr.getClockTime(
                            dlt=True
                        )
                        if tail:
                            tail -= diff
                        if until:
                            until -= diff

                inputParam = [
                    logcmd,
                    "-g%s" % keyword,
                    "-J",
                    "-Q",
                    "-a",
                    "-dL",
                ]

                # set input path #
                inputParam.insert(1, "-I" + (path if path else ""))

                # set time condition #
                if tail or until:
                    inputParam.append(
                        (
                            "-q"
                            + ("TAIL:%s," % tail if tail else "")
                            + ("UNTIL:%s," % until if until else "")
                        ).rstrip(",")
                    )

                # execute filter process #
                """
                use initPkg flag to clear all global packages
                because cast functions are in both ctypes and pylab.
                """
                ret = SysMgr.launchGuider(
                    inputParam,
                    pipe=True,
                    stderr=True,
                    logo=False,
                    copyOpt=False,
                    initPkg=True,
                )

                # read logs from filter process #
                try:
                    pipe = ret[1]

                    while 1:
                        # read a line #
                        line = pipe.readline()
                        if not line:
                            if SysMgr.isAlive(ret[0]):
                                continue
                            else:
                                break

                        # remove useless characters #
                        line = line.strip()
                        if not line or not line.startswith("{"):
                            continue

                        # print progress #
                        UtilMgr.printProgress()

                        # convert string to dict #
                        obj = UtilMgr.convStr2Dict(line)
                        if obj:
                            obj["name"] = name
                            obj["type"] = logtype
                            logEvents.append(obj)

                        if printFlag:
                            SysMgr.printWarn(
                                "<%s> (%s) %s" % (logtype, name, line), True
                            )

                    # delete progress #
                    UtilMgr.deleteProgress()
                except SystemExit:
                    sys.exit(0)
                except:
                    SysMgr.printErr("stopped to read %s logs" % logname, True)

        # sort by seconds #
        try:
            logEvents.sort(key=lambda val: val[sort])
        except SystemExit:
            sys.exit(0)
        except:
            pass

        return logEvents

    @staticmethod
    def initLogWatcher(data):
        logList = {
            "DLT": "printdlt",
            "FILE": "watch",
            "JOURNAL": "printjrl",
            "KERNEL": "printkmsg",
            "ANDROID": "printand",
            "SYSLOG": "printsyslog",
            "TRACE": "printtrace",
        }

        for name, values in data.items():
            # check skip condition #
            if not name in logList:
                SysMgr.printErr("no support '%s' for log monitoring" % name)
                continue

            # check type #
            if isinstance(values, list):
                pass
            elif isinstance(values, dict):
                values = [values]
            else:
                continue

            for value in values:
                # check skip conditions #
                if value.get("apply") != "true":
                    continue
                elif not "filter" in value:
                    SysMgr.printErr("no filter for '%s' monitoring" % name)
                    continue

                # set port opt #
                if not SysMgr.localServObj:
                    SysMgr.printErr("no local network address info")
                    sys.exit(-1)

                portOpt = " -X%s" % SysMgr.localServObj.port

                # get filter #
                target = value["filter"]

                # set event name #
                event = "CMD_NOTIFY:log_%s_%s" % (name, target)

                # get condition for exit #
                if (
                    value.get("oneshot") == "true"
                    or value.get("goneshot") == "true"
                ):
                    exitCond = ",WATCHLOGEXIT"
                else:
                    exitCond = ""

                # set env variables #
                envOpt = (
                    "-qWATCHLOG:%s,TAIL,WATCHLOGCMD:GUIDER event %s%s%s"
                    % (target, event, portOpt, exitCond)
                )
                if name == "FILE":
                    if value.get("taskmon") == "true":
                        envOpt += ",PROCINFO"
                    elif value.get("inmount") == "true":
                        envOpt += ",INMOUNT"
                    elif value.get("allmount") == "true":
                        envOpt += ",ALLMOUNT"
                if value.get("re") == "true":
                    envOpt += ",REFILTER"

                # set default command #
                common = [envOpt, "-dp"]

                # add other options #
                if name == "FILE":
                    if "path" in value:
                        common.append("-g%s" % value["path"])

                # set command #
                cmd = [logList[name]] + common

                # set mute flag #
                mute = not "UNMUTE" in SysMgr.environList

                # execute watcher tasks #
                ret = SysMgr.launchGuider(
                    cmd, mute=mute, pipe=False, stderr=True
                )
                # register the command task #
                if ret > 0:
                    SysMgr.eventCommandList.setdefault(event, [])
                    SysMgr.eventCommandList[event].append(ret)

    @staticmethod
    def initFuncWatcher(data):
        funcList = {
            "NATIVE": "btrace",
            "SYSCALL": "strace",
            "PYTHON": "pytrace",
            "SIGNAL": "sigtrace",
            "DBUS": "printdbus",
        }

        for name, values in data.items():
            # check skip condition #
            if not name in funcList:
                SysMgr.printErr(
                    "no support '%s' for function monitoring" % name
                )
                continue

            # check type #
            if isinstance(values, list):
                pass
            elif isinstance(values, dict):
                values = [values]
            else:
                continue

            for value in values:
                # check skip conditions #
                if value.get("apply") != "true":
                    continue
                elif not "filter" in value:
                    SysMgr.printErr(
                        "no function filter for '%s' monitoring" % name
                    )
                    continue
                elif not "thread" in value:
                    SysMgr.printErr(
                        "no thread filter for '%s' monitoring" % name
                    )
                    continue

                # get function filter #
                target = value["filter"]

                # get task filter #
                task = value["thread"]

                # set event name #
                event = "CMD_NOTIFY:func_%s_%s" % (name, target)

                # get condition for exit #
                if (
                    value.get("oneshot") == "true"
                    or value.get("goneshot") == "true"
                ):
                    logExitCond = ",WATCHLOGEXIT"
                    funcExitCond = "|exit"
                else:
                    logExitCond = ""
                    funcExitCond = ""

                # set mute flag #
                mute = not "UNMUTE" in SysMgr.environList

                # set port opt #
                portOpt = " -X%s" % SysMgr.localServObj.port

                # set handle command #
                handleCmd = "GUIDER event %s%s" % (event, portOpt)

                # set default command #
                if name == "DBUS":
                    common = []
                else:
                    common = ["-g%s" % task]

                # append syscall option #
                if name == "SYSCALL":
                    common += ["-t%s" % target.replace("+", ",")]

                # set filter command #
                if name in ("SIGNAL", "DBUS"):
                    # define mandatory environme